﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGatewayRequest.h>
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/apigateway/model/PatchOperation.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace APIGateway {
namespace Model {

/**
 * <p>The PATCH request to grant a temporary extension to the remaining quota of a
 * usage plan associated with a specified API key.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/UpdateUsageRequest">AWS
 * API Reference</a></p>
 */
class UpdateUsageRequest : public APIGatewayRequest {
 public:
  AWS_APIGATEWAY_API UpdateUsageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateUsage"; }

  AWS_APIGATEWAY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Id of the usage plan associated with the usage data.</p>
   */
  inline const Aws::String& GetUsagePlanId() const { return m_usagePlanId; }
  inline bool UsagePlanIdHasBeenSet() const { return m_usagePlanIdHasBeenSet; }
  template <typename UsagePlanIdT = Aws::String>
  void SetUsagePlanId(UsagePlanIdT&& value) {
    m_usagePlanIdHasBeenSet = true;
    m_usagePlanId = std::forward<UsagePlanIdT>(value);
  }
  template <typename UsagePlanIdT = Aws::String>
  UpdateUsageRequest& WithUsagePlanId(UsagePlanIdT&& value) {
    SetUsagePlanId(std::forward<UsagePlanIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the API key associated with the usage plan in which a
   * temporary extension is granted to the remaining quota.</p>
   */
  inline const Aws::String& GetKeyId() const { return m_keyId; }
  inline bool KeyIdHasBeenSet() const { return m_keyIdHasBeenSet; }
  template <typename KeyIdT = Aws::String>
  void SetKeyId(KeyIdT&& value) {
    m_keyIdHasBeenSet = true;
    m_keyId = std::forward<KeyIdT>(value);
  }
  template <typename KeyIdT = Aws::String>
  UpdateUsageRequest& WithKeyId(KeyIdT&& value) {
    SetKeyId(std::forward<KeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For more information about supported patch operations, see <a
   * href="https://docs.aws.amazon.com/apigateway/latest/api/patch-operations.html">Patch
   * Operations</a>.</p>
   */
  inline const Aws::Vector<PatchOperation>& GetPatchOperations() const { return m_patchOperations; }
  inline bool PatchOperationsHasBeenSet() const { return m_patchOperationsHasBeenSet; }
  template <typename PatchOperationsT = Aws::Vector<PatchOperation>>
  void SetPatchOperations(PatchOperationsT&& value) {
    m_patchOperationsHasBeenSet = true;
    m_patchOperations = std::forward<PatchOperationsT>(value);
  }
  template <typename PatchOperationsT = Aws::Vector<PatchOperation>>
  UpdateUsageRequest& WithPatchOperations(PatchOperationsT&& value) {
    SetPatchOperations(std::forward<PatchOperationsT>(value));
    return *this;
  }
  template <typename PatchOperationsT = PatchOperation>
  UpdateUsageRequest& AddPatchOperations(PatchOperationsT&& value) {
    m_patchOperationsHasBeenSet = true;
    m_patchOperations.emplace_back(std::forward<PatchOperationsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_usagePlanId;

  Aws::String m_keyId;

  Aws::Vector<PatchOperation> m_patchOperations;
  bool m_usagePlanIdHasBeenSet = false;
  bool m_keyIdHasBeenSet = false;
  bool m_patchOperationsHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
