/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*-
 *
 * Copyright 2025 GNOME Foundation, Inc.
 *
 * SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Authors:
 *  - Philip Withnall <pwithnall@gnome.org>
 */

#pragma once

#include <gio/gio.h>
#include <glib.h>
#include <glib-object.h>
#include <libmalcontent/user.h>

G_BEGIN_DECLS

#define MCT_TYPE_USER_MANAGER mct_user_manager_get_type ()
G_DECLARE_FINAL_TYPE (MctUserManager, mct_user_manager, MCT, USER_MANAGER, GObject)

MctUserManager *mct_user_manager_new (GDBusConnection *connection);

void mct_user_manager_load_async (MctUserManager      *self,
                                  GCancellable        *cancellable,
                                  GAsyncReadyCallback  callback,
                                  void                *user_data);
gboolean mct_user_manager_load_finish (MctUserManager  *self,
                                       GAsyncResult    *result,
                                       GError         **error);

gboolean mct_user_manager_get_is_loaded (MctUserManager *self);

void mct_user_manager_get_user_by_uid_async (MctUserManager      *self,
                                             uid_t                uid,
                                             GCancellable        *cancellable,
                                             GAsyncReadyCallback  callback,
                                             void                *user_data);
MctUser *mct_user_manager_get_user_by_uid_finish (MctUserManager  *self,
                                                  GAsyncResult    *result,
                                                  GError         **error);
void mct_user_manager_get_user_by_username_async (MctUserManager      *self,
                                                  const char          *username,
                                                  GCancellable        *cancellable,
                                                  GAsyncReadyCallback  callback,
                                                  void                *user_data);
MctUser *mct_user_manager_get_user_by_username_finish (MctUserManager  *self,
                                                       GAsyncResult    *result,
                                                       GError         **error);

void mct_user_manager_get_family_members_for_user_async (MctUserManager      *self,
                                                         MctUser             *user,
                                                         GCancellable        *cancellable,
                                                         GAsyncReadyCallback  callback,
                                                         void                *user_data);
MctUser **mct_user_manager_get_family_members_for_user_finish (MctUserManager  *self,
                                                               GAsyncResult    *result,
                                                               size_t          *out_len,
                                                               GError         **error);

void mct_user_manager_get_all_users_async (MctUserManager      *self,
                                           GCancellable        *cancellable,
                                           GAsyncReadyCallback  callback,
                                           void                *user_data);
MctUser **mct_user_manager_get_all_users_finish (MctUserManager  *self,
                                                 GAsyncResult    *result,
                                                 size_t          *out_len,
                                                 GError         **error);

/**
 * MctUserArray:
 *
 * Convenience type to allow use of `g_autoptr()` with arrays of
 * [class@Malcontent.User].
 *
 * Use it as:
 * ```c
 * g_autoptr(MctUserArray) users = mct_user_manager_get_family_members_for_user (…);
 * ```
 *
 * Since: 0.14.0
 */
typedef MctUser* MctUserArray;

/**
 * mct_user_array_free:
 * @users: (transfer full) (array zero-terminated=1): array of users
 *
 * Convenience free function for arrays of [class@Malcontent.User].
 *
 * See [type@Malcontent.UserArray].
 *
 * Since: 0.14.0
 */
static inline void
mct_user_array_free (MctUser **users)
{
  size_t i;

  if (users == NULL)
    return;

  for (i = 0; users[i] != NULL; i++)
    g_object_unref (users[i]);
  g_free (users);
}

G_DEFINE_AUTOPTR_CLEANUP_FUNC (MctUserArray, mct_user_array_free)

G_END_DECLS
