#!/usr/bin/perl
#
# XFace.pm:
#
# Copyright (c) 2002 Chris Lightfoot. All rights reserved.
# Email: chris@ex-parrot.com; WWW: http://www.ex-parrot.com/~chris/
#
# $Id: XFace.pm,v 1.1.1.1 2002/02/17 23:09:57 chris Exp $
#

# This stuff was automatically generated by SWIG
package Image::XFace;
require Exporter;
require DynaLoader;

$VERSION = 0.10;

=head1 NAME

Image::XFace - encode and decode `X-Face' 48x48x1 face bitmaps

=head1 SYNOPSIS

    use Image::XFace;

    # obtain $xface data...
    @bits = Image::XFace::uncompface($xface);
    if (@bits != 48) {
        # data were invalid
    }

    # create xface data
    $xface = Image::XFace::compface(@bits);

    # get xface data as an image
    use GD;
    $img = Image::XFace::uncompface_gd($xface);
    # do with img as you will....

=head1 DESCRIPTION

Simple interface to compressing and uncompressing X-Face header data. For
instance, a small bitmap of me is represented by

 "kUA_=&I|(by86eXgYc|U}5`O%<xlo,~+JN9uk"Z`A.UCf2\1KKZ
 {FY-IIOqH/IS"=5=cb`U,mDyyf8a6BzVgYT~pRtqze]%s#\(J{/u
 m"(r,Ol^4J*Y%aWe-9`ZKGEYjG}d?#u2jzP,x37.%A~Qa;Yy6Fz`
 i/vu{}?y8%cI)RJpLnW=$yTs=TDM'MGjX`/LDw%p;EK;[ww;9_;U
 nRa+JZYO}[-j]O08X\Nm/K>M(P#,)y`g7N}Boz4b^JTFYHPz:s%i
 dl@t$\Vv$3OL6:>GEGwFHrV$/bfnL=6uO/ggqZfet:&D3Q=9c

=head1 PUBLIC INTERFACE

=over 4

=cut

@ISA = qw(Exporter DynaLoader);

bootstrap Image::XFace;
var_XFace_init();

@EXPORT = qw( );

use Carp;
use GD;

=item uncompface XFACE

Decode XFACE data. Returns in list context 48 bit-strings of image data as
generated by vec.

=cut
sub uncompface ($) {
    my $r = do_uncompface($_[0]);
    return ( ) unless ($r);
    my @ss = split("\n", $r);
    my @ret;
    my $i;
    for ($i = 0; $i < 48; ++$i) {
        my @hh = split(",", $ss[$i]);
        $ret[$i] = '';
        vec($ret[$i], 0, 16) = hex(substr($hh[0], 2));
        vec($ret[$i], 1, 16) = hex(substr($hh[1], 2));
        vec($ret[$i], 2, 16) = hex(substr($hh[2], 2));
    }
    return @ret;
}

=item compface IMAGE

Encodes binary image data encoded in bit-strings as XFACE data, returning it
in scalar context.

=cut
sub compface (@) {
    my $text = '';
    croak "Usage: compface(list of 48 bit-vectors)" unless (@_ == 48);
    foreach (@_) {
        $text .= sprintf("0x%04x,0x%04x,0x%04x,\n", vec($_, 0, 16),
                                                    vec($_, 1, 16),
                                                    vec($_, 2, 16));
    }
    return do_compface($text);
}

=item uncompface_gd XFACE

Decode XFACE data into a GD image object.

=cut
sub uncompface_gd ($) {
    my ($xface) = shift;
    my @res = uncompface($xface);
    return undef unless(@res == 48);
    my $gd;
    $gd = new GD::Image(48, 48);
    # This is pretty poor but it's what we've got....
    $gd->colorAllocate(0, 0, 0);            # 0
    $gd->colorAllocate(255, 255, 255);      # 1
    use integer;
    for (my $y = 0; $y < 48; ++$y) {
        for (my $x = 0; $x < 48; ++$x) {
            $gd->setPixel($x, $y, vec($res[$y], $x / 16, 16) & (32768 >> ($x & 15)) ? 0 : 1);
        }
    }

    return $gd;
}

1;

__END__

=head1 AUTHOR

Chris Lightfoot <chris@ex-parrot.com>, though all the real work is done by
the compface library of James Ashton.

=head1 COPYING

Copyright (c) 2002 Chris Lightfoot.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 VERSION

$Id: XFace.pm,v 1.1.1.1 2002/02/17 23:09:57 chris Exp $

=head1 SEE ALSO

L<compface(3)>

