#!/usr/bin/perl

use strict;
use warnings;

use English qw(-no_match_vars);
use Test::More;
use Parse::EDID;

my %tests = (
    'crt.13' => {
        manufacturer_name   => 'LTN',
        monitor_name        => 'A1554NEL',
        EISA_ID             => 'LTNa543',
        serial_number       => '16843009',
        serial_number2      => '926750447',
        VertRefresh         => '50-100',
        HorizSync           => '30-54',
        max_size_horizontal => 28,
        max_size_vertical   => 20,
        diagonal_size       => 13.5469689244766,
    },
    'crt.dell-d1626ht' => {
        manufacturer_name   => 'DEL',
        monitor_name        => 'DELL D1626HT',
        EISA_ID             => 'DEL515b',
        serial_number       => '808868404',
        serial_number2      => '55347B06Z418',
        VertRefresh         => '50-160',
        HorizSync           => '30-107',
        max_size_horizontal => 38,
        max_size_vertical   => 29,
        diagonal_size       => 18.819556539712,
    },
    'crt.dell-p1110' => {
        manufacturer_name   => 'DEL',
        monitor_name        => 'DELL P1110',
        EISA_ID             => 'DEL50ab',
        serial_number       => '810173271',
        serial_number2      => '9171RB0JCW89',
        VertRefresh         => '48-160',
        HorizSync           => '30-121',
        max_size_horizontal => 40,
        max_size_vertical   => 30,
        diagonal_size       => 19.6850393700787,
    },
    'crt.dell-p790' => {
        manufacturer_name   => 'DEL',
        monitor_name        => 'DELL P790',
        EISA_ID             => 'DEL62f5',
        serial_number       => '961631577',
        serial_number2      => '8757RH9QUY80',
        VertRefresh         => '50-150',
        HorizSync           => '31-92',
        max_size_horizontal => 32,
        max_size_vertical   => 24,
        diagonal_size       => 15.748031496063,
    },
    'crt.dell-p190s' => {
        manufacturer_name   => 'DEL',
        monitor_name        => 'DELL P190S',
        EISA_ID             => 'DEL405b',
        serial_number       => '1095193939',
        serial_number2      => 'CHRYK07UAGUS',
        VertRefresh         => '56-76',
        HorizSync           => '30-81',
        max_size_horizontal => 37.6,
        max_size_vertical   => 30.1,
        diagonal_size       => 18.9622010624
    },
    'crt.dell-e190s' => {
        manufacturer_name   => 'DEL',
        monitor_name        => 'DELL E190S',
        EISA_ID             => 'DELa04b',
        serial_number       => '810703187',
        serial_number2      => 'G448N08G0RYS',
        VertRefresh         => '56-76',
        HorizSync           => '30-81',
        max_size_horizontal => 37.6,
        max_size_vertical   => 30.1,
        diagonal_size       => 18.9622010624
    },
    'crt.E55' => {
        manufacturer_name   => 'MEI',
        monitor_name        => undef,
        EISA_ID             => 'MEI264a',
        serial_number       => '6310',
        serial_number2      => undef,
        VertRefresh         => '50-120',
        HorizSync           => '30-61',
        max_size_horizontal => 28,
        max_size_vertical   => 21,
        diagonal_size       => 13.7795275590551,
    },
    'crt.emc0313' => {
        manufacturer_name   => 'EMC',
        monitor_name        => undef,
        EISA_ID             => 'EMC0313',
        serial_number       => '6538',
        serial_number2      => undef,
        VertRefresh         => '50-160',
        HorizSync           => '30-86',
        max_size_horizontal => 32,
        max_size_vertical   => 24,
        diagonal_size       => 15.748031496063,
    },
    'crt.hyundai-ImageQuest-L70S+' => {
        manufacturer_name   => 'IQT',
        monitor_name        => 'L70S+',
        EISA_ID             => 'IQT0704',
        serial_number       => '57579',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 33,
        max_size_vertical   => 27,
        diagonal_size       => 16.7866185868723,
    },
    'crt.iiyama-1451' => {
        manufacturer_name   => 'IVM',
        monitor_name        => 'LS902U',
        EISA_ID             => 'IVM1938',
        serial_number       => '128958',
        serial_number2      => undef,
        VertRefresh         => '50-160',
        HorizSync           => '30-96',
        max_size_horizontal => 36,
        max_size_vertical   => 27,
        diagonal_size       => 17.7165354330709,
    },
    'crt.iiyama-404' => {
        manufacturer_name   => 'IVM',
        monitor_name        => undef,
        EISA_ID             => 'IVM1744',
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 32,
        max_size_vertical   => 24,
        diagonal_size       => 15.748031496063,
    },
    'crt.iiyama-410pro' => {
        manufacturer_name   => 'IVM',
        monitor_name        => undef,
        EISA_ID             => 'IVM1740',
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 32,
        max_size_vertical   => 24,
        diagonal_size       => 15.748031496063,
    },
    'crt.leia' => {
        manufacturer_name   => 'CPQ',
        monitor_name        => 'COMPAQ P710',
        EISA_ID             => 'CPQ1384',
        serial_number       => '1630548021',
        serial_number2      => '047ch67ha005',
        VertRefresh         => '50-140',
        HorizSync           => '30-96',
        max_size_horizontal => 31.2,
        max_size_vertical   => 23.4,
        diagonal_size       => 15.3543307086614,
    },
    'crt.LG-Studioworks-N2200P' => {
        manufacturer_name   => 'GSM',
        monitor_name        => 'Studioworks N 2200P',
        EISA_ID             => 'GSM55f3',
        serial_number       => '52846',
        serial_number2      => undef,
        VertRefresh         => '50-160',
        HorizSync           => '30-124',
        max_size_horizontal => 40,
        max_size_vertical   => 30,
        diagonal_size       => 19.6850393700787,
    },
    'crt.med2914' => {
        manufacturer_name   => 'MED',
        monitor_name        => undef,
        EISA_ID             => 'MED2914',
        serial_number       => '16843009',
        serial_number2      => '108371572',
        VertRefresh         => '50-120',
        HorizSync           => '30-98',
        max_size_horizontal => 36,
        max_size_vertical   => 27,
        diagonal_size       => 17.7165354330709,
    },
    'crt.nokia-valuegraph-447w' => {
        manufacturer_name   => 'NOK',
        monitor_name        => undef,
        EISA_ID             => 'NOK00b7',
        serial_number       => '3355',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 30,
        max_size_vertical   => 23,
        diagonal_size       => 14.8827229696345,
    },
    'crt.SM550S' => {
        manufacturer_name   => 'SAM',
        monitor_name        => undef,
        EISA_ID             => 'SAM1035',
        serial_number       => '1146106165',
        serial_number2      => 'HXAKB13419',
        VertRefresh         => '50-120',
        HorizSync           => '30-61',
        max_size_horizontal => 28,
        max_size_vertical   => 21,
        diagonal_size       => 13.7795275590551,
    },
    'crt.SM550V' => {
        manufacturer_name   => 'SAM',
        monitor_name        => 'S/M 550v',
        EISA_ID             => 'SAM12b6',
        serial_number       => '1146106165',
        serial_number2      => 'HXBN407938',
        VertRefresh         => '50-120',
        HorizSync           => '30-55',
        max_size_horizontal => 28,
        max_size_vertical   => 21,
        diagonal_size       => 13.7795275590551,
    },
    'crt.sony-gdm400ps' => {
        manufacturer_name   => 'SNY',
        monitor_name        => 'GDM-400PST9',
        EISA_ID             => 'SNY0090',
        serial_number       => '6005379',
        serial_number2      => '6005379',
        VertRefresh         => '48-160',
        HorizSync           => '30-95',
        max_size_horizontal => 37,
        max_size_vertical   => 27,
        diagonal_size       => 18.0330432922816,
    },
    'crt.sony-gdm420' => {
        manufacturer_name   => 'SNY',
        monitor_name        => 'CPD-G420',
        EISA_ID             => 'SNY0490',
        serial_number       => '16843009',
        serial_number2      => '6017706',
        VertRefresh         => '48-170',
        HorizSync           => '30-110',
        max_size_horizontal => 36,
        max_size_vertical   => 27,
        diagonal_size       => 17.7165354330709,
    },
    'crt.test_box_lmontel' => {
        manufacturer_name   => 'CPQ',
        monitor_name        => 'COMPAQ MV920',
        EISA_ID             => 'CPQ3027',
        serial_number       => '1094268470',
        serial_number2      => '008GA23MA966',
        VertRefresh         => '50-160',
        HorizSync           => '30-96',
        max_size_horizontal => 36,
        max_size_vertical   => 27,
        diagonal_size       => 17.7165354330709,
    },
    'lcd.20inches' => {
        manufacturer_name   => 'MAX',
        monitor_name        => 'B102005',
        EISA_ID             => 'MAX07d3',
        serial_number       => '831',
        serial_number2      => undef,
        VertRefresh         => '50-85',
        HorizSync           => '30-83',
        max_size_horizontal => 40.8,
        max_size_vertical   => 30.6,
        diagonal_size       => 20.0787401574803,
    },
    'lcd.acer-al1921' => {
        manufacturer_name   => 'ACR',
        monitor_name        => 'Acer AL1921',
        EISA_ID             => 'ACRad25',
        serial_number       => '1146094041',
        serial_number2      => 'ETL2508043',
        VertRefresh         => '55-75',
        HorizSync           => '30-83',
        max_size_horizontal => 37.6,
        max_size_vertical   => 30.1,
        diagonal_size       => 18.9622010623803,
    },
    'lcd.acer-al19161.1' => {
        manufacturer_name   => 'ACR',
        monitor_name        => 'Acer AL1916',
        EISA_ID             => 'ACRad49',
        serial_number       => '1905275078',
        serial_number2      => 'L49086694237',
        VertRefresh         => '55-75',
        HorizSync           => '30-83',
        max_size_horizontal => 37.6,
        max_size_vertical   => 30.1,
        diagonal_size       => 18.9622010624
    },
    'lcd.acer-al19161.2' => {
        manufacturer_name   => 'ACR',
        monitor_name        => 'Acer AL1916',
        EISA_ID             => 'ACRad49',
        serial_number       => '1905275535',
        serial_number2      => 'L49086694237',
        VertRefresh         => '55-75',
        HorizSync           => '30-83',
        max_size_horizontal => 37.6,
        max_size_vertical   => 30.1,
        diagonal_size       => 18.9622010624
    },
    'lcd.acer-al19161.3' => {
        manufacturer_name   => 'ACR',
        monitor_name        => 'Acer AL1916',
        EISA_ID             => 'ACRad49',
        serial_number       => '1905275537',
        serial_number2      => 'L49086694237',
        VertRefresh         => '55-75',
        HorizSync           => '30-83',
        max_size_horizontal => 37.6,
        max_size_vertical   => 30.1,
        diagonal_size       => 18.9622010624
    },
    'lcd.acer-al19161.4' => {
        manufacturer_name   => 'ACR',
        monitor_name        => 'Acer AL1916',
        EISA_ID             => 'ACRad49',
        serial_number       => '1905275536',
        serial_number2      => 'L49086694237',
        VertRefresh         => '55-75',
        HorizSync           => '30-83',
        max_size_horizontal => 37.6,
        max_size_vertical   => 30.1,
        diagonal_size       => 18.9622010624
    },
    'lcd.acer-asp1680' => {
        manufacturer_name   => 'QDS',
        monitor_name        => undef,
        EISA_ID             => 'QDS0014',
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 33.1,
        max_size_vertical   => 20.7,
        diagonal_size       => 15.3699698266439,
    },
    'lcd.acer-x193hq' => {
        manufacturer_name   => 'ACR',
        monitor_name        => 'X193HQ',
        EISA_ID             => 'ACR0069',
        serial_number       => '2483043340',
        serial_number2      => 'LEK0D0998545',
        VertRefresh         => '55-76',
        HorizSync           => '31-80',
        max_size_horizontal => 41,
        max_size_vertical   => 23,
        diagonal_size       => 18.5081248402
    },
    'lcd.b-101750' => {
        manufacturer_name   => 'MAX',
        monitor_name        => 'B_101750',
        EISA_ID             => 'MAX06ac',
        serial_number       => '537',
        serial_number2      => undef,
        VertRefresh         => '56-75',
        HorizSync           => '30-84',
        max_size_horizontal => 33.8,
        max_size_vertical   => 27,
        diagonal_size       => 17.0315524878844,
    },
    'lcd.benq-t904' => {
        manufacturer_name   => 'BNQ',
        monitor_name        => 'BenQ T904',
        EISA_ID             => 'BNQ7681',
        serial_number       => '6522',
        serial_number2      => undef,
        VertRefresh         => '56-76',
        HorizSync           => '31-83',
        max_size_horizontal => 37.6,
        max_size_vertical   => 30.1,
        diagonal_size       => 18.9622010623803,
    },
    'lcd.blino' => {
        manufacturer_name   => 'AUO',
        monitor_name        => undef,
        EISA_ID             => 'AUO0f03',
        serial_number       => '657',
        serial_number2      => undef,
        VertRefresh         => '40-69',
        HorizSync           => '63-73',
        max_size_horizontal => 30.5,
        max_size_vertical   => 22.8,
        diagonal_size       => 14.9921445940465,
    },
    'lcd.cmc-17-AD' => {
        manufacturer_name   => 'CMO',
        monitor_name        => 'CMC 17" AD',
        EISA_ID             => 'CMO7801',
        serial_number       => '0',
        serial_number2      => '0',
        VertRefresh         => '50-75',
        HorizSync           => '30-82',
        max_size_horizontal => 33.8,
        max_size_vertical   => 27,
        diagonal_size       => 17.0315524878844,
    },
    'lcd.compaq-evo-n1020v' => {
        manufacturer_name   => 'LGP',
        monitor_name        => undef,
        EISA_ID             => 'LGP6e54',
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 30,
        max_size_vertical   => 22,
        diagonal_size       => 14.646515935227,
    },
    'lcd.dell-2001fp' => {
        manufacturer_name   => 'DEL',
        monitor_name        => 'DELL 2001FP',
        EISA_ID             => 'DELa007',
        serial_number       => '860574796',
        serial_number2      => 'C064652L3KTL',
        VertRefresh         => '56-76',
        HorizSync           => '31-80',
        max_size_horizontal => 41,
        max_size_vertical   => 31,
        diagonal_size       => 20.236373662829,
    },
    'lcd.dell-inspiron-6400' => {
        manufacturer_name   => 'LPL',
        monitor_name        => undef,
        EISA_ID             => undef,
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 33.1,
        max_size_vertical   => 20.7,
        diagonal_size       => 15.3699698266439,
    },
    'lcd.eizo-l997' => {
        manufacturer_name   => 'ENC',
        monitor_name        => 'L997',
        EISA_ID             => 'ENC1749',
        serial_number       => '16843009',
        serial_number2      => '21211015',
        VertRefresh         => '59-61',
        HorizSync           => '31-100',
        max_size_horizontal => 44,
        max_size_vertical   => 33,
        diagonal_size       => 21.6535433071
    },
    'lcd.Elonex-PR600' => {
        manufacturer_name   => 'CMO',
        monitor_name        => undef,
        EISA_ID             => 'CMO1526',
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 33.1,
        max_size_vertical   => 20.7,
        diagonal_size       => 15.3699698266439,
    },
    'lcd.fujitsu-a171' => {
        manufacturer_name   => 'FUS',
        monitor_name        => 'A17-1',
        EISA_ID             => 'FUS0565',
        serial_number       => '525344',
        serial_number2      => 'YEEP525344',
        VertRefresh         => '56-76',
        HorizSync           => '30-83',
        max_size_horizontal => 34,
        max_size_vertical   => 27,
        diagonal_size       => 17.0931443670
    },
    'lcd.gericom-cy-96' => {
        manufacturer_name   => 'PTS',
        monitor_name        => 'CY965',
        EISA_ID             => 'PTS03c5',
        serial_number       => '19190',
        serial_number2      => 'F3AJ3A0019190',
        VertRefresh         => '60-75',
        HorizSync           => '30-80',
        max_size_horizontal => 37.6,
        max_size_vertical   => 30.1,
        diagonal_size       => 18.9622010623803,
    },
    'lcd.hp-nx-7000' => {
        manufacturer_name   => 'LGP',
        monitor_name        => undef,
        EISA_ID             => undef,
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 33.1,
        max_size_vertical   => 20.7,
        diagonal_size       => 15.3699698266439,
    },
    'lcd.hp-nx-7010' => {
        manufacturer_name   => 'LGP',
        monitor_name        => undef,
        EISA_ID             => undef,
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 33.1,
        max_size_vertical   => 20.7,
        diagonal_size       => 15.3699698266439,
    },
    'lcd.hp-l1950' => {
        manufacturer_name   => 'HWP',
        monitor_name        => 'HP L1950',
        EISA_ID             => 'HWP26e7',
        serial_number       => '16843009',
        serial_number2      => 'CNK7420237',
        VertRefresh         => '50-77',
        HorizSync           => '24-83',
        max_size_horizontal => 38,
        max_size_vertical   => 30,
        diagonal_size       => 19.0609739164
    },
    'lcd.HP-Pavilion-ZV6000' => {
        manufacturer_name   => 'QDS',
        monitor_name        => undef,
        EISA_ID             => 'QDS0014',
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 33.1,
        max_size_vertical   => 20.7,
        diagonal_size       => 15.3699698266439,
    },
    'lcd.iiyama-pl2409hd' => {
        manufacturer_name   => 'IVM',
        monitor_name        => 'PL2409HD',
        EISA_ID             => 'IVM560c',
        serial_number       => '313',
        serial_number2      => '11004M0C00313',
        VertRefresh         => '56-76',
        HorizSync           => '30-83',
        max_size_horizontal => 52.1,
        max_size_vertical   => 29.3,
        diagonal_size       => 23.5329685251
    },
    'lcd.lenovo-3000-v100' => {
        manufacturer_name   => 'AUO',
        monitor_name        => undef,
        EISA_ID             => 'AUO3214',
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 26.1,
        max_size_vertical   => 16.3,
        diagonal_size       => 12.1148583788498,
    },
    'lcd.lenovo-w500' => {
        manufacturer_name   => 'LEN',
        monitor_name        => undef,
        EISA_ID             => 'LEN4055',
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 33.1,
        max_size_vertical   => 20.7,
        diagonal_size       => 15.3699698266439,
    },
    'lcd.philips-150s' => {
        manufacturer_name   => 'PHL',
        monitor_name        => 'PHILIPS  150S',
        EISA_ID             => 'PHL0805',
        serial_number       => '237',
        serial_number2      => ' HD  000237',
        VertRefresh         => '56-75',
        HorizSync           => '30-61',
        max_size_horizontal => 30,
        max_size_vertical   => 22,
        diagonal_size       => 14.646515935227,
    },
    'lcd.philips-180b2' => {
        manufacturer_name   => 'PHL',
        monitor_name        => 'Philips 180B2',
        EISA_ID             => 'PHL0810',
        serial_number       => '21838',
        serial_number2      => ' HD  021838',
        VertRefresh         => '56-76',
        HorizSync           => '30-82',
        max_size_horizontal => 35.9,
        max_size_vertical   => 28.7,
        diagonal_size       => 18.0952522744589,
    },
    'lcd.presario-R4000' => {
        manufacturer_name   => 'LPL',
        monitor_name        => undef,
        EISA_ID             => undef,
        serial_number       => '0',
        serial_number2      => undef,
        VertRefresh         => undef,
        HorizSync           => undef,
        max_size_horizontal => 33,
        max_size_vertical   => 21,
        diagonal_size       => 15.3996907209511,
    },
    'lcd.rafael' => {
        manufacturer_name   => 'MAX',
        monitor_name        => 'B101715',
        EISA_ID             => 'MAX06b5',
        serial_number       => '1509',
        serial_number2      => undef,
        VertRefresh         => '56-75',
        HorizSync           => '31-83',
        max_size_horizontal => 34,
        max_size_vertical   => 28,
        diagonal_size       => 17.3407209020041,
    },
    'lcd.regis' => {
        manufacturer_name   => 'ENC',
        monitor_name        => 'L557',
        EISA_ID             => 'ENC1689',
        serial_number       => '16843009',
        serial_number2      => '82522083',
        VertRefresh         => '59-61',
        HorizSync           => '31-64',
        max_size_horizontal => 33.7,
        max_size_vertical   => 27,
        diagonal_size       => 17.0008096293987,
    },
    'lcd.samsung-191n' => {
        manufacturer_name   => 'SAM',
        monitor_name        => 'SyncMaster',
        EISA_ID             => 'SAM0080',
        serial_number       => '1195913529',
        serial_number2      => 'HCHW600639',
        VertRefresh         => '56-85',
        HorizSync           => '30-81',
        max_size_horizontal => 38,
        max_size_vertical   => 31,
        diagonal_size       => 19.3074013153374,
    },
    'lcd.samsung-2494hm' => {
        manufacturer_name   => 'SAM',
        monitor_name        => 'SyncMaster',
        EISA_ID             => 'SAM04d3',
        serial_number       => '1263088180',
        serial_number2      => 'H9XS933672',
        VertRefresh         => '56-60',
        HorizSync           => '30-81',
        max_size_horizontal => 53.1,
        max_size_vertical   => 29.8,
        diagonal_size       => 23.9726281282
    },
    'lcd.tv.VQ32-1T' => {
        manufacturer_name   => 'FUS',
        monitor_name        => 'VQ32-1T',
        EISA_ID             => 'FUS065b',
        serial_number       => '1',
        serial_number2      => undef,
        VertRefresh         => '49-76',
        HorizSync           => '28-65',
        max_size_horizontal => 69.8,
        max_size_vertical   => 39.2,
        diagonal_size       => 31.5174140229615,
    },
    'lcd.viewsonic-vx715' => {
        manufacturer_name   => 'VSC',
        monitor_name        => 'VX715',
        EISA_ID             => 'VSC4319',
        serial_number       => '16843009',
        serial_number2      => 'P21044404507',
        VertRefresh         => '50-85',
        HorizSync           => '30-82',
        max_size_horizontal => 33.8,
        max_size_vertical   => 27,
        diagonal_size       => 17.0315524879
    },
    'IMP2262' => {
        manufacturer_name   => 'IMP',
        monitor_name        => '*22W1*',
        EISA_ID             => 'IMP2262',
        serial_number       => '1944',
        serial_number2      => '74701944',
        VertRefresh         => '56-76',
        HorizSync           => '31-83',
        max_size_horizontal => 47,
        max_size_vertical   => 29,
        diagonal_size       => 21.7428366480
    },
);

plan tests => 2 * scalar keys %tests;

foreach my $test (sort keys %tests) {
    my $edid = read_file("t/edid/$test");
    my $parsed_edid = parse_edid($edid);
    my $message = check_parsed_edid($parsed_edid);
    is($message, "", "$test: parsing succesful");

    subtest "$test: expected results" => sub {
        check_result($parsed_edid, $tests{$test});
    };

}


sub read_file {
    my ($file) = @_;
    local $RS;
    open (my $handle, '<', $file) or die "Can't open $file: $ERRNO";
    my $content = <$handle>;
    close $handle;
    return $content;
}

sub check_result {
    my ($got, $expected) = @_;

    foreach my $field (qw/manufacturer_name monitor_name EISA_ID serial_number VertRefresh HorizSync max_size_horizontal max_size_vertical/) {
        is($got->{$field}, $expected->{$field}, $field);
    }

    # truncate to 10 decimals, to avoid failures due to precision differences
    # between some platforms
    foreach my $field (qw/diagonal_size/) {
        is(
            sprintf("%.10f", $got->{$field}),
            sprintf("%.10f", $expected->{$field}),
            $field
        );
    }

    is(
        $got->{serial_number2}->[0],
        $expected->{serial_number2},
        'serial_number2'
    );
}
