#!/usr/bin/env perl

use strict;
use warnings;
use feature qw/ say /;
use lib 'lib';
use YAGL;
use Cwd;
use Test::More tests => 3;

my $cwd = getcwd;

=head2 Test 1. All paths from the Sedgewick example graph

=cut

my $g = YAGL->new;
$g->read_csv("$cwd/t/24-ham-02.csv");
my @got = $g->paths_between('a', 'm');

my $expected = [
    ['a', 'f', 'e', 'l', 'm'],
    ['a', 'g', 'e', 'l', 'm'],
    ['a', 'b', 'c', 'e', 'l', 'm'],
    ['a', 'f', 'e', 'l', 'j', 'm'],
    ['a', 'g', 'e', 'l', 'j', 'm'],
    ['a', 'b', 'c', 'e', 'l', 'j', 'm'],
    ['a', 'b', 'd', 'f', 'e', 'l', 'm'],
    ['a', 'g', 'h', 'i', 'k', 'j', 'm'],
    ['a', 'b', 'd', 'f', 'e', 'l', 'j', 'm'],
    ['a', 'f', 'd', 'b', 'c', 'e', 'l', 'm'],
    ['a', 'g', 'h', 'i', 'k', 'j', 'l', 'm'],
    ['a', 'f', 'd', 'b', 'c', 'e', 'l', 'j', 'm'],
    ['a', 'f', 'e', 'g', 'h', 'i', 'k', 'j', 'm'],
    ['a', 'b', 'c', 'e', 'g', 'h', 'i', 'k', 'j', 'm'],
    ['a', 'f', 'e', 'g', 'h', 'i', 'k', 'j', 'l', 'm'],
    ['a', 'b', 'c', 'e', 'g', 'h', 'i', 'k', 'j', 'l', 'm'],
    ['a', 'b', 'd', 'f', 'e', 'g', 'h', 'i', 'k', 'j', 'm'],
    ['a', 'b', 'd', 'f', 'e', 'g', 'h', 'i', 'k', 'j', 'l', 'm'],
    ['a', 'f', 'd', 'b', 'c', 'e', 'g', 'h', 'i', 'k', 'j', 'm'],
    ['a', 'f', 'd', 'b', 'c', 'e', 'g', 'h', 'i', 'k', 'j', 'l', 'm']
];

is_deeply(\@got, $expected, "All paths from the Sedgewick example graph");

=head2 Test 2. An interesting Hamiltonion graph from hog.grinvin.org

=cut

my $g2 = YAGL->new;
$g2->read_lst("$cwd/data/graph_32806.lst");
my @got2 = $g2->paths_between('1', '7');

my $expected2 = [
    ['1', '2', '8',  '7'],
    ['1', '2', '9',  '7'],
    ['1', '3', '8',  '7'],
    ['1', '3', '9',  '7'],
    ['1', '4', '10', '7'],
    ['1', '4', '11', '7'],
    ['1', '5', '10', '7'],
    ['1', '5', '11', '7'],
    ['1', '2', '8',  '3', '9',  '7'],
    ['1', '2', '8',  '6', '9',  '7'],
    ['1', '2', '8',  '6', '10', '7'],
    ['1', '2', '8',  '6', '11', '7'],
    ['1', '2', '9',  '3', '8',  '7'],
    ['1', '2', '9',  '6', '8',  '7'],
    ['1', '2', '9',  '6', '10', '7'],
    ['1', '2', '9',  '6', '11', '7'],
    ['1', '2', '12', '3', '8',  '7'],
    ['1', '2', '12', '3', '9',  '7'],
    ['1', '2', '12', '4', '10', '7'],
    ['1', '2', '12', '4', '11', '7'],
    ['1', '2', '12', '5', '10', '7'],
    ['1', '2', '12', '5', '11', '7'],
    ['1', '3', '8',  '2', '9',  '7'],
    ['1', '3', '8',  '6', '9',  '7'],
    ['1', '3', '8',  '6', '10', '7'],
    ['1', '3', '8',  '6', '11', '7'],
    ['1', '3', '9',  '2', '8',  '7'],
    ['1', '3', '9',  '6', '8',  '7'],
    ['1', '3', '9',  '6', '10', '7'],
    ['1', '3', '9',  '6', '11', '7'],
    ['1', '3', '12', '2', '8',  '7'],
    ['1', '3', '12', '2', '9',  '7'],
    ['1', '3', '12', '4', '10', '7'],
    ['1', '3', '12', '4', '11', '7'],
    ['1', '3', '12', '5', '10', '7'],
    ['1', '3', '12', '5', '11', '7'],
    ['1', '4', '10', '5', '11', '7'],
    ['1', '4', '10', '6', '8',  '7'],
    ['1', '4', '10', '6', '9',  '7'],
    ['1', '4', '10', '6', '11', '7'],
    ['1', '4', '11', '5', '10', '7'],
    ['1', '4', '11', '6', '8',  '7'],
    ['1', '4', '11', '6', '9',  '7'],
    ['1', '4', '11', '6', '10', '7'],
    ['1', '4', '12', '2', '8',  '7'],
    ['1', '4', '12', '2', '9',  '7'],
    ['1', '4', '12', '3', '8',  '7'],
    ['1', '4', '12', '3', '9',  '7'],
    ['1', '4', '12', '5', '10', '7'],
    ['1', '4', '12', '5', '11', '7'],
    ['1', '5', '10', '4', '11', '7'],
    ['1', '5', '10', '6', '8',  '7'],
    ['1', '5', '10', '6', '9',  '7'],
    ['1', '5', '10', '6', '11', '7'],
    ['1', '5', '11', '4', '10', '7'],
    ['1', '5', '11', '6', '8',  '7'],
    ['1', '5', '11', '6', '9',  '7'],
    ['1', '5', '11', '6', '10', '7'],
    ['1', '5', '12', '2', '8',  '7'],
    ['1', '5', '12', '2', '9',  '7'],
    ['1', '5', '12', '3', '8',  '7'],
    ['1', '5', '12', '3', '9',  '7'],
    ['1', '5', '12', '4', '10', '7'],
    ['1', '5', '12', '4', '11', '7'],
    ['1', '2', '8',  '3', '9',  '6', '10', '7'],
    ['1', '2', '8',  '3', '9',  '6', '11', '7'],
    ['1', '2', '8',  '3', '12', '4', '10', '7'],
    ['1', '2', '8',  '3', '12', '4', '11', '7'],
    ['1', '2', '8',  '3', '12', '5', '10', '7'],
    ['1', '2', '8',  '3', '12', '5', '11', '7'],
    ['1', '2', '8',  '6', '10', '4', '11', '7'],
    ['1', '2', '8',  '6', '10', '5', '11', '7'],
    ['1', '2', '8',  '6', '11', '4', '10', '7'],
    ['1', '2', '8',  '6', '11', '5', '10', '7'],
    ['1', '2', '9',  '3', '8',  '6', '10', '7'],
    ['1', '2', '9',  '3', '8',  '6', '11', '7'],
    ['1', '2', '9',  '3', '12', '4', '10', '7'],
    ['1', '2', '9',  '3', '12', '4', '11', '7'],
    ['1', '2', '9',  '3', '12', '5', '10', '7'],
    ['1', '2', '9',  '3', '12', '5', '11', '7'],
    ['1', '2', '9',  '6', '10', '4', '11', '7'],
    ['1', '2', '9',  '6', '10', '5', '11', '7'],
    ['1', '2', '9',  '6', '11', '4', '10', '7'],
    ['1', '2', '9',  '6', '11', '5', '10', '7'],
    ['1', '2', '12', '3', '8',  '6', '9',  '7'],
    ['1', '2', '12', '3', '8',  '6', '10', '7'],
    ['1', '2', '12', '3', '8',  '6', '11', '7'],
    ['1', '2', '12', '3', '9',  '6', '8',  '7'],
    ['1', '2', '12', '3', '9',  '6', '10', '7'],
    ['1', '2', '12', '3', '9',  '6', '11', '7'],
    ['1', '2', '12', '4', '10', '5', '11', '7'],
    ['1', '2', '12', '4', '10', '6', '8',  '7'],
    ['1', '2', '12', '4', '10', '6', '9',  '7'],
    ['1', '2', '12', '4', '10', '6', '11', '7'],
    ['1', '2', '12', '4', '11', '5', '10', '7'],
    ['1', '2', '12', '4', '11', '6', '8',  '7'],
    ['1', '2', '12', '4', '11', '6', '9',  '7'],
    ['1', '2', '12', '4', '11', '6', '10', '7'],
    ['1', '2', '12', '5', '10', '4', '11', '7'],
    ['1', '2', '12', '5', '10', '6', '8',  '7'],
    ['1', '2', '12', '5', '10', '6', '9',  '7'],
    ['1', '2', '12', '5', '10', '6', '11', '7'],
    ['1', '2', '12', '5', '11', '4', '10', '7'],
    ['1', '2', '12', '5', '11', '6', '8',  '7'],
    ['1', '2', '12', '5', '11', '6', '9',  '7'],
    ['1', '2', '12', '5', '11', '6', '10', '7'],
    ['1', '3', '8',  '2', '9',  '6', '10', '7'],
    ['1', '3', '8',  '2', '9',  '6', '11', '7'],
    ['1', '3', '8',  '2', '12', '4', '10', '7'],
    ['1', '3', '8',  '2', '12', '4', '11', '7'],
    ['1', '3', '8',  '2', '12', '5', '10', '7'],
    ['1', '3', '8',  '2', '12', '5', '11', '7'],
    ['1', '3', '8',  '6', '10', '4', '11', '7'],
    ['1', '3', '8',  '6', '10', '5', '11', '7'],
    ['1', '3', '8',  '6', '11', '4', '10', '7'],
    ['1', '3', '8',  '6', '11', '5', '10', '7'],
    ['1', '3', '9',  '2', '8',  '6', '10', '7'],
    ['1', '3', '9',  '2', '8',  '6', '11', '7'],
    ['1', '3', '9',  '2', '12', '4', '10', '7'],
    ['1', '3', '9',  '2', '12', '4', '11', '7'],
    ['1', '3', '9',  '2', '12', '5', '10', '7'],
    ['1', '3', '9',  '2', '12', '5', '11', '7'],
    ['1', '3', '9',  '6', '10', '4', '11', '7'],
    ['1', '3', '9',  '6', '10', '5', '11', '7'],
    ['1', '3', '9',  '6', '11', '4', '10', '7'],
    ['1', '3', '9',  '6', '11', '5', '10', '7'],
    ['1', '3', '12', '2', '8',  '6', '9',  '7'],
    ['1', '3', '12', '2', '8',  '6', '10', '7'],
    ['1', '3', '12', '2', '8',  '6', '11', '7'],
    ['1', '3', '12', '2', '9',  '6', '8',  '7'],
    ['1', '3', '12', '2', '9',  '6', '10', '7'],
    ['1', '3', '12', '2', '9',  '6', '11', '7'],
    ['1', '3', '12', '4', '10', '5', '11', '7'],
    ['1', '3', '12', '4', '10', '6', '8',  '7'],
    ['1', '3', '12', '4', '10', '6', '9',  '7'],
    ['1', '3', '12', '4', '10', '6', '11', '7'],
    ['1', '3', '12', '4', '11', '5', '10', '7'],
    ['1', '3', '12', '4', '11', '6', '8',  '7'],
    ['1', '3', '12', '4', '11', '6', '9',  '7'],
    ['1', '3', '12', '4', '11', '6', '10', '7'],
    ['1', '3', '12', '5', '10', '4', '11', '7'],
    ['1', '3', '12', '5', '10', '6', '8',  '7'],
    ['1', '3', '12', '5', '10', '6', '9',  '7'],
    ['1', '3', '12', '5', '10', '6', '11', '7'],
    ['1', '3', '12', '5', '11', '4', '10', '7'],
    ['1', '3', '12', '5', '11', '6', '8',  '7'],
    ['1', '3', '12', '5', '11', '6', '9',  '7'],
    ['1', '3', '12', '5', '11', '6', '10', '7'],
    ['1', '4', '10', '5', '11', '6', '8',  '7'],
    ['1', '4', '10', '5', '11', '6', '9',  '7'],
    ['1', '4', '10', '5', '12', '2', '8',  '7'],
    ['1', '4', '10', '5', '12', '2', '9',  '7'],
    ['1', '4', '10', '5', '12', '3', '8',  '7'],
    ['1', '4', '10', '5', '12', '3', '9',  '7'],
    ['1', '4', '10', '6', '8',  '2', '9',  '7'],
    ['1', '4', '10', '6', '8',  '3', '9',  '7'],
    ['1', '4', '10', '6', '9',  '2', '8',  '7'],
    ['1', '4', '10', '6', '9',  '3', '8',  '7'],
    ['1', '4', '11', '5', '10', '6', '8',  '7'],
    ['1', '4', '11', '5', '10', '6', '9',  '7'],
    ['1', '4', '11', '5', '12', '2', '8',  '7'],
    ['1', '4', '11', '5', '12', '2', '9',  '7'],
    ['1', '4', '11', '5', '12', '3', '8',  '7'],
    ['1', '4', '11', '5', '12', '3', '9',  '7'],
    ['1', '4', '11', '6', '8',  '2', '9',  '7'],
    ['1', '4', '11', '6', '8',  '3', '9',  '7'],
    ['1', '4', '11', '6', '9',  '2', '8',  '7'],
    ['1', '4', '11', '6', '9',  '3', '8',  '7'],
    ['1', '4', '12', '2', '8',  '3', '9',  '7'],
    ['1', '4', '12', '2', '8',  '6', '9',  '7'],
    ['1', '4', '12', '2', '8',  '6', '10', '7'],
    ['1', '4', '12', '2', '8',  '6', '11', '7'],
    ['1', '4', '12', '2', '9',  '3', '8',  '7'],
    ['1', '4', '12', '2', '9',  '6', '8',  '7'],
    ['1', '4', '12', '2', '9',  '6', '10', '7'],
    ['1', '4', '12', '2', '9',  '6', '11', '7'],
    ['1', '4', '12', '3', '8',  '2', '9',  '7'],
    ['1', '4', '12', '3', '8',  '6', '9',  '7'],
    ['1', '4', '12', '3', '8',  '6', '10', '7'],
    ['1', '4', '12', '3', '8',  '6', '11', '7'],
    ['1', '4', '12', '3', '9',  '2', '8',  '7'],
    ['1', '4', '12', '3', '9',  '6', '8',  '7'],
    ['1', '4', '12', '3', '9',  '6', '10', '7'],
    ['1', '4', '12', '3', '9',  '6', '11', '7'],
    ['1', '4', '12', '5', '10', '6', '8',  '7'],
    ['1', '4', '12', '5', '10', '6', '9',  '7'],
    ['1', '4', '12', '5', '10', '6', '11', '7'],
    ['1', '4', '12', '5', '11', '6', '8',  '7'],
    ['1', '4', '12', '5', '11', '6', '9',  '7'],
    ['1', '4', '12', '5', '11', '6', '10', '7'],
    ['1', '5', '10', '4', '11', '6', '8',  '7'],
    ['1', '5', '10', '4', '11', '6', '9',  '7'],
    ['1', '5', '10', '4', '12', '2', '8',  '7'],
    ['1', '5', '10', '4', '12', '2', '9',  '7'],
    ['1', '5', '10', '4', '12', '3', '8',  '7'],
    ['1', '5', '10', '4', '12', '3', '9',  '7'],
    ['1', '5', '10', '6', '8',  '2', '9',  '7'],
    ['1', '5', '10', '6', '8',  '3', '9',  '7'],
    ['1', '5', '10', '6', '9',  '2', '8',  '7'],
    ['1', '5', '10', '6', '9',  '3', '8',  '7'],
    ['1', '5', '11', '4', '10', '6', '8',  '7'],
    ['1', '5', '11', '4', '10', '6', '9',  '7'],
    ['1', '5', '11', '4', '12', '2', '8',  '7'],
    ['1', '5', '11', '4', '12', '2', '9',  '7'],
    ['1', '5', '11', '4', '12', '3', '8',  '7'],
    ['1', '5', '11', '4', '12', '3', '9',  '7'],
    ['1', '5', '11', '6', '8',  '2', '9',  '7'],
    ['1', '5', '11', '6', '8',  '3', '9',  '7'],
    ['1', '5', '11', '6', '9',  '2', '8',  '7'],
    ['1', '5', '11', '6', '9',  '3', '8',  '7'],
    ['1', '5', '12', '2', '8',  '3', '9',  '7'],
    ['1', '5', '12', '2', '8',  '6', '9',  '7'],
    ['1', '5', '12', '2', '8',  '6', '10', '7'],
    ['1', '5', '12', '2', '8',  '6', '11', '7'],
    ['1', '5', '12', '2', '9',  '3', '8',  '7'],
    ['1', '5', '12', '2', '9',  '6', '8',  '7'],
    ['1', '5', '12', '2', '9',  '6', '10', '7'],
    ['1', '5', '12', '2', '9',  '6', '11', '7'],
    ['1', '5', '12', '3', '8',  '2', '9',  '7'],
    ['1', '5', '12', '3', '8',  '6', '9',  '7'],
    ['1', '5', '12', '3', '8',  '6', '10', '7'],
    ['1', '5', '12', '3', '8',  '6', '11', '7'],
    ['1', '5', '12', '3', '9',  '2', '8',  '7'],
    ['1', '5', '12', '3', '9',  '6', '8',  '7'],
    ['1', '5', '12', '3', '9',  '6', '10', '7'],
    ['1', '5', '12', '3', '9',  '6', '11', '7'],
    ['1', '5', '12', '4', '10', '6', '8',  '7'],
    ['1', '5', '12', '4', '10', '6', '9',  '7'],
    ['1', '5', '12', '4', '10', '6', '11', '7'],
    ['1', '5', '12', '4', '11', '6', '8',  '7'],
    ['1', '5', '12', '4', '11', '6', '9',  '7'],
    ['1', '5', '12', '4', '11', '6', '10', '7'],
    ['1', '2', '8',  '3', '9',  '6', '10', '4', '11', '7'],
    ['1', '2', '8',  '3', '9',  '6', '10', '5', '11', '7'],
    ['1', '2', '8',  '3', '9',  '6', '11', '4', '10', '7'],
    ['1', '2', '8',  '3', '9',  '6', '11', '5', '10', '7'],
    ['1', '2', '8',  '3', '12', '4', '10', '5', '11', '7'],
    ['1', '2', '8',  '3', '12', '4', '10', '6', '9',  '7'],
    ['1', '2', '8',  '3', '12', '4', '10', '6', '11', '7'],
    ['1', '2', '8',  '3', '12', '4', '11', '5', '10', '7'],
    ['1', '2', '8',  '3', '12', '4', '11', '6', '9',  '7'],
    ['1', '2', '8',  '3', '12', '4', '11', '6', '10', '7'],
    ['1', '2', '8',  '3', '12', '5', '10', '4', '11', '7'],
    ['1', '2', '8',  '3', '12', '5', '10', '6', '9',  '7'],
    ['1', '2', '8',  '3', '12', '5', '10', '6', '11', '7'],
    ['1', '2', '8',  '3', '12', '5', '11', '4', '10', '7'],
    ['1', '2', '8',  '3', '12', '5', '11', '6', '9',  '7'],
    ['1', '2', '8',  '3', '12', '5', '11', '6', '10', '7'],
    ['1', '2', '8',  '6', '9',  '3', '12', '4', '10', '7'],
    ['1', '2', '8',  '6', '9',  '3', '12', '4', '11', '7'],
    ['1', '2', '8',  '6', '9',  '3', '12', '5', '10', '7'],
    ['1', '2', '8',  '6', '9',  '3', '12', '5', '11', '7'],
    ['1', '2', '8',  '6', '10', '4', '12', '3', '9',  '7'],
    ['1', '2', '8',  '6', '10', '4', '12', '5', '11', '7'],
    ['1', '2', '8',  '6', '10', '5', '12', '3', '9',  '7'],
    ['1', '2', '8',  '6', '10', '5', '12', '4', '11', '7'],
    ['1', '2', '8',  '6', '11', '4', '12', '3', '9',  '7'],
    ['1', '2', '8',  '6', '11', '4', '12', '5', '10', '7'],
    ['1', '2', '8',  '6', '11', '5', '12', '3', '9',  '7'],
    ['1', '2', '8',  '6', '11', '5', '12', '4', '10', '7'],
    ['1', '2', '9',  '3', '8',  '6', '10', '4', '11', '7'],
    ['1', '2', '9',  '3', '8',  '6', '10', '5', '11', '7'],
    ['1', '2', '9',  '3', '8',  '6', '11', '4', '10', '7'],
    ['1', '2', '9',  '3', '8',  '6', '11', '5', '10', '7'],
    ['1', '2', '9',  '3', '12', '4', '10', '5', '11', '7'],
    ['1', '2', '9',  '3', '12', '4', '10', '6', '8',  '7'],
    ['1', '2', '9',  '3', '12', '4', '10', '6', '11', '7'],
    ['1', '2', '9',  '3', '12', '4', '11', '5', '10', '7'],
    ['1', '2', '9',  '3', '12', '4', '11', '6', '8',  '7'],
    ['1', '2', '9',  '3', '12', '4', '11', '6', '10', '7'],
    ['1', '2', '9',  '3', '12', '5', '10', '4', '11', '7'],
    ['1', '2', '9',  '3', '12', '5', '10', '6', '8',  '7'],
    ['1', '2', '9',  '3', '12', '5', '10', '6', '11', '7'],
    ['1', '2', '9',  '3', '12', '5', '11', '4', '10', '7'],
    ['1', '2', '9',  '3', '12', '5', '11', '6', '8',  '7'],
    ['1', '2', '9',  '3', '12', '5', '11', '6', '10', '7'],
    ['1', '2', '9',  '6', '8',  '3', '12', '4', '10', '7'],
    ['1', '2', '9',  '6', '8',  '3', '12', '4', '11', '7'],
    ['1', '2', '9',  '6', '8',  '3', '12', '5', '10', '7'],
    ['1', '2', '9',  '6', '8',  '3', '12', '5', '11', '7'],
    ['1', '2', '9',  '6', '10', '4', '12', '3', '8',  '7'],
    ['1', '2', '9',  '6', '10', '4', '12', '5', '11', '7'],
    ['1', '2', '9',  '6', '10', '5', '12', '3', '8',  '7'],
    ['1', '2', '9',  '6', '10', '5', '12', '4', '11', '7'],
    ['1', '2', '9',  '6', '11', '4', '12', '3', '8',  '7'],
    ['1', '2', '9',  '6', '11', '4', '12', '5', '10', '7'],
    ['1', '2', '9',  '6', '11', '5', '12', '3', '8',  '7'],
    ['1', '2', '9',  '6', '11', '5', '12', '4', '10', '7'],
    ['1', '2', '12', '3', '8',  '6', '10', '4', '11', '7'],
    ['1', '2', '12', '3', '8',  '6', '10', '5', '11', '7'],
    ['1', '2', '12', '3', '8',  '6', '11', '4', '10', '7'],
    ['1', '2', '12', '3', '8',  '6', '11', '5', '10', '7'],
    ['1', '2', '12', '3', '9',  '6', '10', '4', '11', '7'],
    ['1', '2', '12', '3', '9',  '6', '10', '5', '11', '7'],
    ['1', '2', '12', '3', '9',  '6', '11', '4', '10', '7'],
    ['1', '2', '12', '3', '9',  '6', '11', '5', '10', '7'],
    ['1', '2', '12', '4', '10', '5', '11', '6', '8',  '7'],
    ['1', '2', '12', '4', '10', '5', '11', '6', '9',  '7'],
    ['1', '2', '12', '4', '10', '6', '8',  '3', '9',  '7'],
    ['1', '2', '12', '4', '10', '6', '9',  '3', '8',  '7'],
    ['1', '2', '12', '4', '11', '5', '10', '6', '8',  '7'],
    ['1', '2', '12', '4', '11', '5', '10', '6', '9',  '7'],
    ['1', '2', '12', '4', '11', '6', '8',  '3', '9',  '7'],
    ['1', '2', '12', '4', '11', '6', '9',  '3', '8',  '7'],
    ['1', '2', '12', '5', '10', '4', '11', '6', '8',  '7'],
    ['1', '2', '12', '5', '10', '4', '11', '6', '9',  '7'],
    ['1', '2', '12', '5', '10', '6', '8',  '3', '9',  '7'],
    ['1', '2', '12', '5', '10', '6', '9',  '3', '8',  '7'],
    ['1', '2', '12', '5', '11', '4', '10', '6', '8',  '7'],
    ['1', '2', '12', '5', '11', '4', '10', '6', '9',  '7'],
    ['1', '2', '12', '5', '11', '6', '8',  '3', '9',  '7'],
    ['1', '2', '12', '5', '11', '6', '9',  '3', '8',  '7'],
    ['1', '3', '8',  '2', '9',  '6', '10', '4', '11', '7'],
    ['1', '3', '8',  '2', '9',  '6', '10', '5', '11', '7'],
    ['1', '3', '8',  '2', '9',  '6', '11', '4', '10', '7'],
    ['1', '3', '8',  '2', '9',  '6', '11', '5', '10', '7'],
    ['1', '3', '8',  '2', '12', '4', '10', '5', '11', '7'],
    ['1', '3', '8',  '2', '12', '4', '10', '6', '9',  '7'],
    ['1', '3', '8',  '2', '12', '4', '10', '6', '11', '7'],
    ['1', '3', '8',  '2', '12', '4', '11', '5', '10', '7'],
    ['1', '3', '8',  '2', '12', '4', '11', '6', '9',  '7'],
    ['1', '3', '8',  '2', '12', '4', '11', '6', '10', '7'],
    ['1', '3', '8',  '2', '12', '5', '10', '4', '11', '7'],
    ['1', '3', '8',  '2', '12', '5', '10', '6', '9',  '7'],
    ['1', '3', '8',  '2', '12', '5', '10', '6', '11', '7'],
    ['1', '3', '8',  '2', '12', '5', '11', '4', '10', '7'],
    ['1', '3', '8',  '2', '12', '5', '11', '6', '9',  '7'],
    ['1', '3', '8',  '2', '12', '5', '11', '6', '10', '7'],
    ['1', '3', '8',  '6', '9',  '2', '12', '4', '10', '7'],
    ['1', '3', '8',  '6', '9',  '2', '12', '4', '11', '7'],
    ['1', '3', '8',  '6', '9',  '2', '12', '5', '10', '7'],
    ['1', '3', '8',  '6', '9',  '2', '12', '5', '11', '7'],
    ['1', '3', '8',  '6', '10', '4', '12', '2', '9',  '7'],
    ['1', '3', '8',  '6', '10', '4', '12', '5', '11', '7'],
    ['1', '3', '8',  '6', '10', '5', '12', '2', '9',  '7'],
    ['1', '3', '8',  '6', '10', '5', '12', '4', '11', '7'],
    ['1', '3', '8',  '6', '11', '4', '12', '2', '9',  '7'],
    ['1', '3', '8',  '6', '11', '4', '12', '5', '10', '7'],
    ['1', '3', '8',  '6', '11', '5', '12', '2', '9',  '7'],
    ['1', '3', '8',  '6', '11', '5', '12', '4', '10', '7'],
    ['1', '3', '9',  '2', '8',  '6', '10', '4', '11', '7'],
    ['1', '3', '9',  '2', '8',  '6', '10', '5', '11', '7'],
    ['1', '3', '9',  '2', '8',  '6', '11', '4', '10', '7'],
    ['1', '3', '9',  '2', '8',  '6', '11', '5', '10', '7'],
    ['1', '3', '9',  '2', '12', '4', '10', '5', '11', '7'],
    ['1', '3', '9',  '2', '12', '4', '10', '6', '8',  '7'],
    ['1', '3', '9',  '2', '12', '4', '10', '6', '11', '7'],
    ['1', '3', '9',  '2', '12', '4', '11', '5', '10', '7'],
    ['1', '3', '9',  '2', '12', '4', '11', '6', '8',  '7'],
    ['1', '3', '9',  '2', '12', '4', '11', '6', '10', '7'],
    ['1', '3', '9',  '2', '12', '5', '10', '4', '11', '7'],
    ['1', '3', '9',  '2', '12', '5', '10', '6', '8',  '7'],
    ['1', '3', '9',  '2', '12', '5', '10', '6', '11', '7'],
    ['1', '3', '9',  '2', '12', '5', '11', '4', '10', '7'],
    ['1', '3', '9',  '2', '12', '5', '11', '6', '8',  '7'],
    ['1', '3', '9',  '2', '12', '5', '11', '6', '10', '7'],
    ['1', '3', '9',  '6', '8',  '2', '12', '4', '10', '7'],
    ['1', '3', '9',  '6', '8',  '2', '12', '4', '11', '7'],
    ['1', '3', '9',  '6', '8',  '2', '12', '5', '10', '7'],
    ['1', '3', '9',  '6', '8',  '2', '12', '5', '11', '7'],
    ['1', '3', '9',  '6', '10', '4', '12', '2', '8',  '7'],
    ['1', '3', '9',  '6', '10', '4', '12', '5', '11', '7'],
    ['1', '3', '9',  '6', '10', '5', '12', '2', '8',  '7'],
    ['1', '3', '9',  '6', '10', '5', '12', '4', '11', '7'],
    ['1', '3', '9',  '6', '11', '4', '12', '2', '8',  '7'],
    ['1', '3', '9',  '6', '11', '4', '12', '5', '10', '7'],
    ['1', '3', '9',  '6', '11', '5', '12', '2', '8',  '7'],
    ['1', '3', '9',  '6', '11', '5', '12', '4', '10', '7'],
    ['1', '3', '12', '2', '8',  '6', '10', '4', '11', '7'],
    ['1', '3', '12', '2', '8',  '6', '10', '5', '11', '7'],
    ['1', '3', '12', '2', '8',  '6', '11', '4', '10', '7'],
    ['1', '3', '12', '2', '8',  '6', '11', '5', '10', '7'],
    ['1', '3', '12', '2', '9',  '6', '10', '4', '11', '7'],
    ['1', '3', '12', '2', '9',  '6', '10', '5', '11', '7'],
    ['1', '3', '12', '2', '9',  '6', '11', '4', '10', '7'],
    ['1', '3', '12', '2', '9',  '6', '11', '5', '10', '7'],
    ['1', '3', '12', '4', '10', '5', '11', '6', '8',  '7'],
    ['1', '3', '12', '4', '10', '5', '11', '6', '9',  '7'],
    ['1', '3', '12', '4', '10', '6', '8',  '2', '9',  '7'],
    ['1', '3', '12', '4', '10', '6', '9',  '2', '8',  '7'],
    ['1', '3', '12', '4', '11', '5', '10', '6', '8',  '7'],
    ['1', '3', '12', '4', '11', '5', '10', '6', '9',  '7'],
    ['1', '3', '12', '4', '11', '6', '8',  '2', '9',  '7'],
    ['1', '3', '12', '4', '11', '6', '9',  '2', '8',  '7'],
    ['1', '3', '12', '5', '10', '4', '11', '6', '8',  '7'],
    ['1', '3', '12', '5', '10', '4', '11', '6', '9',  '7'],
    ['1', '3', '12', '5', '10', '6', '8',  '2', '9',  '7'],
    ['1', '3', '12', '5', '10', '6', '9',  '2', '8',  '7'],
    ['1', '3', '12', '5', '11', '4', '10', '6', '8',  '7'],
    ['1', '3', '12', '5', '11', '4', '10', '6', '9',  '7'],
    ['1', '3', '12', '5', '11', '6', '8',  '2', '9',  '7'],
    ['1', '3', '12', '5', '11', '6', '9',  '2', '8',  '7'],
    ['1', '4', '10', '5', '11', '6', '8',  '2', '9',  '7'],
    ['1', '4', '10', '5', '11', '6', '8',  '3', '9',  '7'],
    ['1', '4', '10', '5', '11', '6', '9',  '2', '8',  '7'],
    ['1', '4', '10', '5', '11', '6', '9',  '3', '8',  '7'],
    ['1', '4', '10', '5', '12', '2', '8',  '3', '9',  '7'],
    ['1', '4', '10', '5', '12', '2', '8',  '6', '9',  '7'],
    ['1', '4', '10', '5', '12', '2', '8',  '6', '11', '7'],
    ['1', '4', '10', '5', '12', '2', '9',  '3', '8',  '7'],
    ['1', '4', '10', '5', '12', '2', '9',  '6', '8',  '7'],
    ['1', '4', '10', '5', '12', '2', '9',  '6', '11', '7'],
    ['1', '4', '10', '5', '12', '3', '8',  '2', '9',  '7'],
    ['1', '4', '10', '5', '12', '3', '8',  '6', '9',  '7'],
    ['1', '4', '10', '5', '12', '3', '8',  '6', '11', '7'],
    ['1', '4', '10', '5', '12', '3', '9',  '2', '8',  '7'],
    ['1', '4', '10', '5', '12', '3', '9',  '6', '8',  '7'],
    ['1', '4', '10', '5', '12', '3', '9',  '6', '11', '7'],
    ['1', '4', '10', '6', '8',  '2', '12', '3', '9',  '7'],
    ['1', '4', '10', '6', '8',  '2', '12', '5', '11', '7'],
    ['1', '4', '10', '6', '8',  '3', '12', '2', '9',  '7'],
    ['1', '4', '10', '6', '8',  '3', '12', '5', '11', '7'],
    ['1', '4', '10', '6', '9',  '2', '12', '3', '8',  '7'],
    ['1', '4', '10', '6', '9',  '2', '12', '5', '11', '7'],
    ['1', '4', '10', '6', '9',  '3', '12', '2', '8',  '7'],
    ['1', '4', '10', '6', '9',  '3', '12', '5', '11', '7'],
    ['1', '4', '10', '6', '11', '5', '12', '2', '8',  '7'],
    ['1', '4', '10', '6', '11', '5', '12', '2', '9',  '7'],
    ['1', '4', '10', '6', '11', '5', '12', '3', '8',  '7'],
    ['1', '4', '10', '6', '11', '5', '12', '3', '9',  '7'],
    ['1', '4', '11', '5', '10', '6', '8',  '2', '9',  '7'],
    ['1', '4', '11', '5', '10', '6', '8',  '3', '9',  '7'],
    ['1', '4', '11', '5', '10', '6', '9',  '2', '8',  '7'],
    ['1', '4', '11', '5', '10', '6', '9',  '3', '8',  '7'],
    ['1', '4', '11', '5', '12', '2', '8',  '3', '9',  '7'],
    ['1', '4', '11', '5', '12', '2', '8',  '6', '9',  '7'],
    ['1', '4', '11', '5', '12', '2', '8',  '6', '10', '7'],
    ['1', '4', '11', '5', '12', '2', '9',  '3', '8',  '7'],
    ['1', '4', '11', '5', '12', '2', '9',  '6', '8',  '7'],
    ['1', '4', '11', '5', '12', '2', '9',  '6', '10', '7'],
    ['1', '4', '11', '5', '12', '3', '8',  '2', '9',  '7'],
    ['1', '4', '11', '5', '12', '3', '8',  '6', '9',  '7'],
    ['1', '4', '11', '5', '12', '3', '8',  '6', '10', '7'],
    ['1', '4', '11', '5', '12', '3', '9',  '2', '8',  '7'],
    ['1', '4', '11', '5', '12', '3', '9',  '6', '8',  '7'],
    ['1', '4', '11', '5', '12', '3', '9',  '6', '10', '7'],
    ['1', '4', '11', '6', '8',  '2', '12', '3', '9',  '7'],
    ['1', '4', '11', '6', '8',  '2', '12', '5', '10', '7'],
    ['1', '4', '11', '6', '8',  '3', '12', '2', '9',  '7'],
    ['1', '4', '11', '6', '8',  '3', '12', '5', '10', '7'],
    ['1', '4', '11', '6', '9',  '2', '12', '3', '8',  '7'],
    ['1', '4', '11', '6', '9',  '2', '12', '5', '10', '7'],
    ['1', '4', '11', '6', '9',  '3', '12', '2', '8',  '7'],
    ['1', '4', '11', '6', '9',  '3', '12', '5', '10', '7'],
    ['1', '4', '11', '6', '10', '5', '12', '2', '8',  '7'],
    ['1', '4', '11', '6', '10', '5', '12', '2', '9',  '7'],
    ['1', '4', '11', '6', '10', '5', '12', '3', '8',  '7'],
    ['1', '4', '11', '6', '10', '5', '12', '3', '9',  '7'],
    ['1', '4', '12', '2', '8',  '3', '9',  '6', '10', '7'],
    ['1', '4', '12', '2', '8',  '3', '9',  '6', '11', '7'],
    ['1', '4', '12', '2', '8',  '6', '10', '5', '11', '7'],
    ['1', '4', '12', '2', '8',  '6', '11', '5', '10', '7'],
    ['1', '4', '12', '2', '9',  '3', '8',  '6', '10', '7'],
    ['1', '4', '12', '2', '9',  '3', '8',  '6', '11', '7'],
    ['1', '4', '12', '2', '9',  '6', '10', '5', '11', '7'],
    ['1', '4', '12', '2', '9',  '6', '11', '5', '10', '7'],
    ['1', '4', '12', '3', '8',  '2', '9',  '6', '10', '7'],
    ['1', '4', '12', '3', '8',  '2', '9',  '6', '11', '7'],
    ['1', '4', '12', '3', '8',  '6', '10', '5', '11', '7'],
    ['1', '4', '12', '3', '8',  '6', '11', '5', '10', '7'],
    ['1', '4', '12', '3', '9',  '2', '8',  '6', '10', '7'],
    ['1', '4', '12', '3', '9',  '2', '8',  '6', '11', '7'],
    ['1', '4', '12', '3', '9',  '6', '10', '5', '11', '7'],
    ['1', '4', '12', '3', '9',  '6', '11', '5', '10', '7'],
    ['1', '4', '12', '5', '10', '6', '8',  '2', '9',  '7'],
    ['1', '4', '12', '5', '10', '6', '8',  '3', '9',  '7'],
    ['1', '4', '12', '5', '10', '6', '9',  '2', '8',  '7'],
    ['1', '4', '12', '5', '10', '6', '9',  '3', '8',  '7'],
    ['1', '4', '12', '5', '11', '6', '8',  '2', '9',  '7'],
    ['1', '4', '12', '5', '11', '6', '8',  '3', '9',  '7'],
    ['1', '4', '12', '5', '11', '6', '9',  '2', '8',  '7'],
    ['1', '4', '12', '5', '11', '6', '9',  '3', '8',  '7'],
    ['1', '5', '10', '4', '11', '6', '8',  '2', '9',  '7'],
    ['1', '5', '10', '4', '11', '6', '8',  '3', '9',  '7'],
    ['1', '5', '10', '4', '11', '6', '9',  '2', '8',  '7'],
    ['1', '5', '10', '4', '11', '6', '9',  '3', '8',  '7'],
    ['1', '5', '10', '4', '12', '2', '8',  '3', '9',  '7'],
    ['1', '5', '10', '4', '12', '2', '8',  '6', '9',  '7'],
    ['1', '5', '10', '4', '12', '2', '8',  '6', '11', '7'],
    ['1', '5', '10', '4', '12', '2', '9',  '3', '8',  '7'],
    ['1', '5', '10', '4', '12', '2', '9',  '6', '8',  '7'],
    ['1', '5', '10', '4', '12', '2', '9',  '6', '11', '7'],
    ['1', '5', '10', '4', '12', '3', '8',  '2', '9',  '7'],
    ['1', '5', '10', '4', '12', '3', '8',  '6', '9',  '7'],
    ['1', '5', '10', '4', '12', '3', '8',  '6', '11', '7'],
    ['1', '5', '10', '4', '12', '3', '9',  '2', '8',  '7'],
    ['1', '5', '10', '4', '12', '3', '9',  '6', '8',  '7'],
    ['1', '5', '10', '4', '12', '3', '9',  '6', '11', '7'],
    ['1', '5', '10', '6', '8',  '2', '12', '3', '9',  '7'],
    ['1', '5', '10', '6', '8',  '2', '12', '4', '11', '7'],
    ['1', '5', '10', '6', '8',  '3', '12', '2', '9',  '7'],
    ['1', '5', '10', '6', '8',  '3', '12', '4', '11', '7'],
    ['1', '5', '10', '6', '9',  '2', '12', '3', '8',  '7'],
    ['1', '5', '10', '6', '9',  '2', '12', '4', '11', '7'],
    ['1', '5', '10', '6', '9',  '3', '12', '2', '8',  '7'],
    ['1', '5', '10', '6', '9',  '3', '12', '4', '11', '7'],
    ['1', '5', '10', '6', '11', '4', '12', '2', '8',  '7'],
    ['1', '5', '10', '6', '11', '4', '12', '2', '9',  '7'],
    ['1', '5', '10', '6', '11', '4', '12', '3', '8',  '7'],
    ['1', '5', '10', '6', '11', '4', '12', '3', '9',  '7'],
    ['1', '5', '11', '4', '10', '6', '8',  '2', '9',  '7'],
    ['1', '5', '11', '4', '10', '6', '8',  '3', '9',  '7'],
    ['1', '5', '11', '4', '10', '6', '9',  '2', '8',  '7'],
    ['1', '5', '11', '4', '10', '6', '9',  '3', '8',  '7'],
    ['1', '5', '11', '4', '12', '2', '8',  '3', '9',  '7'],
    ['1', '5', '11', '4', '12', '2', '8',  '6', '9',  '7'],
    ['1', '5', '11', '4', '12', '2', '8',  '6', '10', '7'],
    ['1', '5', '11', '4', '12', '2', '9',  '3', '8',  '7'],
    ['1', '5', '11', '4', '12', '2', '9',  '6', '8',  '7'],
    ['1', '5', '11', '4', '12', '2', '9',  '6', '10', '7'],
    ['1', '5', '11', '4', '12', '3', '8',  '2', '9',  '7'],
    ['1', '5', '11', '4', '12', '3', '8',  '6', '9',  '7'],
    ['1', '5', '11', '4', '12', '3', '8',  '6', '10', '7'],
    ['1', '5', '11', '4', '12', '3', '9',  '2', '8',  '7'],
    ['1', '5', '11', '4', '12', '3', '9',  '6', '8',  '7'],
    ['1', '5', '11', '4', '12', '3', '9',  '6', '10', '7'],
    ['1', '5', '11', '6', '8',  '2', '12', '3', '9',  '7'],
    ['1', '5', '11', '6', '8',  '2', '12', '4', '10', '7'],
    ['1', '5', '11', '6', '8',  '3', '12', '2', '9',  '7'],
    ['1', '5', '11', '6', '8',  '3', '12', '4', '10', '7'],
    ['1', '5', '11', '6', '9',  '2', '12', '3', '8',  '7'],
    ['1', '5', '11', '6', '9',  '2', '12', '4', '10', '7'],
    ['1', '5', '11', '6', '9',  '3', '12', '2', '8',  '7'],
    ['1', '5', '11', '6', '9',  '3', '12', '4', '10', '7'],
    ['1', '5', '11', '6', '10', '4', '12', '2', '8',  '7'],
    ['1', '5', '11', '6', '10', '4', '12', '2', '9',  '7'],
    ['1', '5', '11', '6', '10', '4', '12', '3', '8',  '7'],
    ['1', '5', '11', '6', '10', '4', '12', '3', '9',  '7'],
    ['1', '5', '12', '2', '8',  '3', '9',  '6', '10', '7'],
    ['1', '5', '12', '2', '8',  '3', '9',  '6', '11', '7'],
    ['1', '5', '12', '2', '8',  '6', '10', '4', '11', '7'],
    ['1', '5', '12', '2', '8',  '6', '11', '4', '10', '7'],
    ['1', '5', '12', '2', '9',  '3', '8',  '6', '10', '7'],
    ['1', '5', '12', '2', '9',  '3', '8',  '6', '11', '7'],
    ['1', '5', '12', '2', '9',  '6', '10', '4', '11', '7'],
    ['1', '5', '12', '2', '9',  '6', '11', '4', '10', '7'],
    ['1', '5', '12', '3', '8',  '2', '9',  '6', '10', '7'],
    ['1', '5', '12', '3', '8',  '2', '9',  '6', '11', '7'],
    ['1', '5', '12', '3', '8',  '6', '10', '4', '11', '7'],
    ['1', '5', '12', '3', '8',  '6', '11', '4', '10', '7'],
    ['1', '5', '12', '3', '9',  '2', '8',  '6', '10', '7'],
    ['1', '5', '12', '3', '9',  '2', '8',  '6', '11', '7'],
    ['1', '5', '12', '3', '9',  '6', '10', '4', '11', '7'],
    ['1', '5', '12', '3', '9',  '6', '11', '4', '10', '7'],
    ['1', '5', '12', '4', '10', '6', '8',  '2', '9',  '7'],
    ['1', '5', '12', '4', '10', '6', '8',  '3', '9',  '7'],
    ['1', '5', '12', '4', '10', '6', '9',  '2', '8',  '7'],
    ['1', '5', '12', '4', '10', '6', '9',  '3', '8',  '7'],
    ['1', '5', '12', '4', '11', '6', '8',  '2', '9',  '7'],
    ['1', '5', '12', '4', '11', '6', '8',  '3', '9',  '7'],
    ['1', '5', '12', '4', '11', '6', '9',  '2', '8',  '7'],
    ['1', '5', '12', '4', '11', '6', '9',  '3', '8',  '7'],
    ['1', '2', '8',  '3', '9',  '6', '10', '4', '12', '5', '11', '7'],
    ['1', '2', '8',  '3', '9',  '6', '10', '5', '12', '4', '11', '7'],
    ['1', '2', '8',  '3', '9',  '6', '11', '4', '12', '5', '10', '7'],
    ['1', '2', '8',  '3', '9',  '6', '11', '5', '12', '4', '10', '7'],
    ['1', '2', '8',  '3', '12', '4', '10', '5', '11', '6', '9', '7'],
    ['1', '2', '8',  '3', '12', '4', '11', '5', '10', '6', '9', '7'],
    ['1', '2', '8',  '3', '12', '5', '10', '4', '11', '6', '9', '7'],
    ['1', '2', '8',  '3', '12', '5', '11', '4', '10', '6', '9', '7'],
    ['1', '2', '8',  '6', '9',  '3', '12', '4', '10', '5', '11', '7'],
    ['1', '2', '8',  '6', '9',  '3', '12', '4', '11', '5', '10', '7'],
    ['1', '2', '8',  '6', '9',  '3', '12', '5', '10', '4', '11', '7'],
    ['1', '2', '8',  '6', '9',  '3', '12', '5', '11', '4', '10', '7'],
    ['1', '2', '8',  '6', '10', '4', '11', '5', '12', '3', '9', '7'],
    ['1', '2', '8',  '6', '10', '5', '11', '4', '12', '3', '9', '7'],
    ['1', '2', '8',  '6', '11', '4', '10', '5', '12', '3', '9', '7'],
    ['1', '2', '8',  '6', '11', '5', '10', '4', '12', '3', '9', '7'],
    ['1', '2', '9',  '3', '8',  '6', '10', '4', '12', '5', '11', '7'],
    ['1', '2', '9',  '3', '8',  '6', '10', '5', '12', '4', '11', '7'],
    ['1', '2', '9',  '3', '8',  '6', '11', '4', '12', '5', '10', '7'],
    ['1', '2', '9',  '3', '8',  '6', '11', '5', '12', '4', '10', '7'],
    ['1', '2', '9',  '3', '12', '4', '10', '5', '11', '6', '8', '7'],
    ['1', '2', '9',  '3', '12', '4', '11', '5', '10', '6', '8', '7'],
    ['1', '2', '9',  '3', '12', '5', '10', '4', '11', '6', '8', '7'],
    ['1', '2', '9',  '3', '12', '5', '11', '4', '10', '6', '8', '7'],
    ['1', '2', '9',  '6', '8',  '3', '12', '4', '10', '5', '11', '7'],
    ['1', '2', '9',  '6', '8',  '3', '12', '4', '11', '5', '10', '7'],
    ['1', '2', '9',  '6', '8',  '3', '12', '5', '10', '4', '11', '7'],
    ['1', '2', '9',  '6', '8',  '3', '12', '5', '11', '4', '10', '7'],
    ['1', '2', '9',  '6', '10', '4', '11', '5', '12', '3', '8', '7'],
    ['1', '2', '9',  '6', '10', '5', '11', '4', '12', '3', '8', '7'],
    ['1', '2', '9',  '6', '11', '4', '10', '5', '12', '3', '8', '7'],
    ['1', '2', '9',  '6', '11', '5', '10', '4', '12', '3', '8', '7'],
    ['1', '2', '12', '4', '10', '5', '11', '6', '8',  '3', '9', '7'],
    ['1', '2', '12', '4', '10', '5', '11', '6', '9',  '3', '8', '7'],
    ['1', '2', '12', '4', '11', '5', '10', '6', '8',  '3', '9', '7'],
    ['1', '2', '12', '4', '11', '5', '10', '6', '9',  '3', '8', '7'],
    ['1', '2', '12', '5', '10', '4', '11', '6', '8',  '3', '9', '7'],
    ['1', '2', '12', '5', '10', '4', '11', '6', '9',  '3', '8', '7'],
    ['1', '2', '12', '5', '11', '4', '10', '6', '8',  '3', '9', '7'],
    ['1', '2', '12', '5', '11', '4', '10', '6', '9',  '3', '8', '7'],
    ['1', '3', '8',  '2', '9',  '6', '10', '4', '12', '5', '11', '7'],
    ['1', '3', '8',  '2', '9',  '6', '10', '5', '12', '4', '11', '7'],
    ['1', '3', '8',  '2', '9',  '6', '11', '4', '12', '5', '10', '7'],
    ['1', '3', '8',  '2', '9',  '6', '11', '5', '12', '4', '10', '7'],
    ['1', '3', '8',  '2', '12', '4', '10', '5', '11', '6', '9', '7'],
    ['1', '3', '8',  '2', '12', '4', '11', '5', '10', '6', '9', '7'],
    ['1', '3', '8',  '2', '12', '5', '10', '4', '11', '6', '9', '7'],
    ['1', '3', '8',  '2', '12', '5', '11', '4', '10', '6', '9', '7'],
    ['1', '3', '8',  '6', '9',  '2', '12', '4', '10', '5', '11', '7'],
    ['1', '3', '8',  '6', '9',  '2', '12', '4', '11', '5', '10', '7'],
    ['1', '3', '8',  '6', '9',  '2', '12', '5', '10', '4', '11', '7'],
    ['1', '3', '8',  '6', '9',  '2', '12', '5', '11', '4', '10', '7'],
    ['1', '3', '8',  '6', '10', '4', '11', '5', '12', '2', '9', '7'],
    ['1', '3', '8',  '6', '10', '5', '11', '4', '12', '2', '9', '7'],
    ['1', '3', '8',  '6', '11', '4', '10', '5', '12', '2', '9', '7'],
    ['1', '3', '8',  '6', '11', '5', '10', '4', '12', '2', '9', '7'],
    ['1', '3', '9',  '2', '8',  '6', '10', '4', '12', '5', '11', '7'],
    ['1', '3', '9',  '2', '8',  '6', '10', '5', '12', '4', '11', '7'],
    ['1', '3', '9',  '2', '8',  '6', '11', '4', '12', '5', '10', '7'],
    ['1', '3', '9',  '2', '8',  '6', '11', '5', '12', '4', '10', '7'],
    ['1', '3', '9',  '2', '12', '4', '10', '5', '11', '6', '8', '7'],
    ['1', '3', '9',  '2', '12', '4', '11', '5', '10', '6', '8', '7'],
    ['1', '3', '9',  '2', '12', '5', '10', '4', '11', '6', '8', '7'],
    ['1', '3', '9',  '2', '12', '5', '11', '4', '10', '6', '8', '7'],
    ['1', '3', '9',  '6', '8',  '2', '12', '4', '10', '5', '11', '7'],
    ['1', '3', '9',  '6', '8',  '2', '12', '4', '11', '5', '10', '7'],
    ['1', '3', '9',  '6', '8',  '2', '12', '5', '10', '4', '11', '7'],
    ['1', '3', '9',  '6', '8',  '2', '12', '5', '11', '4', '10', '7'],
    ['1', '3', '9',  '6', '10', '4', '11', '5', '12', '2', '8', '7'],
    ['1', '3', '9',  '6', '10', '5', '11', '4', '12', '2', '8', '7'],
    ['1', '3', '9',  '6', '11', '4', '10', '5', '12', '2', '8', '7'],
    ['1', '3', '9',  '6', '11', '5', '10', '4', '12', '2', '8', '7'],
    ['1', '3', '12', '4', '10', '5', '11', '6', '8',  '2', '9', '7'],
    ['1', '3', '12', '4', '10', '5', '11', '6', '9',  '2', '8', '7'],
    ['1', '3', '12', '4', '11', '5', '10', '6', '8',  '2', '9', '7'],
    ['1', '3', '12', '4', '11', '5', '10', '6', '9',  '2', '8', '7'],
    ['1', '3', '12', '5', '10', '4', '11', '6', '8',  '2', '9', '7'],
    ['1', '3', '12', '5', '10', '4', '11', '6', '9',  '2', '8', '7'],
    ['1', '3', '12', '5', '11', '4', '10', '6', '8',  '2', '9', '7'],
    ['1', '3', '12', '5', '11', '4', '10', '6', '9',  '2', '8', '7'],
    ['1', '4', '10', '5', '11', '6', '8',  '2', '12', '3', '9', '7'],
    ['1', '4', '10', '5', '11', '6', '8',  '3', '12', '2', '9', '7'],
    ['1', '4', '10', '5', '11', '6', '9',  '2', '12', '3', '8', '7'],
    ['1', '4', '10', '5', '11', '6', '9',  '3', '12', '2', '8', '7'],
    ['1', '4', '10', '5', '12', '2', '8',  '3', '9',  '6', '11', '7'],
    ['1', '4', '10', '5', '12', '2', '9',  '3', '8',  '6', '11', '7'],
    ['1', '4', '10', '5', '12', '3', '8',  '2', '9',  '6', '11', '7'],
    ['1', '4', '10', '5', '12', '3', '9',  '2', '8',  '6', '11', '7'],
    ['1', '4', '10', '6', '8',  '2', '9',  '3', '12', '5', '11', '7'],
    ['1', '4', '10', '6', '8',  '3', '9',  '2', '12', '5', '11', '7'],
    ['1', '4', '10', '6', '9',  '2', '8',  '3', '12', '5', '11', '7'],
    ['1', '4', '10', '6', '9',  '3', '8',  '2', '12', '5', '11', '7'],
    ['1', '4', '10', '6', '11', '5', '12', '2', '8',  '3', '9', '7'],
    ['1', '4', '10', '6', '11', '5', '12', '2', '9',  '3', '8', '7'],
    ['1', '4', '10', '6', '11', '5', '12', '3', '8',  '2', '9', '7'],
    ['1', '4', '10', '6', '11', '5', '12', '3', '9',  '2', '8', '7'],
    ['1', '4', '11', '5', '10', '6', '8',  '2', '12', '3', '9', '7'],
    ['1', '4', '11', '5', '10', '6', '8',  '3', '12', '2', '9', '7'],
    ['1', '4', '11', '5', '10', '6', '9',  '2', '12', '3', '8', '7'],
    ['1', '4', '11', '5', '10', '6', '9',  '3', '12', '2', '8', '7'],
    ['1', '4', '11', '5', '12', '2', '8',  '3', '9',  '6', '10', '7'],
    ['1', '4', '11', '5', '12', '2', '9',  '3', '8',  '6', '10', '7'],
    ['1', '4', '11', '5', '12', '3', '8',  '2', '9',  '6', '10', '7'],
    ['1', '4', '11', '5', '12', '3', '9',  '2', '8',  '6', '10', '7'],
    ['1', '4', '11', '6', '8',  '2', '9',  '3', '12', '5', '10', '7'],
    ['1', '4', '11', '6', '8',  '3', '9',  '2', '12', '5', '10', '7'],
    ['1', '4', '11', '6', '9',  '2', '8',  '3', '12', '5', '10', '7'],
    ['1', '4', '11', '6', '9',  '3', '8',  '2', '12', '5', '10', '7'],
    ['1', '4', '11', '6', '10', '5', '12', '2', '8',  '3', '9', '7'],
    ['1', '4', '11', '6', '10', '5', '12', '2', '9',  '3', '8', '7'],
    ['1', '4', '11', '6', '10', '5', '12', '3', '8',  '2', '9', '7'],
    ['1', '4', '11', '6', '10', '5', '12', '3', '9',  '2', '8', '7'],
    ['1', '4', '12', '2', '8',  '3', '9',  '6', '10', '5', '11', '7'],
    ['1', '4', '12', '2', '8',  '3', '9',  '6', '11', '5', '10', '7'],
    ['1', '4', '12', '2', '9',  '3', '8',  '6', '10', '5', '11', '7'],
    ['1', '4', '12', '2', '9',  '3', '8',  '6', '11', '5', '10', '7'],
    ['1', '4', '12', '3', '8',  '2', '9',  '6', '10', '5', '11', '7'],
    ['1', '4', '12', '3', '8',  '2', '9',  '6', '11', '5', '10', '7'],
    ['1', '4', '12', '3', '9',  '2', '8',  '6', '10', '5', '11', '7'],
    ['1', '4', '12', '3', '9',  '2', '8',  '6', '11', '5', '10', '7'],
    ['1', '5', '10', '4', '11', '6', '8',  '2', '12', '3', '9', '7'],
    ['1', '5', '10', '4', '11', '6', '8',  '3', '12', '2', '9', '7'],
    ['1', '5', '10', '4', '11', '6', '9',  '2', '12', '3', '8', '7'],
    ['1', '5', '10', '4', '11', '6', '9',  '3', '12', '2', '8', '7'],
    ['1', '5', '10', '4', '12', '2', '8',  '3', '9',  '6', '11', '7'],
    ['1', '5', '10', '4', '12', '2', '9',  '3', '8',  '6', '11', '7'],
    ['1', '5', '10', '4', '12', '3', '8',  '2', '9',  '6', '11', '7'],
    ['1', '5', '10', '4', '12', '3', '9',  '2', '8',  '6', '11', '7'],
    ['1', '5', '10', '6', '8',  '2', '9',  '3', '12', '4', '11', '7'],
    ['1', '5', '10', '6', '8',  '3', '9',  '2', '12', '4', '11', '7'],
    ['1', '5', '10', '6', '9',  '2', '8',  '3', '12', '4', '11', '7'],
    ['1', '5', '10', '6', '9',  '3', '8',  '2', '12', '4', '11', '7'],
    ['1', '5', '10', '6', '11', '4', '12', '2', '8',  '3', '9', '7'],
    ['1', '5', '10', '6', '11', '4', '12', '2', '9',  '3', '8', '7'],
    ['1', '5', '10', '6', '11', '4', '12', '3', '8',  '2', '9', '7'],
    ['1', '5', '10', '6', '11', '4', '12', '3', '9',  '2', '8', '7'],
    ['1', '5', '11', '4', '10', '6', '8',  '2', '12', '3', '9', '7'],
    ['1', '5', '11', '4', '10', '6', '8',  '3', '12', '2', '9', '7'],
    ['1', '5', '11', '4', '10', '6', '9',  '2', '12', '3', '8', '7'],
    ['1', '5', '11', '4', '10', '6', '9',  '3', '12', '2', '8', '7'],
    ['1', '5', '11', '4', '12', '2', '8',  '3', '9',  '6', '10', '7'],
    ['1', '5', '11', '4', '12', '2', '9',  '3', '8',  '6', '10', '7'],
    ['1', '5', '11', '4', '12', '3', '8',  '2', '9',  '6', '10', '7'],
    ['1', '5', '11', '4', '12', '3', '9',  '2', '8',  '6', '10', '7'],
    ['1', '5', '11', '6', '8',  '2', '9',  '3', '12', '4', '10', '7'],
    ['1', '5', '11', '6', '8',  '3', '9',  '2', '12', '4', '10', '7'],
    ['1', '5', '11', '6', '9',  '2', '8',  '3', '12', '4', '10', '7'],
    ['1', '5', '11', '6', '9',  '3', '8',  '2', '12', '4', '10', '7'],
    ['1', '5', '11', '6', '10', '4', '12', '2', '8',  '3', '9', '7'],
    ['1', '5', '11', '6', '10', '4', '12', '2', '9',  '3', '8', '7'],
    ['1', '5', '11', '6', '10', '4', '12', '3', '8',  '2', '9', '7'],
    ['1', '5', '11', '6', '10', '4', '12', '3', '9',  '2', '8', '7'],
    ['1', '5', '12', '2', '8',  '3', '9',  '6', '10', '4', '11', '7'],
    ['1', '5', '12', '2', '8',  '3', '9',  '6', '11', '4', '10', '7'],
    ['1', '5', '12', '2', '9',  '3', '8',  '6', '10', '4', '11', '7'],
    ['1', '5', '12', '2', '9',  '3', '8',  '6', '11', '4', '10', '7'],
    ['1', '5', '12', '3', '8',  '2', '9',  '6', '10', '4', '11', '7'],
    ['1', '5', '12', '3', '8',  '2', '9',  '6', '11', '4', '10', '7'],
    ['1', '5', '12', '3', '9',  '2', '8',  '6', '10', '4', '11', '7'],
    ['1', '5', '12', '3', '9',  '2', '8',  '6', '11', '4', '10', '7']
];

is_deeply(\@got2, $expected2,
    "An interesting Hamiltonion graph from hog.grinvin.org");

=head2 Another example from my Hamiltonian path tests

=cut

my $g3 = YAGL->new;
$g3->read_lst("$cwd/t/24-ham-03.lst");
my @got3 = $g3->paths_between('1', '17');

my $expected3 = [
    ['1', '8', '17'],
    ['1', '14', '11', '5',  '17'],
    ['1', '16', '2',  '10', '17'],
    ['1', '14', '7',  '15', '8', '17'],
    ['1', '16', '18', '13', '3', '17'],
    ['1', '16', '2',  '9',  '3', '17'],
    ['1', '16', '6',  '11', '5', '17'],
    ['1', '16', '6',  '15', '8', '17'],
    ['1', '8',  '15', '6',  '11', '5',  '17'],
    ['1', '14', '7',  '18', '13', '3',  '17'],
    ['1', '14', '11', '6',  '15', '8',  '17'],
    ['1', '16', '18', '7',  '15', '8',  '17'],
    ['1', '16', '18', '13', '4',  '10', '17'],
    ['1', '16', '2',  '9',  '12', '5',  '17'],
    ['1', '8',  '15', '6',  '16', '2',  '10', '17'],
    ['1', '8',  '15', '7',  '18', '13', '3',  '17'],
    ['1', '8',  '15', '7',  '14', '11', '5',  '17'],
    ['1', '14', '7',  '18', '13', '4',  '10', '17'],
    ['1', '14', '7',  '18', '16', '2',  '10', '17'],
    ['1', '14', '7',  '15', '6',  '11', '5',  '17'],
    ['1', '14', '11', '5',  '12', '4',  '10', '17'],
    ['1', '14', '11', '5',  '12', '9',  '3',  '17'],
    ['1', '14', '11', '6',  '16', '2',  '10', '17'],
    ['1', '16', '18', '7',  '14', '11', '5',  '17'],
    ['1', '16', '18', '13', '4',  '12', '5',  '17'],
    ['1', '16', '2',  '9',  '12', '4',  '10', '17'],
    ['1', '16', '2',  '10', '4',  '12', '5',  '17'],
    ['1', '16', '2',  '10', '4',  '13', '3',  '17'],
    ['1', '8',  '15', '6',  '16', '18', '13', '3',  '17'],
    ['1', '8',  '15', '6',  '16', '2',  '9',  '3',  '17'],
    ['1', '8',  '15', '7',  '18', '13', '4',  '10', '17'],
    ['1', '8',  '15', '7',  '18', '16', '2',  '10', '17'],
    ['1', '14', '7',  '18', '13', '4',  '12', '5',  '17'],
    ['1', '14', '7',  '18', '16', '2',  '9',  '3',  '17'],
    ['1', '14', '7',  '18', '16', '6',  '11', '5',  '17'],
    ['1', '14', '7',  '18', '16', '6',  '15', '8',  '17'],
    ['1', '14', '7',  '15', '6',  '16', '2',  '10', '17'],
    ['1', '14', '11', '5',  '12', '4',  '13', '3',  '17'],
    ['1', '14', '11', '5',  '12', '9',  '2',  '10', '17'],
    ['1', '14', '11', '6',  '16', '18', '13', '3',  '17'],
    ['1', '14', '11', '6',  '16', '2',  '9',  '3',  '17'],
    ['1', '16', '18', '7',  '15', '6',  '11', '5',  '17'],
    ['1', '16', '18', '13', '3',  '9',  '2',  '10', '17'],
    ['1', '16', '18', '13', '3',  '9',  '12', '5',  '17'],
    ['1', '16', '18', '13', '4',  '12', '9',  '3',  '17'],
    ['1', '16', '2',  '9',  '3',  '13', '4',  '10', '17'],
    ['1', '16', '2',  '9',  '12', '4',  '13', '3',  '17'],
    ['1', '16', '2',  '10', '4',  '12', '9',  '3',  '17'],
    ['1', '16', '6',  '11', '5',  '12', '4',  '10', '17'],
    ['1', '16', '6',  '11', '5',  '12', '9',  '3',  '17'],
    ['1', '16', '6',  '11', '14', '7',  '15', '8',  '17'],
    ['1', '16', '6',  '15', '7',  '18', '13', '3',  '17'],
    ['1', '16', '6',  '15', '7',  '14', '11', '5',  '17'],
    ['1', '8',  '15', '6',  '11', '5',  '12', '4',  '10', '17'],
    ['1', '8',  '15', '6',  '11', '5',  '12', '9',  '3',  '17'],
    ['1', '8',  '15', '6',  '16', '18', '13', '4',  '10', '17'],
    ['1', '8',  '15', '6',  '16', '2',  '9',  '12', '5',  '17'],
    ['1', '8',  '15', '7',  '18', '13', '4',  '12', '5',  '17'],
    ['1', '8',  '15', '7',  '18', '16', '2',  '9',  '3',  '17'],
    ['1', '8',  '15', '7',  '18', '16', '6',  '11', '5',  '17'],
    ['1', '14', '7',  '18', '13', '3',  '9',  '2',  '10', '17'],
    ['1', '14', '7',  '18', '13', '3',  '9',  '12', '5',  '17'],
    ['1', '14', '7',  '18', '13', '4',  '12', '9',  '3',  '17'],
    ['1', '14', '7',  '18', '16', '2',  '9',  '12', '5',  '17'],
    ['1', '14', '7',  '15', '6',  '16', '18', '13', '3',  '17'],
    ['1', '14', '7',  '15', '6',  '16', '2',  '9',  '3',  '17'],
    ['1', '14', '11', '6',  '15', '7',  '18', '13', '3',  '17'],
    ['1', '14', '11', '6',  '16', '18', '7',  '15', '8',  '17'],
    ['1', '14', '11', '6',  '16', '18', '13', '4',  '10', '17'],
    ['1', '14', '11', '6',  '16', '2',  '9',  '12', '5',  '17'],
    ['1', '16', '18', '7',  '14', '11', '6',  '15', '8',  '17'],
    ['1', '16', '18', '13', '3',  '9',  '12', '4',  '10', '17'],
    ['1', '16', '18', '13', '4',  '10', '2',  '9',  '3',  '17'],
    ['1', '16', '18', '13', '4',  '12', '9',  '2',  '10', '17'],
    ['1', '16', '2',  '9',  '3',  '13', '4',  '12', '5',  '17'],
    ['1', '16', '6',  '11', '5',  '12', '4',  '13', '3',  '17'],
    ['1', '16', '6',  '11', '5',  '12', '9',  '2',  '10', '17'],
    ['1', '16', '6',  '11', '14', '7',  '18', '13', '3',  '17'],
    ['1', '16', '6',  '15', '7',  '18', '13', '4',  '10', '17'],
    ['1', '8',  '15', '6',  '11', '5',  '12', '4',  '13', '3',  '17'],
    ['1', '8',  '15', '6',  '11', '5',  '12', '9',  '2',  '10', '17'],
    ['1', '8',  '15', '6',  '11', '14', '7',  '18', '13', '3',  '17'],
    ['1', '8',  '15', '6',  '16', '18', '7',  '14', '11', '5',  '17'],
    ['1', '8',  '15', '6',  '16', '18', '13', '4',  '12', '5',  '17'],
    ['1', '8',  '15', '6',  '16', '2',  '9',  '12', '4',  '10', '17'],
    ['1', '8',  '15', '6',  '16', '2',  '10', '4',  '12', '5',  '17'],
    ['1', '8',  '15', '6',  '16', '2',  '10', '4',  '13', '3',  '17'],
    ['1', '8',  '15', '7',  '18', '13', '3',  '9',  '2',  '10', '17'],
    ['1', '8',  '15', '7',  '18', '13', '3',  '9',  '12', '5',  '17'],
    ['1', '8',  '15', '7',  '18', '13', '4',  '12', '9',  '3',  '17'],
    ['1', '8',  '15', '7',  '18', '16', '2',  '9',  '12', '5',  '17'],
    ['1', '8',  '15', '7',  '14', '11', '5',  '12', '4',  '10', '17'],
    ['1', '8',  '15', '7',  '14', '11', '5',  '12', '9',  '3',  '17'],
    ['1', '8',  '15', '7',  '14', '11', '6',  '16', '2',  '10', '17'],
    ['1', '14', '7',  '18', '13', '3',  '9',  '12', '4',  '10', '17'],
    ['1', '14', '7',  '18', '13', '4',  '10', '2',  '9',  '3',  '17'],
    ['1', '14', '7',  '18', '13', '4',  '12', '9',  '2',  '10', '17'],
    ['1', '14', '7',  '18', '16', '2',  '9',  '12', '4',  '10', '17'],
    ['1', '14', '7',  '18', '16', '2',  '10', '4',  '12', '5',  '17'],
    ['1', '14', '7',  '18', '16', '2',  '10', '4',  '13', '3',  '17'],
    ['1', '14', '7',  '15', '6',  '11', '5',  '12', '4',  '10', '17'],
    ['1', '14', '7',  '15', '6',  '11', '5',  '12', '9',  '3',  '17'],
    ['1', '14', '7',  '15', '6',  '16', '18', '13', '4',  '10', '17'],
    ['1', '14', '7',  '15', '6',  '16', '2',  '9',  '12', '5',  '17'],
    ['1', '14', '11', '5',  '12', '4',  '10', '2',  '9',  '3',  '17'],
    ['1', '14', '11', '5',  '12', '9',  '3',  '13', '4',  '10', '17'],
    ['1', '14', '11', '6',  '15', '7',  '18', '13', '4',  '10', '17'],
    ['1', '14', '11', '6',  '15', '7',  '18', '16', '2',  '10', '17'],
    ['1', '14', '11', '6',  '16', '18', '13', '4',  '12', '5',  '17'],
    ['1', '14', '11', '6',  '16', '2',  '9',  '12', '4',  '10', '17'],
    ['1', '14', '11', '6',  '16', '2',  '10', '4',  '12', '5',  '17'],
    ['1', '14', '11', '6',  '16', '2',  '10', '4',  '13', '3',  '17'],
    ['1', '16', '18', '7',  '14', '11', '5',  '12', '4',  '10', '17'],
    ['1', '16', '18', '7',  '14', '11', '5',  '12', '9',  '3',  '17'],
    ['1', '16', '18', '13', '4',  '10', '2',  '9',  '12', '5',  '17'],
    ['1', '16', '2',  '9',  '3',  '13', '18', '7',  '15', '8',  '17'],
    ['1', '16', '2',  '9',  '12', '5',  '11', '6',  '15', '8',  '17'],
    ['1', '16', '2',  '10', '4',  '13', '18', '7',  '15', '8',  '17'],
    ['1', '16', '2',  '10', '4',  '13', '3',  '9',  '12', '5',  '17'],
    ['1', '16', '6',  '11', '14', '7',  '18', '13', '4',  '10', '17'],
    ['1', '16', '6',  '15', '7',  '18', '13', '4',  '12', '5',  '17'],
    ['1', '8',  '15', '6',  '11', '14', '7',  '18', '13', '4',  '10', '17'],
    ['1', '8',  '15', '6',  '11', '14', '7',  '18', '16', '2',  '10', '17'],
    ['1', '8',  '15', '6',  '16', '18', '13', '3',  '9',  '2',  '10', '17'],
    ['1', '8',  '15', '6',  '16', '18', '13', '3',  '9',  '12', '5',  '17'],
    ['1', '8',  '15', '6',  '16', '18', '13', '4',  '12', '9',  '3',  '17'],
    ['1', '8',  '15', '6',  '16', '2',  '9',  '3',  '13', '4',  '10', '17'],
    ['1', '8',  '15', '6',  '16', '2',  '9',  '12', '4',  '13', '3',  '17'],
    ['1', '8',  '15', '6',  '16', '2',  '10', '4',  '12', '9',  '3',  '17'],
    ['1', '8',  '15', '7',  '18', '13', '3',  '9',  '12', '4',  '10', '17'],
    ['1', '8',  '15', '7',  '18', '13', '4',  '10', '2',  '9',  '3',  '17'],
    ['1', '8',  '15', '7',  '18', '13', '4',  '12', '9',  '2',  '10', '17'],
    ['1', '8',  '15', '7',  '18', '16', '2',  '9',  '12', '4',  '10', '17'],
    ['1', '8',  '15', '7',  '18', '16', '2',  '10', '4',  '12', '5',  '17'],
    ['1', '8',  '15', '7',  '18', '16', '2',  '10', '4',  '13', '3',  '17'],
    ['1', '8',  '15', '7',  '14', '11', '5',  '12', '4',  '13', '3',  '17'],
    ['1', '8',  '15', '7',  '14', '11', '5',  '12', '9',  '2',  '10', '17'],
    ['1', '8',  '15', '7',  '14', '11', '6',  '16', '18', '13', '3',  '17'],
    ['1', '8',  '15', '7',  '14', '11', '6',  '16', '2',  '9',  '3',  '17'],
    ['1', '14', '7',  '18', '13', '4',  '10', '2',  '9',  '12', '5',  '17'],
    ['1', '14', '7',  '18', '16', '2',  '9',  '3',  '13', '4',  '10', '17'],
    ['1', '14', '7',  '18', '16', '2',  '9',  '12', '4',  '13', '3',  '17'],
    ['1', '14', '7',  '18', '16', '2',  '10', '4',  '12', '9',  '3',  '17'],
    ['1', '14', '7',  '18', '16', '6',  '11', '5',  '12', '4',  '10', '17'],
    ['1', '14', '7',  '18', '16', '6',  '11', '5',  '12', '9',  '3',  '17'],
    ['1', '14', '7',  '15', '6',  '11', '5',  '12', '4',  '13', '3',  '17'],
    ['1', '14', '7',  '15', '6',  '11', '5',  '12', '9',  '2',  '10', '17'],
    ['1', '14', '7',  '15', '6',  '16', '18', '13', '4',  '12', '5',  '17'],
    ['1', '14', '7',  '15', '6',  '16', '2',  '9',  '12', '4',  '10', '17'],
    ['1', '14', '7',  '15', '6',  '16', '2',  '10', '4',  '12', '5',  '17'],
    ['1', '14', '7',  '15', '6',  '16', '2',  '10', '4',  '13', '3',  '17'],
    ['1', '14', '11', '5',  '12', '4',  '13', '18', '7',  '15', '8',  '17'],
    ['1', '14', '11', '5',  '12', '4',  '13', '18', '16', '2',  '10', '17'],
    ['1', '14', '11', '5',  '12', '4',  '13', '3',  '9',  '2',  '10', '17'],
    ['1', '14', '11', '5',  '12', '9',  '2',  '10', '4',  '13', '3',  '17'],
    ['1', '14', '11', '5',  '12', '9',  '2',  '16', '18', '13', '3',  '17'],
    ['1', '14', '11', '5',  '12', '9',  '2',  '16', '6',  '15', '8',  '17'],
    ['1', '14', '11', '6',  '15', '7',  '18', '13', '4',  '12', '5',  '17'],
    ['1', '14', '11', '6',  '15', '7',  '18', '16', '2',  '9',  '3',  '17'],
    ['1', '14', '11', '6',  '16', '18', '13', '3',  '9',  '2',  '10', '17'],
    ['1', '14', '11', '6',  '16', '18', '13', '3',  '9',  '12', '5',  '17'],
    ['1', '14', '11', '6',  '16', '18', '13', '4',  '12', '9',  '3',  '17'],
    ['1', '14', '11', '6',  '16', '2',  '9',  '3',  '13', '4',  '10', '17'],
    ['1', '14', '11', '6',  '16', '2',  '9',  '12', '4',  '13', '3',  '17'],
    ['1', '14', '11', '6',  '16', '2',  '10', '4',  '12', '9',  '3',  '17'],
    ['1', '16', '18', '7',  '14', '11', '5',  '12', '4',  '13', '3',  '17'],
    ['1', '16', '18', '7',  '14', '11', '5',  '12', '9',  '2',  '10', '17'],
    ['1', '16', '18', '7',  '15', '6',  '11', '5',  '12', '4',  '10', '17'],
    ['1', '16', '18', '7',  '15', '6',  '11', '5',  '12', '9',  '3',  '17'],
    ['1', '16', '18', '13', '3',  '9',  '2',  '10', '4',  '12', '5',  '17'],
    ['1', '16', '18', '13', '4',  '12', '5',  '11', '6',  '15', '8',  '17'],
    ['1', '16', '2',  '9',  '3',  '13', '18', '7',  '14', '11', '5',  '17'],
    ['1', '16', '2',  '9',  '12', '4',  '13', '18', '7',  '15', '8',  '17'],
    ['1', '16', '2',  '9',  '12', '5',  '11', '14', '7',  '15', '8',  '17'],
    ['1', '16', '2',  '10', '4',  '12', '5',  '11', '6',  '15', '8',  '17'],
    ['1', '16', '2',  '10', '4',  '13', '18', '7',  '14', '11', '5',  '17'],
    ['1', '16', '6',  '11', '5',  '12', '4',  '10', '2',  '9',  '3',  '17'],
    ['1', '16', '6',  '11', '5',  '12', '9',  '3',  '13', '4',  '10', '17'],
    ['1', '16', '6',  '11', '14', '7',  '18', '13', '4',  '12', '5',  '17'],
    ['1', '16', '6',  '15', '7',  '18', '13', '3',  '9',  '2',  '10', '17'],
    ['1', '16', '6',  '15', '7',  '18', '13', '3',  '9',  '12', '5',  '17'],
    ['1', '16', '6',  '15', '7',  '18', '13', '4',  '12', '9',  '3',  '17'],
    ['1', '16', '6',  '15', '7',  '14', '11', '5',  '12', '4',  '10', '17'],
    ['1', '16', '6',  '15', '7',  '14', '11', '5',  '12', '9',  '3',  '17'],
    ['1', '8', '15', '6', '11', '5', '12', '4', '10', '2',  '9', '3',  '17'],
    ['1', '8', '15', '6', '11', '5', '12', '9', '3',  '13', '4', '10', '17'],
    ['1', '8', '15', '6', '11', '14', '7', '18', '13', '4', '12', '5', '17'],
    ['1', '8', '15', '6', '11', '14', '7', '18', '16', '2', '9',  '3', '17'],
    ['1', '8', '15', '6', '16', '18', '13', '3', '9',  '12', '4', '10', '17'],
    ['1', '8', '15', '6', '16', '18', '13', '4', '10', '2',  '9', '3',  '17'],
    ['1', '8', '15', '6', '16', '18', '13', '4', '12', '9',  '2', '10', '17'],
    ['1', '8', '15', '6', '16', '2',  '9', '3',  '13', '4', '12', '5', '17'],
    ['1', '8', '15', '7', '18', '13', '4', '10', '2',  '9', '12', '5', '17'],
    ['1', '8', '15', '7', '18', '16', '2', '9', '3',  '13', '4',  '10', '17'],
    ['1', '8', '15', '7', '18', '16', '2', '9', '12', '4',  '13', '3',  '17'],
    ['1', '8', '15', '7', '18', '16', '2', '10', '4',  '12', '9', '3',  '17'],
    ['1', '8', '15', '7', '18', '16', '6', '11', '5',  '12', '4', '10', '17'],
    ['1', '8', '15', '7', '18', '16', '6', '11', '5',  '12', '9', '3',  '17'],
    ['1', '8', '15', '7', '14', '11', '6', '16', '18', '13', '4', '10', '17'],
    ['1', '8', '15', '7', '14', '11', '6', '16', '2', '9', '12', '5', '17'],
    ['1', '14', '7', '18', '13', '3', '9',  '2', '10', '4', '12', '5',  '17'],
    ['1', '14', '7', '18', '13', '3', '9',  '2', '16', '6', '11', '5',  '17'],
    ['1', '14', '7', '18', '13', '3', '9',  '2', '16', '6', '15', '8',  '17'],
    ['1', '14', '7', '18', '13', '4', '10', '2', '16', '6', '11', '5',  '17'],
    ['1', '14', '7', '18', '13', '4', '10', '2', '16', '6', '15', '8',  '17'],
    ['1', '14', '7', '18', '13', '4', '12', '5', '11', '6', '15', '8',  '17'],
    ['1', '14', '7', '18', '16', '2', '9',  '3', '13', '4', '12', '5',  '17'],
    ['1', '14', '7', '18', '16', '6', '11', '5', '12', '4', '13', '3',  '17'],
    ['1', '14', '7', '18', '16', '6', '11', '5', '12', '9', '2',  '10', '17'],
    ['1', '14', '7', '15', '6', '16', '18', '13', '3', '9', '2',  '10', '17'],
    ['1', '14', '7', '15', '6', '16', '18', '13', '3', '9', '12', '5',  '17'],
    ['1', '14', '7', '15', '6', '16', '18', '13', '4', '12', '9', '3',  '17'],
    ['1', '14', '7', '15', '6', '16', '2',  '9',  '3', '13', '4', '10', '17'],
    ['1', '14', '7', '15', '6', '16', '2', '9',  '12', '4',  '13', '3', '17'],
    ['1', '14', '7', '15', '6', '16', '2', '10', '4',  '12', '9',  '3', '17'],
    ['1', '14', '11', '5', '12', '4', '10', '2', '16', '18', '13', '3', '17'],
    ['1', '14', '11', '5', '12', '4', '10', '2', '16', '6',  '15', '8', '17'],
    ['1', '14', '11', '5', '12', '4', '13', '18', '16', '2', '9',  '3', '17'],
    ['1', '14', '11', '5', '12', '4', '13', '18', '16', '6', '15', '8', '17'],
    ['1', '14', '11', '5', '12', '9', '2',  '16', '18', '7', '15', '8', '17'],
    ['1', '14', '11', '5', '12', '9', '2', '16', '18', '13', '4', '10', '17'],
    ['1', '14', '11', '5', '12', '9', '3', '13', '18', '7', '15', '8', '17'],
    ['1', '14', '11', '5', '12', '9', '3', '13', '18', '16', '2', '10', '17'],
    ['1', '14', '11', '6', '15', '7', '18', '13', '3', '9', '2',  '10', '17'],
    ['1', '14', '11', '6', '15', '7', '18', '13', '3', '9', '12', '5',  '17'],
    ['1', '14', '11', '6', '15', '7', '18', '13', '4', '12', '9',  '3', '17'],
    ['1', '14', '11', '6', '15', '7', '18', '16', '2', '9',  '12', '5', '17'],
    ['1', '14', '11', '6', '16', '18', '13', '3', '9', '12', '4', '10', '17'],
    ['1', '14', '11', '6', '16', '18', '13', '4', '10', '2', '9', '3',  '17'],
    ['1', '14', '11', '6', '16', '18', '13', '4', '12', '9', '2', '10', '17'],
    ['1', '14', '11', '6', '16', '2', '9',  '3', '13', '4', '12', '5',  '17'],
    ['1', '16', '18', '7', '15', '6', '11', '5', '12', '4', '13', '3',  '17'],
    ['1', '16', '18', '7', '15', '6', '11', '5', '12', '9', '2',  '10', '17'],
    ['1', '16', '18', '13', '3', '9', '12', '5', '11', '6', '15', '8', '17'],
    ['1', '16', '18', '13', '4', '12', '5', '11', '14', '7', '15', '8', '17'],
    ['1', '16', '2', '9', '3', '13', '18', '7', '15', '6', '11', '5', '17'],
    ['1', '16', '2', '9', '12', '4', '13', '18', '7', '14', '11', '5', '17'],
    ['1', '16', '2', '9', '12', '5', '11', '14', '7', '18', '13', '3', '17'],
    ['1', '16', '2', '10', '4', '12', '5',  '11', '14', '7', '15', '8', '17'],
    ['1', '16', '2', '10', '4', '13', '18', '7',  '15', '6', '11', '5', '17'],
    ['1', '16', '6', '11', '5', '12', '4',  '13', '18', '7', '15', '8', '17'],
    ['1', '16', '6', '11', '5', '12', '4', '13', '3',  '9', '2',  '10', '17'],
    ['1', '16', '6', '11', '5', '12', '9', '2',  '10', '4', '13', '3',  '17'],
    ['1', '16', '6', '11', '14', '7', '18', '13', '3', '9', '2',  '10', '17'],
    ['1', '16', '6', '11', '14', '7', '18', '13', '3', '9', '12', '5',  '17'],
    ['1', '16', '6', '11', '14', '7', '18', '13', '4', '12', '9', '3', '17'],
    ['1', '16', '6', '15', '7', '18', '13', '3', '9',  '12', '4', '10', '17'],
    ['1', '16', '6', '15', '7', '18', '13', '4', '10', '2',  '9', '3',  '17'],
    ['1', '16', '6', '15', '7', '18', '13', '4', '12', '9',  '2', '10', '17'],
    ['1', '16', '6', '15', '7', '14', '11', '5', '12', '4', '13', '3',  '17'],
    ['1', '16', '6', '15', '7', '14', '11', '5', '12', '9', '2',  '10', '17'],
    [
        '1',  '8',  '15', '6', '11', '5', '12', '4',
        '13', '18', '16', '2', '10', '17'
    ],
    [
        '1',  '8', '15', '6', '11', '5', '12', '4',
        '13', '3', '9',  '2', '10', '17'
    ],
    [
        '1', '8',  '15', '6',  '11', '5', '12', '9',
        '2', '10', '4',  '13', '3',  '17'
    ],
    [
        '1', '8',  '15', '6',  '11', '5', '12', '9',
        '2', '16', '18', '13', '3',  '17'
    ],
    [
        '1',  '8', '15', '6', '11', '14', '7', '18',
        '13', '3', '9',  '2', '10', '17'
    ],
    [
        '1',  '8', '15', '6',  '11', '14', '7', '18',
        '13', '3', '9',  '12', '5',  '17'
    ],
    [
        '1',  '8', '15', '6', '11', '14', '7', '18',
        '13', '4', '12', '9', '3',  '17'
    ],
    [
        '1',  '8', '15', '6',  '11', '14', '7', '18',
        '16', '2', '9',  '12', '5',  '17'
    ],
    [
        '1',  '8', '15', '6', '16', '18', '7', '14',
        '11', '5', '12', '4', '10', '17'
    ],
    [
        '1',  '8', '15', '6', '16', '18', '7', '14',
        '11', '5', '12', '9', '3',  '17'
    ],
    [
        '1',  '8', '15', '6',  '16', '18', '13', '4',
        '10', '2', '9',  '12', '5',  '17'
    ],
    [
        '1',  '8', '15', '6',  '16', '2', '10', '4',
        '13', '3', '9',  '12', '5',  '17'
    ],
    [
        '1', '8',  '15', '7',  '18', '13', '3', '9',
        '2', '10', '4',  '12', '5',  '17'
    ],
    [
        '1', '8',  '15', '7',  '18', '13', '3', '9',
        '2', '16', '6',  '11', '5',  '17'
    ],
    [
        '1', '8',  '15', '7',  '18', '13', '4', '10',
        '2', '16', '6',  '11', '5',  '17'
    ],
    [
        '1', '8',  '15', '7',  '18', '16', '2', '9',
        '3', '13', '4',  '12', '5',  '17'
    ],
    [
        '1', '8',  '15', '7',  '18', '16', '6', '11',
        '5', '12', '4',  '13', '3',  '17'
    ],
    [
        '1', '8',  '15', '7', '18', '16', '6', '11',
        '5', '12', '9',  '2', '10', '17'
    ],
    [
        '1', '8',  '15', '7', '14', '11', '5', '12',
        '4', '10', '2',  '9', '3',  '17'
    ],
    [
        '1', '8', '15', '7', '14', '11', '5', '12',
        '9', '3', '13', '4', '10', '17'
    ],
    [
        '1',  '8',  '15', '7',  '14', '11', '6', '16',
        '18', '13', '4',  '12', '5',  '17'
    ],
    [
        '1', '8', '15', '7', '14', '11', '6', '16',
        '2', '9', '12', '4', '10', '17'
    ],
    [
        '1', '8',  '15', '7',  '14', '11', '6', '16',
        '2', '10', '4',  '12', '5',  '17'
    ],
    [
        '1', '8',  '15', '7',  '14', '11', '6', '16',
        '2', '10', '4',  '13', '3',  '17'
    ],
    [
        '1', '14', '7', '18', '13', '3', '9', '12',
        '5', '11', '6', '15', '8',  '17'
    ],
    [
        '1',  '14', '7',  '18', '13', '4', '12', '5',
        '11', '6',  '16', '2',  '10', '17'
    ],
    [
        '1', '14', '7', '18', '13', '4', '12', '9',
        '2', '16', '6', '11', '5',  '17'
    ],
    [
        '1', '14', '7', '18', '13', '4', '12', '9',
        '2', '16', '6', '15', '8',  '17'
    ],
    [
        '1', '14', '7', '18', '16', '2', '9', '12',
        '5', '11', '6', '15', '8',  '17'
    ],
    [
        '1',  '14', '7', '18', '16', '2', '10', '4',
        '13', '3',  '9', '12', '5',  '17'
    ],
    [
        '1', '14', '7', '15', '6', '11', '5', '12',
        '4', '10', '2', '9',  '3', '17'
    ],
    [
        '1', '14', '7',  '15', '6',  '11', '5', '12',
        '9', '3',  '13', '4',  '10', '17'
    ],
    [
        '1', '14', '7',  '15', '6',  '16', '18', '13',
        '3', '9',  '12', '4',  '10', '17'
    ],
    [
        '1', '14', '7', '15', '6', '16', '18', '13',
        '4', '10', '2', '9',  '3', '17'
    ],
    [
        '1', '14', '7', '15', '6',  '16', '18', '13',
        '4', '12', '9', '2',  '10', '17'
    ],
    [
        '1', '14', '7', '15', '6', '16', '2', '9',
        '3', '13', '4', '12', '5', '17'
    ],
    [
        '1',  '14', '11', '5',  '12', '4', '10', '2',
        '16', '18', '7',  '15', '8',  '17'
    ],
    [
        '1',  '14', '11', '5',  '12', '9', '3', '13',
        '18', '16', '6',  '15', '8',  '17'
    ],
    [
        '1', '14', '11', '6', '15', '7', '18', '13',
        '3', '9',  '12', '4', '10', '17'
    ],
    [
        '1', '14', '11', '6', '15', '7', '18', '13',
        '4', '10', '2',  '9', '3',  '17'
    ],
    [
        '1', '14', '11', '6', '15', '7', '18', '13',
        '4', '12', '9',  '2', '10', '17'
    ],
    [
        '1', '14', '11', '6', '15', '7', '18', '16',
        '2', '9',  '12', '4', '10', '17'
    ],
    [
        '1', '14', '11', '6',  '15', '7', '18', '16',
        '2', '10', '4',  '12', '5',  '17'
    ],
    [
        '1', '14', '11', '6',  '15', '7', '18', '16',
        '2', '10', '4',  '13', '3',  '17'
    ],
    [
        '1',  '14', '11', '6',  '16', '18', '13', '4',
        '10', '2',  '9',  '12', '5',  '17'
    ],
    [
        '1',  '14', '11', '6',  '16', '2', '9', '3',
        '13', '18', '7',  '15', '8',  '17'
    ],
    [
        '1',  '14', '11', '6',  '16', '2', '10', '4',
        '13', '18', '7',  '15', '8',  '17'
    ],
    [
        '1',  '14', '11', '6',  '16', '2', '10', '4',
        '13', '3',  '9',  '12', '5',  '17'
    ],
    [
        '1', '16', '18', '7', '14', '11', '5', '12',
        '4', '10', '2',  '9', '3',  '17'
    ],
    [
        '1', '16', '18', '7', '14', '11', '5', '12',
        '9', '3',  '13', '4', '10', '17'
    ],
    [
        '1',  '16', '18', '13', '3', '9', '12', '5',
        '11', '14', '7',  '15', '8', '17'
    ],
    [
        '1',  '16', '2', '9',  '3', '13', '18', '7',
        '14', '11', '6', '15', '8', '17'
    ],
    [
        '1', '16', '2', '9',  '3', '13', '4', '12',
        '5', '11', '6', '15', '8', '17'
    ],
    [
        '1', '16', '2', '9',  '12', '4', '13', '18',
        '7', '15', '6', '11', '5',  '17'
    ],
    [
        '1',  '16', '2',  '9',  '12', '5', '11', '6',
        '15', '7',  '18', '13', '3',  '17'
    ],
    [
        '1', '16', '2',  '9', '12', '5', '11', '14',
        '7', '18', '13', '4', '10', '17'
    ],
    [
        '1',  '16', '2',  '10', '4', '12', '5', '11',
        '14', '7',  '18', '13', '3', '17'
    ],
    [
        '1',  '16', '2', '10', '4', '12', '9', '3',
        '13', '18', '7', '15', '8', '17'
    ],
    [
        '1',  '16', '2', '10', '4', '13', '18', '7',
        '14', '11', '6', '15', '8', '17'
    ],
    [
        '1',  '16', '6', '11', '5', '12', '9', '3',
        '13', '18', '7', '15', '8', '17'
    ],
    [
        '1', '16', '6',  '11', '14', '7', '18', '13',
        '3', '9',  '12', '4',  '10', '17'
    ],
    [
        '1', '16', '6', '11', '14', '7', '18', '13',
        '4', '10', '2', '9',  '3',  '17'
    ],
    [
        '1', '16', '6', '11', '14', '7', '18', '13',
        '4', '12', '9', '2',  '10', '17'
    ],
    [
        '1',  '16', '6', '15', '7', '18', '13', '4',
        '10', '2',  '9', '12', '5', '17'
    ],
    [
        '1',  '8', '15', '6',  '11', '5', '12', '4',
        '10', '2', '16', '18', '13', '3', '17'
    ],
    [
        '1',  '8',  '15', '6', '11', '5', '12', '4',
        '13', '18', '16', '2', '9',  '3', '17'
    ],
    [
        '1', '8',  '15', '6',  '11', '5',  '12', '9',
        '2', '16', '18', '13', '4',  '10', '17'
    ],
    [
        '1', '8',  '15', '6',  '11', '5',  '12', '9',
        '3', '13', '18', '16', '2',  '10', '17'
    ],
    [
        '1',  '8', '15', '6',  '11', '14', '7', '18',
        '13', '3', '9',  '12', '4',  '10', '17'
    ],
    [
        '1',  '8', '15', '6', '11', '14', '7', '18',
        '13', '4', '10', '2', '9',  '3',  '17'
    ],
    [
        '1',  '8', '15', '6', '11', '14', '7', '18',
        '13', '4', '12', '9', '2',  '10', '17'
    ],
    [
        '1',  '8', '15', '6',  '11', '14', '7', '18',
        '16', '2', '9',  '12', '4',  '10', '17'
    ],
    [
        '1',  '8', '15', '6', '11', '14', '7', '18',
        '16', '2', '10', '4', '12', '5',  '17'
    ],
    [
        '1',  '8', '15', '6', '11', '14', '7', '18',
        '16', '2', '10', '4', '13', '3',  '17'
    ],
    [
        '1',  '8', '15', '6', '16', '18', '7', '14',
        '11', '5', '12', '4', '13', '3',  '17'
    ],
    [
        '1',  '8', '15', '6', '16', '18', '7', '14',
        '11', '5', '12', '9', '2',  '10', '17'
    ],
    [
        '1', '8', '15', '6', '16', '18', '13', '3',
        '9', '2', '10', '4', '12', '5',  '17'
    ],
    [
        '1',  '8',  '15', '6',  '16', '2', '9', '3',
        '13', '18', '7',  '14', '11', '5', '17'
    ],
    [
        '1',  '8',  '15', '6',  '16', '2', '10', '4',
        '13', '18', '7',  '14', '11', '5', '17'
    ],
    [
        '1', '8',  '15', '7',  '18', '13', '4', '12',
        '5', '11', '6',  '16', '2',  '10', '17'
    ],
    [
        '1', '8', '15', '7', '18', '13', '4', '12',
        '9', '2', '16', '6', '11', '5',  '17'
    ],
    [
        '1', '8',  '15', '7', '18', '16', '2', '10',
        '4', '13', '3',  '9', '12', '5',  '17'
    ],
    [
        '1', '8',  '15', '7',  '14', '11', '5', '12',
        '4', '13', '18', '16', '2',  '10', '17'
    ],
    [
        '1', '8',  '15', '7', '14', '11', '5', '12',
        '4', '13', '3',  '9', '2',  '10', '17'
    ],
    [
        '1', '8', '15', '7', '14', '11', '5', '12',
        '9', '2', '10', '4', '13', '3',  '17'
    ],
    [
        '1', '8', '15', '7',  '14', '11', '5', '12',
        '9', '2', '16', '18', '13', '3',  '17'
    ],
    [
        '1',  '8',  '15', '7', '14', '11', '6', '16',
        '18', '13', '3',  '9', '2',  '10', '17'
    ],
    [
        '1',  '8',  '15', '7', '14', '11', '6', '16',
        '18', '13', '3',  '9', '12', '5',  '17'
    ],
    [
        '1',  '8',  '15', '7',  '14', '11', '6', '16',
        '18', '13', '4',  '12', '9',  '3',  '17'
    ],
    [
        '1', '8', '15', '7',  '14', '11', '6', '16',
        '2', '9', '3',  '13', '4',  '10', '17'
    ],
    [
        '1', '8', '15', '7', '14', '11', '6', '16',
        '2', '9', '12', '4', '13', '3',  '17'
    ],
    [
        '1', '8',  '15', '7',  '14', '11', '6', '16',
        '2', '10', '4',  '12', '9',  '3',  '17'
    ],
    [
        '1', '14', '7', '18', '13', '3',  '9', '12',
        '5', '11', '6', '16', '2',  '10', '17'
    ],
    [
        '1',  '14', '7',  '18', '13', '4', '12', '5',
        '11', '6',  '16', '2',  '9',  '3', '17'
    ],
    [
        '1',  '14', '7',  '18', '16', '2', '10', '4',
        '12', '5',  '11', '6',  '15', '8', '17'
    ],
    [
        '1',  '14', '7',  '18', '16', '6', '11', '5',
        '12', '4',  '10', '2',  '9',  '3', '17'
    ],
    [
        '1',  '14', '7', '18', '16', '6',  '11', '5',
        '12', '9',  '3', '13', '4',  '10', '17'
    ],
    [
        '1', '14', '7',  '15', '6', '11', '5', '12',
        '4', '13', '18', '16', '2', '10', '17'
    ],
    [
        '1', '14', '7', '15', '6', '11', '5', '12',
        '4', '13', '3', '9',  '2', '10', '17'
    ],
    [
        '1', '14', '7',  '15', '6',  '11', '5', '12',
        '9', '2',  '10', '4',  '13', '3',  '17'
    ],
    [
        '1', '14', '7',  '15', '6',  '11', '5', '12',
        '9', '2',  '16', '18', '13', '3',  '17'
    ],
    [
        '1', '14', '7', '15', '6',  '16', '18', '13',
        '4', '10', '2', '9',  '12', '5',  '17'
    ],
    [
        '1', '14', '7', '15', '6',  '16', '2', '10',
        '4', '13', '3', '9',  '12', '5',  '17'
    ],
    [
        '1', '14', '11', '5',  '12', '4',  '13', '18',
        '7', '15', '6',  '16', '2',  '10', '17'
    ],
    [
        '1', '14', '11', '5', '12', '4', '13', '3',
        '9', '2',  '16', '6', '15', '8', '17'
    ],
    [
        '1', '14', '11', '5', '12', '9', '2', '10',
        '4', '13', '18', '7', '15', '8', '17'
    ],
    [
        '1', '14', '11', '5',  '12', '9', '2', '16',
        '6', '15', '7',  '18', '13', '3', '17'
    ],
    [
        '1', '14', '11', '6', '15', '7', '18', '13',
        '4', '10', '2',  '9', '12', '5', '17'
    ],
    [
        '1', '14', '11', '6',  '15', '7',  '18', '16',
        '2', '9',  '3',  '13', '4',  '10', '17'
    ],
    [
        '1', '14', '11', '6', '15', '7', '18', '16',
        '2', '9',  '12', '4', '13', '3', '17'
    ],
    [
        '1', '14', '11', '6',  '15', '7', '18', '16',
        '2', '10', '4',  '12', '9',  '3', '17'
    ],
    [
        '1', '14', '11', '6', '16', '18', '13', '3',
        '9', '2',  '10', '4', '12', '5',  '17'
    ],
    [
        '1', '14', '11', '6', '16', '2', '9', '12',
        '4', '13', '18', '7', '15', '8', '17'
    ],
    [
        '1', '16', '18', '7', '14', '11', '5', '12',
        '4', '13', '3',  '9', '2',  '10', '17'
    ],
    [
        '1', '16', '18', '7', '14', '11', '5', '12',
        '9', '2',  '10', '4', '13', '3',  '17'
    ],
    [
        '1',  '16', '18', '7', '15', '6', '11', '5',
        '12', '4',  '10', '2', '9',  '3', '17'
    ],
    [
        '1',  '16', '18', '7',  '15', '6',  '11', '5',
        '12', '9',  '3',  '13', '4',  '10', '17'
    ],
    [
        '1',  '16', '18', '13', '4',  '10', '2', '9',
        '12', '5',  '11', '6',  '15', '8',  '17'
    ],
    [
        '1',  '16', '2', '9',  '3', '13', '18', '7',
        '14', '11', '5', '12', '4', '10', '17'
    ],
    [
        '1', '16', '2',  '9', '3',  '13', '4', '12',
        '5', '11', '14', '7', '15', '8',  '17'
    ],
    [
        '1', '16', '2',  '9', '12', '4', '13', '18',
        '7', '14', '11', '6', '15', '8', '17'
    ],
    [
        '1',  '16', '2',  '9',  '12', '5',  '11', '6',
        '15', '7',  '18', '13', '4',  '10', '17'
    ],
    [
        '1', '16', '2', '10', '4',  '12', '5', '11',
        '6', '15', '7', '18', '13', '3',  '17'
    ],
    [
        '1',  '16', '2', '10', '4',  '12', '9', '3',
        '13', '18', '7', '14', '11', '5',  '17'
    ],
    [
        '1',  '16', '2', '10', '4', '13', '18', '7',
        '14', '11', '5', '12', '9', '3',  '17'
    ],
    [
        '1',  '16', '2',  '10', '4',  '13', '3', '9',
        '12', '5',  '11', '6',  '15', '8',  '17'
    ],
    [
        '1', '16', '6', '11', '14', '7', '18', '13',
        '4', '10', '2', '9',  '12', '5', '17'
    ],
    [
        '1', '16', '6',  '15', '7',  '18', '13', '3',
        '9', '2',  '10', '4',  '12', '5',  '17'
    ],
    [
        '1',  '16', '6',  '15', '7', '14', '11', '5',
        '12', '4',  '10', '2',  '9', '3',  '17'
    ],
    [
        '1',  '16', '6', '15', '7', '14', '11', '5',
        '12', '9',  '3', '13', '4', '10', '17'
    ],
    [
        '1',  '8', '15', '6', '11', '14', '7', '18',
        '13', '4', '10', '2', '9',  '12', '5', '17'
    ],
    [
        '1',  '8', '15', '6', '11', '14', '7',  '18',
        '16', '2', '9',  '3', '13', '4',  '10', '17'
    ],
    [
        '1',  '8', '15', '6',  '11', '14', '7', '18',
        '16', '2', '9',  '12', '4',  '13', '3', '17'
    ],
    [
        '1',  '8', '15', '6', '11', '14', '7', '18',
        '16', '2', '10', '4', '12', '9',  '3', '17'
    ],
    [
        '1', '8',  '15', '6', '16', '2',  '9', '12',
        '4', '13', '18', '7', '14', '11', '5', '17'
    ],
    [
        '1', '8',  '15', '6', '16', '2',  '9', '12',
        '5', '11', '14', '7', '18', '13', '3', '17'
    ],
    [
        '1',  '8', '15', '7', '18', '13', '3',  '9',
        '12', '5', '11', '6', '16', '2',  '10', '17'
    ],
    [
        '1', '8',  '15', '7',  '18', '13', '4', '12',
        '5', '11', '6',  '16', '2',  '9',  '3', '17'
    ],
    [
        '1', '8',  '15', '7',  '18', '16', '6', '11',
        '5', '12', '4',  '10', '2',  '9',  '3', '17'
    ],
    [
        '1', '8',  '15', '7', '18', '16', '6',  '11',
        '5', '12', '9',  '3', '13', '4',  '10', '17'
    ],
    [
        '1', '8',  '15', '7',  '14', '11', '5', '12',
        '4', '10', '2',  '16', '18', '13', '3', '17'
    ],
    [
        '1', '8',  '15', '7',  '14', '11', '5', '12',
        '4', '13', '18', '16', '2',  '9',  '3', '17'
    ],
    [
        '1', '8', '15', '7',  '14', '11', '5',  '12',
        '9', '2', '16', '18', '13', '4',  '10', '17'
    ],
    [
        '1', '8', '15', '7',  '14', '11', '5',  '12',
        '9', '3', '13', '18', '16', '2',  '10', '17'
    ],
    [
        '1',  '8',  '15', '7', '14', '11', '6',  '16',
        '18', '13', '3',  '9', '12', '4',  '10', '17'
    ],
    [
        '1',  '8',  '15', '7',  '14', '11', '6', '16',
        '18', '13', '4',  '10', '2',  '9',  '3', '17'
    ],
    [
        '1',  '8',  '15', '7',  '14', '11', '6',  '16',
        '18', '13', '4',  '12', '9',  '2',  '10', '17'
    ],
    [
        '1', '8', '15', '7',  '14', '11', '6', '16',
        '2', '9', '3',  '13', '4',  '12', '5', '17'
    ],
    [
        '1',  '14', '7',  '18', '13', '3', '9',  '2',
        '16', '6',  '11', '5',  '12', '4', '10', '17'
    ],
    [
        '1', '14', '7', '18', '13', '3',  '9', '12',
        '4', '10', '2', '16', '6',  '11', '5', '17'
    ],
    [
        '1', '14', '7', '18', '13', '3',  '9', '12',
        '4', '10', '2', '16', '6',  '15', '8', '17'
    ],
    [
        '1', '14', '7', '18', '13', '4',  '10', '2',
        '9', '12', '5', '11', '6',  '15', '8',  '17'
    ],
    [
        '1',  '14', '7',  '18', '13', '4', '10', '2',
        '16', '6',  '11', '5',  '12', '9', '3',  '17'
    ],
    [
        '1',  '14', '7',  '18', '16', '6', '11', '5',
        '12', '4',  '13', '3',  '9',  '2', '10', '17'
    ],
    [
        '1',  '14', '7', '18', '16', '6',  '11', '5',
        '12', '9',  '2', '10', '4',  '13', '3',  '17'
    ],
    [
        '1', '14', '7', '15', '6',  '11', '5', '12',
        '4', '10', '2', '16', '18', '13', '3', '17'
    ],
    [
        '1', '14', '7',  '15', '6', '11', '5', '12',
        '4', '13', '18', '16', '2', '9',  '3', '17'
    ],
    [
        '1', '14', '7',  '15', '6',  '11', '5',  '12',
        '9', '2',  '16', '18', '13', '4',  '10', '17'
    ],
    [
        '1', '14', '7',  '15', '6',  '11', '5',  '12',
        '9', '3',  '13', '18', '16', '2',  '10', '17'
    ],
    [
        '1', '14', '7', '15', '6', '16', '18', '13',
        '3', '9',  '2', '10', '4', '12', '5',  '17'
    ],
    [
        '1', '14', '11', '5',  '12', '4',  '10', '2',
        '9', '3',  '13', '18', '7',  '15', '8',  '17'
    ],
    [
        '1',  '14', '11', '5', '12', '4',  '10', '2',
        '16', '6',  '15', '7', '18', '13', '3',  '17'
    ],
    [
        '1', '14', '11', '5',  '12', '4', '13', '18',
        '7', '15', '6',  '16', '2',  '9', '3',  '17'
    ],
    [
        '1', '14', '11', '5',  '12', '4',  '13', '3',
        '9', '2',  '16', '18', '7',  '15', '8',  '17'
    ],
    [
        '1', '14', '11', '5',  '12', '9',  '2', '10',
        '4', '13', '18', '16', '6',  '15', '8', '17'
    ],
    [
        '1', '14', '11', '5',  '12', '9', '2',  '16',
        '6', '15', '7',  '18', '13', '4', '10', '17'
    ],
    [
        '1',  '14', '11', '5', '12', '9', '3',  '13',
        '18', '7',  '15', '6', '16', '2', '10', '17'
    ],
    [
        '1', '14', '11', '5',  '12', '9',  '3', '13',
        '4', '10', '2',  '16', '6',  '15', '8', '17'
    ],
    [
        '1', '14', '11', '6',  '15', '7',  '18', '13',
        '3', '9',  '2',  '10', '4',  '12', '5',  '17'
    ],
    [
        '1', '14', '11', '6',  '15', '7',  '18', '16',
        '2', '9',  '3',  '13', '4',  '12', '5',  '17'
    ],
    [
        '1',  '16', '18', '7', '15', '6', '11', '5',
        '12', '4',  '13', '3', '9',  '2', '10', '17'
    ],
    [
        '1',  '16', '18', '7',  '15', '6',  '11', '5',
        '12', '9',  '2',  '10', '4',  '13', '3',  '17'
    ],
    [
        '1', '16', '18', '13', '3', '9',  '2', '10',
        '4', '12', '5',  '11', '6', '15', '8', '17'
    ],
    [
        '1',  '16', '18', '13', '4', '10', '2', '9',
        '12', '5',  '11', '14', '7', '15', '8', '17'
    ],
    [
        '1',  '16', '2',  '9', '3',  '13', '18', '7',
        '15', '6',  '11', '5', '12', '4',  '10', '17'
    ],
    [
        '1',  '16', '2', '10', '4', '12', '9', '3',
        '13', '18', '7', '15', '6', '11', '5', '17'
    ],
    [
        '1',  '16', '2',  '10', '4',  '13', '18', '7',
        '15', '6',  '11', '5',  '12', '9',  '3',  '17'
    ],
    [
        '1',  '16', '2',  '10', '4', '13', '3', '9',
        '12', '5',  '11', '14', '7', '15', '8', '17'
    ],
    [
        '1',  '16', '6',  '11', '5', '12', '9', '2',
        '10', '4',  '13', '18', '7', '15', '8', '17'
    ],
    [
        '1', '16', '6', '11', '14', '7',  '18', '13',
        '3', '9',  '2', '10', '4',  '12', '5',  '17'
    ],
    [
        '1',  '16', '6',  '15', '7', '14', '11', '5',
        '12', '4',  '13', '3',  '9', '2',  '10', '17'
    ],
    [
        '1',  '16', '6', '15', '7', '14', '11', '5',
        '12', '9',  '2', '10', '4', '13', '3',  '17'
    ],
    [
        '1', '8', '15', '6', '11', '14', '7', '18', '13', '3',
        '9', '2', '10', '4', '12', '5',  '17'
    ],
    [
        '1', '8', '15', '6', '11', '14', '7', '18', '16', '2',
        '9', '3', '13', '4', '12', '5',  '17'
    ],
    [
        '1',  '8', '15', '6', '16', '18', '7', '14', '11', '5',
        '12', '4', '10', '2', '9',  '3',  '17'
    ],
    [
        '1',  '8', '15', '6',  '16', '18', '7', '14', '11', '5',
        '12', '9', '3',  '13', '4',  '10', '17'
    ],
    [
        '1',  '8', '15', '6',  '16', '2',  '9', '12', '5', '11',
        '14', '7', '18', '13', '4',  '10', '17'
    ],
    [
        '1',  '8',  '15', '6',  '16', '2', '10', '4', '12', '5',
        '11', '14', '7',  '18', '13', '3', '17'
    ],
    [
        '1', '8',  '15', '7',  '18', '13', '3', '9', '2', '16',
        '6', '11', '5',  '12', '4',  '10', '17'
    ],
    [
        '1',  '8', '15', '7', '18', '13', '3', '9', '12', '4',
        '10', '2', '16', '6', '11', '5',  '17'
    ],
    [
        '1', '8',  '15', '7',  '18', '13', '4', '10', '2', '16',
        '6', '11', '5',  '12', '9',  '3',  '17'
    ],
    [
        '1', '8',  '15', '7', '18', '16', '6', '11', '5', '12',
        '4', '13', '3',  '9', '2',  '10', '17'
    ],
    [
        '1', '8', '15', '7', '18', '16', '6', '11', '5', '12',
        '9', '2', '10', '4', '13', '3',  '17'
    ],
    [
        '1', '8',  '15', '7', '14', '11', '6', '16', '18', '13',
        '4', '10', '2',  '9', '12', '5',  '17'
    ],
    [
        '1', '8',  '15', '7', '14', '11', '6', '16', '2', '10',
        '4', '13', '3',  '9', '12', '5',  '17'
    ],
    [
        '1',  '14', '7',  '18', '13', '3', '9', '2', '10', '4',
        '12', '5',  '11', '6',  '15', '8', '17'
    ],
    [
        '1',  '14', '7',  '18', '16', '2', '9', '3', '13', '4',
        '12', '5',  '11', '6',  '15', '8', '17'
    ],
    [
        '1',  '14', '11', '5', '12', '4', '10', '2', '9', '3',
        '13', '18', '16', '6', '15', '8', '17'
    ],
    [
        '1', '14', '11', '5', '12', '9', '3', '13', '4', '10',
        '2', '16', '18', '7', '15', '8', '17'
    ],
    [
        '1', '14', '11', '6', '15', '7', '18', '16', '2', '10',
        '4', '13', '3',  '9', '12', '5', '17'
    ],
    [
        '1', '14', '11', '6', '16', '2', '10', '4', '12', '9',
        '3', '13', '18', '7', '15', '8', '17'
    ],
    [
        '1', '16', '18', '13', '3',  '9', '2', '10', '4', '12',
        '5', '11', '14', '7',  '15', '8', '17'
    ],
    [
        '1', '16', '2',  '10', '4',  '12', '9', '3', '13', '18',
        '7', '14', '11', '6',  '15', '8',  '17'
    ],
    [
        '1', '16', '6',  '11', '5',  '12', '4', '10', '2', '9',
        '3', '13', '18', '7',  '15', '8',  '17'
    ],
    [
        '1',  '8', '15', '6', '11', '14', '7', '18', '16', '2',
        '10', '4', '13', '3', '9',  '12', '5', '17'
    ],
    [
        '1',  '8', '15', '6', '16', '18', '7',  '14', '11', '5',
        '12', '4', '13', '3', '9',  '2',  '10', '17'
    ],
    [
        '1',  '8', '15', '6',  '16', '18', '7', '14', '11', '5',
        '12', '9', '2',  '10', '4',  '13', '3', '17'
    ],
    [
        '1', '8',  '15', '6', '16', '2', '9',  '3', '13', '18',
        '7', '14', '11', '5', '12', '4', '10', '17'
    ],
    [
        '1', '8',  '15', '6', '16', '2',  '10', '4', '12', '9',
        '3', '13', '18', '7', '14', '11', '5',  '17'
    ],
    [
        '1', '8',  '15', '6', '16', '2', '10', '4', '13', '18',
        '7', '14', '11', '5', '12', '9', '3',  '17'
    ],
    [
        '1', '8', '15', '7',  '14', '11', '6', '16', '18', '13',
        '3', '9', '2',  '10', '4',  '12', '5', '17'
    ],
    [
        '1', '14', '7', '18', '16', '2',  '10', '4', '13', '3',
        '9', '12', '5', '11', '6',  '15', '8',  '17'
    ]
];

is_deeply(\@got3, $expected3,
    "Another example from my Hamiltonian path tests");

# Local Variables:
# compile-command: "cd .. && perl t/31-all-paths.t"
# End:
