package Text::AsciiTeX;

use strict;
use warnings;

use parent 'Exporter';

our $VERSION = '0.05';
$VERSION = eval $VERSION;

require XSLoader;
XSLoader::load('Text::AsciiTeX', $VERSION);

our @EXPORT = ( qw/ render / );

sub render {
  my ($eq, $columns) = @_;

  # defaults
  $eq ||= '';
  $columns = (defined $columns) ? $columns : 80;

  my $text_array = c_render($eq, $columns);

  my $wantarray = wantarray;

  if ($wantarray) {
    return @$text_array;
  }

  my $text_string = join( "\n", @$text_array ) . "\n";

  if (defined $wantarray) {
    return $text_string;
  } 

  print $text_string;
  return;
}

1;

__END__

__POD__

=head1 NAME

Text::AsciiTeX - Convert (La)TeX formulas to ASCII art

=head1 SYNOPSIS

 use Text::AsciiArt;

 #equivalent examples

 my @text_array = render('\frac{1}{e}');
 print "$_\n" for @text_array;

 my $text_array = render('\frac{1}{e}');
 print $text_array;

 print scalar render('\frac{1}{e}');

 render('\frac{1}{e}');

=head1 DESCRIPTION

This module provides a mechanism to render (La)TeX formulae to ASCII art. It is based solely on F<AsciiTeX> written by Bart Pieters (See L</"UNDERLYING TECHNOLOGIES">).

=head1 EXPORTED FUNCTION

This module exports the C<render> function.

=head2 C<< render( $latex [, $columns] ) >>

=head3 Argument(s)

The function C<render> accepts a string containing a formula in (La)TeX formatting. Optionally, an integer may be given to specify the number of columns for the output or zero for no-breaking. The default number of columns is 80.

=head3 Return

Since version 0.03 the return value is context aware.

=over

=item *

In list context, C<render> returns a list whose elements are strings, one for each row of the art. Printing each line, terminated by a newline will probably do what you expect.

=item *

In scalar context, C<render> will return a string of the concatenated lines, each ended with a newline.

=item *

In void context, C<render> will print the scalar context return directly to the C<select>ed file handle (usually C<STDOUT>).

=back

=head1 EXAMPLES

For use examples see L</SYNOPSIS>. For a list of allowed syntax and syntax examples read L<Text::AsciiTeX::Syntax>.

=head1 UNDERLYING TECHNOLOGIES

This module is basically just a C-level Perl wrapper of F<AsciiTeX> written by Bart Pieters. That project is hosted at L<http://asciitex.sourceforge.net/>. F<AsciiTeX> in turn was a fork of F<eqascii> which was written by Przemek Borys. This module owes a debt of thanks to both authors.

=head1 SOURCE REPOSITORY

L<http://github.com/jberger/Text::AsciiTeX>

=head1 AUTHOR

Joel Berger, E<lt>joel.a.berger@gmail.comE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2012 by Joel Berger

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.
