#
# $Id: LLC.pm,v 1.1 2007/01/03 21:51:25 gomor Exp $
#
package Net::Frame::Layer::LLC;
use strict;
use warnings;

our $VERSION = '1.00';

use Net::Frame::Layer qw(:consts);
require Exporter;
our @ISA = qw(Net::Frame::Layer Exporter);

our %EXPORT_TAGS = (
   consts => [qw(
      NF_LLC_HDR_LEN
      NF_LLC_OUI_CISCO
      NF_LLC_PID_CDP
      NF_LLC_PID_STP
      NF_LLC_DSAP_SNAP
      NF_LLC_SSAP_SNAP
   )],
);
our @EXPORT_OK = (
   @{$EXPORT_TAGS{consts}},
);

use constant NF_LLC_HDR_LEN   => 8;
use constant NF_LLC_OUI_CISCO => 0x00000c;
use constant NF_LLC_PID_CDP   => 0x2000;
use constant NF_LLC_PID_STP   => 0x010b;
use constant NF_LLC_DSAP_SNAP => 0x2a;
use constant NF_LLC_SSAP_SNAP => NF_LLC_DSAP_SNAP();

our @AS = qw(
   dsap
   ig
   ssap
   cr
   control
   oui
   pid
);
__PACKAGE__->cgBuildIndices;
__PACKAGE__->cgBuildAccessorsScalar(\@AS);

no strict 'vars';

require Bit::Vector;

sub new {
   shift->SUPER::new(
      dsap    => NF_LLC_DSAP_SNAP,
      ig      => 1,
      ssap    => NF_LLC_SSAP_SNAP,
      cr      => 1,
      control => 0x03,
      oui     => NF_LLC_OUI_CISCO,
      pid     => NF_LLC_PID_CDP,
      @_,
   );
}

sub getLength { NF_LLC_HDR_LEN }

sub pack {
   my $self = shift;

   my $dsap = Bit::Vector->new_Dec(7, $self->[$__dsap]);
   my $ig   = Bit::Vector->new_Dec(1, $self->[$__ig]);
   my $ssap = Bit::Vector->new_Dec(7, $self->[$__ssap]);
   my $cr   = Bit::Vector->new_Dec(1, $self->[$__cr]);
   my $v16  = $dsap->Concat_List($ig, $ssap, $cr);

   my $oui = Bit::Vector->new_Dec(24, $self->[$__oui]);

   $self->[$__raw] = $self->SUPER::pack('nCB24n',
      $v16->to_Dec,
      $self->[$__control],
      $oui->to_Bin,
      $self->[$__pid],
   ) or return undef;

   $self->[$__raw];
}

sub unpack {
   my $self = shift;

   my ($dsapIgSsapCr, $control, $oui, $pid, $payload) =
      $self->SUPER::unpack('nCB24n a*', $self->[$__raw])
         or return undef;

   my $v16 = Bit::Vector->new_Dec(16, $dsapIgSsapCr);
   $self->[$__dsap] = $v16->Chunk_Read(7, 0);
   $self->[$__ig]   = $v16->Chunk_Read(1, 7);
   $self->[$__ssap] = $v16->Chunk_Read(7, 8);
   $self->[$__cr]   = $v16->Chunk_Read(1, 15);

   $self->[$__control] = $control;

   my $v24 = Bit::Vector->new_Bin(24, $oui);
   $self->[$__oui] = $v24->to_Dec;

   $self->[$__pid]     = $pid;
   $self->[$__payload] = $payload;

   $self;
}

sub encapsulate {
   my $self = shift;

   return $self->[$__nextLayer] if $self->[$__nextLayer];

   my $types = {
      NF_LLC_PID_CDP() => 'CDP',
      NF_LLC_PID_STP() => 'STP',
   };

   $types->{$self->[$__pid]} || NF_LAYER_UNKNOWN;
}

sub print {
   my $self = shift;

   my $l = $self->layer;
   sprintf "$l: dsap:0x%02x  ig:%d  ssap:0x%02x  cr:%d  control:0x%02x\n".
           "$l: oui:0x%06x  pid:0x%04x",
      $self->[$__dsap], $self->[$__ig], $self->[$__ssap], $self->[$__cr],
      $self->[$__control], $self->[$__oui], $self->[$__pid];
}

1;

__END__

=head1 NAME

Net::Frame::Layer::LLC - Logical-Link Control layer object

=head1 SYNOPSIS

   use Net::Frame::Layer::LLC qw(:consts);

   # Build a layer
   my $layer = Net::Frame::Layer::LLC->new(
      dsap    => NF_LLC_DSAP_SNAP,
      ig      => 1,
      ssap    => NF_LLC_SSAP_SNAP,
      cr      => 1,
      control => 0x03,
      oui     => NF_LLC_OUI_CISCO,
      pid     => NF_LLC_PID_CDP,
   );
   $layer->pack;

   print 'RAW: '.$layer->dump."\n";

   # Read a raw layer
   my $layer = Net::Frame::Layer::LLC->new(raw => $raw);

   print $layer->print."\n";
   print 'PAYLOAD: '.unpack('H*', $layer->payload)."\n"
      if $layer->payload;

=head1 DESCRIPTION

This modules implements the encoding and decoding of the Logical-Link Control layer.

See also B<Net::Frame::Layer> for other attributes and methods.

=head1 ATTRIBUTES

=over 4

=item B<dsap> - 7 bits

=item B<ig> - 1 bit

=item B<ssap> - 7 bits

=item B<cr> - 1 bit

=item B<control> - 8 bits

=item B<oui> - 24 bits

=item B<pid> - 16 bits

=back

The following are inherited attributes. See B<Net::Frame::Layer> for more information.

=over 4

=item B<raw>

=item B<payload>

=item B<nextLayer>

=back

=head1 METHODS

=over 4

=item B<new>

=item B<new> (hash)

Object constructor. You can pass attributes that will overwrite default ones. See B<SYNOPSIS> for default values.

=back

The following are inherited methods. Some of them may be overriden in this layer, and some others may not be meaningful in this layer. See B<Net::Frame::Layer> for more information.

=over 4

=item B<layer>

=item B<computeLengths>

=item B<computeChecksums>

=item B<pack>

=item B<unpack>

=item B<encapsulate>

=item B<getLength>

=item B<getPayloadLength>

=item B<print>

=item B<dump>

=back

=head1 CONSTANTS

Load them: use Net::Frame::Layer::LLC qw(:consts);

=over 4

=item B<NF_LLC_HDR_LEN>

LLC header length.

=item B<NF_LLC_OUI_CISCO>

Oui attribute constants.

=item B<NF_LLC_PID_CDP>

=item B<NF_LLC_PID_STP>

Pid attribute constants.

=item B<NF_LLC_DSAP_SNAP>

Dsap attribute constants.

=item B<NF_LLC_SSAP_SNAP>

Ssap attribute constants.

=back

=head1 SEE ALSO

L<Net::Frame::Layer>

=head1 AUTHOR

Patrice E<lt>GomoRE<gt> Auffret

=head1 COPYRIGHT AND LICENSE

Copyright (c) 2006-2007, Patrice E<lt>GomoRE<gt> Auffret

You may distribute this module under the terms of the Artistic license.
See LICENSE.Artistic file in the source distribution archive.

=cut
