/* -----------------------------------------------------------------------------
 *
 * (c) The GHC Team, 1998-2021
 *
 * Support for mapping info table pointers to source locations
 *
 * ---------------------------------------------------------------------------*/


#include "rts/PosixSource.h"
#include "Rts.h"

#include "Capability.h"
#include "Hash.h"
#include "IPE.h"
#include "Printer.h"
#include "Profiling.h"
#include "RtsUtils.h"

#include <fs_rts.h>
#include <string.h>

#if HAVE_LIBZSTD == 1
#include <zstd.h>
#endif

#if defined(TRACING)
#include "Trace.h"
#endif

/*
Note [The Info Table Provenance Entry (IPE) Map]
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
IPEs are stored in a hash map from info table address (pointer) to IPE. This
ensures cheap lookup and traversal.

Unfortunately, inserting into the hash map is relatively expensive. To keep
startup times low, there's a temporary data structure that is optimized for
collecting IPE lists on registration.

It's a singly linked list of IPE list buffers (IpeBufferListNode). These are
emitted by the code generator, with generally one produced per module. Each
contains a pointer to a list of IPE entries, a pointer to a list of info
table pointers, and a link field (which is used to link buffers onto the
pending list.

For reasons of space efficiency, IPE entries are represented slightly
differently in the object file than the InfoProvEnt which we ultimately expose
to the user. Specifically, the IPEs in IpeBufferListNode are represented by
IpeBufferEntrys, along with a corresponding string table. The string fields
of InfoProvEnt are represented in IpeBufferEntry as 32-bit offsets into the
string table. This allows us to halve the size of the buffer entries on
64-bit machines while significantly reducing the number of needed
relocations, reducing linking cost. Moreover, the code generator takes care
to deduplicate strings when generating the string table. When we insert a
set of IpeBufferEntrys into the IPE hash-map we convert them to InfoProvEnts,
which contain proper string pointers.

Building the hash map is done lazily, i.e. on first lookup or traversal. For
this all IPE lists of all IpeBufferListNode are traversed to insert all IPEs.

After the content of a IpeBufferListNode has been inserted, it's freed.
*/

#if defined(THREADED_RTS)
static Mutex ipeMapLock;
#endif
// Protected by ipeMapLock
static HashTable *ipeMap = NULL;

// Accessed atomically
static IpeBufferListNode *ipeBufferList = NULL;

#if defined(THREADED_RTS)

void initIpe(void) { initMutex(&ipeMapLock); }

void exitIpe(void) { closeMutex(&ipeMapLock); }

#else

void initIpe(void) { }

void exitIpe(void) { }

#endif // THREADED_RTS

static InfoProvEnt ipeBufferEntryToIpe(const char *strings, const StgInfoTable *tbl, const IpeBufferEntry ent)
{
    return (InfoProvEnt) {
            .info = tbl,
            .prov = {
                .table_name = &strings[ent.table_name],
                .closure_desc = &strings[ent.closure_desc],
                .ty_desc = &strings[ent.ty_desc],
                .label = &strings[ent.label],
                .module = &strings[ent.module_name],
                .src_file = &strings[ent.src_file],
                .src_span = &strings[ent.src_span]
            }
    };
}


#if defined(TRACING)
static void traceIPEFromHashTable(void *data STG_UNUSED, StgWord key STG_UNUSED,
                                  const void *value) {
    InfoProvEnt *ipe = (InfoProvEnt *)value;
    traceIPE(ipe);
}

void dumpIPEToEventLog(void) {
    // Dump pending entries
    IpeBufferListNode *cursor = RELAXED_LOAD(&ipeBufferList);
    while (cursor != NULL) {
        IpeBufferEntry *entries;
        char *strings;

        // Decompress if compressed
        decompressIPEBufferListNodeIfCompressed(cursor, &entries, &strings);

        for (uint32_t i = 0; i < cursor->count; i++) {
            const InfoProvEnt ent = ipeBufferEntryToIpe(
                strings,
                cursor->tables[i],
                entries[i]
            );
            traceIPE(&ent);
        }
        cursor = cursor->next;
    }

    // Dump entries already in hashmap
    ACQUIRE_LOCK(&ipeMapLock);
    if (ipeMap != NULL) {
        mapHashTable(ipeMap, NULL, &traceIPEFromHashTable);
    }
    RELEASE_LOCK(&ipeMapLock);
}


#else

void dumpIPEToEventLog(void) { }

#endif

/* Registering IPEs

Adds the ent_list to the temporary buffer structure described in
Note [The Info Table Provenance Entry (IPE) Map].

Statically initialized IPE lists are registered at startup by a C constructor
function generated by the compiler (CodeOutput.hs) in a *.c file for each
module. Since this is called in a static initializer we cannot rely on
ipeMapLock; we instead use atomic CAS operations to add to the list.

A performance test for IPE registration and lookup can be found here:
https://gitlab.haskell.org/ghc/ghc/-/merge_requests/5724#note_370806
*/
void registerInfoProvList(IpeBufferListNode *node) {
    while (true) {
        IpeBufferListNode *old = RELAXED_LOAD(&ipeBufferList);
        node->next = old;
        if (cas_ptr((volatile void **) &ipeBufferList, old, node) == (void *) old) {
            return;
        }
    }
}

InfoProvEnt *lookupIPE(const StgInfoTable *info) {
    updateIpeMap();
    return lookupHashTable(ipeMap, (StgWord)info);
}

void updateIpeMap(void) {
    // Check if there's any work at all. If not so, we can circumvent locking,
    // which decreases performance.
    IpeBufferListNode *pending = xchg_ptr((void **) &ipeBufferList, NULL);
    if (ipeMap != NULL && pending == NULL) {
        return;
    }

    ACQUIRE_LOCK(&ipeMapLock);

    if (ipeMap == NULL) {
        ipeMap = allocHashTable();
    }

    while (pending != NULL) {
        IpeBufferListNode *current_node = pending;
        IpeBufferEntry *entries;
        char *strings;

        // Decompress if compressed
        decompressIPEBufferListNodeIfCompressed(current_node, &entries, &strings);

        // Convert the on-disk IPE buffer entry representation (IpeBufferEntry)
        // into the runtime representation (InfoProvEnt)
        InfoProvEnt *ip_ents = stgMallocBytes(
            sizeof(InfoProvEnt) * current_node->count,
            "updateIpeMap: ip_ents"
        );
        for (uint32_t i = 0; i < current_node->count; i++) {
            const IpeBufferEntry ent = entries[i];
            const StgInfoTable *tbl = current_node->tables[i];
            ip_ents[i] = ipeBufferEntryToIpe(strings, tbl, ent);
            insertHashTable(ipeMap, (StgWord) tbl, &ip_ents[i]);
        }

        pending = current_node->next;
    }

    RELEASE_LOCK(&ipeMapLock);
}

/* Decompress the IPE data and strings table referenced by an IPE buffer list
node if it is compressed. No matter whether the data is compressed, the pointers
referenced by the 'entries_dst' and 'string_table_dst' parameters will point at
the decompressed IPE data and string table for the given node, respectively,
upon return from this function.
*/
void decompressIPEBufferListNodeIfCompressed(IpeBufferListNode *node, IpeBufferEntry **entries_dst, char **string_table_dst) {
    if (node->compressed == 1) {
        // The IPE list buffer node indicates that the strings table and
        // entries list has been compressed. If zstd is not available, fail.
        // If zstd is available, decompress.
#if HAVE_LIBZSTD == 0
        barf("An IPE buffer list node has been compressed, but the "
             "decompression library (zstd) is not available."
);
#else
        size_t compressed_sz = ZSTD_findFrameCompressedSize(
            node->string_table,
            node->string_table_size
        );
        char *decompressed_strings = stgMallocBytes(
            node->string_table_size,
            "updateIpeMap: decompressed_strings"
        );
        ZSTD_decompress(
            decompressed_strings,
            node->string_table_size,
            node->string_table,
            compressed_sz
        );
        *string_table_dst = decompressed_strings;

        // Decompress the IPE data
        compressed_sz = ZSTD_findFrameCompressedSize(
            node->entries,
            node->entries_size
        );
        void *decompressed_entries = stgMallocBytes(
            node->entries_size,
            "updateIpeMap: decompressed_entries"
        );
        ZSTD_decompress(
            decompressed_entries,
            node->entries_size,
            node->entries,
            compressed_sz
        );
        *entries_dst = decompressed_entries;
#endif // HAVE_LIBZSTD == 0

    } else {
        // Not compressed, no need to decompress
        *entries_dst = node->entries;
        *string_table_dst = node->string_table;
    }
}
