package models

import (
	"sort"

	"github.com/anchore/grype/grype/version"
	"github.com/anchore/grype/grype/vulnerability"
	"github.com/anchore/grype/internal/log"
)

type Vulnerability struct {
	VulnerabilityMetadata
	Fix        Fix        `json:"fix"`
	Advisories []Advisory `json:"advisories"`
}

type Fix struct {
	Versions []string `json:"versions"`
	State    string   `json:"state"`
}

type Advisory struct {
	ID   string `json:"id"`
	Link string `json:"link"`
}

func NewVulnerability(vuln vulnerability.Vulnerability, metadata *vulnerability.Metadata, versionFormat version.Format) Vulnerability {
	if metadata == nil {
		return Vulnerability{
			VulnerabilityMetadata: NewVulnerabilityMetadata(vuln.ID, vuln.Namespace, metadata),
		}
	}

	advisories := make([]Advisory, len(vuln.Advisories))
	for idx, advisory := range vuln.Advisories {
		advisories[idx] = Advisory{
			ID:   advisory.ID,
			Link: advisory.Link,
		}
	}

	fixedInVersions := vuln.Fix.Versions
	if fixedInVersions == nil {
		// always allocate collections
		fixedInVersions = make([]string, 0)
	}

	return Vulnerability{
		VulnerabilityMetadata: NewVulnerabilityMetadata(vuln.ID, vuln.Namespace, metadata),
		Fix: Fix{
			Versions: sortVersions(fixedInVersions, versionFormat),
			State:    string(vuln.Fix.State),
		},
		Advisories: advisories,
	}
}
func sortVersions(fixedVersions []string, format version.Format) []string {
	if len(fixedVersions) <= 1 {
		return fixedVersions
	}

	// First, create Version objects from strings (only once)
	versionObjs := make([]*version.Version, 0, len(fixedVersions))
	for _, vStr := range fixedVersions {
		v, err := version.NewVersion(vStr, format)
		if err != nil {
			log.WithFields("version", vStr, "error", err).Trace("error parsing version, skipping")
			continue
		}
		versionObjs = append(versionObjs, v)
	}

	// Sort the Version objects
	sort.Slice(versionObjs, func(i, j int) bool {
		// Compare returns -1 if v[i] < v[j], so we negate for descending order
		// (higher versions first)
		comparison, err := versionObjs[i].Compare(versionObjs[j])
		if err != nil {
			log.WithFields("error", err).Trace("error comparing versions")
			return false
		}
		return comparison > 0 // Descending order
	})

	// Convert back to strings
	result := make([]string, len(versionObjs))
	for i, v := range versionObjs {
		result[i] = v.Raw
	}

	return result
}
