/*************************************************************************************************
 * The command line utility of the core API
 *                                                               Copyright (C) 2007-2010 FAL Labs
 * This file is part of Tokyo Dystopia.
 * Tokyo Dystopia is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License or any later version.  Tokyo Dystopia is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with Tokyo
 * Dystopia; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA.
 *************************************************************************************************/


#include <dystopia.h>
#include "myconf.h"

#define SEARCHWORDMAX  256               // maximum number of search words
#define DEFSEARCHMAX   10                // default maximum number of printed IDs

enum {                                   // enumeration for expression mode
  EMCOMP,                                // compound
  EMUNION,                               // union
  EMISECT,                               // intersection
  EMDIFF,                                // difference
};


/* global variables */
const char *g_progname;                  // program name
int g_dbgfd;                             // debugging output


/* function prototypes */
int main(int argc, char **argv);
static void usage(void);
static void printerr(TCIDB *idb);
static char *mygetline(FILE *ifp);
static bool mysynccb(int total, int current, const char *msg, void *opq);
static void printresult(TCIDB *idb, const uint64_t *res, int rnum, int max,
                        bool ph, bool pv, double etime);
static int runcreate(int argc, char **argv);
static int runinform(int argc, char **argv);
static int runput(int argc, char **argv);
static int runout(int argc, char **argv);
static int runget(int argc, char **argv);
static int runsearch(int argc, char **argv);
static int runlist(int argc, char **argv);
static int runoptimize(int argc, char **argv);
static int runimporttsv(int argc, char **argv);
static int runversion(int argc, char **argv);
static int proccreate(const char *path, int64_t ernum, int64_t etnum, int64_t iusiz, int opts);
static int procinform(const char *path, int omode);
static int procput(const char *path, int64_t id, const char *text, int omode);
static int procout(const char *path, int64_t id, int omode);
static int procget(const char *path, int64_t id, int omode);
static int procsearch(const char *path, const char **words, int wnum, int omode,
                      int emode, int smode, int max, bool ph, bool pv);
static int proclist(const char *path, int omode, int max, bool pv);
static int procoptimize(const char *path, int omode);
static int procimporttsv(const char *path, const char *file, int64_t icsiz, int omode);
static int procversion(void);


/* main routine */
int main(int argc, char **argv){
  g_progname = argv[0];
  g_dbgfd = -1;
  const char *ebuf = getenv("TCDBGFD");
  if(ebuf) g_dbgfd = tcatoi(ebuf);
  if(argc < 2) usage();
  int rv = 0;
  if(!strcmp(argv[1], "create")){
    rv = runcreate(argc, argv);
  } else if(!strcmp(argv[1], "inform")){
    rv = runinform(argc, argv);
  } else if(!strcmp(argv[1], "put")){
    rv = runput(argc, argv);
  } else if(!strcmp(argv[1], "out")){
    rv = runout(argc, argv);
  } else if(!strcmp(argv[1], "get")){
    rv = runget(argc, argv);
  } else if(!strcmp(argv[1], "search")){
    rv = runsearch(argc, argv);
  } else if(!strcmp(argv[1], "list")){
    rv = runlist(argc, argv);
  } else if(!strcmp(argv[1], "optimize")){
    rv = runoptimize(argc, argv);
  } else if(!strcmp(argv[1], "importtsv")){
    rv = runimporttsv(argc, argv);
  } else if(!strcmp(argv[1], "version") || !strcmp(argv[1], "--version")){
    rv = runversion(argc, argv);
  } else {
    usage();
  }
  return rv;
}


/* print the usage and exit */
static void usage(void){
  fprintf(stderr, "%s: the command line utility of the core API\n", g_progname);
  fprintf(stderr, "\n");
  fprintf(stderr, "usage:\n");
  fprintf(stderr, "  %s create [-tl] [-td|-tb|-tt] path [ernum [etnum]]\n", g_progname);
  fprintf(stderr, "  %s inform [-nl|-nb] path\n", g_progname);
  fprintf(stderr, "  %s put [-nl|-nb] path id text\n", g_progname);
  fprintf(stderr, "  %s out [-nl|-nb] path id\n", g_progname);
  fprintf(stderr, "  %s get [-nl|-nb] path id\n", g_progname);
  fprintf(stderr, "  %s search [-nl|-nb] [-eu|-ei|-ed] [-sp|-ss|-sf|-st|-stp|-sts] [-max num]"
          " [-ph] [-pv] path [word...]\n", g_progname);
  fprintf(stderr, "  %s list [-nl|-nb] [-max num] [-pv] path\n", g_progname);
  fprintf(stderr, "  %s optimize [-nl|-nb] path\n", g_progname);
  fprintf(stderr, "  %s importtsv [-ic num] [-nl|-nb] path [file]\n", g_progname);
  fprintf(stderr, "  %s version\n", g_progname);
  fprintf(stderr, "\n");
  exit(1);
}


/* print error information */
static void printerr(TCIDB *idb){
  const char *path = tcidbpath(idb);
  int ecode = tcidbecode(idb);
  fprintf(stderr, "%s: %s: %d: %s\n", g_progname, path ? path : "-", ecode, tcidberrmsg(ecode));
}


/* read a line from a file descriptor */
static char *mygetline(FILE *ifp){
  int len = 0;
  int blen = 1024;
  char *buf = tcmalloc(blen);
  bool end = true;
  int c;
  while((c = fgetc(ifp)) != EOF){
    end = false;
    if(c == '\0') continue;
    if(blen <= len){
      blen *= 2;
      buf = tcrealloc(buf, blen + 1);
    }
    if(c == '\n' || c == '\r') c = '\0';
    buf[len++] = c;
    if(c == '\0') break;
  }
  if(end){
    tcfree(buf);
    return NULL;
  }
  buf[len] = '\0';
  return buf;
}


/* callback function for sync progression */
static bool mysynccb(int total, int current, const char *msg, void *opq){
  if(total < 10 || current % (total / 10) == 0) printf("[sync:%d:%d:%s]\n", total, current, msg);
  return true;
}


/* print search result */
static void printresult(TCIDB *idb, const uint64_t *res, int rnum, int max,
                        bool ph, bool pv, double etime){
  if(ph){
    printf("hits: %d\n", rnum);
    printf("time: %.6f\n", etime);
    printf("\n");
  }
  max = tclmin(max, rnum);
  for(int i = 0; i < max; i++){
    if(pv){
      printf("%llu", (unsigned long long)res[i]);
      char *text = tcidbget(idb, res[i]);
      if(text){
        printf("\t%s", text);
        tcfree(text);
      }
      printf("\n");
    } else {
      printf("%llu\n", (unsigned long long)res[i]);
    }
  }
}


/* parse arguments of create command */
static int runcreate(int argc, char **argv){
  char *path = NULL;
  char *erstr = NULL;
  char *etstr = NULL;
  char *iustr = NULL;
  int opts = 0;
  for(int i = 2; i < argc; i++){
    if(!path && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-tl")){
        opts |= IDBTLARGE;
      } else if(!strcmp(argv[i], "-td")){
        opts |= IDBTDEFLATE;
      } else if(!strcmp(argv[i], "-tb")){
        opts |= IDBTBZIP;
      } else if(!strcmp(argv[i], "-tt")){
        opts |= IDBTTCBS;
      } else {
        usage();
      }
    } else if(!path){
      path = argv[i];
    } else if(!erstr){
      erstr = argv[i];
    } else if(!etstr){
      etstr = argv[i];
    } else if(!iustr){
      iustr = argv[i];
    } else {
      usage();
    }
  }
  if(!path) usage();
  int64_t ernum = erstr ? strtoll(erstr, NULL, 10) : -1;
  int64_t etnum = etstr ? strtoll(etstr, NULL, 10) : -1;
  int64_t iusiz = iustr ? strtoll(iustr, NULL, 10) : -1;
  int rv = proccreate(path, ernum, etnum, iusiz, opts);
  return rv;
}


/* parse arguments of inform command */
static int runinform(int argc, char **argv){
  char *path = NULL;
  int omode = 0;
  for(int i = 2; i < argc; i++){
    if(!path && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-nl")){
        omode |= IDBONOLCK;
      } else if(!strcmp(argv[i], "-nb")){
        omode |= IDBOLCKNB;
      } else {
        usage();
      }
    } else if(!path){
      path = argv[i];
    } else {
      usage();
    }
  }
  if(!path) usage();
  int rv = procinform(path, omode);
  return rv;
}


/* parse arguments of put command */
static int runput(int argc, char **argv){
  char *path = NULL;
  char *idstr = NULL;
  char *text = NULL;
  int omode = 0;
  for(int i = 2; i < argc; i++){
    if(!path && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-nl")){
        omode |= IDBONOLCK;
      } else if(!strcmp(argv[i], "-nb")){
        omode |= IDBOLCKNB;
      } else {
        usage();
      }
    } else if(!path){
      path = argv[i];
    } else if(!idstr){
      idstr = argv[i];
    } else if(!text){
      text = argv[i];
    } else {
      usage();
    }
  }
  if(!path || !idstr || !text) usage();
  int64_t id = strtoll(idstr, NULL, 10);
  if(id < 1) usage();
  int rv = procput(path, id, text, omode);
  return rv;
}


/* parse arguments of out command */
static int runout(int argc, char **argv){
  char *path = NULL;
  char *idstr = NULL;
  int omode = 0;
  for(int i = 2; i < argc; i++){
    if(!path && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-nl")){
        omode |= IDBONOLCK;
      } else if(!strcmp(argv[i], "-nb")){
        omode |= IDBOLCKNB;
      } else {
        usage();
      }
    } else if(!path){
      path = argv[i];
    } else if(!idstr){
      idstr = argv[i];
    } else {
      usage();
    }
  }
  if(!path || !idstr) usage();
  int64_t id = strtoll(idstr, NULL, 10);
  if(id < 1) usage();
  int rv = procout(path, id, omode);
  return rv;
}


/* parse arguments of get command */
static int runget(int argc, char **argv){
  char *path = NULL;
  char *idstr = NULL;
  int omode = 0;
  for(int i = 2; i < argc; i++){
    if(!path && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-nl")){
        omode |= IDBONOLCK;
      } else if(!strcmp(argv[i], "-nb")){
        omode |= IDBOLCKNB;
      } else {
        usage();
      }
    } else if(!path){
      path = argv[i];
    } else if(!idstr){
      idstr = argv[i];
    } else {
      usage();
    }
  }
  if(!path || !idstr) usage();
  int64_t id = strtoll(idstr, NULL, 10);
  if(id < 1) usage();
  int rv = procget(path, id, omode);
  return rv;
}


/* parse arguments of search command */
static int runsearch(int argc, char **argv){
  char *path = NULL;
  char *words[SEARCHWORDMAX];
  int wnum = 0;
  int omode = 0;
  int emode = EMCOMP;
  int smode = IDBSSUBSTR;
  int max = DEFSEARCHMAX;
  bool ph = false;
  bool pv = false;
  for(int i = 2; i < argc; i++){
    if(!path && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-nl")){
        omode |= IDBONOLCK;
      } else if(!strcmp(argv[i], "-nb")){
        omode |= IDBOLCKNB;
      } else if(!strcmp(argv[i], "-eu")){
        emode = EMUNION;
      } else if(!strcmp(argv[i], "-ei")){
        emode = EMISECT;
      } else if(!strcmp(argv[i], "-ed")){
        emode = EMDIFF;
      } else if(!strcmp(argv[i], "-sp")){
        smode = IDBSPREFIX;
      } else if(!strcmp(argv[i], "-ss")){
        smode = IDBSSUFFIX;
      } else if(!strcmp(argv[i], "-sf")){
        smode = IDBSFULL;
      } else if(!strcmp(argv[i], "-st")){
        smode = IDBSTOKEN;
      } else if(!strcmp(argv[i], "-stp")){
        smode = IDBSTOKPRE;
      } else if(!strcmp(argv[i], "-sts")){
        smode = IDBSTOKSUF;
      } else if(!strcmp(argv[i], "-max")){
        if(++i >= argc) usage();
        max = tcatoi(argv[i]);
      } else if(!strcmp(argv[i], "-ph")){
        ph = true;
      } else if(!strcmp(argv[i], "-pv")){
        pv = true;
      } else {
        usage();
      }
    } else if(!path){
      path = argv[i];
    } else {
      if(wnum < SEARCHWORDMAX) words[wnum++] = argv[i];
    }
  }
  if(!path || wnum < 1) usage();
  if(emode == EMCOMP && smode != IDBSSUBSTR) emode = EMISECT;
  if(max < 0) max = INT_MAX;
  int rv = procsearch(path, (const char **)words, wnum, omode, emode, smode, max, ph, pv);
  return rv;
}


/* parse arguments of optimize command */
static int runoptimize(int argc, char **argv){
  char *path = NULL;
  int omode = 0;
  for(int i = 2; i < argc; i++){
    if(!path && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-nl")){
        omode |= IDBONOLCK;
      } else if(!strcmp(argv[i], "-nb")){
        omode |= IDBOLCKNB;
      } else {
        usage();
      }
    } else if(!path){
      path = argv[i];
    } else {
      usage();
    }
  }
  if(!path) usage();
  int rv = procoptimize(path, omode);
  return rv;
}


/* parse arguments of list command */
static int runlist(int argc, char **argv){
  char *path = NULL;
  int omode = 0;
  int max = -1;
  bool pv = false;
  for(int i = 2; i < argc; i++){
    if(!path && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-nl")){
        omode |= IDBONOLCK;
      } else if(!strcmp(argv[i], "-nb")){
        omode |= IDBOLCKNB;
      } else if(!strcmp(argv[i], "-max")){
        if(++i >= argc) usage();
        max = tcatoi(argv[i]);
      } else if(!strcmp(argv[i], "-pv")){
        pv = true;
      } else {
        usage();
      }
    } else if(!path){
      path = argv[i];
    } else {
      usage();
    }
  }
  if(!path) usage();
  if(max < 0) max = INT_MAX;
  int rv = proclist(path, omode, max, pv);
  return rv;
}


/* parse arguments of importtsv command */
static int runimporttsv(int argc, char **argv){
  char *path = NULL;
  char *file = NULL;
  int64_t icsiz = 0;
  int omode = 0;
  for(int i = 2; i < argc; i++){
    if(!path && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-ic")){
        if(++i >= argc) usage();
        icsiz = tcatoix(argv[i]);
      } else if(!strcmp(argv[i], "-nl")){
        omode |= HDBONOLCK;
      } else if(!strcmp(argv[i], "-nb")){
        omode |= HDBOLCKNB;
      } else {
        usage();
      }
    } else if(!path){
      path = argv[i];
    } else if(!file){
      file = argv[i];
    } else {
      usage();
    }
  }
  if(!path) usage();
  int rv = procimporttsv(path, file, icsiz, omode);
  return rv;
}


/* parse arguments of version command */
static int runversion(int argc, char **argv){
  int rv = procversion();
  return rv;
}


/* perform create command */
static int proccreate(const char *path, int64_t ernum, int64_t etnum, int64_t iusiz, int opts){
  TCIDB *idb = tcidbnew();
  if(g_dbgfd >= 0) tcidbsetdbgfd(idb, g_dbgfd);
  if(!tcidbtune(idb, ernum, etnum, iusiz, opts)){
    printerr(idb);
    tcidbdel(idb);
    return 1;
  }
  if(!tcidbopen(idb, path, IDBOWRITER | IDBOCREAT | IDBOTRUNC)){
    printerr(idb);
    tcidbdel(idb);
    return 1;
  }
  bool err = false;
  if(!tcidbclose(idb)){
    printerr(idb);
    err = true;
  }
  tcidbdel(idb);
  return err ? 1 : 0;
}


/* perform inform command */
static int procinform(const char *path, int omode){
  TCIDB *idb = tcidbnew();
  if(g_dbgfd >= 0) tcidbsetdbgfd(idb, g_dbgfd);
  if(!tcidbopen(idb, path, IDBOREADER | omode)){
    printerr(idb);
    tcidbdel(idb);
    return 1;
  }
  bool err = false;
  const char *npath = tcidbpath(idb);
  if(!npath) npath = "(unknown)";
  printf("path: %s\n", npath);
  printf("database type: indexed\n");
  printf("inode number: %lld\n", (long long)tcidbinode(idb));
  char date[48];
  tcdatestrwww(tcidbmtime(idb), INT_MAX, date);
  printf("modified time: %s\n", date);
  uint8_t opts = tcidbopts(idb);
  printf("options:");
  if(opts & IDBTLARGE) printf(" large");
  if(opts & IDBTDEFLATE) printf(" deflate");
  if(opts & IDBTTCBS) printf(" tcbs");
  printf("\n");
  printf("record number: %llu\n", (unsigned long long)tcidbrnum(idb));
  printf("file size: %llu\n", (unsigned long long)tcidbfsiz(idb));
  if(!tcidbclose(idb)){
    if(!err) printerr(idb);
    err = true;
  }
  tcidbdel(idb);
  return err ? 1 : 0;
}


/* perform put command */
static int procput(const char *path, int64_t id, const char *text, int omode){
  TCIDB *idb = tcidbnew();
  if(g_dbgfd >= 0) tcidbsetdbgfd(idb, g_dbgfd);
  tcidbsetsynccb(idb, mysynccb, NULL);
  if(!tcidbopen(idb, path, IDBOWRITER | omode)){
    printerr(idb);
    tcidbdel(idb);
    return 1;
  }
  bool err = false;
  if(!tcidbput(idb, id, text)){
    printerr(idb);
    err = true;
  }
  if(!tcidbclose(idb)){
    if(!err) printerr(idb);
    err = true;
  }
  tcidbdel(idb);
  return err ? 1 : 0;
}


/* perform out command */
static int procout(const char *path, int64_t id, int omode){
  TCIDB *idb = tcidbnew();
  if(g_dbgfd >= 0) tcidbsetdbgfd(idb, g_dbgfd);
  tcidbsetsynccb(idb, mysynccb, NULL);
  if(!tcidbopen(idb, path, IDBOWRITER | omode)){
    printerr(idb);
    tcidbdel(idb);
    return 1;
  }
  bool err = false;
  if(!tcidbout(idb, id)){
    printerr(idb);
    err = true;
  }
  if(!tcidbclose(idb)){
    if(!err) printerr(idb);
    err = true;
  }
  tcidbdel(idb);
  return err ? 1 : 0;
}


/* perform get command */
static int procget(const char *path, int64_t id, int omode){
  TCIDB *idb = tcidbnew();
  if(g_dbgfd >= 0) tcidbsetdbgfd(idb, g_dbgfd);
  tcidbsetsynccb(idb, mysynccb, NULL);
  if(!tcidbopen(idb, path, IDBOREADER | omode)){
    printerr(idb);
    tcidbdel(idb);
    return 1;
  }
  bool err = false;
  char *text = tcidbget(idb, id);
  if(text){
    printf("%s\n", text);
    tcfree(text);
  } else {
    printerr(idb);
    err = true;
  }
  if(!tcidbclose(idb)){
    if(!err) printerr(idb);
    err = true;
  }
  tcidbdel(idb);
  return err ? 1 : 0;
}


/* perform search command */
static int procsearch(const char *path, const char **words, int wnum, int omode,
                      int emode, int smode, int max, bool ph, bool pv){
  TCIDB *idb = tcidbnew();
  if(g_dbgfd >= 0) tcidbsetdbgfd(idb, g_dbgfd);
  if(!tcidbopen(idb, path, IDBOREADER | omode)){
    printerr(idb);
    tcidbdel(idb);
    return 1;
  }
  bool err = false;
  if(emode == EMCOMP){
    TCXSTR *expr = tcxstrnew();
    for(int i = 0; i < wnum; i++){
      if(i > 0) tcxstrcat(expr, " ", 1);
      tcxstrcat2(expr, words[i]);
    }
    double stime = tctime();
    int rnum;
    uint64_t *res = tcidbsearch2(idb, tcxstrptr(expr), &rnum);
    if(res){
      printresult(idb, res, rnum, max, ph, pv, tctime() - stime);
      tcfree(res);
    } else {
      printerr(idb);
      err = true;
    }
    tcxstrdel(expr);
  } else if(wnum == 1){
    double stime = tctime();
    int rnum;
    uint64_t *res = tcidbsearch(idb, words[0], smode, &rnum);
    if(res){
      printresult(idb, res, rnum, max, ph, pv, tctime() - stime);
      tcfree(res);
    } else {
      printerr(idb);
      err = true;
    }
  } else {
    double stime = tctime();
    QDBRSET rsets[wnum];
    for(int i = 0; i < wnum; i++){
      rsets[i].ids = tcidbsearch(idb, words[i], smode, &rsets[i].num);
    }
    int rnum;
    uint64_t *res;
    switch(emode){
      case EMISECT:
        res = tcqdbresisect(rsets, wnum, &rnum);
        break;
      case EMDIFF:
        res = tcqdbresdiff(rsets, wnum, &rnum);
        break;
      default:
        res = tcqdbresunion(rsets, wnum, &rnum);
        break;
    }
    printresult(idb, res, rnum, max, ph, pv, tctime() - stime);
    tcfree(res);
    for(int i = 0; i < wnum; i++){
      tcfree(rsets[i].ids);
    }
  }
  if(!tcidbclose(idb)){
    if(!err) printerr(idb);
    err = true;
  }
  tcidbdel(idb);
  return err ? 1 : 0;
}


/* perform list command */
static int proclist(const char *path, int omode, int max, bool pv){
  TCIDB *idb = tcidbnew();
  if(g_dbgfd >= 0) tcidbsetdbgfd(idb, g_dbgfd);
  if(!tcidbopen(idb, path, IDBOREADER | omode)){
    printerr(idb);
    tcidbdel(idb);
    return 1;
  }
  bool err = false;
  if(!tcidbiterinit(idb)){
    printerr(idb);
    err = true;
  }
  uint64_t id;
  while(max-- > 0 && (id = tcidbiternext(idb)) > 0){
    if(pv){
      printf("%llu", (unsigned long long)id);
      char *text = tcidbget(idb, id);
      if(text){
        printf("\t%s", text);
        tcfree(text);
      }
      printf("\n");
    } else {
      printf("%llu\n", (unsigned long long)id);
    }
  }
  if(!tcidbclose(idb)){
    if(!err) printerr(idb);
    err = true;
  }
  tcidbdel(idb);
  return err ? 1 : 0;
}


/* perform optimize command */
static int procoptimize(const char *path, int omode){
  TCIDB *idb = tcidbnew();
  if(g_dbgfd >= 0) tcidbsetdbgfd(idb, g_dbgfd);
  if(!tcidbopen(idb, path, IDBOWRITER | omode)){
    printerr(idb);
    tcidbdel(idb);
    return 1;
  }
  bool err = false;
  if(!tcidboptimize(idb)){
    printerr(idb);
    err = true;
  }
  if(!tcidbclose(idb)){
    if(!err) printerr(idb);
    err = true;
  }
  tcidbdel(idb);
  return err ? 1 : 0;
}


/* perform importtsv command */
static int procimporttsv(const char *path, const char *file, int64_t icsiz, int omode){
  TCIDB *idb = tcidbnew();
  if(g_dbgfd >= 0) tcidbsetdbgfd(idb, g_dbgfd);
  FILE *ifp = file ? fopen(file, "rb") : stdin;
  if(!ifp){
    fprintf(stderr, "%s: could not open\n", file ? file : "(stdin)");
    tcidbdel(idb);
    return 1;
  }
  tcidbsetsynccb(idb, mysynccb, NULL);
  if(!tcidbsetcache(idb, icsiz, -1)) printerr(idb);
  if(!tcidbopen(idb, path, IDBOWRITER | IDBOCREAT | omode)){
    printerr(idb);
    tcidbdel(idb);
    return 1;
  }
  bool err = false;
  char *line;
  int cnt = 0;
  while(!err && (line = mygetline(ifp)) != NULL){
    int64_t id = strtoll(line, NULL, 10);
    char *pv = strchr(line, '\t');
    if(id == 0 || !pv){
      tcfree(line);
      continue;
    }
    *(pv++) = '\0';
    tcstrsqzspc(pv);
    if(id > 0){
      if(!tcidbput(idb, id, pv)){
        printerr(idb);
        err = true;
      }
    } else {
      if(!tcidbout(idb, -id)){
        printerr(idb);
        err = true;
      }
    }
    tcfree(line);
    if(cnt > 0 && cnt % 100 == 0){
      putchar('.');
      fflush(stdout);
      if(cnt % 5000 == 0) printf(" (%08d)\n", cnt);
    }
    cnt++;
  }
  printf(" (%08d)\n", cnt);
  if(!tcidbclose(idb)){
    if(!err) printerr(idb);
    err = true;
  }
  tcidbdel(idb);
  if(ifp != stdin) fclose(ifp);
  return err ? 1 : 0;
}


/* perform version command */
static int procversion(void){
  printf("Tokyo Dystopia version %s (%d:%s) for %s\n",
         tdversion, _TD_LIBVER, _TD_FORMATVER, TDSYSNAME);
  printf("Copyright (C) 2007-2010 FAL Labs\n");
  return 0;
}



// END OF FILE
