/*********************************************************************
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ===================================================================
 * Revision History ::
 * YYYY.MM.DD   Change ID       Developer
 *              Description
 * -------------------------------------------------------------------
 * 2002.04.25                   Vlad Skarzhevskyy
 *              Initial implementation.
 *
 * ===================================================================
 *
 ********************************************************************/

#include "pure-sfv.h"

int sfvCreate(pure_sfv_params* params, char* sfv_dir, crc_info_type* sfv_info)
{
    int                rc = 0;
    crc_info_type      dir_info;
    int                i;
    unsigned long      crc_val;
    char              *work_on_file;

    sfvInfoInit(&dir_info);

    if (sfv_dir[0] != '\0') {
        if (chdir(sfv_dir) != 0) {
            fprintf(stderr, "sfv: %s: %s\n", sfv_dir, strerror(errno));
            return ERROR_DIR_ACCESS;
        }
    }
    /* Read list of all Files  */
    rc = dirRead("", &dir_info);
    if (rc) return rc;
    rc = dirReadAttributes("", &dir_info);
    if (rc) return rc;

    DBUG_PRINT("sfvCreate", ("dir_info.len [%i]", dir_info.len));
    for(i = 0; i < dir_info.len; i++)
    {
        if (dir_info.items[i].is_dir) {
            DBUG_PRINT("sfvCreate", ("Skip dir"));
            continue;
        }
        work_on_file = dir_info.items[i].file_name;
        if (strcasecmp(sfv_info->name, work_on_file) == 0) {
            /* Exclude SFV itself */
            DBUG_PRINT("sfvCreate", ("Exclude SFV itself"));
            continue;
        }
        if (!includeFile(work_on_file, params)) {
            /* Exclude files not in Wildcard */
            DBUG_PRINT("sfvCreate", ("Exclude by Wildcard:%s", work_on_file));
            continue;
        }
        if (!params->quiet) {
            /* TODO */
            printf("Adding: %-49s\n", work_on_file);
        }
        rc = getFileCRC(work_on_file, params, &crc_val);
        if (rc) return rc;
        dir_info.items[i].found = 1;
        dir_info.items[i].tested = 1;
        dir_info.items[i].crc = crc_val;
        if (sfvInfoAdd(sfv_info, &(dir_info.items[i]))) {
            return ERROR_MEMORY;
        }
        sfv_info->cnt.files_added ++;
    }

    sfvInfoFree(&dir_info);
    return rc;
}

int m3uCreate(pure_sfv_params* params, crc_info_type* sfv_info)
{
    char    m3u_name[PATH_MAX];
    char    current_dir_name[PATH_MAX];
    FILE   *m3ufile;
    int     i, k;

    getcwd(current_dir_name, PATH_MAX);
    strcpy(m3u_name, getBaseName(current_dir_name));
    strcat(m3u_name, ".m3u");

    m3ufile = fopen (m3u_name, "wt");
    if (m3ufile == NULL)
    {
        fprintf(stderr, "Eror Can\'t write to %s (check permissions!), aborting ...\n", m3u_name);
        return ERROR_FILE_ACCESS;
    }

    k = 0;
    for (i = 0; i < sfv_info->len; i++) {
        if (strcasecmp(sfv_info->items[i].ext, mp3_extension) != 0) {
            continue;
        }
        k ++;
        fprintf(m3ufile, "%s" DOS_LINE_END, sfv_info->items[i].file_name);
    }

    fclose(m3ufile);

    printf("M3U file successfully created (%d files added) ...\n", k);


    return 0;
}

int sfvCreateFile(pure_sfv_params* params, char* sfv_file, char* sfv_dir)
{
    int             rc = 0;
    crc_info_type   sfv_info;
    int             i;
    FILE            *sfvfile;
    time_t          curtime;
    struct tm       tm_local;
    char            time_str[80];
    char            tmp_str[80];
    char            current_dir_name[PATH_MAX];
    char           *sfv_file_name;


    if (sfv_dir[0] != '\0') {
        if (chdir(sfv_dir) != 0) {
            fprintf(stderr, "sfv: %s: %s\n", sfv_dir, strerror(errno));
            return ERROR_DIR_ACCESS;
        }
        getcwd(current_dir_name, PATH_MAX);
        if (!params->quiet) {
            printf("Directory %s\n", current_dir_name);
        }
    } else {
        getcwd(current_dir_name, PATH_MAX);
    }
    sfvInfoInit(&sfv_info);

    if (sfv_file[0] == '\0') {
        if (params->sfv_files.names_cnt) {
            for(i = 0; i < params->sfv_files.len; i++) {
                if (attr_base(params->sfv_files.attr[i]) == PARAM_NAME) {
                    strncpy(sfv_info.name, params->sfv_files.strs[i], MAX_STRING_LEN);
                    getFileExtension(sfv_info.name, tmp_str);
                    if (!isSfvFile(tmp_str)) {
                        strcat(sfv_info.name, "." sfv_extension);
                    }
                    break;
                }
            }
        } else {
            strncpy(sfv_info.name, getBaseName(current_dir_name), MAX_STRING_LEN);
            strcat(sfv_info.name, "." sfv_extension);
        }
    } else {
        strncpy(sfv_info.name, sfv_file, MAX_STRING_LEN);
    }
    sfv_file_name = sfv_info.name;

    if (!params->quiet) {
            /* TODO */
        printf("Curent directory: %s \n", current_dir_name);
        printf("----( Creating: %s )--------\n", sfv_file_name);
    }

    rc = sfvCreate(params, "", &sfv_info);
    if (rc) return rc;

    if (sfv_info.len == 0) {
        fprintf(stderr, "No files to add.\n");
        return 0;
    }

    sfvfile = fopen (sfv_file_name, "wt");
    if (sfvfile == NULL)
    {
        fprintf(stderr, "Eror Can\'t write to %s (check permissions!), aborting ...\n", sfv_file_name);
        return ERROR_FILE_ACCESS;
    }

    /* Write header */
    time (&curtime);
    tm_local = *localtime (&curtime);
    strftime (time_str, 50, "%Y-%m-%d at %H:%M.%S", &tm_local);

    if (params->win_sfv) {
        fprintf(sfvfile, "; Generated by WIN-SFV32 v1.1a (pure-sfv v%s) on %s" DOS_LINE_END ";" DOS_LINE_END,
                PURE_SFV_VERSION, time_str);
    } else {
        fprintf(sfvfile, "; Generated by pure-sfv v%s on %s" DOS_LINE_END ";" DOS_LINE_END,
               PURE_SFV_VERSION, time_str);
    }

    for (i = 0; i < sfv_info.len; i++) {
        strftime(time_str, 50, "%H:%M.%S %Y-%m-%d", &(sfv_info.items[i].time));
        fprintf(sfvfile, "; %12d  %s %s" DOS_LINE_END, (int)sfv_info.items[i].size, time_str, sfv_info.items[i].file_name);
    }

    for (i = 0; i < sfv_info.len; i++) {
        fprintf(sfvfile, "%s %08lX" DOS_LINE_END, sfv_info.items[i].file_name, sfv_info.items[i].crc);
    }

    fclose(sfvfile);

    printf("SFV file successfully created (%d files processed) ...\n", sfv_info.len);

    sfv_info.cnt.sfv_created ++;
    addCnt(&(params->cnt), &(sfv_info.cnt));

    if (params->create_m3u) {
        rc = m3uCreate(params, &sfv_info);
    }
    if (params->par_create) {
        rc = pparCreate(params, &sfv_info);
        if (rc) { rc = ERROR_CREATING_PAR; }
    }


    sfvInfoFree(&sfv_info);
    return rc;
}

static char place_holder = '_';

int is_removable(char c)
{
    if ((c == place_holder) ||
        (c == ' ') ||
        (ispunct((int)c)) ||
        (isdigit((int)c))) {
        return 1;
    }
    return 0;
}

int has_of(char* name)
{
    if ((tolower(name[0]) == 'o') &&
        (tolower(name[1]) == 'f') &&
        ((name[2] == '\0') || (is_removable(name[2])))) {
            return 1;
    }
    return 0;
}

int makeSFVname(pure_sfv_params* params, crc_info_type *dir_info, char *current_dir_name, char * name_out)
{
    int   i, k;
    int   name_created = 0;
    char  *name2, *end;
    char  tmp_str[80];
    char  c;

    place_holder = params->name_place_holder;

    name_out[0] = '\0';
    if (params->sfv_files.names_cnt) {
        for(i = 0; i < params->sfv_files.len; i++) {
            if (attr_base(params->sfv_files.attr[i]) == PARAM_NAME) {
                strncpy(name_out, params->sfv_files.strs[i], MAX_STRING_LEN);
                getFileExtension(name_out, tmp_str);
                if (!isSfvFile(tmp_str)) {
                    strcat(name_out, "." sfv_extension);
                }
                return 0;
            }
        }
    }

    for(i = 0; i < dir_info->len; i++)
    {
        if (dir_info->items[i].found != 1) {
            continue;
        }
        if (name_out[0] == '\0') {
            strncpy(name_out, dir_info->items[i].file_name, MAX_STRING_LEN);
            end = strrchr(name_out, '.');
            if (end != NULL) {
                /* Remove Extension */
                *end = '\0';
            }

        } else {
            name2 = dir_info->items[i].file_name;
            DBUG_PRINT("makeSFVname", ("    name_out [%s]", name_out));
            DBUG_PRINT("makeSFVname", ("add name_out [%s]", name2));
            k = 0;
            while (name2[k] && name_out[k]) {
                if (name_out[k] == place_holder) {
                    k ++;
                    continue;
                }
                if (tolower(name2[k]) == tolower(name_out[k])) {
                    k ++;
                    continue;
                }
                name_out[k] = place_holder;
                k ++;
            }
            DBUG_PRINT("makeSFVname", ("res name_out [%s]", name_out));
        }
    }
    DBUG_PRINT("makeSFVname", ("nextname_out [%s]", name_out));
    /* Strip spaces and punctuation from front*/
    k = 0; end = NULL;
    while (name_out[k]) {
        c = name_out[k];
        DBUG_PRINT("makeSFVname", ("front[%c]", c));
        if (is_removable(c)) {
            end = name_out + k;
            k ++;
        } else if (has_of( name_out + k)) {
            /* remove of */
            end = name_out + (k + 1);
            k += 2;
        } else {
            break;
        }
    }
    if (end != NULL) {
         k = 0;
         while (*end) {
            end ++;
            name_out[k] = *end;
            k++;
         }
    }
    DBUG_PRINT("makeSFVname", ("nsp1name_out [%s]", name_out));
    /* Strip spaces and punctuation from end*/
    k = strlen(name_out) - 1;
    while ((k > 0) && (name_out[k]))  {
        c = name_out[k];
        DBUG_PRINT("makeSFVname", ("end[%c]", c));
        if (is_removable(c)) {
            name_out[k] = '\0';
            k --;
        } else if (has_of(name_out + k - 1)) {
            /* remove of */
            name_out[k-1] = '\0';
            k -= 2;
        } else {
            break;
        }
    }
    DBUG_PRINT("makeSFVname", ("nsp2name_out [%s]", name_out));
    if (strlen(name_out) > 1) {
        name_created = 1;
    }

    if (!name_created) {
        strncpy(name_out, getBaseName(current_dir_name), MAX_STRING_LEN);
    }
    strcat(name_out, "." sfv_extension);
    return 0;
}

int sfvCreateRecursive(pure_sfv_params* params, char* sfv_dir)
{
    int             rc;
    int             i;
    crc_info_type   dir_info;
    char            current_dir_name[PATH_MAX];
    string          sfv_name;
    int             sfv_exists = 0;
    int             files_count = 0;

    sfvInfoInit(&dir_info);

    getcwd(current_dir_name, PATH_MAX);

    if (sfv_dir[0] != '\0') {
        if (chdir(sfv_dir) != 0) {
            fprintf(stderr, "sfv: directory [%s], cd [%s]: %s\n", current_dir_name, sfv_dir, strerror(errno));
            return ERROR_DIR_ACCESS;
        }
        getcwd(current_dir_name, PATH_MAX);
    }
    DBUG_PRINT("DirRecursive", ("enter dir [%s]", current_dir_name));

    rc = dirRead("", &dir_info);
    if (rc) return rc;
    rc = dirReadAttributes("", &dir_info);
    if (rc) return rc;

    for(i = 0; i < dir_info.len; i++)
    {
        if (!dir_info.items[i].is_dir) {
            if (( (params->sfv_files.len == 0) &&
                  (isSfvFile(dir_info.items[i].ext)) &&
                  (isFile("", dir_info.items[i].file_name))) ||
                ( (params->sfv_files.len != 0) &&
                  (includeFileSFV(dir_info.items[i].file_name, params))))  {
                sfv_exists = 1;
            }
            if (includeFile(dir_info.items[i].file_name, params)) {
                files_count ++;
                /* Use this flag for common name making */
                dir_info.items[i].found = 1;
            }
        }
    }

    if ( (files_count) &&
         ((params->force_create) || (!sfv_exists) ||
          (!params->recurse_subdirectories))) {
        makeSFVname(params, &dir_info, current_dir_name, sfv_name);
        rc = sfvCreateFile(params, sfv_name, "");
        if (rc) return rc;
    }

    params->cnt.directories_scaned ++;

    if (params->recurse_subdirectories) {
    for(i = 0; i < dir_info.len; i++)
    {
        if (dir_info.items[i].is_dir) {

            if (chdir(dir_info.items[i].file_name) != 0) {
                fprintf(stderr, "sfv: directory [%s], cd [%s]: %s\n", current_dir_name, dir_info.items[i].file_name, strerror(errno));
                /* return ERROR_DIR_ACCESS; */
                continue;
            }

            rc = sfvCreateRecursive(params, "");
            if (rc) return rc;

            if ((chdir(current_dir_name) != 0)) {
                fprintf(stderr, "sfv: cd [%s] %s\n", current_dir_name,  strerror(errno));
                return ERROR_DIR_ACCESS;
            }
        }
    }
    }

    sfvInfoFree(&dir_info);
    return 0;
}

/* EOF */
