// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief gamma gamma -> K*0K*0
  class ARGUS_1987_I248680 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ARGUS_1987_I248680);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      // book histos
      for (size_t ih=0; ih<5; ++ih) {
        book(_est[ih], 5, 1, ih+1);
        book(_xsec[ih], "_aux_xsec_"+toString(ih), _est[ih].binning().edges<0>());
      }
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        for (size_t ix=0; ix<9; ++ix) {
          book(_nMeson[en+toString(ix)],"TMP/nMeson_"+en+"_"+toString(ix+1));
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle& child : p.children()) {
        if (child.children().empty()) {
          nRes[child.pid()]-=1;
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      // find the final-state particles
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      // find any K* mesons
      int ires=-1;
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      Particles Kstar=ufs.particles(Cuts::abspid==313);
      for (size_t ix=0; ix<Kstar.size(); ++ix) {
        if (Kstar[ix].children().empty()) continue;
        map<long,int> nRes=nCount;
        int ncount = ntotal;
        findChildren(Kstar[ix],nRes,ncount);
        bool matched=false;
        // K*K*
        for (size_t iy=ix+1; iy<Kstar.size(); ++iy) {
          if (Kstar[iy].children().empty()) continue;
          if (Kstar[ix].pid()!=-Kstar[iy].pid()) continue;
          map<long,int> nRes2=nRes;
          int ncount2 = ncount;
          findChildren(Kstar[iy],nRes2,ncount2);
          if (ncount2 !=0 ) continue;
          matched = true;
          for (const auto& val : nRes2) {
            if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched) {
            break;
          }
        }
        if (matched) {
          _nMeson[_sqs+"1"s]->fill();
          ires=7;
          break;
        }
        int sign = Kstar[ix].pid()/Kstar[ix].abspid();
        // three body intermediate states
        if (ncount==2) {
          // K*0 K- pi+ +ccd
          matched=true;
          for (const auto& val : nRes) {
            if (val.first==sign*211 || val.first==-sign*321) {
              if (val.second!=1) {
                matched = false;
                break;
              }
            }
            else {
              if (val.second!=0) {
                matched = false;
                break;
              }
            }
          }
          if (matched) {
            _nMeson[_sqs+"2"s]->fill();
            ires=6;
            break;
          }
        }
      }
      // look for phi modes
      for (const Particle& p : ufs.particles(Cuts::pid==PID::PHI)) {
       	if (p.children().empty()) continue;
       	map<long,int> nRes=nCount;
       	int ncount = ntotal;
       	findChildren(p,nRes,ncount);
        if (ncount==2) {
          bool matched=true;
          for (const auto& val : nRes) {
            if (abs(val.first)==211) {
              if (val.second!=1) {
                matched = false;
                break;
              }
            }
            else {
              if (val.second!=0) {
                matched = false;
                break;
              }
            }
          }
          if (matched) {
            ires=8;
            break;
          }
        }
      }
      // 4 meson modes
      if (ntotal==4 &&
         nCount[PID::KPLUS ]==1 && nCount[PID::KMINUS ]==1 &&
         nCount[PID::PIPLUS]==1 && nCount[PID::PIMINUS]==1 ) {
        _nMeson[_sqs+"0"s]->fill();
        _nMeson[_sqs+"4"s]->fill();
        _xsec[0]->fill(sqrtS()/GeV);
        if (ires<0) {
          _nMeson[_sqs+"3"s]->fill();
          _nMeson[_sqs+"5"s]->fill();
          _xsec[1]->fill(sqrtS()/GeV);
        }
        else {
          _nMeson[_sqs+toString(ires)]->fill();
          if (ires > 3)  _xsec[ires-4]->fill(sqrtS()/GeV);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      const double sf = crossSection()/nanobarn/sumOfWeights();
      scale(_nMeson, sf);
      scale(_xsec, sf);
      for (size_t ih=0; ih<4; ++ih) {
        barchart(_xsec[ih], _est[ih]);
      }
      // loop over tables in paper
      for (size_t ih=1; ih<5; ++ih) {
       	for (size_t iy=1; iy<2; ++iy) {
          string iloc = toString(ih+iy-2);
       	  BinnedEstimatePtr<string> mult;
       	  book(mult, ih, 1, iy);
          for (auto& b : mult->bins()) {
            const double eVal = stod(b.xEdge());
            const string en = toString(round(eVal/MeV));
            b.set(_nMeson[en+iloc]->val(), _nMeson[en+iloc]->err());
          }
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,CounterPtr> _nMeson;
    Histo1DPtr _xsec[5];
    Estimate1DPtr _est[5];
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(ARGUS_1987_I248680);

}
