// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Xi_b- 7,8 and 13 TeV
  class LHCB_2019_I1716259 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(LHCB_2019_I1716259);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projections
      declare(UnstableParticles(), "UFS");
      // CMS energy
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal)) {
          _sqs = (en == "13000"s)? "13"s : "7 & 8"s;
        }
      }
      raiseBeamErrorIf(_sqs.empty());

      // histograms
      for (const string& en : vector<string>{"7 & 8"s, "13"s}) {
        for (size_t ix=0; ix<2; ++ix) {
          book(_c[en+"xi"+toString(ix)] ,"TMP/c_xi_" +toString(ix+1)+"_"+en);
          book(_c[en+"lam"+toString(ix)],"TMP/c_lam_"+toString(ix+1)+"_"+en);
        }
      }
      const auto& ref = refData<YODA::BinnedEstimate<string>>(1,1,1);
      book(_h_xi ,"TMP/h_xi",  ref);
      book(_h_lam,"TMP/h_lam", ref);
    }

    void findDecayProducts(const Particle& mother, double sign, Particles& Lambda,
                           Particles& Xi, Particles& Jpsi, unsigned int& nstable) const {
      for (const Particle& p: mother.children()) {
        if (p.pid() == 3122*sign)      Lambda += p;
        else if (p.pid() == 3312*sign) Xi += p;
        else if (p.pid()==443)         Jpsi += p;
        else if (p.pid()==111 || p.children().empty()) ++nstable;
        else  findDecayProducts(p,sign,Lambda,Xi,Jpsi,nstable);
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      for (const Particle& p : ufs.particles(Cuts::abspid==5122 || Cuts::abspid==5132)) {
        // decay modes
        Particles Lambda,Xi,Jpsi;
        unsigned int nstable=0;
        double sign = p.pid()>0 ? 1. : -1.;
        findDecayProducts(p,sign,Lambda,Xi,Jpsi,nstable);
        if (p.abspid()==5112) {
          _c[_sqs+"lam1"s]->fill();
          if (Lambda.size()==1 && Jpsi.size()==1 && nstable==0) _c[_sqs+"lam0"s]->fill();
        }
        else {
          _c[_sqs+"xi1"]->fill();
          if (Xi.size()==1 && Jpsi.size()==1 && nstable==0) _c[_sqs+"xi0"]->fill();
        }
        // pT and rapidity cuts
        if (p.pT()>20*GeV) continue;
        const double eta = p.abseta();
        if (eta<2. || eta> 6.) continue;
        if (p.abspid()==5122) {
          _h_lam->fill(_sqs);
        }
        else {
          _h_xi->fill(_sqs);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      // first the simple ratio
      BinnedEstimatePtr<string> tmp;
      book(tmp,2,1,1);
      divide(_h_xi, _h_lam, tmp);
      // and the one with brs included
      book(tmp,1,1,1);
      for (const auto& b : tmp->bins()) {
        const string edge = b.xEdge();
        if (_c[edge+"xi1"]->sumW()>0) {
          _h_xi->bin(b.index()).scaleW( dbl(*_c[edge+"xi0"] / *_c[edge+"xi1"]) );
        }
        if (_c[edge+"lam1"]->sumW()>0) {
          _h_lam->bin(b.index()).scaleW( dbl(*_c[edge+"lam0"] / *_c[edge+"lam1"]) );
        }
      }
      divide(_h_xi, _h_lam, tmp);
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _h_xi, _h_lam;
    map<string,CounterPtr> _c;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(LHCB_2019_I1716259);

}
