{-# LANGUAGE OverloadedStrings #-}
{-# LANGUAGE BangPatterns #-}
module KAT_Ed448 ( tests ) where

import           Crypto.Error
import qualified Crypto.PubKey.Ed448 as Ed448
import           Imports

data Vec = Vec
    { vecSec :: ByteString
    , vecPub :: ByteString
    , vecMsg :: ByteString
    , vecSig :: ByteString
    } deriving (Show,Eq)

vectors =
    [ Vec
        { vecSec = "\x6c\x82\xa5\x62\xcb\x80\x8d\x10\xd6\x32\xbe\x89\xc8\x51\x3e\xbf\x6c\x92\x9f\x34\xdd\xfa\x8c\x9f\x63\xc9\x96\x0e\xf6\xe3\x48\xa3\x52\x8c\x8a\x3f\xcc\x2f\x04\x4e\x39\xa3\xfc\x5b\x94\x49\x2f\x8f\x03\x2e\x75\x49\xa2\x00\x98\xf9\x5b"
        , vecPub = "\x5f\xd7\x44\x9b\x59\xb4\x61\xfd\x2c\xe7\x87\xec\x61\x6a\xd4\x6a\x1d\xa1\x34\x24\x85\xa7\x0e\x1f\x8a\x0e\xa7\x5d\x80\xe9\x67\x78\xed\xf1\x24\x76\x9b\x46\xc7\x06\x1b\xd6\x78\x3d\xf1\xe5\x0f\x6c\xd1\xfa\x1a\xbe\xaf\xe8\x25\x61\x80"
        , vecMsg = ""
        , vecSig = "\x53\x3a\x37\xf6\xbb\xe4\x57\x25\x1f\x02\x3c\x0d\x88\xf9\x76\xae\x2d\xfb\x50\x4a\x84\x3e\x34\xd2\x07\x4f\xd8\x23\xd4\x1a\x59\x1f\x2b\x23\x3f\x03\x4f\x62\x82\x81\xf2\xfd\x7a\x22\xdd\xd4\x7d\x78\x28\xc5\x9b\xd0\xa2\x1b\xfd\x39\x80\xff\x0d\x20\x28\xd4\xb1\x8a\x9d\xf6\x3e\x00\x6c\x5d\x1c\x2d\x34\x5b\x92\x5d\x8d\xc0\x0b\x41\x04\x85\x2d\xb9\x9a\xc5\xc7\xcd\xda\x85\x30\xa1\x13\xa0\xf4\xdb\xb6\x11\x49\xf0\x5a\x73\x63\x26\x8c\x71\xd9\x58\x08\xff\x2e\x65\x26\x00"
        }
    , Vec
        { vecSec = "\xc4\xea\xb0\x5d\x35\x70\x07\xc6\x32\xf3\xdb\xb4\x84\x89\x92\x4d\x55\x2b\x08\xfe\x0c\x35\x3a\x0d\x4a\x1f\x00\xac\xda\x2c\x46\x3a\xfb\xea\x67\xc5\xe8\xd2\x87\x7c\x5e\x3b\xc3\x97\xa6\x59\x94\x9e\xf8\x02\x1e\x95\x4e\x0a\x12\x27\x4e"
        , vecPub = "\x43\xba\x28\xf4\x30\xcd\xff\x45\x6a\xe5\x31\x54\x5f\x7e\xcd\x0a\xc8\x34\xa5\x5d\x93\x58\xc0\x37\x2b\xfa\x0c\x6c\x67\x98\xc0\x86\x6a\xea\x01\xeb\x00\x74\x28\x02\xb8\x43\x8e\xa4\xcb\x82\x16\x9c\x23\x51\x60\x62\x7b\x4c\x3a\x94\x80"
        , vecMsg = "\x03"
        , vecSig = "\x26\xb8\xf9\x17\x27\xbd\x62\x89\x7a\xf1\x5e\x41\xeb\x43\xc3\x77\xef\xb9\xc6\x10\xd4\x8f\x23\x35\xcb\x0b\xd0\x08\x78\x10\xf4\x35\x25\x41\xb1\x43\xc4\xb9\x81\xb7\xe1\x8f\x62\xde\x8c\xcd\xf6\x33\xfc\x1b\xf0\x37\xab\x7c\xd7\x79\x80\x5e\x0d\xbc\xc0\xaa\xe1\xcb\xce\xe1\xaf\xb2\xe0\x27\xdf\x36\xbc\x04\xdc\xec\xbf\x15\x43\x36\xc1\x9f\x0a\xf7\xe0\xa6\x47\x29\x05\xe7\x99\xf1\x95\x3d\x2a\x0f\xf3\x34\x8a\xb2\x1a\xa4\xad\xaf\xd1\xd2\x34\x44\x1c\xf8\x07\xc0\x3a\x00"
        }
    , Vec
        { vecSec = "\xcd\x23\xd2\x4f\x71\x42\x74\xe7\x44\x34\x32\x37\xb9\x32\x90\xf5\x11\xf6\x42\x5f\x98\xe6\x44\x59\xff\x20\x3e\x89\x85\x08\x3f\xfd\xf6\x05\x00\x55\x3a\xbc\x0e\x05\xcd\x02\x18\x4b\xdb\x89\xc4\xcc\xd6\x7e\x18\x79\x51\x26\x7e\xb3\x28"
        , vecPub = "\xdc\xea\x9e\x78\xf3\x5a\x1b\xf3\x49\x9a\x83\x1b\x10\xb8\x6c\x90\xaa\xc0\x1c\xd8\x4b\x67\xa0\x10\x9b\x55\xa3\x6e\x93\x28\xb1\xe3\x65\xfc\xe1\x61\xd7\x1c\xe7\x13\x1a\x54\x3e\xa4\xcb\x5f\x7e\x9f\x1d\x8b\x00\x69\x64\x47\x00\x14\x00"
        , vecMsg = "\x0c\x3e\x54\x40\x74\xec\x63\xb0\x26\x5e\x0c"
        , vecSig = "\x1f\x0a\x88\x88\xce\x25\xe8\xd4\x58\xa2\x11\x30\x87\x9b\x84\x0a\x90\x89\xd9\x99\xaa\xba\x03\x9e\xaf\x3e\x3a\xfa\x09\x0a\x09\xd3\x89\xdb\xa8\x2c\x4f\xf2\xae\x8a\xc5\xcd\xfb\x7c\x55\xe9\x4d\x5d\x96\x1a\x29\xfe\x01\x09\x94\x1e\x00\xb8\xdb\xde\xea\x6d\x3b\x05\x10\x68\xdf\x72\x54\xc0\xcd\xc1\x29\xcb\xe6\x2d\xb2\xdc\x95\x7d\xbb\x47\xb5\x1f\xd3\xf2\x13\xfb\x86\x98\xf0\x64\x77\x42\x50\xa5\x02\x89\x61\xc9\xbf\x8f\xfd\x97\x3f\xe5\xd5\xc2\x06\x49\x2b\x14\x0e\x00"
        }
    , Vec
        { vecSec = "\x25\x8c\xdd\x4a\xda\x32\xed\x9c\x9f\xf5\x4e\x63\x75\x6a\xe5\x82\xfb\x8f\xab\x2a\xc7\x21\xf2\xc8\xe6\x76\xa7\x27\x68\x51\x3d\x93\x9f\x63\xdd\xdb\x55\x60\x91\x33\xf2\x9a\xdf\x86\xec\x99\x29\xdc\xcb\x52\xc1\xc5\xfd\x2f\xf7\xe2\x1b"
        , vecPub = "\x3b\xa1\x6d\xa0\xc6\xf2\xcc\x1f\x30\x18\x77\x40\x75\x6f\x5e\x79\x8d\x6b\xc5\xfc\x01\x5d\x7c\x63\xcc\x95\x10\xee\x3f\xd4\x4a\xdc\x24\xd8\xe9\x68\xb6\xe4\x6e\x6f\x94\xd1\x9b\x94\x53\x61\x72\x6b\xd7\x5e\x14\x9e\xf0\x98\x17\xf5\x80"
        , vecMsg = "\x64\xa6\x5f\x3c\xde\xdc\xdd\x66\x81\x1e\x29\x15"
        , vecSig = "\x7e\xee\xab\x7c\x4e\x50\xfb\x79\x9b\x41\x8e\xe5\xe3\x19\x7f\xf6\xbf\x15\xd4\x3a\x14\xc3\x43\x89\xb5\x9d\xd1\xa7\xb1\xb8\x5b\x4a\xe9\x04\x38\xac\xa6\x34\xbe\xa4\x5e\x3a\x26\x95\xf1\x27\x0f\x07\xfd\xcd\xf7\xc6\x2b\x8e\xfe\xaf\x00\xb4\x5c\x2c\x96\xba\x45\x7e\xb1\xa8\xbf\x07\x5a\x3d\xb2\x8e\x5c\x24\xf6\xb9\x23\xed\x4a\xd7\x47\xc3\xc9\xe0\x3c\x70\x79\xef\xb8\x7c\xb1\x10\xd3\xa9\x98\x61\xe7\x20\x03\xcb\xae\x6d\x6b\x8b\x82\x7e\x4e\x6c\x14\x30\x64\xff\x3c\x00"
        }
    , Vec
        { vecSec = "\x7e\xf4\xe8\x45\x44\x23\x67\x52\xfb\xb5\x6b\x8f\x31\xa2\x3a\x10\xe4\x28\x14\xf5\xf5\x5c\xa0\x37\xcd\xcc\x11\xc6\x4c\x9a\x3b\x29\x49\xc1\xbb\x60\x70\x03\x14\x61\x17\x32\xa6\xc2\xfe\xa9\x8e\xeb\xc0\x26\x6a\x11\xa9\x39\x70\x10\x0e"
        , vecPub = "\xb3\xda\x07\x9b\x0a\xa4\x93\xa5\x77\x20\x29\xf0\x46\x7b\xae\xbe\xe5\xa8\x11\x2d\x9d\x3a\x22\x53\x23\x61\xda\x29\x4f\x7b\xb3\x81\x5c\x5d\xc5\x9e\x17\x6b\x4d\x9f\x38\x1c\xa0\x93\x8e\x13\xc6\xc0\x7b\x17\x4b\xe6\x5d\xfa\x57\x8e\x80"
        , vecMsg = "\x64\xa6\x5f\x3c\xde\xdc\xdd\x66\x81\x1e\x29\x15\xe7"
        , vecSig = "\x6a\x12\x06\x6f\x55\x33\x1b\x6c\x22\xac\xd5\xd5\xbf\xc5\xd7\x12\x28\xfb\xda\x80\xae\x8d\xec\x26\xbd\xd3\x06\x74\x3c\x50\x27\xcb\x48\x90\x81\x0c\x16\x2c\x02\x74\x68\x67\x5e\xcf\x64\x5a\x83\x17\x6c\x0d\x73\x23\xa2\xcc\xde\x2d\x80\xef\xe5\xa1\x26\x8e\x8a\xca\x1d\x6f\xbc\x19\x4d\x3f\x77\xc4\x49\x86\xeb\x4a\xb4\x17\x79\x19\xad\x8b\xec\x33\xeb\x47\xbb\xb5\xfc\x6e\x28\x19\x6f\xd1\xca\xf5\x6b\x4e\x7e\x0b\xa5\x51\x92\x34\xd0\x47\x15\x5a\xc7\x27\xa1\x05\x31\x00"
        }
    , Vec
        { vecSec = "\xd6\x5d\xf3\x41\xad\x13\xe0\x08\x56\x76\x88\xba\xed\xda\x8e\x9d\xcd\xc1\x7d\xc0\x24\x97\x4e\xa5\xb4\x22\x7b\x65\x30\xe3\x39\xbf\xf2\x1f\x99\xe6\x8c\xa6\x96\x8f\x3c\xca\x6d\xfe\x0f\xb9\xf4\xfa\xb4\xfa\x13\x5d\x55\x42\xea\x3f\x01"
        , vecPub = "\xdf\x97\x05\xf5\x8e\xdb\xab\x80\x2c\x7f\x83\x63\xcf\xe5\x56\x0a\xb1\xc6\x13\x2c\x20\xa9\xf1\xdd\x16\x34\x83\xa2\x6f\x8a\xc5\x3a\x39\xd6\x80\x8b\xf4\xa1\xdf\xbd\x26\x1b\x09\x9b\xb0\x3b\x3f\xb5\x09\x06\xcb\x28\xbd\x8a\x08\x1f\x00"
        , vecMsg = "\xbd\x0f\x6a\x37\x47\xcd\x56\x1b\xdd\xdf\x46\x40\xa3\x32\x46\x1a\x4a\x30\xa1\x2a\x43\x4c\xd0\xbf\x40\xd7\x66\xd9\xc6\xd4\x58\xe5\x51\x22\x04\xa3\x0c\x17\xd1\xf5\x0b\x50\x79\x63\x1f\x64\xeb\x31\x12\x18\x2d\xa3\x00\x58\x35\x46\x11\x13\x71\x8d\x1a\x5e\xf9\x44"
        , vecSig = "\x55\x4b\xc2\x48\x08\x60\xb4\x9e\xab\x85\x32\xd2\xa5\x33\xb7\xd5\x78\xef\x47\x3e\xeb\x58\xc9\x8b\xb2\xd0\xe1\xce\x48\x8a\x98\xb1\x8d\xfd\xe9\xb9\xb9\x07\x75\xe6\x7f\x47\xd4\xa1\xc3\x48\x20\x58\xef\xc9\xf4\x0d\x2c\xa0\x33\xa0\x80\x1b\x63\xd4\x5b\x3b\x72\x2e\xf5\x52\xba\xd3\xb4\xcc\xb6\x67\xda\x35\x01\x92\xb6\x1c\x50\x8c\xf7\xb6\xb5\xad\xad\xc2\xc8\xd9\xa4\x46\xef\x00\x3f\xb0\x5c\xba\x5f\x30\xe8\x8e\x36\xec\x27\x03\xb3\x49\xca\x22\x9c\x26\x70\x83\x39\x00"
        }
    , Vec
        { vecSec = "\x2e\xc5\xfe\x3c\x17\x04\x5a\xbd\xb1\x36\xa5\xe6\xa9\x13\xe3\x2a\xb7\x5a\xe6\x8b\x53\xd2\xfc\x14\x9b\x77\xe5\x04\x13\x2d\x37\x56\x9b\x7e\x76\x6b\xa7\x4a\x19\xbd\x61\x62\x34\x3a\x21\xc8\x59\x0a\xa9\xce\xbc\xa9\x01\x4c\x63\x6d\xf5"
        , vecPub = "\x79\x75\x6f\x01\x4d\xcf\xe2\x07\x9f\x5d\xd9\xe7\x18\xbe\x41\x71\xe2\xef\x24\x86\xa0\x8f\x25\x18\x6f\x6b\xff\x43\xa9\x93\x6b\x9b\xfe\x12\x40\x2b\x08\xae\x65\x79\x8a\x3d\x81\xe2\x2e\x9e\xc8\x0e\x76\x90\x86\x2e\xf3\xd4\xed\x3a\x00"
        , vecMsg = "\x15\x77\x75\x32\xb0\xbd\xd0\xd1\x38\x9f\x63\x6c\x5f\x6b\x9b\xa7\x34\xc9\x0a\xf5\x72\x87\x7e\x2d\x27\x2d\xd0\x78\xaa\x1e\x56\x7c\xfa\x80\xe1\x29\x28\xbb\x54\x23\x30\xe8\x40\x9f\x31\x74\x50\x41\x07\xec\xd5\xef\xac\x61\xae\x75\x04\xda\xbe\x2a\x60\x2e\xde\x89\xe5\xcc\xa6\x25\x7a\x7c\x77\xe2\x7a\x70\x2b\x3a\xe3\x9f\xc7\x69\xfc\x54\xf2\x39\x5a\xe6\xa1\x17\x8c\xab\x47\x38\xe5\x43\x07\x2f\xc1\xc1\x77\xfe\x71\xe9\x2e\x25\xbf\x03\xe4\xec\xb7\x2f\x47\xb6\x4d\x04\x65\xaa\xea\x4c\x7f\xad\x37\x25\x36\xc8\xba\x51\x6a\x60\x39\xc3\xc2\xa3\x9f\x0e\x4d\x83\x2b\xe4\x32\xdf\xa9\xa7\x06\xa6\xe5\xc7\xe1\x9f\x39\x79\x64\xca\x42\x58\x00\x2f\x7c\x05\x41\xb5\x90\x31\x6d\xbc\x56\x22\xb6\xb2\xa6\xfe\x7a\x4a\xbf\xfd\x96\x10\x5e\xca\x76\xea\x7b\x98\x81\x6a\xf0\x74\x8c\x10\xdf\x04\x8c\xe0\x12\xd9\x01\x01\x5a\x51\xf1\x89\xf3\x88\x81\x45\xc0\x36\x50\xaa\x23\xce\x89\x4c\x3b\xd8\x89\xe0\x30\xd5\x65\x07\x1c\x59\xf4\x09\xa9\x98\x1b\x51\x87\x8f\xd6\xfc\x11\x06\x24\xdc\xbc\xde\x0b\xf7\xa6\x9c\xcc\xe3\x8f\xab\xdf\x86\xf3\xbe\xf6\x04\x48\x19\xde\x11"
        , vecSig = "\xc6\x50\xdd\xbb\x06\x01\xc1\x9c\xa1\x14\x39\xe1\x64\x0d\xd9\x31\xf4\x3c\x51\x8e\xa5\xbe\xa7\x0d\x3d\xcd\xe5\xf4\x19\x1f\xe5\x3f\x00\xcf\x96\x65\x46\xb7\x2b\xcc\x7d\x58\xbe\x2b\x9b\xad\xef\x28\x74\x39\x54\xe3\xa4\x4a\x23\xf8\x80\xe8\xd4\xf1\xcf\xce\x2d\x7a\x61\x45\x2d\x26\xda\x05\x89\x6f\x0a\x50\xda\x66\xa2\x39\xa8\xa1\x88\xb6\xd8\x25\xb3\x30\x5a\xd7\x7b\x73\xfb\xac\x08\x36\xec\xc6\x09\x87\xfd\x08\x52\x7c\x1a\x8e\x80\xd5\x82\x3e\x65\xca\xfe\x2a\x3d\x00"
        }
    , Vec
        { vecSec = "\x87\x2d\x09\x37\x80\xf5\xd3\x73\x0d\xf7\xc2\x12\x66\x4b\x37\xb8\xa0\xf2\x4f\x56\x81\x0d\xaa\x83\x82\xcd\x4f\xa3\xf7\x76\x34\xec\x44\xdc\x54\xf1\xc2\xed\x9b\xea\x86\xfa\xfb\x76\x32\xd8\xbe\x19\x9e\xa1\x65\xf5\xad\x55\xdd\x9c\xe8"
        , vecPub = "\xa8\x1b\x2e\x8a\x70\xa5\xac\x94\xff\xdb\xcc\x9b\xad\xfc\x3f\xeb\x08\x01\xf2\x58\x57\x8b\xb1\x14\xad\x44\xec\xe1\xec\x0e\x79\x9d\xa0\x8e\xff\xb8\x1c\x5d\x68\x5c\x0c\x56\xf6\x4e\xec\xae\xf8\xcd\xf1\x1c\xc3\x87\x37\x83\x8c\xf4\x00"
        , vecMsg = "\x6d\xdf\x80\x2e\x1a\xae\x49\x86\x93\x5f\x7f\x98\x1b\xa3\xf0\x35\x1d\x62\x73\xc0\xa0\xc2\x2c\x9c\x0e\x83\x39\x16\x8e\x67\x54\x12\xa3\xde\xbf\xaf\x43\x5e\xd6\x51\x55\x80\x07\xdb\x43\x84\xb6\x50\xfc\xc0\x7e\x3b\x58\x6a\x27\xa4\xf7\xa0\x0a\xc8\xa6\xfe\xc2\xcd\x86\xae\x4b\xf1\x57\x0c\x41\xe6\xa4\x0c\x93\x1d\xb2\x7b\x2f\xaa\x15\xa8\xce\xdd\x52\xcf\xf7\x36\x2c\x4e\x6e\x23\xda\xec\x0f\xbc\x3a\x79\xb6\x80\x6e\x31\x6e\xfc\xc7\xb6\x81\x19\xbf\x46\xbc\x76\xa2\x60\x67\xa5\x3f\x29\x6d\xaf\xdb\xdc\x11\xc7\x7f\x77\x77\xe9\x72\x66\x0c\xf4\xb6\xa9\xb3\x69\xa6\x66\x5f\x02\xe0\xcc\x9b\x6e\xdf\xad\x13\x6b\x4f\xab\xe7\x23\xd2\x81\x3d\xb3\x13\x6c\xfd\xe9\xb6\xd0\x44\x32\x2f\xee\x29\x47\x95\x2e\x03\x1b\x73\xab\x5c\x60\x33\x49\xb3\x07\xbd\xc2\x7b\xc6\xcb\x8b\x8b\xbd\x7b\xd3\x23\x21\x9b\x80\x33\xa5\x81\xb5\x9e\xad\xeb\xb0\x9b\x3c\x4f\x3d\x22\x77\xd4\xf0\x34\x36\x24\xac\xc8\x17\x80\x47\x28\xb2\x5a\xb7\x97\x17\x2b\x4c\x5c\x21\xa2\x2f\x9c\x78\x39\xd6\x43\x00\x23\x2e\xb6\x6e\x53\xf3\x1c\x72\x3f\xa3\x7f\xe3\x87\xc7\xd3\xe5\x0b\xdf\x98\x13\xa3\x0e\x5b\xb1\x2c\xf4\xcd\x93\x0c\x40\xcf\xb4\xe1\xfc\x62\x25\x92\xa4\x95\x88\x79\x44\x94\xd5\x6d\x24\xea\x4b\x40\xc8\x9f\xc0\x59\x6c\xc9\xeb\xb9\x61\xc8\xcb\x10\xad\xde\x97\x6a\x5d\x60\x2b\x1c\x3f\x85\xb9\xb9\xa0\x01\xed\x3c\x6a\x4d\x3b\x14\x37\xf5\x20\x96\xcd\x19\x56\xd0\x42\xa5\x97\xd5\x61\xa5\x96\xec\xd3\xd1\x73\x5a\x8d\x57\x0e\xa0\xec\x27\x22\x5a\x2c\x4a\xaf\xf2\x63\x06\xd1\x52\x6c\x1a\xf3\xca\x6d\x9c\xf5\xa2\xc9\x8f\x47\xe1\xc4\x6d\xb9\xa3\x32\x34\xcf\xd4\xd8\x1f\x2c\x98\x53\x8a\x09\xeb\xe7\x69\x98\xd0\xd8\xfd\x25\x99\x7c\x7d\x25\x5c\x6d\x66\xec\xe6\xfa\x56\xf1\x11\x44\x95\x0f\x02\x77\x95\xe6\x53\x00\x8f\x4b\xd7\xca\x2d\xee\x85\xd8\xe9\x0f\x3d\xc3\x15\x13\x0c\xe2\xa0\x03\x75\xa3\x18\xc7\xc3\xd9\x7b\xe2\xc8\xce\x5b\x6d\xb4\x1a\x62\x54\xff\x26\x4f\xa6\x15\x5b\xae\xe3\xb0\x77\x3c\x0f\x49\x7c\x57\x3f\x19\xbb\x4f\x42\x40\x28\x1f\x0b\x1f\x4f\x7b\xe8\x57\xa4\xe5\x9d\x41\x6c\x06\xb4\xc5\x0f\xa0\x9e\x18\x10\xdd\xc6\xb1\x46\x7b\xae\xac\x5a\x36\x68\xd1\x1b\x6e\xca\xa9\x01\x44\x00\x16\xf3\x89\xf8\x0a\xcc\x4d\xb9\x77\x02\x5e\x7f\x59\x24\x38\x8c\x7e\x34\x0a\x73\x2e\x55\x44\x40\xe7\x65\x70\xf8\xdd\x71\xb7\xd6\x40\xb3\x45\x0d\x1f\xd5\xf0\x41\x0a\x18\xf9\xa3\x49\x4f\x70\x7c\x71\x7b\x79\xb4\xbf\x75\xc9\x84\x00\xb0\x96\xb2\x16\x53\xb5\xd2\x17\xcf\x35\x65\xc9\x59\x74\x56\xf7\x07\x03\x49\x7a\x07\x87\x63\x82\x9b\xc0\x1b\xb1\xcb\xc8\xfa\x04\xea\xdc\x9a\x6e\x3f\x66\x99\x58\x7a\x9e\x75\xc9\x4e\x5b\xab\x00\x36\xe0\xb2\xe7\x11\x39\x2c\xff\x00\x47\xd0\xd6\xb0\x5b\xd2\xa5\x88\xbc\x10\x97\x18\x95\x42\x59\xf1\xd8\x66\x78\xa5\x79\xa3\x12\x0f\x19\xcf\xb2\x96\x3f\x17\x7a\xeb\x70\xf2\xd4\x84\x48\x26\x26\x2e\x51\xb8\x02\x71\x27\x20\x68\xef\x5b\x38\x56\xfa\x85\x35\xaa\x2a\x88\xb2\xd4\x1f\x2a\x0e\x2f\xda\x76\x24\xc2\x85\x02\x72\xac\x4a\x2f\x56\x1f\x8f\x2f\x7a\x31\x8b\xfd\x5c\xaf\x96\x96\x14\x9e\x4a\xc8\x24\xad\x34\x60\x53\x8f\xdc\x25\x42\x1b\xee\xc2\xcc\x68\x18\x16\x2d\x06\xbb\xed\x0c\x40\xa3\x87\x19\x23\x49\xdb\x67\xa1\x18\xba\xda\x6c\xd5\xab\x01\x40\xee\x27\x32\x04\xf6\x28\xaa\xd1\xc1\x35\xf7\x70\x27\x9a\x65\x1e\x24\xd8\xc1\x4d\x75\xa6\x05\x9d\x76\xb9\x6a\x6f\xd8\x57\xde\xf5\xe0\xb3\x54\xb2\x7a\xb9\x37\xa5\x81\x5d\x16\xb5\xfa\xe4\x07\xff\x18\x22\x2c\x6d\x1e\xd2\x63\xbe\x68\xc9\x5f\x32\xd9\x08\xbd\x89\x5c\xd7\x62\x07\xae\x72\x64\x87\x56\x7f\x9a\x67\xda\xd7\x9a\xbe\xc3\x16\xf6\x83\xb1\x7f\x2d\x02\xbf\x07\xe0\xac\x8b\x5b\xc6\x16\x2c\xf9\x46\x97\xb3\xc2\x7c\xd1\xfe\xa4\x9b\x27\xf2\x3b\xa2\x90\x18\x71\x96\x25\x06\x52\x0c\x39\x2d\xa8\xb6\xad\x0d\x99\xf7\x01\x3f\xbc\x06\xc2\xc1\x7a\x56\x95\x00\xc8\xa7\x69\x64\x81\xc1\xcd\x33\xe9\xb1\x4e\x40\xb8\x2e\x79\xa5\xf5\xdb\x82\x57\x1b\xa9\x7b\xae\x3a\xd3\xe0\x47\x95\x15\xbb\x0e\x2b\x0f\x3b\xfc\xd1\xfd\x33\x03\x4e\xfc\x62\x45\xed\xdd\x7e\xe2\x08\x6d\xda\xe2\x60\x0d\x8c\xa7\x3e\x21\x4e\x8c\x2b\x0b\xdb\x2b\x04\x7c\x6a\x46\x4a\x56\x2e\xd7\x7b\x73\xd2\xd8\x41\xc4\xb3\x49\x73\x55\x12\x57\x71\x3b\x75\x36\x32\xef\xba\x34\x81\x69\xab\xc9\x0a\x68\xf4\x26\x11\xa4\x01\x26\xd7\xcb\x21\xb5\x86\x95\x56\x81\x86\xf7\xe5\x69\xd2\xff\x0f\x9e\x74\x5d\x04\x87\xdd\x2e\xb9\x97\xca\xfc\x5a\xbf\x9d\xd1\x02\xe6\x2f\xf6\x6c\xba\x87"
        , vecSig = "\xe3\x01\x34\x5a\x41\xa3\x9a\x4d\x72\xff\xf8\xdf\x69\xc9\x80\x75\xa0\xcc\x08\x2b\x80\x2f\xc9\xb2\xb6\xbc\x50\x3f\x92\x6b\x65\xbd\xdf\x7f\x4c\x8f\x1c\xb4\x9f\x63\x96\xaf\xc8\xa7\x0a\xbe\x6d\x8a\xef\x0d\xb4\x78\xd4\xc6\xb2\x97\x00\x76\xc6\xa0\x48\x4f\xe7\x6d\x76\xb3\xa9\x76\x25\xd7\x9f\x1c\xe2\x40\xe7\xc5\x76\x75\x0d\x29\x55\x28\x28\x6f\x71\x9b\x41\x3d\xe9\xad\xa3\xe8\xeb\x78\xed\x57\x36\x03\xce\x30\xd8\xbb\x76\x17\x85\xdc\x30\xdb\xc3\x20\x86\x9e\x1a\x00"
        }
    ]


doPublicKeyTest i vec = testCase (show i) (pub @=? Ed448.toPublic sec)
  where
        !pub = throwCryptoError $ Ed448.publicKey (vecPub vec)
        !sec = throwCryptoError $ Ed448.secretKey (vecSec vec)

doSignatureTest i vec = testCase (show i) (sig @=? Ed448.sign sec pub (vecMsg vec))
  where
        !sig = throwCryptoError $ Ed448.signature (vecSig vec)
        !pub = throwCryptoError $ Ed448.publicKey (vecPub vec)
        !sec = throwCryptoError $ Ed448.secretKey (vecSec vec)

doVerifyTest i vec = testCase (show i) (True @=? Ed448.verify pub (vecMsg vec) sig)
  where
        !sig = throwCryptoError $ Ed448.signature (vecSig vec)
        !pub = throwCryptoError $ Ed448.publicKey (vecPub vec)


tests = testGroup "Ed448"
    [ testCase  "gen secretkey" (Ed448.generateSecretKey *> pure ())
    , testGroup "gen publickey" $ zipWith doPublicKeyTest [katZero..] vectors
    , testGroup "gen signature" $ zipWith doSignatureTest [katZero..] vectors
    , testGroup "verify sig" $ zipWith doVerifyTest [katZero..] vectors
    ]
