// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Polyform License
// that can be found in the LICENSE file.

package runtime

import (
	"bytes"
	"encoding/json"
	"testing"
)

//
// runtime policy unit tests.
//

func TestRunPolicy_Marshal(t *testing.T) {
	tests := []struct {
		policy RunPolicy
		data   string
	}{
		{
			policy: RunAlways,
			data:   `"always"`,
		},
		{
			policy: RunOnFailure,
			data:   `"on-failure"`,
		},
		{
			policy: RunOnSuccess,
			data:   `"on-success"`,
		},
		{
			policy: RunNever,
			data:   `"never"`,
		},
	}
	for _, test := range tests {
		data, err := json.Marshal(&test.policy)
		if err != nil {
			t.Error(err)
			return
		}
		if bytes.Equal([]byte(test.data), data) == false {
			t.Errorf("Failed to marshal policy %s", test.policy)
		}
	}
}

func TestRunPolicy_Unmarshal(t *testing.T) {
	tests := []struct {
		policy RunPolicy
		data   string
	}{
		{
			policy: RunAlways,
			data:   `"always"`,
		},
		{
			policy: RunOnFailure,
			data:   `"on-failure"`,
		},
		{
			policy: RunOnSuccess,
			data:   `"on-success"`,
		},
		{
			policy: RunNever,
			data:   `"never"`,
		},
		{
			// no policy should default to on-success
			policy: RunOnSuccess,
			data:   `""`,
		},
	}
	for _, test := range tests {
		var policy RunPolicy
		err := json.Unmarshal([]byte(test.data), &policy)
		if err != nil {
			t.Error(err)
			return
		}
		if got, want := policy, test.policy; got != want {
			t.Errorf("Want policy %q, got %q", want, got)
		}
	}
}

func TestRunPolicy_UnmarshalTypeError(t *testing.T) {
	var policy RunPolicy
	err := json.Unmarshal([]byte("[]"), &policy)
	if _, ok := err.(*json.UnmarshalTypeError); !ok {
		t.Errorf("Expect unmarshal error return when JSON invalid")
	}
}

func TestRunPolicy_String(t *testing.T) {
	tests := []struct {
		policy RunPolicy
		value  string
	}{
		{
			policy: RunAlways,
			value:  "always",
		},
		{
			policy: RunOnFailure,
			value:  "on-failure",
		},
		{
			policy: RunOnSuccess,
			value:  "on-success",
		},
	}
	for _, test := range tests {
		if got, want := test.policy.String(), test.value; got != want {
			t.Errorf("Want policy string %q, got %q", want, got)
		}
	}
}
