/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.services.kinesis.leases.impl;

import com.amazonaws.services.cloudwatch.model.StandardUnit;
import com.amazonaws.services.kinesis.leases.exceptions.DependencyException;
import com.amazonaws.services.kinesis.leases.exceptions.InvalidStateException;
import com.amazonaws.services.kinesis.leases.exceptions.ProvisionedThroughputException;
import com.amazonaws.services.kinesis.leases.impl.GenericLeaseSelector;
import com.amazonaws.services.kinesis.leases.impl.Lease;
import com.amazonaws.services.kinesis.leases.interfaces.ILeaseManager;
import com.amazonaws.services.kinesis.leases.interfaces.ILeaseTaker;
import com.amazonaws.services.kinesis.leases.interfaces.LeaseSelector;
import com.amazonaws.services.kinesis.metrics.impl.MetricsHelper;
import com.amazonaws.services.kinesis.metrics.interfaces.IMetricsScope;
import com.amazonaws.services.kinesis.metrics.interfaces.MetricsLevel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class LeaseTaker<T extends Lease>
implements ILeaseTaker<T> {
    private static final Log LOG = LogFactory.getLog(LeaseTaker.class);
    private static final int TAKE_RETRIES = 3;
    private static final int SCAN_RETRIES = 1;
    private static final Callable<Long> SYSTEM_CLOCK_CALLABLE = new Callable<Long>(){

        @Override
        public Long call() {
            return System.nanoTime();
        }
    };
    private final ILeaseManager<T> leaseManager;
    private final LeaseSelector<T> leaseSelector;
    private final String workerIdentifier;
    private final Map<String, T> allLeases = new HashMap<String, T>();
    private final long leaseDurationNanos;
    private int maxLeasesForWorker = Integer.MAX_VALUE;
    private int maxLeasesToStealAtOneTime = 1;
    private long lastScanTimeNanos = 0L;

    private static <T extends Lease> LeaseSelector<T> getDefaultLeaseSelector() {
        return new GenericLeaseSelector();
    }

    public LeaseTaker(ILeaseManager<T> leaseManager, String workerIdentifier, long leaseDurationMillis) {
        this(leaseManager, LeaseTaker.getDefaultLeaseSelector(), workerIdentifier, leaseDurationMillis);
    }

    public LeaseTaker(ILeaseManager<T> leaseManager, LeaseSelector<T> leaseSelector, String workerIdentifier, long leaseDurationMillis) {
        this.leaseManager = leaseManager;
        this.leaseSelector = leaseSelector;
        this.workerIdentifier = workerIdentifier;
        this.leaseDurationNanos = TimeUnit.MILLISECONDS.toNanos(leaseDurationMillis);
    }

    public LeaseTaker<T> withMaxLeasesForWorker(int maxLeasesForWorker) {
        if (maxLeasesForWorker <= 0) {
            throw new IllegalArgumentException("maxLeasesForWorker should be >= 1");
        }
        this.maxLeasesForWorker = maxLeasesForWorker;
        return this;
    }

    public LeaseTaker<T> withMaxLeasesToStealAtOneTime(int maxLeasesToStealAtOneTime) {
        if (maxLeasesToStealAtOneTime <= 0) {
            throw new IllegalArgumentException("maxLeasesToStealAtOneTime should be >= 1");
        }
        this.maxLeasesToStealAtOneTime = maxLeasesToStealAtOneTime;
        return this;
    }

    @Override
    public Map<String, T> takeLeases() throws DependencyException, InvalidStateException {
        return this.takeLeases(SYSTEM_CLOCK_CALLABLE);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized Map<String, T> takeLeases(Callable<Long> timeProvider) throws DependencyException, InvalidStateException {
        HashMap<String, Lease> takenLeases = new HashMap<String, Lease>();
        long startTime = System.currentTimeMillis();
        boolean success = false;
        ProvisionedThroughputException lastException = null;
        try {
            for (int i = 1; i <= 1; ++i) {
                try {
                    this.updateAllLeases(timeProvider);
                    success = true;
                    continue;
                }
                catch (ProvisionedThroughputException e) {
                    LOG.info(String.format("Worker %s could not find expired leases on try %d out of %d", this.workerIdentifier, i, 3));
                    lastException = e;
                }
            }
        }
        finally {
            MetricsHelper.addSuccessAndLatency("ListLeases", startTime, success, MetricsLevel.DETAILED);
        }
        if (lastException != null) {
            LOG.error("Worker " + this.workerIdentifier + " could not scan leases table, aborting takeLeases. Exception caught by last retry:", lastException);
            return takenLeases;
        }
        List<T> expiredLeases = this.getExpiredLeases();
        Set<T> leasesToTake = this.computeLeasesToTake(expiredLeases);
        HashSet<String> untakenLeaseKeys = new HashSet<String>();
        block11: for (Lease lease : leasesToTake) {
            String leaseKey = lease.getLeaseKey();
            startTime = System.currentTimeMillis();
            success = false;
            try {
                for (int i = 1; i <= 3; ++i) {
                    try {
                        if (this.leaseManager.takeLease(lease, this.workerIdentifier)) {
                            lease.setLastCounterIncrementNanos(System.nanoTime());
                            takenLeases.put(leaseKey, lease);
                        } else {
                            untakenLeaseKeys.add(leaseKey);
                        }
                        success = true;
                        continue block11;
                    }
                    catch (ProvisionedThroughputException e) {
                        LOG.info(String.format("Could not take lease with key %s for worker %s on try %d out of %d due to capacity", leaseKey, this.workerIdentifier, i, 3));
                        continue;
                    }
                }
            }
            finally {
                MetricsHelper.addSuccessAndLatency("TakeLease", startTime, success, MetricsLevel.DETAILED);
            }
        }
        if (takenLeases.size() > 0) {
            LOG.info(String.format("Worker %s successfully took %d leases: %s", this.workerIdentifier, takenLeases.size(), LeaseTaker.stringJoin(takenLeases.keySet(), ", ")));
        }
        if (untakenLeaseKeys.size() > 0) {
            LOG.info(String.format("Worker %s failed to take %d leases: %s", this.workerIdentifier, untakenLeaseKeys.size(), LeaseTaker.stringJoin(untakenLeaseKeys, ", ")));
        }
        MetricsHelper.getMetricsScope().addData("TakenLeases", takenLeases.size(), StandardUnit.Count, MetricsLevel.SUMMARY);
        return takenLeases;
    }

    static String stringJoin(Collection<String> strings, String delimiter) {
        StringBuilder builder = new StringBuilder();
        boolean needDelimiter = false;
        for (String string : strings) {
            if (needDelimiter) {
                builder.append(delimiter);
            }
            builder.append(string);
            needDelimiter = true;
        }
        return builder.toString();
    }

    private void updateAllLeases(Callable<Long> timeProvider) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        List<T> freshList = this.leaseManager.listLeases();
        try {
            this.lastScanTimeNanos = timeProvider.call();
        }
        catch (Exception e) {
            throw new DependencyException("Exception caught from timeProvider", e);
        }
        HashSet<String> notUpdated = new HashSet<String>(this.allLeases.keySet());
        for (Lease lease : freshList) {
            String leaseKey = lease.getLeaseKey();
            Lease oldLease = (Lease)this.allLeases.get(leaseKey);
            this.allLeases.put(leaseKey, lease);
            notUpdated.remove(leaseKey);
            if (oldLease != null) {
                if (oldLease.getLeaseCounter().equals(lease.getLeaseCounter())) {
                    lease.setLastCounterIncrementNanos(oldLease.getLastCounterIncrementNanos());
                    continue;
                }
                lease.setLastCounterIncrementNanos(this.lastScanTimeNanos);
                continue;
            }
            if (lease.getLeaseOwner() == null) {
                lease.setLastCounterIncrementNanos(0L);
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Treating new lease with key " + leaseKey + " as never renewed because it is new and unowned.");
                continue;
            }
            lease.setLastCounterIncrementNanos(this.lastScanTimeNanos);
            if (!LOG.isDebugEnabled()) continue;
            LOG.debug("Treating new lease with key " + leaseKey + " as recently renewed because it is new and owned.");
        }
        for (String key : notUpdated) {
            this.allLeases.remove(key);
        }
    }

    private List<T> getExpiredLeases() {
        ArrayList<Lease> expiredLeases = new ArrayList<Lease>();
        for (Lease lease : this.allLeases.values()) {
            if (!lease.isExpired(this.leaseDurationNanos, this.lastScanTimeNanos)) continue;
            expiredLeases.add(lease);
        }
        return expiredLeases;
    }

    private Set<T> computeLeasesToTake(List<T> expiredLeases) {
        int target;
        Map<String, Integer> leaseCounts = this.computeLeaseCounts(expiredLeases);
        Set<Object> leasesToTake = new HashSet();
        IMetricsScope metrics = MetricsHelper.getMetricsScope();
        int numLeases = this.leaseSelector.getLeaseCountThatCanBeTaken(this.allLeases.values());
        int numWorkers = leaseCounts.size();
        if (numLeases == 0) {
            return leasesToTake;
        }
        if (numWorkers >= numLeases) {
            target = 1;
        } else {
            target = numLeases / numWorkers + (numLeases % numWorkers == 0 ? 0 : 1);
            int leaseSpillover = Math.max(0, target - this.maxLeasesForWorker);
            if (target > this.maxLeasesForWorker) {
                LOG.warn(String.format("Worker %s target is %d leases and maxLeasesForWorker is %d. Resetting target to %d, lease spillover is %d.  Note that some shards may not be processed if no other workers are able to pick them up.", this.workerIdentifier, target, this.maxLeasesForWorker, this.maxLeasesForWorker, leaseSpillover));
                target = this.maxLeasesForWorker;
            }
            metrics.addData("LeaseSpillover", leaseSpillover, StandardUnit.Count, MetricsLevel.SUMMARY);
        }
        int myCount = leaseCounts.get(this.workerIdentifier);
        int numLeasesToReachTarget = target - myCount;
        if (numLeasesToReachTarget <= 0) {
            return leasesToTake;
        }
        Collections.shuffle(expiredLeases);
        int originalExpiredLeasesSize = expiredLeases.size();
        if (expiredLeases.size() > 0) {
            leasesToTake = this.leaseSelector.getLeasesToTakeFromExpiredLeases(expiredLeases, numLeasesToReachTarget);
        } else {
            List<T> leasesToSteal = this.chooseLeasesToSteal(leaseCounts, numLeasesToReachTarget, target);
            for (Lease leaseToSteal : leasesToSteal) {
                LOG.info(String.format("Worker %s needed %d leases but none were expired, so it will steal lease %s from %s", this.workerIdentifier, numLeasesToReachTarget, leaseToSteal.getLeaseKey(), leaseToSteal.getLeaseOwner()));
                leasesToTake.add(leaseToSteal);
            }
        }
        if (!leasesToTake.isEmpty()) {
            LOG.info(String.format("Worker %s saw %d total leases, %d available leases, %d workers. Target is %d leases, I have %d leases, I will take %d leases", this.workerIdentifier, numLeases, originalExpiredLeasesSize, numWorkers, target, myCount, leasesToTake.size()));
        }
        metrics.addData("TotalLeases", numLeases, StandardUnit.Count, MetricsLevel.DETAILED);
        metrics.addData("ExpiredLeases", originalExpiredLeasesSize, StandardUnit.Count, MetricsLevel.SUMMARY);
        metrics.addData("NumWorkers", numWorkers, StandardUnit.Count, MetricsLevel.SUMMARY);
        metrics.addData("NeededLeases", numLeasesToReachTarget, StandardUnit.Count, MetricsLevel.DETAILED);
        metrics.addData("LeasesToTake", leasesToTake.size(), StandardUnit.Count, MetricsLevel.DETAILED);
        return leasesToTake;
    }

    private List<T> chooseLeasesToSteal(Map<String, Integer> leaseCounts, int needed, int target) {
        ArrayList leasesToSteal = new ArrayList();
        Map.Entry<String, Integer> mostLoadedWorker = null;
        for (Map.Entry<String, Integer> worker : leaseCounts.entrySet()) {
            if (mostLoadedWorker != null && (Integer)mostLoadedWorker.getValue() >= worker.getValue()) continue;
            mostLoadedWorker = worker;
        }
        int numLeasesToSteal = 0;
        if ((Integer)mostLoadedWorker.getValue() >= target && needed > 0) {
            int leasesOverTarget = mostLoadedWorker.getValue() - target;
            numLeasesToSteal = Math.min(needed, leasesOverTarget);
            if (needed > 1 && numLeasesToSteal == 0) {
                numLeasesToSteal = 1;
            }
            numLeasesToSteal = Math.min(numLeasesToSteal, this.maxLeasesToStealAtOneTime);
        }
        if (numLeasesToSteal <= 0) {
            if (LOG.isDebugEnabled()) {
                LOG.debug(String.format("Worker %s not stealing from most loaded worker %s.  He has %d, target is %d, and I need %d", this.workerIdentifier, mostLoadedWorker.getKey(), mostLoadedWorker.getValue(), target, needed));
            }
            return leasesToSteal;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("Worker %s will attempt to steal %d leases from most loaded worker %s.  He has %d leases, target is %d, I need %d, maxLeasesToSteatAtOneTime is %d.", this.workerIdentifier, numLeasesToSteal, mostLoadedWorker.getKey(), mostLoadedWorker.getValue(), target, needed, this.maxLeasesToStealAtOneTime));
        }
        String mostLoadedWorkerIdentifier = mostLoadedWorker.getKey();
        ArrayList<Lease> candidates = new ArrayList<Lease>();
        for (Lease lease : this.allLeases.values()) {
            if (!mostLoadedWorkerIdentifier.equals(lease.getLeaseOwner())) continue;
            candidates.add(lease);
        }
        Collections.shuffle(candidates);
        int toIndex = Math.min(candidates.size(), numLeasesToSteal);
        leasesToSteal.addAll(candidates.subList(0, toIndex));
        return leasesToSteal;
    }

    private Map<String, Integer> computeLeaseCounts(List<T> expiredLeases) {
        HashMap<String, Integer> leaseCounts = new HashMap<String, Integer>();
        for (Lease lease : this.allLeases.values()) {
            if (expiredLeases.contains(lease)) continue;
            String leaseOwner = lease.getLeaseOwner();
            Integer oldCount = (Integer)leaseCounts.get(leaseOwner);
            if (oldCount == null) {
                leaseCounts.put(leaseOwner, 1);
                continue;
            }
            leaseCounts.put(leaseOwner, oldCount + 1);
        }
        Integer myCount = (Integer)leaseCounts.get(this.workerIdentifier);
        if (myCount == null) {
            myCount = 0;
            leaseCounts.put(this.workerIdentifier, myCount);
        }
        return leaseCounts;
    }

    @Override
    public String getWorkerIdentifier() {
        return this.workerIdentifier;
    }
}

