/*---------------------------------------------------------------------+
| Module-name:                                                         |
|        callbuff.c                                                    |
|                                                                      |
| NOTE: This program is provided as a sample for the proper sequence   |
|       of calls to the TSM API, and in no way is intended to provide  |
|       a complete consumer/producer implementation.                   |
|       Also, remember that if failures occur it is the responsibility |
|       of the application to return all buffers it is holding before  |
|       calling dsmTerminate.                                          |
|       The pseudo code should be :                                    |
|       while (app has TSM buffers)                                    |        
|         for each buffer call dsmReleaseBuffer                        |
|       call dsmTerminate                                              |
|                                                                      |    
| Function:                                                            |
|        Invokes series of the IBM Storage Protect API's necessary  |
|        to move data from a local file to the TSM server              |
|        while using API buffers for read/write                        |
|        See Using the API Book for more details on memcpy elimination.|
|        usage : callbuff <infile> <outfile>                           |
|        The program performs the following actions                    |
|        o Initialize a session                                        |
|        o Backup <infile> to TSM using API buffers                    |
|        o Query backup                                                |
|        o Restore data to <outfile> using TSM buffers                 | 
|                                                                      |
|                                                                      |
| Setup:                                                               |
|        Set up the environment variables to point to the correct      |
|        directories and the dsm.opt file. DSMI_CONFIG has to contain  |
|        path to the dsm.opt file. DSMI_DIR should point to the        |
|        directory containing message text repository (dscenu.txt)     |
|        and the system preferences file dsm.sys (on UNIX).            |
|        DSMI_LOG specifies directory to store the error log.          |
|                                                                      |
|           set DSMI_CONFIG=\yourDirectoryPath\dsm.opt                 |
|           set DSMI_DIR=\InstallationPath                             |
|           set DSMI_LOG=\AnyDesiredDirectory                          |
|                                                                      |
|        Update dsm.opt or dsm.sys file so that TCPSERVERADDRESS       |
|        contains a correct host name or an IP address of the TSM      |
|        server                                                        |
|                                                                      |
|           COMMMETHOD       TCPip                                     |
|           TCPSERVERADDRESS host.company.com                          |
|                                                                      |
|                                                                      |
|        In the TSM server specified in the dsm.opt or dsm.sys file,   |
|        register node CALLBUFF.                                       |
|                                                                      |
|        Using 'dsmadmc', log on to the IBM Storage Protect server     |
|        and execute the following command:                            |
|                                                                      |
|           REGISTER NODE CALLBUFF CALLBUFF DOMAIN=STANDARD            |
|                                                                      |
|        In order to compile the source code, use makesmp with buff    |
|        argument.                                                     |
|                                                                      |
+---------------------------------------------------------------------*/
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "dsmapitd.h"
#include "dsmapifp.h"
#include "dsmrc.h"
#include "dpsthread.h"

#define NODE1 "CALLBUFF"

/*=== thread function prototype ===*/
void *consumerSend(void *arg);
void *consumerGet(void *arg);

#define READSIZE 64 *1024  /* sets read size from file */
#define BUFCNT  10         /* sets number of buffers */

/*============================================================== 
   this is the shared data structure that is used between the producer
   and consumer threads                                    
===============================================================*/

static struct {
        requestBufferOut_t  buffer[BUFCNT];
        int                 byteinbuf[BUFCNT];
        dpsMutex_t          buflockMutex;
        dpsCondition_t      adddataCond;
        dpsCondition_t      remdataCond;
        int                 nextadd;
        int                 nextrem;
        int                 occ;
        int                 done;
        dsUint32_t          tsmHandle;
} BuffArray;


int NumberOfThreads = 0;
int totalBytes =0;

#if _OPSYS_TYPE == DS_WINNT
   #define FSNAME     "CALLBUFF_FS"
   #define HLNAME     "\\TESTHL"
   #define LLNAME     "\\TESTLL1"
   #define SLEEP(a)   _sleep(a * 1000)
   #define PLATFORM   "Windows"
#else
   #define FSNAME     "/CALLBUFF_FS"
   #define HLNAME     "/TESTHL"
   #define LLNAME     "/TESTLL1"
   #define SLEEP(a)   sleep(a)
   #define PLATFORM   "Unix"
#endif
/*----------------------------------------------------------------------+
| Name:    main()  (producer)      
|
| Action:  o Performs initialization
|          
|          Backup :
|          o Creates an object on the TSM server.
|          o Requests buffers from TSM API and reads data into those 
|              buffers.
|          o Signals consumer to start consuming.
|          o When all the data has been read from file signals done.
|          o Waits till consumer is finished.
|          o Commits txn for the send.
|
|          o Performs a query for the object that was backed up.
|
|          Restore :
|          o Initiates a restore from TSM server.
|          o Recieves data from the TSM server places buffers in buffer
|               array. 
|          
+----------------------------------------------------------------------*/
int main(int argc, char **argv)
{
   FILE                *ifd, *ofd;
   dsInt16_t           apiver=0;
   dsUint32_t          apiVersion, applVersion;
   dsmApiVersionEx     apiLibVer;
   envSetUp            dsmEnvSetUp;
   char                uniStr[20];
   dsInt16_t           rc =0, rc1;
   char                options[100];
   char                configfile[10];
   mcBindKey           mcBindKey;
   ObjAttr             objAttrArea;
   DataBlk             qDataBlkArea;
   qryBackupData       queryBuffer;
   qryRespBackupData   qbDataArea;
   dsUint16_t          reason;
   dsmObjName          objName;
   dsmQueryType        queryType;
   dsmGetList           dsmObjList;
   regFSData           regFilespace;
   dsmApiVersion       dsmApiVer;
   char                rcMsg[DSM_MAX_RC_MSG_LENGTH];
   dsUint32_t          handle = 0;
   requestBufferIn_t   requestBuffIn;
   requestBufferOut_t  requestBuffOut1, requestBuffOut2, requestBuffOut3;
   sendBufferDataIn_t  sendBufferDataIn;
   sendBufferDataOut_t sendBufferDataOut;
   releaseBufferIn_t   relBuffIn;
   releaseBufferOut_t  relBuffOut;
   getBufferDataIn_t   getBufferDataIn;
   getBufferDataOut_t  getBufferDataOut;
   struct stat         st_buffer; 

   dsmInitExIn_t       initIn;
   dsmInitExOut_t      initOut;
   dsmApiVersionEx     apiApplVer;
   
   dsmEndSendObjExIn_t    dsmEndSendObjExIn;
   dsmEndSendObjExOut_t   dsmEndSendObjExOut;

   NumberOfThreads = 1;

   if ((rc = dpsCreateCondition(&BuffArray.adddataCond)) != 0)
      exit (rc);

   if ((rc = dpsCreateCondition(&BuffArray.remdataCond)) != 0)
      exit (rc);

   if ((rc = dpsMutexInit(&BuffArray.buflockMutex)) != 0)
      exit (rc);

    

   memset(&initIn,            0x00, sizeof(dsmInitExIn_t));
   memset(&initOut,           0x00, sizeof(dsmInitExOut_t));
   memset(&apiApplVer,        0x00, sizeof(dsmApiVersionEx));
   memset(&requestBuffIn,     0x00, sizeof(requestBufferIn_t));
   memset(&requestBuffOut1,   0x00, sizeof(requestBufferOut_t));
   memset(&requestBuffOut2,   0x00, sizeof(requestBufferOut_t));
   memset(&requestBuffOut3,   0x00, sizeof(requestBufferOut_t));
   memset(&sendBufferDataIn,  0x00, sizeof(sendBufferDataIn_t));
   memset(&sendBufferDataOut, 0x00, sizeof(sendBufferDataOut_t));
   memset(&relBuffIn,         0x00, sizeof(releaseBufferIn_t));
   memset(&relBuffOut,        0x00, sizeof(releaseBufferOut_t));
   memset(&getBufferDataIn,   0x00, sizeof(getBufferDataIn_t));
   memset(&getBufferDataOut,  0x00, sizeof(getBufferDataOut_t));
   memset(&dsmEndSendObjExIn, 0x00, sizeof(dsmEndSendObjExIn_t));
   memset(&dsmEndSendObjExOut,0x00, sizeof(dsmEndSendObjExOut_t));

   /* check the command line arguments */
   if (argc != 3)
        printf("Usage: %s <infile> <outfile>\n", argv[0]), exit(0);

   memset(&apiLibVer, 0x00, sizeof(dsmApiVersionEx));
   apiLibVer.stVersion = apiVersionExVer;
   dsmQueryApiVersionEx(&apiLibVer);
   
   applVersion = (10000 * DSM_API_VERSION) + (1000 * DSM_API_RELEASE) + (100 * DSM_API_LEVEL)
                 + DSM_API_SUBLEVEL;

   apiVersion = (10000 * apiLibVer.version) + (1000 * apiLibVer.release) + (100 * apiLibVer.level)
                 + apiLibVer.subLevel;

   /* check for compatibility problems */
   if (apiVersion < applVersion)
   { 
      printf("The IBM Storage Protect API library Version = %d.%d.%d.%d is at a lower version\n",
        apiLibVer.version,
        apiLibVer.release,
        apiLibVer.level,
        apiLibVer.subLevel);
     printf("than the application version = %d.%d.%d.%d.\n",
        DSM_API_VERSION,
        DSM_API_RELEASE,
        DSM_API_LEVEL,
        DSM_API_SUBLEVEL);
     printf("Please upgrade the API accordingly.\n");
     return -1;
   }

   printf("\n*************************************************************\n");
   printf(  "* Welcome to the sample Memcopy Elimination for             *\n");
   printf(  "* IBM Storage Protect API.                                 *\n");
   printf(  "* API Library Version = %d.%d.%d.%d                            *\n",
   apiLibVer.version,
   apiLibVer.release,
   apiLibVer.level,
   apiLibVer.subLevel,
   uniStr);
   printf(  "*************************************************************\n");

   memset(&dsmEnvSetUp, 0x00, sizeof(envSetUp));

   dsmEnvSetUp.stVersion = envSetUpVersion;
   strcpy(dsmEnvSetUp.dsmiDir, ""); 
   strcpy(dsmEnvSetUp.dsmiConfig, ""); 
   strcpy(dsmEnvSetUp.dsmiLog, "");
   strcpy(dsmEnvSetUp.logName, "");
   dsmEnvSetUp.argv = argv;

   if ((rc = dsmSetUp(DSM_MULTITHREAD, &dsmEnvSetUp)) != DSM_RC_OK) 
   {
      dsmRCMsg(0,rc,rcMsg);
      printf("Error in dsmSetUp rcMsg=%s\n", rcMsg);
      dsmCleanUp(DSM_MULTITHREAD);
      return 0;
   }
   
      printf("producer started\n");

   dsmApiVer.version = DSM_API_VERSION;
   dsmApiVer.release = DSM_API_RELEASE;
   dsmApiVer.level   = DSM_API_LEVEL  ;

   apiApplVer.version  = DSM_API_VERSION;
   apiApplVer.release  = DSM_API_RELEASE;
   apiApplVer.level    = DSM_API_LEVEL  ;
   apiApplVer.subLevel = DSM_API_SUBLEVEL;

   /************** Initialize the API session ******************************/
   /************ The following 2 can be updated optionally *****************/
   configfile[0]= '\0';    /* API config file for dsmInit */
   options[0] = '\0';

   initIn.stVersion        = dsmInitExInVersion;
   initIn.apiVersionExP    = &apiApplVer;
   initIn.clientNodeNameP  = NODE1;
   initIn.clientOwnerNameP = NULL;
   initIn.clientPasswordP  = NODE1;
   initIn.applicationTypeP = PLATFORM;
   initIn.configfile       = configfile;
   initIn.options          = options;
   initIn.userNameP        = NULL;
   initIn.userPasswordP    = NULL;

   initIn.useTsmBuffers    = bTrue;
   initIn.numTsmBuffers    = BUFCNT;

   if ((rc = dsmInitEx(&handle,&initIn,&initOut)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("producer error in dsmInit. rcMsg=%s\n",rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      return rc;
   }

   printf("producer after dsmInit, session handle is %d\n",handle);

   BuffArray.tsmHandle = handle;


   strcpy(objName.fs,FSNAME);
   strcpy(objName.hl,HLNAME);
   strcpy(objName.ll,LLNAME);
   objName.objType = DSM_OBJ_FILE;

   /************ Build File Space ****************************************/
   regFilespace.fsName = (char *)malloc(100);
   regFilespace.fsType = (char *)malloc(100);

   strcpy(regFilespace.fsName,FSNAME);
   strcpy(regFilespace.fsType,"CALLER1_FS");
   regFilespace.capacity.lo = 0;
   regFilespace.capacity.hi = 0;
   regFilespace.occupancy.lo = 0;
   regFilespace.occupancy.hi = 0;
   regFilespace.stVersion = regFSDataVersion;
#if _OPSYS_TYPE == DS_WINNT
   regFilespace.fsAttr.dosFSAttr.fsInfoLength = 6;
   strcpy(regFilespace.fsAttr.dosFSAttr.fsInfo,"fsinfo");
   regFilespace.fsAttr.dosFSAttr.driveLetter = 'G';
#else
   regFilespace.fsAttr.unixFSAttr.fsInfoLength = 6;
   strcpy(regFilespace.fsAttr.unixFSAttr.fsInfo,"fsinfo");
#endif

   /******************** Register the file space ************************/
   rc = dsmRegisterFS(handle, &regFilespace);

   if ((rc != DSM_RC_OK) && (rc != DSM_RC_FS_ALREADY_REGED)) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("producer error in dsmRegisterFS. rcMsg=%s\n", rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return rc ;
   }

   if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("producer error in dsmBeginTxn. rcMsg=%s\n", rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      return rc;
   }

   mcBindKey.stVersion = mcBindKeyVersion;

   if ((rc = dsmBindMC(handle, &objName, stBackup, &mcBindKey)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("producer error in dsmBindMC. rcMsg=%s\n", rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      return rc;
   }

   /* open the input file for the producer to use */
   if ((ifd = fopen(argv[1], "rb")) == NULL)
   {
      printf("Can't open file %s\n", argv[1]);
      exit(1);
   }
   stat(argv[1], &st_buffer);

   printf ("The length of the file is %u\n", (unsigned)st_buffer.st_size);

   /****************** Backup the data ******************************/
   strcpy(objAttrArea.owner,"");
   objAttrArea.sizeEstimate.hi = 0;                 /* size estimate of object hi*/
   objAttrArea.sizeEstimate.lo = st_buffer.st_size; /* size estimate of object lo*/
   objAttrArea.objCompressed = bFalse;
   objAttrArea.objInfoLength = 11;            /* length of objInfo */
   objAttrArea.objInfo = (char *)malloc(objAttrArea.objInfoLength + 1);
   strcpy(objAttrArea.objInfo,"OBJECT INFO");  /* object-dependent info */
   objAttrArea.stVersion = ObjAttrVersion;
   objAttrArea.mcNameP = NULL;

   if ((rc = dsmSendObj(handle, stBackup, NULL, &objName, &objAttrArea, 
        NULL)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("producer error in dsmBackup. rcMsg=%s\n",rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      return rc;
   }

   requestBuffIn.stVersion = requestBufferInVersion;
   requestBuffIn.dsmHandle = handle;


   /* zero the counters */
   BuffArray.nextadd = BuffArray.nextrem = BuffArray.occ = BuffArray.done = 0;

   /* create the consumer thread */
   NumberOfThreads++;
   dpsThreadCreate(&consumerSend, (void *)NULL);

   /* the producer ! */
   while (1) 
   {
      /* lock the mutex */
      if ((rc = dpsMutexLock(&BuffArray.buflockMutex)) != 0)
         exit (rc);

      /* check to see if any buffers are empty */
      /* If not then wait for that condition to become true */
      while (BuffArray.occ == BUFCNT)
         if ((rc = dpsWaitCondition(&BuffArray.remdataCond, &BuffArray.buflockMutex)) != 0)
            exit (rc);

      /* request a buffer from tsm and set it in bufferArray*/
      printf("call dsmRequestBuffer for slot %d\n", BuffArray.nextadd);
      requestBuffIn.stVersion = requestBufferInVersion;
      requestBuffIn.dsmHandle = handle;
      rc = dsmRequestBuffer(&requestBuffIn, &BuffArray.buffer[BuffArray.nextadd]);
      if (rc != DSM_RC_OK) 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("producer error in dsmSendData. rcMsg=%s\n",rcMsg);
         dsmTerminate(handle);
         dsmCleanUp(DSM_MULTITHREAD);
         exit (rc);
      }
      if (BuffArray.buffer[BuffArray.nextadd].bufferLen < READSIZE)
      {
         printf("producer error bufferLen: %d smaller than READSIZE: %d use a smaller READSIZE\n",
                 BuffArray.buffer[BuffArray.nextadd].bufferLen,  READSIZE);
         dsmTerminate(handle);
         dsmCleanUp(DSM_MULTITHREAD);
         exit (rc);
      }
      /*=== read from the file and put data into a buffer ===*/
      BuffArray.byteinbuf[BuffArray.nextadd] = fread(BuffArray.buffer[BuffArray.nextadd].dataPtr,1,READSIZE, ifd);
      printf("read slot %d numBytes %d \n", BuffArray.nextadd, BuffArray.byteinbuf[BuffArray.nextadd]);

      /* check to see if done reading */
      if (BuffArray.byteinbuf[BuffArray.nextadd] == 0)
      {

         /*=== return last request buffer ===*/
         relBuffIn.stVersion = releaseBufferInVersion;
         relBuffIn.dsmHandle = handle;
         relBuffIn.tsmBufferHandle = BuffArray.buffer[BuffArray.nextadd].tsmBufferHandle;
         relBuffIn.dataPtr = BuffArray.buffer[BuffArray.nextadd].dataPtr;
         relBuffOut.stVersion = releaseBufferOutVersion;
         if ((rc1 = dsmReleaseBuffer(&relBuffIn, &relBuffOut)) != DSM_RC_OK) 
         {
            dsmRCMsg(handle,rc1,rcMsg);
            printf("producer error from dsmEndGetData. rcMsg=%s\n", rcMsg);
            dsmTerminate(handle);
            dsmCleanUp(DSM_MULTITHREAD);
            exit (rc);
         }
 
         /* set the done flag and release the mutex lock */
         BuffArray.done = 1;
         printf("**** set DONE flag ****\n");

         /* signal the consumer to start consuming */
         if ((rc = dpsSignalCondition(&BuffArray.adddataCond)) != 0)
            exit (rc);
        
         /* release the buffer mutex */
         if ((rc = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
            exit (rc);
                
         /* leave the while looop */
         break;
      }

      /* set the next buffer to fill */
      BuffArray.nextadd = ++BuffArray.nextadd % BUFCNT;

      /* increment the number of buffers that are filled */
      BuffArray.occ++;

      /* signal the consumer to start consuming */
      if ((rc = dpsSignalCondition(&BuffArray.adddataCond)) != 0)
         exit (rc);
        
      /* release the mutex */
      if ((rc = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
         exit (rc);
   }

   fclose(ifd);

   /* wait for the consumer to finish */
   while(NumberOfThreads > 1)
      SLEEP(1);

   dsmEndSendObjExIn.stVersion  = dsmEndSendObjExInVersion;
   dsmEndSendObjExIn.dsmHandle  = handle;
   dsmEndSendObjExOut.stVersion = dsmEndSendObjExOutVersion;

   if ((rc = dsmEndSendObjEx(&dsmEndSendObjExIn, &dsmEndSendObjExOut)) != DSM_RC_OK)
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("producer error in dsmEndSendObjEx. rcMsg=%s\n",rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      exit (rc);
   }

   if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("producer error in dsmEndTxn. rcMsg=%s\n", rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      exit (rc);
   }
   printf("\n*******************************************************\n"
                "   Finished send\n"
                "   dsmEndSendObjEx: Total bytes send %d %d\n"
                "   encrypt is %d compress is %d\n"
                "   totalCompress is %d %d totalLFBytesSent %d %d\n"
                "*******************************************************", 
          dsmEndSendObjExOut.totalBytesSent.hi,
          dsmEndSendObjExOut.totalBytesSent.lo,
          dsmEndSendObjExOut.encryptionType,
          dsmEndSendObjExOut.objCompressed,
          dsmEndSendObjExOut.totalCompressSize.hi,
          dsmEndSendObjExOut.totalCompressSize.lo,
          dsmEndSendObjExOut.totalLFBytesSent.hi,
          dsmEndSendObjExOut.totalLFBytesSent.lo);

   /*=== Excute Query ===*/
   queryType = qtBackup;

   qbDataArea.stVersion = qryRespBackupDataVersion;
   qDataBlkArea.stVersion = DataBlkVersion;
   qDataBlkArea.bufferPtr = (char *)&qbDataArea;
   qDataBlkArea.bufferLen = sizeof(qryRespBackupData);

   queryBuffer.objName = (dsmObjName *)malloc(sizeof(dsmObjName));
   queryBuffer.owner = (char *)malloc(100);

   strcpy(queryBuffer.objName->fs, objName.fs);
   strcpy(queryBuffer.objName->hl, objName.hl);
   strcpy(queryBuffer.objName->ll, objName.ll);
   queryBuffer.objName->objType = objName.objType;
   strcpy(queryBuffer.owner, "");
   queryBuffer.objState = DSM_ACTIVE;
   queryBuffer.stVersion = qryBackupDataVersion;

   if ((rc=dsmBeginQuery(handle,queryType, (void *)&queryBuffer )) 
        != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error in dsmBeginQuery. rcMsg=%s\n",rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      exit (rc);
   }

   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
         printf("\n*******************************************************\n"
                "   query response\n"
                "   %s%s%s\n"
                "   sizeEstimate %d %d\n"
                "******************************************************* \n\n", 
	              qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll,
                 qbDataArea.sizeEstimate.hi, qbDataArea.sizeEstimate.lo);
      ;

   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA)) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error from dsmGetNextQObj. rcMsg=%s\n", rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      exit (rc);
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error from dsmEndQuery. rcMsg=%s\n", rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      exit (rc);
   }
   /* reset BuffArray variables */
   BuffArray.nextrem = 0;
   BuffArray.nextadd = 0;
   BuffArray.done    = 0;
   /*=== restore the data ===*/
      /* open the output file for the consumer to use */
   if ((ofd = fopen(argv[2], "wb")) == NULL)
   {
      printf("Can't open file %s\n", argv[2]);
      exit(1);
   }

   /* create the consumerGet thread */
   NumberOfThreads++;
   dpsThreadCreate(&consumerGet, (void *)ofd);

   dsmObjList.stVersion = dsmGetListVersion;
   dsmObjList.numObjId = 1;
   dsmObjList.objId = (ObjID *)malloc(sizeof(ObjID) * dsmObjList.numObjId);

   dsmObjList.objId[0].hi = qbDataArea.objId.hi;
   dsmObjList.objId[0].lo = qbDataArea.objId.lo;

   if ((rc = dsmBeginGetData(handle,bTrue,gtBackup,&dsmObjList)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error from dsmBeginGetData. rcMsg=%s\n", rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      exit (rc);
   }

   rc = dsmGetObj(handle, &(dsmObjList.objId[0]), NULL);

   if (rc == DSM_RC_FINISHED) 
   {
      if ((rc = dsmEndGetObj(handle)) != DSM_RC_OK) 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("producer error from dsmEndGetObj. rcMsg=%s\n", rcMsg);
         dsmTerminate(handle);
         dsmCleanUp(DSM_MULTITHREAD);
         exit (rc);
      }
   }
   else if (rc == DSM_RC_MORE_DATA) 
   {
      getBufferDataIn.stVersion = getBufferDataOutVersion;
      getBufferDataIn.dsmHandle = handle;
      while (rc == DSM_RC_MORE_DATA)
      {
         /* lock the mutex */
         if ((rc = dpsMutexLock(&BuffArray.buflockMutex)) != 0)
            exit (rc);
      
         /* check to see if any buffers are empty */
         /* If not then wait for that condition to become true */
         while (BuffArray.occ == BUFCNT)
            if ((rc = dpsWaitCondition(&BuffArray.remdataCond, &BuffArray.buflockMutex)) != 0)
               exit (rc);
         printf("call dsmGetBufferData for slot %d\n", BuffArray.nextadd);

         rc = dsmGetBufferData(&getBufferDataIn, &getBufferDataOut);
         if (rc == DSM_RC_MORE_DATA)
         {
             /* write the data from the buffer to the file */
             /* place buffer in pool */
            
            BuffArray.buffer[BuffArray.nextadd].tsmBufferHandle = getBufferDataOut.tsmBufferHandle;
            BuffArray.buffer[BuffArray.nextadd].dataPtr = getBufferDataOut.dataPtr;
            BuffArray.buffer[BuffArray.nextadd].bufferLen = getBufferDataOut.numBytes;
            BuffArray.byteinbuf[BuffArray.nextadd] = getBufferDataOut.numBytes; 
            printf("Recv buffer %d %d bytes\n",BuffArray.nextadd, BuffArray.buffer[BuffArray.nextadd].bufferLen);    
            
            /* set the next buffer to fill */
            BuffArray.nextadd = ++BuffArray.nextadd % BUFCNT;

            /* increment the number of buffers that are filled */
            BuffArray.occ++;

            /* signal the consumer to start consuming */
            if ((rc1 = dpsSignalCondition(&BuffArray.adddataCond)) != 0)
               exit (rc1);
        
            /* release the mutex */
            if ((rc1 = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
               exit (rc1);
         }
      }
    
      /* set the done flag and release the mutex lock */
      BuffArray.done = 1;
       printf("**** set DONE flag ****\n");

      /* signal the consumer to start consuming */
      if ((rc1 = dpsSignalCondition(&BuffArray.adddataCond)) != 0)
          exit (rc1);
        
      /* release the buffer mutex */
      if ((rc1 = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
         exit (rc1);
                
      /* wait for the consumer to finish */
      while(NumberOfThreads > 1)
         SLEEP(1);


      if (rc == DSM_RC_FINISHED) 
      {
         if ((rc = dsmEndGetObj(handle)) != DSM_RC_OK) 
         {
            dsmRCMsg(handle,rc,rcMsg);
            printf("producer error from dsmEndGetObj. rcMsg=%s\n", rcMsg);
            dsmTerminate(handle);
            dsmCleanUp(DSM_MULTITHREAD);
            exit (rc);
         }
      }
      else  /* dsmGetData did not finish */ 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("producer error from dsmGetData. rcMsg=%s\n", rcMsg);
         dsmTerminate(handle);
         dsmCleanUp(DSM_MULTITHREAD);
         exit (rc);
      }
   }
   else  /* dsmGetObj did not finish */
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("producer error from dsmGetObj. rcMsg=%s\n", rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      exit (rc);
   }
   if ((rc = dsmEndGetData(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("producer error from dsmEndGetData. rcMsg=%s\n", rcMsg);
      dsmTerminate(handle);
      dsmCleanUp(DSM_MULTITHREAD);
      exit (rc);
   }
   fclose(ofd);
   printf("\n\n*******************************************************\n"
                "  dsmGetBufferData TOTAL bytes restored >%d<\n"
                "  callbuff finished successfully\n"
                "******************************************************* \n\n", 
	              totalBytes);
   if ((rc = dsmTerminate(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("producer error from dsmTerminate. rcMsg=%s\n", rcMsg);
      dsmCleanUp(DSM_MULTITHREAD);
      exit (rc);
   }

   dsmCleanUp(DSM_MULTITHREAD);

   free(regFilespace.fsName);
   free(regFilespace.fsType);
   free(objAttrArea.objInfo);
   free(dsmObjList.objId);
   free(queryBuffer.objName);
   free(queryBuffer.owner);
   /* exit the program */
   return(0);
}

/*----------------------------------------------------------------------+
| Name:    consumerSend()        
|
| Action:  The consumer Send thread waits for a signal that there are
|          buffers to be consumed.
|          It gets buffers from the shared buffer array
|          and sends the data to TSM server                       
|          
+----------------------------------------------------------------------*/
void *consumerSend(void *arg)
{
   FILE                *fd;
   dsInt16_t          rc =0;
   sendBufferDataIn_t  sendBufferDataIn;
   sendBufferDataOut_t sendBufferDataOut;
   char                rcMsg[1024];


   memset(&sendBufferDataIn,     0x00, sizeof(sendBufferDataIn_t));
   memset(&sendBufferDataOut,    0x00, sizeof(sendBufferDataOut_t));



   fd = (FILE *)arg;
   
   /* check to see if any buffers are filled or if the done flag is set */
   while (1) 
   {

      /* lock the mutex */
      if ((rc = dpsMutexLock(&BuffArray.buflockMutex)) != 0)
         exit (rc);
        
     if (!BuffArray.occ && BuffArray.done) 
     {
        printf("                                  **** ConsumerSend DONE occ is %d ****\n", 
               BuffArray.occ);
        if ((rc = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
           exit (rc);
        
        break;
     }
     /* check to see if any buffers are filled */
    /* if not then wait for the condition to become true */
     while (BuffArray.occ == 0 && !BuffArray.done)
        if ((rc = dpsWaitCondition(&BuffArray.adddataCond, &BuffArray.buflockMutex)) != 0)
            exit (rc);
     printf("                                  consumerSend called occ is %d\n", BuffArray.occ);
     if (!BuffArray.occ && BuffArray.done) 
     {
        printf("                                  **** ConsumerSend DONE occ is %d ****\n", 
               BuffArray.occ);
        if ((rc = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
           exit (rc);
        
        break;
     }        
    
     /*SendData to tsm server */
     sendBufferDataIn.stVersion = sendBufferDataInVersion;
     sendBufferDataIn.dsmHandle = BuffArray.tsmHandle;
     sendBufferDataIn.tsmBufferHandle = BuffArray.buffer[BuffArray.nextrem].tsmBufferHandle;
     sendBufferDataIn.dataPtr = BuffArray.buffer[BuffArray.nextrem].dataPtr;
     sendBufferDataIn.numBytes = BuffArray.byteinbuf[BuffArray.nextrem];

     sendBufferDataOut.stVersion = sendBufferDataOutVersion;

     if ((rc = dsmSendBufferData(&sendBufferDataIn, &sendBufferDataOut)) != DSM_RC_OK) 
     {
         dsmRCMsg(BuffArray.tsmHandle,rc,rcMsg);
         printf("consumerSend error in dsmSendData. rcMsg=%s\n",rcMsg);
         dsmTerminate(BuffArray.tsmHandle);
         if ((rc = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
           exit (rc);
         NumberOfThreads--;  
         return NULL;
      }
     printf("                                  consumerSend Write slot %d numbytes %d\n",
            BuffArray.nextrem, BuffArray.byteinbuf[BuffArray.nextrem]); 
    
     /* set the next buffer to write from */
     BuffArray.nextrem = ++BuffArray.nextrem % BUFCNT;

     /* decrement the number of buffers that are full */
     BuffArray.occ--;

     /* signal the producer that a buffer is empty */
     if ((rc = dpsSignalCondition(&BuffArray.remdataCond)) != 0)
         exit (rc);

     /* release the mutex */
     if ((rc = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
        exit (rc);
   }

   /* exit the thread */
   NumberOfThreads--;  
   return NULL;
}

/*---------------------------------------------------------------------------+
| Name:    consumerGet()        
|
| Action:  The consumer Get thread waits for a signal that there are
|          buffers to be consumed.
|          It gets buffers from the shared buffer array
|          and writes the data to the local file. 
|          After the write call dsmRelease buffer to return bufffers to TSM.                      
|          
+---------------------------------------------------------------------------*/
void *consumerGet(void *arg)
{
   FILE                *fd;
   dsInt16_t           rc =0;
   releaseBufferIn_t   relBuffIn;
   releaseBufferOut_t  relBuffOut;
   char                rcMsg[1024];

  
   memset(&relBuffIn,         0x00, sizeof(releaseBufferIn_t));
   memset(&relBuffOut,        0x00, sizeof(releaseBufferOut_t));


   fd = (FILE *)arg;
   
   /* check to see if any buffers are filled or if the done flag is set */
   while (1) 
   {

      /* lock the mutex */
      if ((rc = dpsMutexLock(&BuffArray.buflockMutex)) != 0)
         exit (rc);
        
     if (!BuffArray.occ && BuffArray.done) 
     {
        printf("                                  **** consumerGet DONE occ is %d ****\n", 
               BuffArray.occ);
        if ((rc = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
           exit (rc);
        
        break;
     }
     /* check to see if any buffers are filled */
    /* if not then wait for the condition to become true */
     while (BuffArray.occ == 0 && !BuffArray.done)
        if ((rc = dpsWaitCondition(&BuffArray.adddataCond, &BuffArray.buflockMutex)) != 0)
            exit (rc);

     printf("                                  consumerGet called occ is %d\n", BuffArray.occ);
     if (!BuffArray.occ && BuffArray.done) 
     {
        printf("                                  **** consumerGet DONE occ is %d ****\n", 
               BuffArray.occ);
        if ((rc = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
           exit (rc);
        
        break;
     }
             
     /*=== write data to output file ===*/
     fwrite(BuffArray.buffer[BuffArray.nextrem].dataPtr,1, BuffArray.byteinbuf[BuffArray.nextrem], fd);
     printf("                                  consumerGet Write slot %d numbytes %d\n",
            BuffArray.nextrem, BuffArray.byteinbuf[BuffArray.nextrem]); 

     totalBytes += BuffArray.byteinbuf[BuffArray.nextrem];
     
     /* after write return buffer to tsm */
     relBuffIn.stVersion = releaseBufferInVersion;
     relBuffIn.dsmHandle = BuffArray.tsmHandle;
     relBuffIn.tsmBufferHandle = BuffArray.buffer[BuffArray.nextrem].tsmBufferHandle;
     relBuffIn.dataPtr = BuffArray.buffer[BuffArray.nextrem].dataPtr ;
     relBuffOut.stVersion = releaseBufferOutVersion;
     if ((rc = dsmReleaseBuffer(&relBuffIn, &relBuffOut)) != DSM_RC_OK) 
     {
         dsmRCMsg(BuffArray.tsmHandle,rc,rcMsg);
         printf("consumerGet error from dsmReleaseBuffer. rcMsg=%s\n", rcMsg);
         dsmTerminate(BuffArray.tsmHandle);
         if ((rc = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
           exit (rc);
         NumberOfThreads--;  
         return NULL;
     }
     
     /* set the next buffer to write from */
     BuffArray.nextrem = ++BuffArray.nextrem % BUFCNT;

     /* decrement the number of buffers that are full */
     BuffArray.occ--;

     /* signal the producer that a buffer is empty */
     if ((rc = dpsSignalCondition(&BuffArray.remdataCond)) != 0)
         exit (rc);

     /* release the mutex */
     if ((rc = dpsMutexUnlock(&BuffArray.buflockMutex)) != 0)
        exit (rc);
   }

   /* exit the thread */
   NumberOfThreads--;  
   return NULL;
}



