/*---------------------------------------------------------------------+
| Module-name:                                                         |
|        callhold.c                                                    |
|                                                                      |
| Function:                                                            |
|        Invokes series of the IBM Storage Protect APIs necessary   |
|        to hold, release files and perform the following operations:  |
|                                                                      |
|        o Archive 10 files                                            |
|        o Query archive                                               |
|        o Send Hold event to these files                              |
|        o Query archive                                               |
|        o Send Release event to these files                           |
|        o Query archive                                               |
|                                                                      |
| Setup:                                                               |
|        Set up the environment variables to point to the correct      |
|        directories and the dsm.opt file. DSMI_CONFIG has to contain  |
|        path to the dsm.opt file. DSMI_DIR should point to the        |
|        directory containing message text repository (dscenu.txt)     |
|        and the system preferences file dsm.sys (on UNIX).            |
|        DSMI_LOG specifies directory to store the error log.          |
|                                                                      |
|           set DSMI_CONFIG=\yourDirectoryPath\dsm.opt                 |
|           set DSMI_DIR=\InstallationPath                             |
|           set DSMI_LOG=\AnyDesiredDirectory                          |
|                                                                      |
|        Update dsm.opt or dsm.sys file so that TCPSERVERADDRESS       |
|        contains a correct host name or an IP address of the TSM      |
|        server version 5.2.2 or later.                                |
|                                                                      |
|           COMMMETHOD       TCPip                                     |
|           TCPSERVERADDRESS host.company.com                          |
|                                                                      |
|        In the TSM server specified in the dsm.opt or dsm.sys file,   |
|        register node NODET1.                                         |
|                                                                      |
|        Using 'dsmadmc', log on to the IBM Storage Protect server  |
|        and execute the following command:                            |
|                                                                      |
|           REGISTER NODE NODET1 NODET1 DOMAIN=STANDARD                |
|                                                                      |
|        In order to compile the source code, create a make file       |
|        similar to the one provided for the callmt API sample.        |
|                                                                      |
| Note:                                                                |
|        The TSM Server must be licensed to use deletion hold.         |
|        To verify the license compliance:                             |
|        1. Query for one object and get the ID.                       |
|        2. Issue the dsmBeginTxn, dsmRetentionEvent with Hold,        |
|           and dsmEndTxn.                                             |
|        3. If the server is not licensed, the dsmEndTxn function      |
|           call will fail with reason code                            |
|           DSM_RS_ABORT_LICENSE_VIOLATION.                            |
|                                                                      |
+---------------------------------------------------------------------*/

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "dsmapitd.h"
#include "dsmapifp.h"
#include "dsmrc.h"

#define NODE1   "nodet1"
#define MC      "DEFAULT"
#define MTAPI   316
#define SRVR    522

#if _OPSYS_TYPE == DS_WINNT
   #define PLATFORM   "Windows"
#else
   #define PLATFORM   "Unix"
#endif

void *callHold(void);

static void QryPrintObjInfo(qryRespArchiveData qbDataArea);

int main(int argc, char *argv[]) 
{
    dsInt16_t          rc= 0, apiver=0;
    dsUint32_t         apiVersion, applVersion;
    dsmApiVersionEx    apiLibVer;
    envSetUp           dsmEnvSetUp;
    char               *rcMsg;
    char               uniStr[20];

    memset(&apiLibVer, 0x00, sizeof(dsmApiVersionEx));
    apiLibVer.stVersion = apiVersionExVer;
    dsmQueryApiVersionEx(&apiLibVer);

    applVersion = (10000 * DSM_API_VERSION) + (1000 * DSM_API_RELEASE) + (100 * DSM_API_LEVEL)
            + DSM_API_SUBLEVEL;

    apiVersion = (10000 * apiLibVer.version) + (1000 * apiLibVer.release) + (100 * apiLibVer.level)
            + apiLibVer.subLevel;

    /* check for compatibility problems */
    if (apiVersion < applVersion)
    { 
        printf("The IBM Storage Protect API library Version = %d.%d.%d.%d is at a lower version\n",
        apiLibVer.version,
        apiLibVer.release,
        apiLibVer.level,
        apiLibVer.subLevel);
        printf("than the application version = %d.%d.%d.%d.\n",
        DSM_API_VERSION,
        DSM_API_RELEASE,
        DSM_API_LEVEL,
        DSM_API_SUBLEVEL);
        printf("Please upgrade the API accordingly.\n");
        return -1;
    }

    apiver = (apiLibVer.version * 100) + (apiLibVer.release * 10) + apiLibVer.level;
    if (apiver < MTAPI)
    {
        printf("\n***********************************************************\n");
        printf(  "The IBM Storage Protect API library is lower than 3.1.6 \n");
        printf(  "(multithreaded API)                                        \n");
        printf(  "Install the current level.                                 \n");
        printf(  "***********************************************************\n");
        return 0;
    }

    if (apiver >= 412)
    {
        if(apiLibVer.unicode)
            strcpy(uniStr,"(unicode)");
        else
            strcpy(uniStr,"         ");
    }

    printf("\n*************************************************************\n");
    printf(  "* Welcome to the sample application for                     *\n");
    printf(  "* IBM Storage Protect API.                                 *\n");
    printf(  "* API Library Version = %d.%d.%d.%d %s                   *\n",
    apiLibVer.version,
    apiLibVer.release,
    apiLibVer.level,
    apiLibVer.subLevel,
    uniStr);
    printf(  "*************************************************************\n");

    memset(&dsmEnvSetUp, 0x00, sizeof(envSetUp));

    dsmEnvSetUp.stVersion = envSetUpVersion;
    strcpy(dsmEnvSetUp.dsmiDir, ""); 
    strcpy(dsmEnvSetUp.dsmiConfig, ""); 
    strcpy(dsmEnvSetUp.dsmiLog, "");
    strcpy(dsmEnvSetUp.logName, "");
    dsmEnvSetUp.argv = argv;

    if ((rc = dsmSetUp(DSM_SINGLETHREAD, &dsmEnvSetUp)) != DSM_RC_OK) 
    {
        rcMsg = (char *)malloc(DSM_MAX_RC_MSG_LENGTH);
        dsmRCMsg(0,rc,rcMsg);
        printf("Error in dsmSetUp rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmCleanUp(DSM_SINGLETHREAD);
        return 0;
    }

    callHold();
        
    dsmCleanUp(DSM_SINGLETHREAD);
    
    return 0;
}

void *callHold()
{
    int                     i;
    char                    *rcMsg;
    char                    options[100], temp[10];
    char                    configfile[10];
    char                    fsDelimiter[2] = {' ', '\0'};
    char                    hlDelimiter[2] = {' ', '\0'};
    char                    desc[2] = {'*','\0'};
    mcBindKey               mcBindKey;
    ObjAttr                 objAttrArea;
    DataBlk                 dataBlkArea, qDataBlkArea;
    qryArchiveData          queryBuffer;
    qryRespArchiveData      qbDataArea;
    sndArchiveData          archData;
    dsmObjName              objName;
    dsUint16_t              reason;
    dsInt16_t               rc = 0;
    dsmQueryType            queryType;
    regFSData               regFilespace;
    dsmApiVersion           dsmApiVer;
    dsUint32_t              handle = 0;
    ApiSessInfo             dsmSessInfo;
    DataBlk                 qData;
    dsmObjList_t            objList;
    dsmInitExIn_t           initIn;
    dsmInitExOut_t          initOut;
    dsmApiVersionEx         apiApplVer;
    dsStruct64_t            id[1000];
    int                     count, numObj; 
    dsmRetentionEventIn_t   dsmRetentionEventIn;
    dsmRetentionEventOut_t  dsmRetentionEventOut;
    dsUint32_t              srvrVersion;

    memset(&initIn,     0x00, sizeof(dsmInitExIn_t));
    memset(&initOut,    0x00, sizeof(dsmInitExOut_t));
    memset(&apiApplVer, 0x00, sizeof(dsmApiVersionEx));

    printf("callHold started\n");

    dsmApiVer.version   = DSM_API_VERSION;
    dsmApiVer.release   = DSM_API_RELEASE;
    dsmApiVer.level     = DSM_API_LEVEL  ;

    apiApplVer.version  = DSM_API_VERSION;
    apiApplVer.release  = DSM_API_RELEASE;
    apiApplVer.level    = DSM_API_LEVEL  ;
    apiApplVer.subLevel = DSM_API_SUBLEVEL;

    objList.stVersion   = dsmObjlistVersion;
    objList.numObjId    = 0;
    objList.objId       = &id[0];

    /************ Initialize the API session **********************************/
    rcMsg = (char *)malloc(DSM_MAX_RC_MSG_LENGTH);

    /*************** The following 2 can be updated optionally ****************/
    configfile[0]= '\0';
    options[0]= '\0';
    
    initIn.stVersion        = dsmInitExInVersion;
    initIn.apiVersionExP    = &apiApplVer;
    initIn.clientNodeNameP  = NODE1;
    initIn.clientOwnerNameP = NULL;
    initIn.clientPasswordP  = NODE1;
    initIn.applicationTypeP = PLATFORM;
    initIn.configfile       = configfile;
    initIn.options          = options;
    initIn.userNameP        = NULL;
    initIn.userPasswordP    = NULL;

    if ((rc = dsmInitEx(&handle,&initIn,&initOut)) != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmInit. rcMsg=%s\n",rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
	    return NULL;
    }

    printf("After dsmInit, session handle is %d\n",handle);

    memset(&dsmSessInfo,0x00,sizeof(ApiSessInfo));  /* Zero out block.     */
    dsmSessInfo.stVersion = ApiSessInfoVersion;     /* Init struct version */
    if ((rc = dsmQuerySessInfo(handle, &dsmSessInfo)) != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmQuerySessInfo. rcMsg=%s\n",rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
	    return NULL;
    }           

    printf("\n************************************************************\n"
            "After QuerySess:\n"
            "Server's ver/rel/lev       : %d/%d/%d/%d\n"
            "ArchiveRetentionProtection : %s \n"
            "************************************************************\n",
            dsmSessInfo.serverVer, dsmSessInfo.serverRel, 
            dsmSessInfo.serverLev, dsmSessInfo.serverSubLev,
            dsmSessInfo.archiveRetentionProtection ? "Yes" : "No");

    /* Check that the TSM Server version is at least 5.2.2 */
    srvrVersion = dsmSessInfo.serverVer*100 + dsmSessInfo.serverRel*10 + dsmSessInfo.serverLev;
    if (srvrVersion < SRVR)
    {   printf("\nThis sample cannot be used with pre-5.2.2 TSM Server.\n");
        printf("Please update the dsm.opt or dsm.sys file.\n");
        dsmTerminate(handle);
	    return NULL;
    }

    /************ Get file space and hl delimiters *************************/
    fsDelimiter[0] = dsmSessInfo.fsdelim == ':' ? '\0' : dsmSessInfo.fsdelim;
    hlDelimiter[0] = dsmSessInfo.hldelim;

    /************ Build File Space ****************************************/
    memset(&regFilespace,0x00,sizeof(regFSData));
    regFilespace.fsName = (char *)malloc(100);
    regFilespace.fsType = (char *)malloc(100);
    strcpy(regFilespace.fsName, fsDelimiter);
    strcat(regFilespace.fsName,"t5fs");
    strcpy(regFilespace.fsType,"CALLER4_FS");
    regFilespace.capacity.lo = 0;
    regFilespace.capacity.hi = 0;
    regFilespace.occupancy.lo = 0;
    regFilespace.occupancy.hi = 0;
    regFilespace.stVersion = regFSDataVersion;

#if defined(WIN32) || defined(WIN64)
    regFilespace.fsAttr.dosFSAttr.driveLetter = 'G';
    regFilespace.fsAttr.dosFSAttr.fsInfoLength = 7;
    strcpy(regFilespace.fsAttr.dosFSAttr.fsInfo,"fsinfo");
#elif defined(NOVELL)
    regFilespace.fsAttr.netwareFSAttr.fsInfoLength = 7;
    strcpy(regFilespace.fsAttr.netwareFSAttr.fsInfo,"fsinfo");
#else
    regFilespace.fsAttr.unixFSAttr.fsInfoLength = 7;
    strcpy(regFilespace.fsAttr.unixFSAttr.fsInfo,"fsinfo");
#endif
        
    /************ Register the file space ********************************/
    rc = dsmRegisterFS(handle, &regFilespace);

    if ((rc != DSM_RC_OK) && (rc != DSM_RC_FS_ALREADY_REGED)) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmRegisterFS. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
	    return NULL;
    }
    for (numObj = 1; numObj<=10; numObj++)
    {
    if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK) 
    {
	    dsmRCMsg(handle,rc,rcMsg);
	    printf("Error in dsmBeginTxn. rcMsg=%s\n", rcMsg);
	    free(rcMsg);
	    dsmTerminate(handle);
	    return NULL;
    }

    mcBindKey.stVersion = mcBindKeyVersion;

    strcpy(objName.fs,fsDelimiter);
    strcat(objName.fs,"t5fs");
    strcpy(objName.hl,hlDelimiter);
    strcat(objName.hl,"testhl");
    strcpy(objName.ll,hlDelimiter);
    strcat(objName.ll,"testll");
    sprintf(temp,"%d", numObj);
    strcat(objName.ll,temp);
    objName.objType = DSM_OBJ_FILE;

    if ((rc = dsmBindMC(handle, &objName, stArchive, &mcBindKey)) != DSM_RC_OK) 
    {
	    dsmRCMsg(handle,rc,rcMsg);
	    printf("Error in dsmBindMC. rcMsg=%s\n", rcMsg);
	    free(rcMsg);
	    dsmTerminate(handle); 
	    return NULL;
    }

    archData.stVersion = sndArchiveDataVersion;
    archData.descr = desc;

    strcpy(objAttrArea.owner,"");
    objAttrArea.sizeEstimate.hi = 0;         /* size estimate of object */
    objAttrArea.sizeEstimate.lo = 460;       /* size estimate of object */
    objAttrArea.objCompressed = bFalse;
    objAttrArea.objInfoLength = 10;          /* length of objInfo */
    objAttrArea.objInfo = (char *)malloc(objAttrArea.objInfoLength + 1);
    strcpy(objAttrArea.objInfo,"abcdefghij");      /* object-dependent info */
    objAttrArea.stVersion = ObjAttrVersion;
    objAttrArea.mcNameP = MC;


    /***************** Archive the data  *************************************/
    if ((rc = dsmSendObj(handle, stArchive, &archData, &objName, &objAttrArea,
		    NULL)) != DSM_RC_OK) 
    {
	    dsmRCMsg(handle,rc,rcMsg);
	    printf("Error in dsmSendObj. rcMsg=%s\n",rcMsg);
	    free(rcMsg);
	    dsmTerminate(handle);
	    return NULL;
    }

    dataBlkArea.bufferPtr = (char *)malloc(20);
    dataBlkArea.stVersion = DataBlkVersion;
    dataBlkArea.bufferLen = 20;
    memcpy(dataBlkArea.bufferPtr,"T1T1T1T1T1T1T1T1T1T1", 20); 

    for (i=1; i<=24; i++) 
    {
	    if ((rc = dsmSendData(handle,&dataBlkArea)) != DSM_RC_OK) 
	    {
		    dsmRCMsg(handle,rc,rcMsg);
		    printf("Error in dsmSendData. rcMsg=%s\n",rcMsg);
		    free(rcMsg);
		    dsmTerminate(handle);
		    return NULL;
	    }

	    if(i == 23)
		    memcpy(dataBlkArea.bufferPtr,"T1T1T1T1T1T1T1T1T1T\0", 20);
    }

    if ((rc = dsmEndSendObj(handle)) != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmEndSendObj. rcMsg=%s\n",rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
	    return NULL;
    }
    	
    if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK) 
    {
	    dsmRCMsg(handle,rc,rcMsg);
	    printf("Error in dsmEndTxn. rcMsg=%s\n", rcMsg);
	    free(rcMsg);
	    dsmTerminate(handle);
	    return NULL;
    }
    printf("finished sending >%s%s%s<\n",
		    objName.fs, objName.hl,objName.ll);

    }
    /*********************** Archive query  *****************************/
    queryType = qtArchive;

    qbDataArea.stVersion = qryRespArchiveDataVersion;

    qDataBlkArea.stVersion = DataBlkVersion;
    qDataBlkArea.bufferPtr = (char *)&qbDataArea;
    qDataBlkArea.bufferLen = sizeof(qryRespArchiveData); 

    queryBuffer.objName = (dsmObjName *)malloc(sizeof(dsmObjName));
    queryBuffer.owner = (char *)malloc(100);

    strcpy(queryBuffer.objName->fs, objName.fs);
    strcpy(queryBuffer.objName->hl, objName.hl);
    strcpy(queryBuffer.objName->ll, hlDelimiter);
    strcat(queryBuffer.objName->ll, "*");
    queryBuffer.objName->objType = objName.objType;
    strcpy(queryBuffer.owner, "");
    queryBuffer.stVersion = qryArchiveDataVersion;
    queryBuffer.descr = desc;

    queryBuffer.insDateLowerBound.year = DATE_MINUS_INFINITE;
    queryBuffer.insDateUpperBound.year = DATE_PLUS_INFINITE;
    queryBuffer.expDateLowerBound.year = DATE_MINUS_INFINITE;
    queryBuffer.expDateUpperBound.year = DATE_PLUS_INFINITE;

    if ((rc=dsmBeginQuery(handle,queryType, (void *)&queryBuffer )) 
        != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmBeginQuery. rcMsg=%s\n",rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
	    return NULL;
    }

    count =0;
    while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
    {
        QryPrintObjInfo(qbDataArea);
        id[count].hi = qbDataArea.objId.hi;
        id[count].lo = qbDataArea.objId.lo;
        count ++;
    }
    objList.numObjId = count;

    if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA)) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmGetNextQObj. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
        return NULL;
    }

    if ((rc = dsmEndQuery(handle)) != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmEndQuery. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
        return NULL;
    }

    /********************** Send Hold event *************************/
    if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK)
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmBeginTxn. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
        return NULL;
    }

    memset(&dsmRetentionEventIn,  0x00, sizeof(dsmRetentionEventIn_t));
    memset(&dsmRetentionEventOut, 0x00, sizeof(dsmRetentionEventOut_t));

    dsmRetentionEventIn.stVersion  = dsmRetentionEventInVersion;
    dsmRetentionEventIn.dsmHandle  = handle;
    dsmRetentionEventIn.eventType  = eventHoldObj;
    dsmRetentionEventIn.objList    = objList;

    dsmRetentionEventOut.stVersion = dsmRetentionEventOutVersion; 

    if ((rc = dsmRetentionEvent(&dsmRetentionEventIn, &dsmRetentionEventOut)) != DSM_RC_OK)
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmRetentionEvent. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
        return NULL;
    }
    if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmEndTxn. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
	    return NULL;
    }

    printf("\n\nfinished eventHoldObj\n\n");

    /*********************** Archive query  *****************************/
    if ((rc=dsmBeginQuery(handle,queryType, (void *)&queryBuffer )) 
        != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmBeginQuery. rcMsg=%s\n",rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
	    return NULL;
    }

    while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
        QryPrintObjInfo(qbDataArea);
    
    if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA)) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmGetNextQObj. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
        return NULL;
    }

    if ((rc = dsmEndQuery(handle)) != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmEndQuery. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
        return NULL;
    }

    /********************** Send Release event *************************/
    if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK)
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmBeginTxn. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
        return NULL;
    }

    memset(&dsmRetentionEventIn,  0x00, sizeof(dsmRetentionEventIn_t));
    memset(&dsmRetentionEventOut, 0x00, sizeof(dsmRetentionEventOut_t));

    dsmRetentionEventIn.stVersion  = dsmRetentionEventInVersion;
    dsmRetentionEventIn.dsmHandle  = handle;
    dsmRetentionEventIn.eventType  = eventReleaseObj;
    dsmRetentionEventIn.objList    = objList;

    dsmRetentionEventOut.stVersion = dsmRetentionEventOutVersion; 

    if ((rc = dsmRetentionEvent(&dsmRetentionEventIn, &dsmRetentionEventOut)) != DSM_RC_OK)
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmRetentionEvent. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
        return NULL;
    }
    if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmEndTxn. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
	    return NULL;
    }

    printf("\n\nfinished eventReleaseObj\n\n");

    /*********************** Archive query  *****************************/
    if ((rc=dsmBeginQuery(handle,queryType, (void *)&queryBuffer )) 
        != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmBeginQuery. rcMsg=%s\n",rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
	    return NULL;
    }

    while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
        QryPrintObjInfo(qbDataArea);
    
    if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA)) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmGetNextQObj. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
        return NULL;
    }

    if ((rc = dsmEndQuery(handle)) != DSM_RC_OK) 
    {
        dsmRCMsg(handle,rc,rcMsg);
        printf("Error in dsmEndQuery. rcMsg=%s\n", rcMsg);
        free(rcMsg);
        dsmTerminate(handle);
        return NULL;
    }

    dsmTerminate(handle);

    free(regFilespace.fsName);
    free(regFilespace.fsType);
    free(objAttrArea.objInfo);
    free(dataBlkArea.bufferPtr);
    free(queryBuffer.objName);
    free(queryBuffer.owner);
    free(rcMsg);
    
    printf("\ncallHold finished successfully.\n\n");
    return NULL;
}

/*----------------------------------------------------------------------+
| Name:    QryPrintObjInfo
|
| Action:  Output object information.
+----------------------------------------------------------------------*/
static void QryPrintObjInfo(qryRespArchiveData qbDataArea)
{
    char   retInit[10], objHeld[10];

    switch(qbDataArea.retentionInitiated)
    {
        case DSM_ARCH_RETINIT_STARTED:
            strcpy(retInit, "STARTED");
            break;
        case DSM_ARCH_RETINIT_PENDING:
            strcpy(retInit, "PENDING");
            break;
        default:
            strcpy(retInit, "UNKNOWN");
    }
    switch(qbDataArea.objHeld)
    {
        case DSM_ARCH_HELD_FALSE:
            strcpy(objHeld, "FALSE");
            break;
        case DSM_ARCH_HELD_TRUE:
            strcpy(objHeld, "TRUE");
            break;
        default:
            strcpy(objHeld, "UNKNOWN");
    }

    printf("\n*** Query response ***\n"
        "*** objname             : %s%s%s ***\n"
        "*** managment class     : %s ***\n"
        "*** Retention Initiated : >%s< ***\n"
        "*** Object Held         : >%s< ***\n"
        "*** obj id              : %u-%u ***\n",
    qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll,
    qbDataArea.mcName, retInit, objHeld,
    qbDataArea.objId.hi, qbDataArea.objId.lo);
}
