/***********************************************************************
* Name:
*        dapdsStruct64_t.c
*
* Function:
*        Operators on Large (64-bit) Integers
*
* Public Routines:
*        dIncr64()
*        dDecr64()
*        dAdd64()
*        dSub64()
*        dShiftR64()
*        dShiftL64()
*        dMul64()
*        dDiv64()
*        dI64toCh()
*
***********************************************************************/

/* Standard (ANSI) C library header files. */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/* Application header files. */
#include "dsmapitd.h"
#include "dsmapifp.h"
#include "dapint64.h"

/***********************************************************************
*        Public Routines
***********************************************************************/

/*----------------------------------------------------------------------
* Name:    dMake64()
*
* Action:  Construct a 64-bit integer from a 32-bit integer.
*
* Input:   n - 32 bit integer to be used to construct 64-bit integer.
*
* Return
* Value:   Constructed 64-bit integer.
*
* Side
* Effects: None.
*
* Notes:   All dsStruct64_t's are assumed to be unsigned.
*---------------------------------------------------------------------*/
dsStruct64_t dMake64( dsUint32_t n )

{
  dsStruct64_t result;

  result.hi = 0;
  result.lo = n;

  return ( result );
} /* dMake64() */

/*----------------------------------------------------------------------
* Name:    dIncr64()
*
* Action:  Increment a 64-bit integer by some 32-bit integer.
*
* Input:   x - 64-bit integer to be incremented.
*          n - 32-bit increment.
*
* Return
* Value:   Result of increment operation.
*
* Side
* Effects: None.
*
* Notes:   All dsStruct64_t's are assumed to be unsigned.
*          Assume no overflow of sum.
*          Rewritten for efficiency
*---------------------------------------------------------------------*/
dsStruct64_t dIncr64( dsStruct64_t *x, dsUint32_t n )
{
  dsStruct64_t   xCopy;

  xCopy.hi = x->hi;
  xCopy.lo = x->lo;

  xCopy.lo += n;                    /* add lo order word */

  if ( xCopy.lo < n )               /* did the add cause an overflow? */
    xCopy.hi++;                     /* yes, adjust hi word for carry */

  return ( xCopy );
} /* dIncr64() */

/*----------------------------------------------------------------------
* Name:    dDecr64()
*
* Action:  Decrement a 64-bit integer by some 32-bit integer.
*
* Input:   x - 64-bit integer to be decremented.
*          n - 32-bit decrement.
*
* Return
* Value:   Result of decrement operation.
*
* Side
* Effects: largeIntP - replaced by the result of the decrement
*                      operation.
*
* Notes:   All dsStruct64_t's are assumed to be unsigned.
*          Assume no underflow of difference.
*          Rewritten for efficiency
*---------------------------------------------------------------------*/
dsStruct64_t dDecr64( dsStruct64_t *x, dsUint32_t n )
{

  dsStruct64_t   xCopy;

  xCopy.hi = x->hi;
  xCopy.lo = x->lo;

  if ( xCopy.lo < n )               /* is a borrow required? */
    xCopy.hi--;                     /* yes, borrow from hi word */

  xCopy.lo -= n;                    /* subtract lo order word */

  return ( xCopy );
} /* dDecr64() */

/*----------------------------------------------------------------------
* Name:    dAdd64()
*
* Action:  Add two 64-bit integers.
*
* Input:   x - first 64-bit integer.
*          y - second 64-bit integer.
*
* Return
* Value:   Result of addition operation.
*
* Side
* Effects: None.
*
* Notes:   All dsStruct64_t's are assumed to be unsigned.
*          Assume no overflow of sum.
*          Rewritten for efficiency
*---------------------------------------------------------------------*/
dsStruct64_t dAdd64( dsStruct64_t *x, dsStruct64_t *y )
{
  dsStruct64_t   xCopy, yCopy;

  xCopy.hi = x->hi;
  xCopy.lo = x->lo;
  yCopy.hi = y->hi;
  yCopy.lo = y->lo;

  xCopy = dIncr64( &xCopy, yCopy.lo);
  xCopy.hi += yCopy.hi;

  return ( xCopy );

} /* dAdd64() */

/*----------------------------------------------------------------------
* Name:    dSub64()
*
* Action:  Subtract two 64-bit integers.
*
* Input:   x - first 64-bit integer.
*          y - second 64-bit integer.
*
* Return
* Value:   Result of subtracting second 64-bit integer from the first.
*
* Side
* Effects: None.
*
* Notes:   All dsStruct64_t's are assumed to be unsigned.
*          Assume no underflow of difference (ie. x >= y)
*          Rewritten for efficiency
*---------------------------------------------------------------------*/
dsStruct64_t dSub64( dsStruct64_t *x, dsStruct64_t *y )
{
  dsStruct64_t   xCopy, yCopy;

  xCopy.hi = x->hi;
  xCopy.lo = x->lo;
  yCopy.hi = y->hi;
  yCopy.lo = y->lo;

  xCopy = dDecr64( &xCopy, yCopy.lo);
  xCopy.hi -= yCopy.hi;

  return ( xCopy );

} /* dSub64() */

/*----------------------------------------------------------------------
* Name:    dShiftR64()
*
* Action:  Performs a logical shift right on an dsStruct64_t by b bits.
*
* Input:   x - the address of the 64-bit integer.
*          b - number bits to shift.
*
* Return
* Value:   the value passed in is shifted
*
* Side
* Effects: None.
*
* Notes:   All dsStruct64_t's are assumed to be unsigned.
*        As expected, bits shifted below bit 0 go into the bit bucket.
*---------------------------------------------------------------------*/
void dShiftR64(dsStruct64_t *x, unsigned int b)
{
  if (b >= 32)
  {
    x->lo = x->hi;
    x->hi = 0;
    b &= 0x1F;                 /* get rid of all bits above value 32  */
  }
  x->lo >>= b;                 /* shift the low word into position    */
  x->lo |= (x->hi << (32-b));  /* OR in the bits remaining of hi word */
  x->hi >>= b;                 /* shift the hi word by x bits         */
}  /* end dShiftR64() */

/*----------------------------------------------------------------------
* Name:    dShiftL64()
*
* Action:  Performs a logical shift left on an dsStruct64_t by b bits.
*
* Input:   x - the address of the 64-bit integer.
*          b - number of bits to shift.
*
* Return
* Value:   The value passed in is shifted
*
* Side
* Effects: None.
*
* Notes:   All dsStruct64_t's are assumed to be unsigned.
*        As expected, bits shifted above bit 63 go into the bit bucket.
*---------------------------------------------------------------------*/
void dShiftL64(dsStruct64_t *x, unsigned int b)
{
  if (b >= 32)
  {
    x->hi = x->lo;
    x->lo = 0;
    b &= 0x1F;                 /* get rid of all bits above value 32  */
  }
  x->hi <<= b;                 /* shift the hi word into position     */
  x->hi |= (x->lo >> (32-b));  /* OR in the bits remaining of lo word */
  x->lo <<= b;                 /* shift the lo word by x bits         */
}  /* end dShiftL64() */

/*----------------------------------------------------------------------
* Name:    dMul64()
*
* Action:  Multiply two 64 bit integers.
*
* Input:   x - first 64 bit integer
*          y - second 64 bit integer
*          overflow - address of an integer
*
* Return
* Values:  1) 64 bit Product (x * y) is returned
*          2) overflow set to 0 or 1 to indicate if Product exceeded
*             64 bits (1 indicates overflow)
*
* Side
* Effects: None.
*
* Notes:   All dsStruct64_t's are assumed to be unsigned.
*          Any overflow above 64 bits will be lost.
*---------------------------------------------------------------------*/
dsStruct64_t dMul64(dsStruct64_t *x, dsStruct64_t *y, int *overflow)
{
dsStruct64_t  Prod, xCopy, yCopy;
int    bits_lost = 0;            /* 1 = bits have been lost on hi end */

  *overflow = 0;                        /* start w/ no overflow */
  Prod.hi = 0;                          /* initialize Product  */
  Prod.lo = 0;

  xCopy.hi = x->hi;
  xCopy.lo = x->lo;
  yCopy.hi = y->hi;
  yCopy.lo = y->lo;

  do {
     if (yCopy.lo & 0x1)  /* if bit 0 is 1, curr power of 2 will be added */
     {
        /* now that we're adding again, if there have been any bits   */
        /* lost from hi end of x, we know there's an overflow.        */
        if (bits_lost)
           *overflow = 1;
        Prod = dAdd64(&Prod,&xCopy);         /* add shifted x to the product */
     }
     if (xCopy.hi & 0x80000000)    /* if bit 63 of x is on before ShiftL, */
        bits_lost = 1;         /* indicate bits have been lost */
     dShiftL64(&xCopy,1);           /* double the value being multiplied   */
     dShiftR64(&yCopy,1);           /* shift next test bit into position   */
  } while (yCopy.hi || yCopy.lo);    /* until all y bits are 0 (ie.Mult done) */

  return ( Prod );
}  /* end dMul64() */

/*----------------------------------------------------------------------
* Name:    dDiv64()
*
* Action:  Divide two 64 bit integers.
*
* Input:   x - first 64 bit integer
*          y - second 64 bit integer
*
* Return
* Values:  1) Quotient is returned
*          2) Remainder is stored in specified address
*
* Side
* Effects: None.
*
* Notes:   All dsStruct64_t's are assumed to be unsigned.
*          As usual, bits shifted below bit 0 go into the bit bucket.
*---------------------------------------------------------------------*/
dsStruct64_t dDiv64(dsStruct64_t *x, dsStruct64_t *y, dsStruct64_t *rmdrP)
{
dsStruct64_t  Quot,    /* the final answer is accumulated here       */
       tQuot,   /* temp. quotient to find a single power of 2 */
                /*  that fits into the dividend on each pass  */
       divisor,
       xCopy, yCopy;

  Quot.hi = 0;                          /* initialize Quotient */
  Quot.lo = 0;

  xCopy.hi = x->hi;
  xCopy.lo = x->lo;
  yCopy.hi = y->hi;
  yCopy.lo = y->lo;

  if (yCopy.hi==0 && yCopy.lo==0)               /* check for divide by zero */
  {
     yCopy.hi = 1/yCopy.lo;         /* force a divide by zero so that error-  */
                            /*  handling will be platform-specific    */
  }
  else  /* do the division */
  {
     while (dCmp64(&xCopy,&yCopy) != DS_LESSTHAN)    /* while x >= y            */

     {
    /* find largest Y (multiplied by powers of 2) that is less than X */
        divisor = yCopy;

        tQuot.hi = 0; /* init temp quotient (at least 1y goes into x) */
        tQuot.lo = 1;
        while (dCmp64(&xCopy,&divisor) != DS_LESSTHAN  &&
                        (divisor.hi <= 0x7FFFFFFF))
        {
           dShiftL64(&divisor,1);      /* double divisor (2y 4y 8y...) */
           dShiftL64(&tQuot,1);        /* count number of y's so far   */
        }
        dShiftR64(&divisor,1); /* back up to a value that's less than X*/
        dShiftR64(&tQuot,1);

        Quot = dAdd64(&Quot,&tQuot);      /* calc cumulative quotient    */
        xCopy = dSub64(&xCopy,&divisor);  /* deduct the multiples so far */
     }
     /*rmdrP = &xCopy;*/               /* save remainder in specified address */
     rmdrP->hi = xCopy.hi;
     rmdrP->lo = xCopy.lo;
  }
  return ( Quot );
}  /* end dDiv64() */

/*----------------------------------------------------------------------
* Name:    dI64toCh()
*
* Action:  Converts an dsStruct64_t to it's character respresentation.
*          Currently only decimal is supported.
*
* Input: value - the 64 bit integer
*        string - a pointer to a character output buffer
*        radix - the base to be used in converting value.  Hexadecimal
*                 can be easily formatted using printf("%08lX", ... )
*
* Return
* Values:  returns the address of the null-terminated string  or
*          NULL if an error occurred
*
* Side
* Effects: None.
*
* Notes: - the space for string must be long enough to hold the
*          converted value AND the null terminator.  For decimal, this
*          may be up to 21 characters.  For octal, it may be up to 23.
*        - tempstr is an internal buffer of maximum length to build the
*          decimal string so that if the caller *knows* they are
*          converting a number that will require less than the maximum,
*          such as 365, they only need space for four characters.
*        - All dsStruct64_t's are assumed to be unsigned.
*        - This routine was originally written to convert to both
*          decimal and octal, but due to it's rarity, octal support has
*          been commented out.  I have left the code for octal, but
*          it is commented out.
*---------------------------------------------------------------------*/
char *dI64toCh(dsStruct64_t *value, char *string, int radix)
{
char *cptr;
char tempstr[24];  /* max. of 23 characters plus one for good measure */
dsStruct64_t valueCopy;

  valueCopy.hi = value->hi;
  valueCopy.lo = value->lo;


  if (radix != 10)
     return NULL;

  cptr = tempstr + sizeof(tempstr)-1;       /* point to end of string */
  *cptr = '\0';                  /* initialize null-terminated string */

  /*--- DECIMAL conversion -------------------------------------------*/
  {
  dsStruct64_t   rem, base;       /* these vars are only used for base 10 */
    base.hi = 0;             /* create an dsStruct64_t divisor */
    base.lo = 10;
    do                  /* repeatedly divide by 10 until nothing left.*/
    {                   /* after each divide, the rmdr is the digit   */
       valueCopy = dDiv64(&valueCopy, &base, &rem);
       cptr--;
       *cptr = (char)(rem.lo + '0');
    } while (valueCopy.lo || valueCopy.hi);
  }
  /*------------------------------------------------------------------*/

  /*--- OCTAL conversion ---------------------------------------------*/
  /** do
   ** {
   **    cptr--;
   **    *cptr = (char)((valueCopy.lo & 0x07) + '0');
   **    ShiftR64(&valueCopy,3);
   ** } while (valueCopy.lo || valueCopy.hi);         */
  /*------------------------------------------------------------------*/

  strcpy(string,cptr);      /* copy new string into caller's string */
  return (string);
}  /* end dI64toCh() */

/*----------------------------------------------------------------------
* Name:    dChtoI64()
*
* Action:  Converts a string of character digits into an dsStruct64_t.
*          Currently only decimal is supported.
*
* Input:   string - a pointer to a character string to be converted.
*          radix  - radix to be used (2 - 10).
*
* Return
* Values:  The equivalent dsStruct64_t value, or zero if a non-digit character
*          was encountered.
*
* Side
* Effects: None.
*
* Notes: - All dsStruct64_t's are assumed to be unsigned.
*---------------------------------------------------------------------*/
dsStruct64_t dChtoI64( const char *str, int radix )

{
  dsStruct64_t power, radix64, result, temp64, temp2;
  int   digit, ovfl, len;

  power   = dMake64( 1 );
  radix64 = dMake64( radix );
  result  = dMake64( 0 );
  len     = strlen( str );

  while ( --len >= 0 )
  {
    digit = ((int) str[len]) - '0';

    if ( digit < 0 || digit >= radix )
      return ( dMake64( 0 ) );



    /*result = dAdd64( result, dMul64( power, dMake64( digit ), &ovfl ) );*/


    temp64 = dMake64(digit);
    temp2 = dMul64( &power, &temp64, &ovfl);
    result = dAdd64( &result, &temp2);



    power  = dMul64( &power, &radix64, &ovfl );
  }

  return ( result );
} /* dChtoI64() */
