#set -x

function usage
{
  if [[ -n ${flag_debug} ]]
  then
    set -x
  fi

  echo "Usage: iscsiMount.sh -i [IP Address] -t [taret name] -m [mount point] -u "
  echo "                     -l -s -v -h -o [output file] -f [path/configFileName] "
  echo "                     -p [patition information]"
  echo ""
  echo "  -i    [Required with -m or -s] IP Address of the ISCSI targets. "
  echo "  -t    [Required with -m] ISCSI target name. "
  echo "  -o    [Optional] stdout will also go to the filename specified."
  echo "  -m    [Optional] Mount. Filesystem mount point"
  echo "  -u    [Optional] UnMount."
  echo "  -l    [Optional] List all pre-discovered ISCSI Devices. Reads te local database."
  echo "        Format: [IP]|[ISCSI target]|[LoggedIn][host:target:id:lun]|[dev]|[mount point]"
  echo "  -s    [Optional] Only used with the -l and -i option. Forces a re-discovery of iscsi targets."
  echo "  -p    [optional] For TSM. Only used with the -m option. A comma seperated list of header:path to append"
  echo "                   to the mount point. One path for each partition. The header is "partx" where x is a "
  echo "                   partion number (ie part1 is partition 1)."
  echo "                   Example: add /boot to partition 1 mount point"
  echo "                            -p part1:/boot"
  echo "                   Example: add /boot to partition 1 mount point and /home to partition 3 mount point"
  echo "                            -p part1:/boot,part3:/home"
  echo "  -f    [Optional] Path and filename of config file. If not set the default config file is used. "
  echo "  -h    [Optional] Display this help. "
  echo "  -v    [Optional] Verbose. "
  echo ""
}

function unMountISCSIDevice
{
  if [[ -n ${flag_debug} ]]
  then
    set -x
  fi

  if [[ ! $# -eq 1 ]]
  then
    exit_print_wFile " Function: unMountISCSIDevice takes exactly 1 argument: MountPoint" 1
  fi

  mountPoint=$1

  verbose_print "UnMounting ISCSI target:${config_ISCSITargetName} from ${mountPoint}"

  if ! umount ${mountPoint}
  then
    exit_print_wFile "Command failed: umount ${mountPoint} with exit code $?" 1
  fi 

  rmdir ${mountPoint}
}

function mountISCSIDevice
{
  if [[ -n ${flag_debug} ]]
  then
    set -x
  fi

  if [[ ! $# -eq 3 ]]
  then
    rm ${tmpFileName}
    exit_print_wFile " Function: mountISCSIDevice takes exactly 3 arguments. IP, target and MountPoint" 1
  fi

  verbose_print "Construcitng Mount command line. Looking for IP:${config_FBSIPAddr} target:${config_ISCSITargetName}"
  {
    tmpFileName=`dirname ${config_commandFile}`/$$_tmpfile
    getISCSITargets > ${tmpFileName}
    tmpLoggedIndevice=`cat ${tmpFileName} | grep -w "${config_FBSIPAddr}" | grep -w "${config_ISCSITargetName}" | awk -F "|" '{print $5}'`
    rm ${tmpFileName}
    verbose_print " Logged In device: ${tmpLoggedIndevice}"
    #mountISCSIDevice `echo ${tmpLoggedIndevice} | tr -d "[" | tr -d "]"` ${config_mountPoint}

    mountDevice=`echo ${tmpLoggedIndevice} | tr -d "[" | tr -d "]"`
    mountPoint=$3

    if [[ ! -d ${mountPoint} ]]
    then
      exit_print_wFile " Mount point: ${mountPoint} is not a directory or does not exist. Cannot mount" 39
    fi

    verbose_print "Finding all partitions for ${tmpLoggedIndevice}"

    sfdisk -l ${mountDevice} | {
      while read_null_lines
      do
        if echo ${REPLY} | grep ^${mountDevice} > /dev/null 2>&1
        then
          tmpSystem=`echo ${REPLY} | awk '{print $7}'`
          if [[ ${tmpSystem} = "Linux" ]]
          then
            tmpPartition=`echo ${REPLY} | awk '{print $1}'`
            tmpDir=`echo ${tmpPartition} | awk -F '/' '{print $NF}'`

            if [[ -n ${config_partitionInfo} ]]
            then
              tmpPartNumber=`echo ${tmpPartition} | awk -F ${mountDevice} '{print $2}'`
              tmpDir="${tmpDir}/`echo "${config_partitionInfo}" | awk -F part${tmpPartNumber}: '{print $2}' | awk -F ',' '{print $1}'`"
            fi

            verbose_print "Creating dir ${mountPoint}/${tmpDir}"
            mkdir -p ${mountPoint}/${tmpDir}
            verbose_print "Mounting ${tmpPartition} on ${mountPoint}/${tmpDir}"
            if [[ -n ${flag_verbose} ]]
            then
              mount ${tmpPartition} ${mountPoint}/${tmpDir}
              tmpRC=$?
              if [[ ! ${tmpRC} = 0 ]]
              then
                exit_print_wFile "Command failed \"mount ${tmpPartition} ${mountPoint}${tmpDir}\" with exit code $?" 1
              fi
            else
              mount ${tmpPartition} ${mountPoint}/${tmpDir} > /dev/null 2>&1
              tmpRC=$?
              if [[ ! ${tmpRC} = 0 ]]
              then
                exit_print_wFile "Command failed \"mount ${tmpPartition} ${mountPoint}${tmpDir}\" with exit code $?" 1
              fi
            fi
          fi
        fi
      done
    }
  }
}

function logoutISCSITarget
{
  if [[ -n ${flag_debug} ]]
  then
    set -x
  fi

  verbose_print "Logging out ISCSI device ${config_FBSIPAddr} ${config_ISCSITargetName}"
  if [[ -n ${flag_verbose} ]]
  then
    iscsiadm -m node -T ${config_ISCSITargetName} -p ${config_FBSIPAddr} --logout
    tmpRC=$?
    if [[ ! ${tmpRC} = 0 ]]
    then
      exit_print_wFile "Command failed \"iscsiadm -m node -T ${config_ISCSITargetName} -p ${config_FBSIPAddr} --logout\" with exit code ${tmpRC}" 1
    fi
  else
    iscsiadm -m node -T ${config_ISCSITargetName} -p ${config_FBSIPAddr} --logout > /dev/null 2>&1
    tmpRC=$?
    if [[ ! ${tmpRC} = 0 ]]
    then
      exit_print_wFile "Command failed \"iscsiadm -m node -T ${config_ISCSITargetName} -p ${config_FBSIPAddr} --logout\" with exit code ${tmpRC}" 1
    fi
  fi
}

function delteISCSIDBEntry
{
  if [[ -n ${flag_debug} ]]
  then
    set -x
  fi

  verbose_print "Deleting ISCSI device ${config_FBSIPAddr} ${config_ISCSITargetName} from database"
  if ! iscsiadm -m node -o delete -T ${config_ISCSITargetName} -p ${config_FBSIPAddr}
  then
    exit_print_wFile "Command failed: iscsiadm -m node -o delete -T ${config_ISCSITargetName} -p ${config_FBSIPAddr} with exit code $?" 1
  fi
}


function logintoISCSITarget
{
  if [[ -n ${flag_debug} ]]
  then
    set -x
  fi

  verbose_print "Logging into ISCSI device ${config_FBSIPAddr} ${config_ISCSITargetName}"
  if [[ -n ${flag_verbose} ]]
  then
    iscsiadm -m node -T ${config_ISCSITargetName} -p ${config_FBSIPAddr} --login
    tmpRC=$?
    if [[ ! ${tmpRC} = 0 ]]
    then
      exit_print_wFile "Command failed: iscsiadm -m node -T ${config_ISCSITargetName} -p ${config_FBSIPAddr} --login with exit code $?" 1
    fi
  else
    iscsiadm -m node -T ${config_ISCSITargetName} -p ${config_FBSIPAddr} --login > /dev/null 2>&1
    tmpRC=$?
    if [[ ! ${tmpRC} = 0 ]]
    then
      exit_print_wFile "Command failed: iscsiadm -m node -T ${config_ISCSITargetName} -p ${config_FBSIPAddr} --login with exit code $?" 1
    fi
  fi
}

# [IP]|[ISCSI target]|[LoggedIn][host:target:id:lun]|[dev]|[mount point]
#
# Example output
# [9.47.98.169]|[ddkiso]|[false]|[]|[]|[]
# [9.47.98.169]|[foo123]|[true]|[2:0:0:0]|[/dev/sda]|[]
# [9.47.98.169]|[foo456]|[true]|[3:0:0:0]|[/dev/sdb]|[/mnt/tmpmnt]
#
function getISCSITargets
{
  if [[ -n ${flag_debug} ]]
  then
    set -x
  fi

  verbose_print "Verify iscsiadm command is functional"
  iscsiadm -h > /dev/null 2>&1
  tmpRC=$?
  if [[ ! ${tmpRC} = 0 ]]
  then
    exit_print_wFile "Command iscsiadm -m node failed with exit code ${tmpRC}" 1
  fi

  if [[ -n ${config_outputFile} ]]
  then
    echo "[IP]|[ISCSI target]|[LoggedIn][host:target:id:lun]|[dev]|[mount point]" | tee ${config_outputFile}
  else
    echo "[IP]|[ISCSI target]|[LoggedIn][host:target:id:lun]|[dev]|[mount point]"
  fi

  #iscsiadm -m node fails if there are no ISCSI records in DB
  iscsiadm -m node > /dev/null 2>&1
  tmpRC=$?
  if [[ ! ${tmpRC} = 0 ]]
  then
    verbose_print "Command iscsiadm -m node failed with exit code ${tmpRC}. This is probably due to a empty ISCSI DB. Ignoring"
    return 0
  fi

  # For every discovered ISCSI target
  verbose_print "Collecting data for discoverd ISCSI targets"
  iscsiadm -m node | {
    while read_null_lines
    do
      ISCSIInitiatorIP=`echo ${REPLY} | awk -F ':' '{print $1}' | awk '{print $NF}'`
      ISCSITargetName=`echo ${REPLY} | awk '{print $NF}'`
      verbose_print "Discovered ISCSI Target Initiator IP: ${ISCSIInitiatorIP}"
      verbose_print "Discovered ISCSI Target Name: ${ISCSITargetName}"
      export IsLoggedIn="false"
      export SCSIInfo=""
      export deviceInfo=""
      export mountpoint=""
     
      isISCSITargetLoggedIn ${ISCSIInitiatorIP} ${ISCSITargetName}

      if [[ ${IsLoggedIn} = "true" ]]
      then
        verbose_print "Looking for mount points for device: ${deviceInfo}"
        for i in `mount | grep ${deviceInfo} | awk '{print $3}'`
        do
          if [[ -n ${mountpoint} ]]
          then
            mountpoint="${mountpoint},"
          fi
          mountpoint="${mountpoint}${i}"
          verbose_print "Found mount point \"${i}\" for device: ${deviceInfo}"
        done
      fi

      if [[ -n ${config_outputFile} ]]
      then
        echo "[${ISCSIInitiatorIP}]|[${ISCSITargetName}]|[${IsLoggedIn}]|[${SCSIInfo}]|[${deviceInfo}]|[${mountpoint}]" | tee -a ${config_outputFile}
      else
        echo "[${ISCSIInitiatorIP}]|[${ISCSITargetName}]|[${IsLoggedIn}]|[${SCSIInfo}]|[${deviceInfo}]|[${mountpoint}]"
      fi

    done
  }
}

function isISCSITargetLoggedIn
{
  if [[ -n ${flag_debug} ]]
  then
    set -x
  fi

  if [[ ! $# -eq 2 ]]
  then
    exit_print_wFile " Function: isISCSITargetLoggedIn takes exactly 2 arguments. IP and Target" 1
  fi

  targetIP=$1
  targetName=$2

  verbose_print "Looking for Logged-In target. IP: ${targetIP} Name:${targetName}"

  if ! lsscsi -l -v > /dev/null 2>&1
  then
    exit_print_wFile "Command failed: lsscsi -l -v with exit code $?." 1
  fi

  # For every SCSI device
  lsscsi -l -v | {
    count=1
    while read_null_lines
    do
      if [[ ${count} -eq 1 ]]
      then
        # Get the SCSI info and device from line one
        tmpSCSIInfo=`echo ${REPLY} | awk '{print $1}'`
        verbose_print "Discovered SCSI device. Host Info: ${tmpSCSIInfo}"
        tmpDevice=`echo ${REPLY} | awk '{print $NF}'`
        verbose_print "Discovered SCSI device.    Device: ${tmpDevice}"
        count=2
      elif [[ ${count} -eq 2 ]]
      then
        # Get the state from line 2
        tmpState=`echo ${REPLY} | awk '{print $1}'`
        verbose_print "Discovered SCSI device.     state: ${tmpState}"
        count=3
      elif [[ ${count} -eq 3 ]]
      then
        # Answer the is this a ISCSI device and is-mounted question from line 3
        tmpPathFile=`echo ${REPLY} | awk '{print $NF}' | awk -F '[' '{print $2}' | awk -F ']' '{print $1}'`
        if [[ -n ${config_OSSuSe11} ]]
        then
          verbose_print "Looking for ISCSI information on SLES11 platform"
          tmpTargetFileName=`echo ${tmpPathFile} | awk -F '/' '{print "/"$2"/"$3"/"$4"/"$5"/"$6"/iscsi_session/"$6"/targetname"}'`
          verbose_print "Looking for Discovered Logged In ISCSI Target name in file: ${tmpTargetFileName}"
          tmpIPAddressFileName=`echo ${tmpPathFile} | awk -F '/' '{print "/"$2"/"$3"/"$4"/"$5"/"$6"/connection*/iscsi_connection/connection*/address"}'`
          verbose_print "Looking for Discovered Logged In ISCSI Initiator IP in file: ${tmpIPAddressFileName}"
        else
          tmpTargetFileName=`echo ${tmpPathFile} | awk -F '/' '{print "/"$2"/"$3"/"$4"/"$5"/"$6"/iscsi_session:"$6"/targetname"}'`
          verbose_print "Looking for Discovered Logged In ISCSI Target name in file: ${tmpTargetFileName}"
          tmpIPAddressFileName=`echo ${tmpPathFile} | awk -F '/' '{print "/"$2"/"$3"/"$4"/"$5"/"$6"/connection*/iscsi_connection:connection*/address"}'`
          verbose_print "Looking for Discovered Logged In ISCSI Initiator IP in file: ${tmpIPAddressFileName}"
        fi
        if [[ ! -f ${tmpTargetFileName} ]]
        then
          verbose_print "Cannot find file: ${tmpTargetFileName}"
          verbose_print "Device ${tmpDevice} is not a ISCSI Device"
          count=1
        elif ! ls ${tmpIPAddressFileName} > /dev/null 2>&1
        then
          exit_print_wFile "Cannot determine if IP: ${targetIP} Name:${targetName} is logged in. File ${tmpIPAddressFileName} does not exist" 1
        else
          tmpTarget=`cat ${tmpTargetFileName}`
          verbose_print "Logged In ISCSI Target name: ${tmpTarget}"
          tmpIP=`cat ${tmpIPAddressFileName}`
          verbose_print "Logged In ISCSI Initiator IP: ${tmpIP}"
          count=1

          if [[ ${targetIP} = ${tmpIP} && ${targetName} = ${tmpTarget} ]]
          then
            verbose_print "ISCSI IP: ${targetIP} TargetName: ${tmpIP} is logged in."
            IsLoggedIn="true"
            SCSIInfo=${tmpSCSIInfo}
            deviceInfo=${tmpDevice}
            break
          fi
        fi
      else
        exit_print_wFile "Invalid count. Unexpected line in lsscci output" 1
      fi
    done
  }
}

config_commandFile=`whence $0`
config_configFile=`dirname ${config_commandFile}`/configFile

. `dirname ${config_commandFile}`/commonFunctions

loadConfigFile

while getopts dhvlsi:t:m:f:o:p:u c 2> /dev/null
do
  case $c in
    i) config_FBSIPAddr=${OPTARG};;
    t) config_ISCSITargetName=${OPTARG};;
    m) config_mountPoint=${OPTARG};;
    o) config_outputFile=${OPTARG};;
    d) set -x
       flag_debug=true;;
    h) usage
       exit;;
    l) flag_listOnly=true;;
    s) flag_rediscover=true;;
    v) flag_verbose=true;;
    f) config_configFile=${OPTARG}
       loadConfigFile;;
    u) config_unMount=true;;
    p) config_partitionInfo=${OPTARG};;
    *) exit_print usage 1;;
  esac
done

if [[ -n ${config_outputFile} ]]
then
  if ! touch ${config_outputFile} > /dev/null 2>&1
  then
    exit_print " Output File ${config_outputFile} is not writable." 24
  fi
fi

if [[ -n ${config_partitionInfo} && -z ${config_mountPoint} ]]
then
  exit_print_wFile "You must provide a filesystem mount point" 36
fi

if [[ -z ${flag_listOnly} ]]
then
  if [[ -z ${config_FBSIPAddr} ]]
  then
    exit_print_wFile " You must provide a IP address of the ISCSI targets." 20
  fi

  if [[ -z ${config_ISCSITargetName} ]]
  then
    exit_print_wFile " You must provide a ISCSI target." 34
  fi

  if [[ -z ${config_mountPoint} && -z ${config_unMount} ]]
  then
    exit_print_wFile" You must provide a filesystem mount point or the unMount option." 36
  fi
  if [[ -n ${config_mountPoint} && -n ${config_unMount} ]]
  then
    exit_print_wFile " You must provide a filesystem mount point or the unMount option. Not both" 37
  fi
fi

if [[ -n ${flag_rediscover} ]]
then
  if [[ -z ${config_FBSIPAddr} ]]
  then
    exit_print_wFile " You must provide a IP address of the ISCSI targets." 20
  fi
fi 

for i in `echo ${configVariableNameList}`
do
  export $i
done

config_OSPlatform=`getOSPlatform`
returnCode=$?
if [[ ${returnCode} = 10 ]]
then
  config_iscsiServiceCmd="service iscsid"
  config_OSRedHat="true"
elif [[ ${returnCode} = 11 ]]
then
  config_iscsiServiceCmd="rcopen-iscsi"
  config_OSSuSe10="true"
elif [[ ${returnCode} = 12 ]]
then
  config_iscsiServiceCmd="rcopen-iscsi"
  config_OSSuSe11="true"
else
  exit_print_wFile "Unsupported OS PLatform: ${config_OSPlatform}" 25
fi
verbose_print "Checking Local Dependancies"
checkLocalDependancies

verbose_print "Verify open-iscsi is configured and running."
if ! ${config_iscsiServiceCmd} status > /dev/null 2>&1
then
  warn_print "iscsid is not running. Attempting to start"
  ${config_iscsiServiceCmd} start
  if ! ${config_iscsiServiceCmd} status > /dev/null 2>&1
  then
    exit_print_wFile " Unable to start iscsi service. Cmd: ${config_iscsiServiceCmd} start" 1
  fi
fi

verbose_print "Verified: iscsi service is running"

if [[ -n ${config_FBSIPAddr} ]]
then 
  verbose_print "Convert Hostname to IP if needed"
  config_FBSIPAddr=`hostnameToIP ${config_FBSIPAddr}`
fi

if [[ -n ${flag_listOnly} ]]
then
  if [[ -n ${flag_rediscover} ]]
  then
    verbose_print "Discover ISCSI Targets"
    if ! iscsiadm -m discovery -t sendtargets -p ${config_FBSIPAddr} > /dev/null 2>&1
    then
      exit_print_wFile "Command failed: iscsiadm -m discovery -t sendtargets -p ${config_FBSIPAddr} with exit code $?" 1
    fi
  fi
  getISCSITargets
else
  verbose_print "Discover ISCSI Targets"
  if ! iscsiadm -m discovery -t sendtargets -p ${config_FBSIPAddr} > /dev/null 2>&1
  then
    exit_print_wFile "Command failed iscsiadm -m discovery -t sendtargets -p ${config_FBSIPAddr} with exit code $?" 1
  fi

  # For every SCSI device
  getISCSITargets | {
    verbose_print "Looking for ISCSI device ${config_FBSIPAddr} ${config_ISCSITargetName}"
    while read_null_lines
    do
      # First line is the Title
      if [[ -z readTileLine ]]
      then
        readTileLine=true
      fi

      tmpIP=`echo ${REPLY} | awk -F "|" '{print $1}'`
      tmptarget=`echo ${REPLY} | awk -F "|" '{print $2}'`
      tmploggedin=`echo ${REPLY} | awk -F "|" '{print $3}'`
      tmpdevice=`echo ${REPLY} | awk -F "|" '{print $5}'`
      tmpmntpoint=`echo ${REPLY} | awk -F "|" '{print $6}'`

      if [[ ${tmpIP} = "[${config_FBSIPAddr}]" ]]
      then
        if [[ ${tmptarget} = "[${config_ISCSITargetName}]" ]]
        then
          if [[ -z ${config_unMount} ]]
          then
            # ISCSI Login
            verbose_print "ISCSI Login"
            if [[ ${tmploggedin} = "[true]" ]]
            then
              warn_print "ISCSI device ${config_FBSIPAddr} ${config_ISCSITargetName} is already logged in. Device ${tmpdevice}"
            else
              logintoISCSITarget
            fi

            # ISCSI Device Mount
            verbose_print "ISCSI Device Mount"
            # Wait for the SG driver to finish
            verbose_print "Wait For SG driver"
            sleep 2
            if [[ ${tmpmntpoint} = "[]" ]]
            then
              verbose_print "Mounting IP:${config_FBSIPAddr} target:${config_ISCSITargetName} on mountpoint: ${config_mountPoint}"
              mountISCSIDevice IP:${config_FBSIPAddr} ${config_ISCSITargetName} ${config_mountPoint}
            else
              warn_print "ISCSI device ${config_FBSIPAddr} ${config_ISCSITargetName} ${tmpLoggedIndevice} is already mounted: ${tmpmntpoint}"
            fi
          else
            # Unmount
            # ISCSI Device UnMount
            if [[ ${tmpmntpoint} != "[]" ]]
            then
              tmpUnMountPoint=`echo ${tmpmntpoint} | tr -d "[" | tr -d "]"`
              for i in `echo "${tmpUnMountPoint}" | awk 'BEGIN{FS=","}{for (i=1; i<=NF; i++) print $i}'`
              do
                verbose_print "Unmounting ${i}"
                unMountISCSIDevice ${i}
                # Wait for the SG driver to finish
                sleep 4
              done
            else
              warn_print "ISCSI device ${config_FBSIPAddr} ${config_ISCSITargetName} ${tmpdevice} is NOT mounted."
            fi

            # ISCSI Logout
            if [[ ${tmploggedin} = "[false]" ]]
            then
            warn_print "ISCSI device ${config_FBSIPAddr} ${config_ISCSITargetName} is NOT logged in."
            else
              logoutISCSITarget
            fi
            
            # ISCSI DB Deletion
            delteISCSIDBEntry
          fi
          if [[ -n ${flag_verbose} ]]
          then
            getISCSITargets
          else
            getISCSITargets > /dev/null 2>&1
          fi
          loggedinDeviceFound=true
          break
        fi
      fi
    done

    if [[ ! -z ${loggedinDeviceFound} ]]
    then
      exit 0
    else
      exit_print_wFile "ISCSI target: ${config_ISCSITargetName} at IP: ${config_FBSIPAddr} was not found" 38
    fi
  }
fi


