#
# Cache up all the info required to display a report...
#
# $Id: report.tcl,v 1.34 2011/02/17 17:00:19 rader Exp rader $
#

###############################################################################
## Wrapper around TryReport...

proc NewReport { } { 
  global cur_task_cmd prev_task_cmd default_task_cmd system_task_cmd 
  global header display error_mode task_id debug

  if { $debug } { puts "NewReport: $cur_task_cmd..." }
  if { [TryReport] } { return }
  if { $debug } { puts "NewReport: $cur_task_cmd failed: 1st punt!" }

  set cur_task_cmd $default_task_cmd
  if { [TryReport] } { return }
  if { $debug } { puts "NewReport: $default_task_cmd failed: 2nd punt!" }

  set cur_task_cmd $system_task_cmd
  if { ! [TryReport] } {
    if { $debug } { puts "NewReport: $system_task_cmd failed: task list is empty!" }
    array unset body
    set error_mode 1
    set task_id 0
    set header(1) ""
    set header(2) ""
    set display(1) "No matches."
  }
}

###############################################################################
## Try to read new report data into memory...

proc TryReport { } {
  global task cur_task_cmd task_args prev_task_cmd pending completed debug
  global header body color_body count_str error_mode win_start win_cur win_size 
  global converg complete display_mode column_one_type column_one_width task_undo
  global rows cols ignore_exit_status

  #--------------------
  # read in global stats...
  if { $debug } { puts "TryReport: exec $task $task_args stat" }
  set FD [open "|$task $task_args rc._forcecolor=off stat"]
  while { [FilteredGets $FD colorl l] >= 0 } {
    regexp {^ *Pending *([0-9]+)} $l junk pending
    regexp {^ *Completed *([0-9]+)} $l junk completed
    if { [regexp {^ *Total *([0-9]+)} $l junk total] } { break }
  }
  catch { close $FD }
  if { $debug } { puts "TryReport: global: completed=$completed pending=$pending" }

  #--------------------
  # read in report stats...
  # FIXME this should probably also somehow work for custom reports...
  set p "UNK"; set t "UNK"
  if { $cur_task_cmd == "ls" || $cur_task_cmd == "list" } { 
    set p $pending;
    set t $total
  } elseif { [regexp {^ls (.*)} $cur_task_cmd junk match] || [regexp {^list (.*)} $cur_task_cmd junk match] } { 
    set FD [open "|$task $task_args rc._forcecolor=off stat $match"]
    while { [FilteredGets $FD colorl l] >= 0 } {
      regexp {^ *Pending *([0-9]+)} $l junk p
      if { [regexp {^ *Total *([0-9]+)} $l junk t] } { break }
    }
    catch { close $FD }
  }
  if { $debug } { puts "TryReport: report: total=$t pending=$p" }
  if { $p != "UNK" && $t != "UNK" && $t != 0 } {
    set p "$p.00"; set t "$t.00"; # yes, tcl/tk is goofy
    set c [expr ($t - $p) / $t * 100]
    regsub {\..*} $c {} c ;# tcl_precision doesn't work!?
    set complete ", $c% complete"
  } else {
    set complete ""
  }
  if { $debug } { puts "TryReport: report: $completed" }

  #--------------------
  # read in convergence...
  if { [regexp {^ac} $cur_task_cmd] || [regexp {^bl} $cur_task_cmd] || [regexp {^w} $cur_task_cmd] } {
    # not when active, blocked or waiting report...
    set converg "" 
  } else {
    # strip any report spec...
    set f $cur_task_cmd
    regsub {^active *} $f {} f; regsub {^all *} $f {} f; regsub {^blocked *} $f {} f;
    regsub {^list *} $f {} f; regsub {^long *} $f {} f; regsub {^ls *} $f {} f;
    regsub {^minimal *} $f {} f; regsub {^newest *} $f {} f; regsub {^next *} $f {} f;
    regsub {^oldest *} $f {} f; regsub {^overdue *} $f {} f; regsub {^recurring *} $f {} f;
    regsub {^unblocked *} $f {} f; regsub {^waiting *} $f {} f

    if { $debug } { puts "TryReport: exec $task $task_args rc.defaultwidth=110 burndown.daily $f" }
    set converg "convergence unknown"
    # FIXME burndown hack: remove defaultwidth here when/if bug #647 gets fixed
    # FIXME us terminal width so convergence matches!
    set FD [open "|$task $task_args rc._forcecolor=off rc.defaultwidth=110 burndown.daily $f"]
    while { [FilteredGets $FD colorl l] >= 0 } {
      if { [regexp {Estimated completion: +(.*)} $l junk match] } { 
        if { $match == "No convergence" } { 
          set converg "no convergence"
        } else {
          set converg "convergence: $match"
        }
      }
    }
    catch { close $FD }
    if { $debug } { puts "TryReport: converg=$converg" }
  }

  set i 1
  set error_mode 0
  set found_burndown 0

  #--------------------
  # start reading in the report...
  set new_header(1) ""; set new_header(2) ""
  if { $debug } { puts "TryReport: exec $task $task_args $cur_task_cmd" }
  regsub -all {"} $cur_task_cmd {\"} cur_task_cmd
  regsub -all {\(} $cur_task_cmd {\(} cur_task_cmd
  regsub -all {\)} $cur_task_cmd {\)} cur_task_cmd
  if { $display_mode == "undo" } { 
    set FD [open "|$task_undo n"]
  } else { 
    set FD [open "|$task $task_args $cur_task_cmd"]
  }

  #--------------------
  # first line... should be blank...
  if { [DisplayGetsHeader] } { 
    FilteredGets $FD colorl l
  } else {
    set l ""
  }
  if { [regexp {^No matches\.} $l] } {  
    Bell
    Feedback "task $cur_task_cmd ... no matches"
    set cur_task_cmd $prev_task_cmd
    return 0
  }
  if { [regexp {^ *No command} $l] } {
    Bell
    Feedback "task $cur_task_cmd ... not supported" 
    set cur_task_cmd $prev_task_cmd
    return 0
  }
  if { [regexp {^ *Ambiguous command} $l] } {  
    Bell
    Feedback $l
    set cur_task_cmd $prev_task_cmd 
    return 0
  }
  if { [regexp {Burndown} $l] } {
    set found_burndown 1
  } elseif { $l != "" && $display_mode == "report" } { 
    set error_mode 1
    if { $debug } { puts "TryReport: error on line 1: \"$l\"" }
    Feedback "task $cur_task_cmd ... command failed" 
    return 0
  } 

  #--------------------
  # second line... should be header...
  # ParseUnderlining creates labels and dashes...
  if { [DisplayGetsHeader] } { 
    FilteredGets $FD colorl l
  } else {
    set colorl ""
  }
  set header_list [ParseUnderlining $colorl]
  set new_header(1) [lindex $header_list 0]
  set new_header(2) [lindex $header_list 1]
  set l [StripControlChars $colorl]
  if { [regexp {^ *Usage:} $l] } {  
    Bell
    Feedback "task $cur_task_cmd ... usage error ... invalid report"
    if { $debug } { puts "TryReport: error on line 2: usage: $l" }
    set cur_task_cmd $prev_task_cmd 
    return 0
  }

  # fix wonky ghistory header... add dashes to last column...
  if { [regexp {^ *ghistory} $cur_task_cmd] } { 
    regsub -all {\- +$} $new_header(2) "" new_header(2)
    set new_header(2) "$new_header(2)- -"
  }

  # muuuunge header...
  if { ! [regexp {^proj} $cur_task_cmd] && ! [regexp {^sum} $cur_task_cmd] } { 
    # trim trailing whitespace...
    regsub -all {\- +$} $new_header(2) "" new_header(2)
    # pad to column N-1...
    set c [expr $cols - 2 - [string length $new_header(2)]]
    for { set j 0 } { $j < $c } {incr j } { set new_header(2) "$new_header(2)-" }
  }

  # fix wonky summary header... pad out dashes for "0% .. 100%"
  if { [regexp {^sum} $cur_task_cmd] } { 
    set s [string first "0%" $new_header(1)]
    set lhs [string range $new_header(2) 0 [expr $s - 1]]
    set rhs [string range $new_header(2) $s end]
    set rhs [string map {" " "-"} $rhs]
    set new_header(2) "${lhs}$rhs"
  }
 
  # untested since migration to ParseUnderlining?  assumably not harmful...
  if { $display_mode == "report" && ! [ regexp {^ *\-} $new_header(2)] } {
    set error_mode 1
    if { $debug } { puts "TryReport: error on line 2: header dashes not parsed from \"$l\"" }
  }

  if { $debug } { puts "TryReport: header 1: $new_header(1)" }
  if { $debug } { puts "TryReport: header 2: $new_header(2)" }

  #--------------------
  # figure out report type...
  set col1 1
  scan $new_header(2) "%s " col1
  set c1w [expr [string length $col1]]
  set col1 [string range $new_header(1) 0 $c1w]
  if { [regexp { *ID *} $col1] } { 
    set display_mode "report"
  } elseif { $found_burndown } { 
    set display_mode "burndown"
  } elseif { $display_mode != "undo" } { 
    set display_mode "info"
  }
  if { $debug } { puts "TryReport: set display_mode=$display_mode" }

  #--------------------
  # body...

  while { [FilteredGets $FD colorl l] >= 0 } {

    # no blank lines in reports...
    if { $l == "" && $display_mode == "report"  } { continue; } 

    # throw away info history... 
    if { [regexp {^ *info} $cur_task_cmd] && [regexp { *Date *} [string range $l 0 $c1w]] } {
      break;
    }

    # clean up undo footer...
    regsub {The undo command is not reversible.*} $colorl "The undo command is not reversible." colorl

    # count...
    if { [regexp {^[0-9]+ tasks*$} $l] } { 
      set count_str $l
      continue;
    }

    # usage oops...
    if { [regexp {^ *Usage:} $l] } {  
      set error_mode 1 
      if { $debug } { puts "TryReport: error on line $i: \"usage\": $ldashes" }
    }

    set new_body($i) $l
    set new_color_body($i) $colorl
    if { $debug } { 
      set istr [format "%-3d" $i]
      puts "TryReport: line $istr txt=\"$l\"" 
    }
    incr i

  }

  if { [catch { close $FD } err] } {
    if { $ignore_exit_status } { 
      puts "Warning: error on file channel close: errorCode is \"$::errorCode\""
    } elseif { $new_body(1) == "No matches." } { 
      Bell
      Feedback "task $cur_task_cmd ... no matches"
      return 0
    } else {
      set error_mode 1 
      if { $debug } { puts "TryReport: error on file channel close: errorCode is \"$::errorCode\"" }
    }
  } 

  # delete empty rows at the end...
  if { $new_body([array size new_body]) == "" } {
    set n [array size new_body]
    array unset new_body $n
    array unset new_color_body $n
  }

  if { [array exists header] } { array unset header }
  array set header [array get new_header]
  array unset new_header

  if { [array exists body] } { array unset body }
  array set body [array get new_body]
  array unset new_body

  if { [array exists color_body] } { array unset color_body }
  array set color_body [array get new_color_body]
  array unset new_color_body

  scan $header(1) "%s " column_one_type
  scan $header(2) "%s " col1
  set column_one_width [expr [string length $col1]]

  if { [expr $win_start + $win_cur] > [expr [array size color_body] + 1] } {
    # cur is no longer valid, adjust to end...
    while { $win_start > [array size color_body] } {
      set win_start [expr $win_start - $win_size]
    }
    if { $win_start < 1 } { set win_start 1 }
    if { $win_cur > [array size color_body] } { 
      set win_start 1
      set win_cur [array size color_body] 
    }
  } 

  set j [array size color_body]
  if { $debug } { puts "TryReport: linesread=$j error_mode=$error_mode win_start=$win_start win_cur=$win_cur" }
  return 1

}

###############################################################################

proc DisplayGetsHeader { } {
  global display_mode cur_task_cmd
  # FIXME maybe this is unnecessary because
  # FIXME display_mode is incomplete or something...
  #puts "DisplayGetsHeader $cur_task_cmd $display_mode"
  if { $display_mode == "undo" }         { return 0 } 
  if { [regexp {^ac} $cur_task_cmd] }    { return 1 }
  if { [regexp {^al} $cur_task_cmd] }    { return 1 }
  if { [regexp {^bl} $cur_task_cmd] }    { return 1 }
  if { [regexp {^burn} $cur_task_cmd] }  { return 0 }
  if { [regexp {^cal} $cur_task_cmd] }   { return 0 }
  if { [regexp {^col} $cur_task_cmd] }   { return 0 }
  if { [regexp {^com} $cur_task_cmd] }   { return 1 }
  if { [regexp {^ghistory} $cur_task_cmd] } { return 1 }
  if { [regexp {^history} $cur_task_cmd] } { return 1 }
  if { [regexp {^in} $cur_task_cmd] }    { return 1 }
  if { [regexp {^li} $cur_task_cmd] }    { return 1 }
  if { [regexp {^lon} $cur_task_cmd] }    { return 1 }
  if { [regexp {^ls} $cur_task_cmd] }    { return 1 }
  if { [regexp {^mi} $cur_task_cmd] }    { return 1 }
  if { [regexp {^ov} $cur_task_cmd] }    { return 1 }
  if { [regexp {^pro} $cur_task_cmd] }   { return 1 }
  if { [regexp {^recur} $cur_task_cmd] } { return 1 }
  if { [regexp {^stats} $cur_task_cmd] } { return 1 }
  if { [regexp {^su} $cur_task_cmd] }    { return 1 }
  if { [regexp {^ta} $cur_task_cmd] }    { return 1 }
  if { [regexp {^un} $cur_task_cmd] }    { return 0 }
  if { [regexp {^w} $cur_task_cmd] }     { return 1 }
  if { $display_mode == "report" }       { return 1 } 
  return 0
}


