
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk.c 2519 2007-07-17 08:39:07Z mschwerin $
 */

#include "config.h"

#include <assert.h>
#include <math.h>
#include <stdio.h>

#include "environment.h"
#include "event.h"
#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "odk.h"
#include "otk.h"
#include "utils.h"
#include "scheduler.h"
#include "xmlparser.h"

#define ABS(x) ((x)<0?-(x):(x))

typedef struct otk_window_s otk_window_t;
struct otk_window_s {
    otk_widget_t widget;

    bool keep;

    otk_widget_t *focus_ptr;

    l_list_t *subs;
};

struct otk_s {
    odk_t *odk;

    otk_window_t *window;
    l_list_t *windows;

    int palettes[7];

    int textpalette_label;

    /**
     * 0: normal button
     * 1: focused button
     * 2: disabled button
     */
    int textpalette_button[3];

    int update_job;
};


static void otk_update (otk_t * otk);


char *
otk_trunc_text_to_width (otk_t * otk, const char *orig, int max_width)
{
    int textwidth;
    int textheight;
    odk_get_text_size (otk->odk, orig, &textwidth, &textheight);

    char *text = ho_strdup (orig);
    if (textwidth > max_width) {
        int i = strlen (text) - 4;

        if (i < 0) {
            return text;
        }

        text[i] = text[i + 1] = text[i + 2] = '.';
        text[i + 3] = 0;

        odk_get_text_size (otk->odk, text, &textwidth, &textheight);
        while ((textwidth > max_width) && (i > 0)) {
            i--;
            text[i] = '.';
            text[i + 3] = 0;
            odk_get_text_size (otk->odk, text, &textwidth, &textheight);
        }
    }

    return text;
}


static void
otk_widget_destroy (void *data)
{
    otk_widget_destructor ((otk_widget_t *) data);
}


/**
 * Returns the widget at position the given position. A widget must be visible
 * and selectable to be returned.
 */
static otk_widget_t *
widget_find_xy (otk_t * otk, int x, int y)
{
    if (!otk->window)
        return NULL;

    otk_widget_t *widget = l_list_first (otk->window->subs);
    while (widget) {
        int x1 = widget->x;
        int x2 = widget->x + widget->w;
        int y1 = widget->y;
        int y2 = widget->y + widget->h;
        if ((widget->selectable != OTK_SELECTABLE_NONE)
            && (widget->is_visible)
            && ((x1 <= x) && (x2 >= x))
            && ((y1 <= y) && (y2 >= y))) {
            return widget;
        }
        widget = l_list_next (otk->window->subs, widget);
    }

    return NULL;
}


/**
 * Searches for the first widget to set the focus to.
 */
static otk_widget_t *
widget_get_first_focus (otk_t * otk)
{
    if (!otk->window)
        return NULL;

    if (l_list_length (otk->window->subs) == 0)
        return NULL;

    otk_widget_t *widget = l_list_first (otk->window->subs);
    while (widget) {
        if ((widget->selectable & OTK_SELECTABLE_KEY)
            && (widget->is_visible)
            && (widget->is_enabled)) {
            return widget;
        }
        widget = l_list_next (otk->window->subs, widget);
    }

    return NULL;
}


/**
 * Searches for the previous widget to pass the focus to.
 */
otk_widget_t *
otk_widget_get_prev_focus (otk_t * otk)
{
    if (!otk->window)
        return NULL;

    if (l_list_length (otk->window->subs) == 0)
        return NULL;

    if (!otk->window->focus_ptr)
        return widget_get_first_focus (otk);

    otk_widget_t *cur_widget = otk->window->focus_ptr;
    otk_widget_t *prev_widget = cur_widget;

    assert (cur_widget);

    do {
        /* get the prev widget */
        prev_widget = l_list_prev (otk->window->subs, prev_widget);
        /* check if we're at the end of the list  */
        if (!prev_widget) {
            prev_widget = l_list_last (otk->window->subs);
        }
        /* check if we've run around once  */
        if (prev_widget == cur_widget) {
            return NULL;
        }
    } while (!(prev_widget->is_visible)
             || !(prev_widget->is_enabled)
             || !(prev_widget->selectable & OTK_SELECTABLE_KEY)
             || ((cur_widget->type == OTK_WIDGET_LISTENTRY)
                 && (prev_widget->type == OTK_WIDGET_LISTENTRY)
                 && (otk_listentry_get_list (cur_widget) ==
                     otk_listentry_get_list (prev_widget))));

    return prev_widget;
}


/**
 * Searches for the next widget to pass the focus to.
 */
otk_widget_t *
otk_widget_get_next_focus (otk_t * otk)
{
    if (!otk->window)
        return NULL;

    if (l_list_length (otk->window->subs) == 0)
        return NULL;

    if (!otk->window->focus_ptr)
        return widget_get_first_focus (otk);

    otk_widget_t *cur_widget = otk->window->focus_ptr;
    otk_widget_t *next_widget = cur_widget;

    assert (cur_widget);

    do {
        /* get the next widget */
        next_widget = l_list_next (otk->window->subs, next_widget);
        /* check if we're at the end of the list */
        if (!next_widget) {
            next_widget = l_list_first (otk->window->subs);
        }
        /* check if we've run around once */
        if (next_widget == cur_widget) {
            return NULL;
        }
    } while (!(next_widget->is_visible)
             || !(next_widget->is_enabled)
             || !(next_widget->selectable & OTK_SELECTABLE_KEY)
             || ((cur_widget->type == OTK_WIDGET_LISTENTRY)
                 && (next_widget->type == OTK_WIDGET_LISTENTRY)
                 && (otk_listentry_get_list (cur_widget) ==
                     otk_listentry_get_list (next_widget))));

    return next_widget;
}


/**
 * Returns the distance between the widgets base and target.
 */
static int
get_distance (otk_widget_t * base, otk_widget_t * target)
{
    int x1 = base->x + (base->w / 2);
    int y1 = base->y + (base->h / 2);
    int x2 = target->x + (target->w / 2);
    int y2 = target->y + (target->h / 2);

    int dist = (int) sqrt (pow ((x1 - x2), 2) + pow ((y1 - y2), 2));

#if 0
    debug ("distance %4d", dist);
#endif

    return dist;
}


/**
 * Returns the vertical angle between the widgets base and target.
 */
static double
get_vertical_angle (otk_widget_t * base, otk_widget_t * target)
{
    double a = 0;

    int x1 = base->x + (base->w / 2);
    int y1 = base->y + (base->h / 2);
    int x2 = target->x + (target->w / 2);
    int y2 = target->y + (target->h / 2);
    int x = ABS (x1 - x2);
    int y = ABS (y1 - y2);

    if (x != 0) {
#if 0
        a = ((atan ((double) y / (double) x) * 90.0) / M_PI_2);
#else
        a = atan ((double) y / (double) x) + 1;
#endif
    }

#if 0
    debug ("   angle %7.2f", a);
#endif

    return a;
}


/**
 * Returns the horizonal angle between the widgets base and target.
 */
static double
get_horizontal_angle (otk_widget_t * base, otk_widget_t * target)
{
    double a = 0;

    int x1 = base->x + (base->w / 2);
    int y1 = base->y + (base->h / 2);
    int x2 = target->x + (target->w / 2);
    int y2 = target->y + (target->h / 2);
    int x = ABS (x1 - x2);
    int y = ABS (y1 - y2);

    if (y != 0) {
#if 0
        a = ((atan ((double) x / (double) y) * 90.0) / M_PI_2);
#else
        a = atan ((double) x / (double) y) + 1;
#endif
    }

#if 0
    debug ("   angle %7.2f", a);
#endif

    return a;
}


otk_widget_t *
otk_widget_find_neighbour (otk_t * otk, otk_direction_t direction)
{

    /**
     * @todo Improve the implementation of this function. It does not always
     *       select the widget the user wants.
     */

    if (!otk->window)
        return NULL;

    if (l_list_length (otk->window->subs) == 0)
        return NULL;

    if (!otk->window->focus_ptr)
        return widget_get_first_focus (otk);

    double best_angle = 0.0;
    double best_ratio = 0.0;

    otk_widget_t *cur_widget = otk->window->focus_ptr;
    otk_widget_t *new_widget = NULL;

    int cx = cur_widget->x + cur_widget->w / 2;
    int cy = cur_widget->y + cur_widget->h / 2;

    otk_widget_t *widget = l_list_first (otk->window->subs);
    while (widget) {
        if ((widget != cur_widget)
            && (widget->is_visible)
            && (widget->is_enabled)
            && (widget->selectable & OTK_SELECTABLE_KEY)) {

            int nx = widget->x + widget->w / 2;
            int ny = widget->y + widget->h / 2;

            double angle = 0.0;
            double ratio = 0.0;

            switch (direction) {
            case OTK_DIRECTION_UP:
                if (ny >= cy)
                    break;
                angle = get_horizontal_angle (cur_widget, widget);
                ratio = angle * get_distance (cur_widget, widget);
                break;
            case OTK_DIRECTION_DOWN:
                if (ny <= cy)
                    break;
                angle = get_horizontal_angle (cur_widget, widget);
                ratio = angle * get_distance (cur_widget, widget);
                break;
            case OTK_DIRECTION_LEFT:
                if (nx >= cx)
                    break;
                angle = get_vertical_angle (cur_widget, widget);
                ratio = angle * get_distance (cur_widget, widget);
                break;
            case OTK_DIRECTION_RIGHT:
                if (nx <= cx)
                    break;
                angle = get_vertical_angle (cur_widget, widget);
                ratio = angle * get_distance (cur_widget, widget);
                break;
            }

            if (((angle == 1.00)
                 && ((best_angle == 0)
                     || (best_angle > angle)))
                || ((ratio > 0)
                    && ((best_ratio == 0)
                        || (ratio < best_ratio)))) {
                best_angle = angle;
                best_ratio = ratio;
                new_widget = widget;
            }
        }
        widget = l_list_next (otk->window->subs, widget);
    }

    return new_widget;
}


/*
 * ***********************************************************************************
 * Window-Widget
 * ***********************************************************************************
 * There is always one current window. This current window is set when
 * creating a new window. It can also be set by calling otk_set_current_window.
 *
 * There are two possibilities of how windows are destroyed:
 *
 * 1. The user keeps a pointer to the window in a variable and marks the
 *    window not to be destroyed by calling otk_window_keep. The user can
 *    himself destroy the window by calling otk_window_destroy.
 *
 * 2. The window is automatically destroyed as soon as a new window is created.
 *
 * When calling otk_free all windows will be destroyed by otk.
 * ***********************************************************************************
 */
static void
window_destroy (otk_widget_t * this)
{
    otk_window_t *window = (otk_window_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_WINDOW))
        return;

    otk_window_t *current_window = this->otk->window;
    this->otk->window = window;

    /* we free all widgets contained in this window */
    l_list_free (window->subs, otk_widget_destroy);
    /* we remove this window from the list of windows */
    l_list_remove (this->otk->windows, window);

    if (current_window == window) {
        this->otk->window = NULL;
    }
    else {
        this->otk->window = current_window;
    }

    ho_free (window);
}


otk_widget_t *
otk_window_new (otk_t * otk, int x, int y, int w, int h)
{
    if (otk->window && !otk->window->keep) {
        window_destroy (&otk->window->widget);
    }

    otk->window = ho_new (otk_window_t);

    otk->window->widget.type = OTK_WIDGET_WINDOW;
    otk->window->widget.x = x;
    otk->window->widget.y = y;
    otk->window->widget.w = w;
    otk->window->widget.h = h;
    otk->window->widget.otk = otk;
    otk->window->widget.odk = otk->odk;
    otk->window->widget.destroy = window_destroy;
    otk->window->widget.need_update = false;
    otk->window->keep = false;
    otk->window->subs = l_list_new ();

    l_list_append (otk->windows, otk->window);

    return (otk_widget_t *) otk->window;
}

void
otk_window_keep (otk_widget_t * this, bool keep)
{
    otk_window_t *window = (otk_window_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_WINDOW))
        return;

    window->keep = keep;
}

void
otk_window_destroy (otk_widget_t * this)
{
    if (!otk_widget_is_correct (this, OTK_WIDGET_WINDOW))
        return;

    window_destroy (this);
}

void
otk_window_append (otk_widget_t * this, otk_widget_t * widget)
{
    if (!widget)
        return;
    if (!otk_widget_is_correct (this, OTK_WIDGET_WINDOW))
        return;

    otk_window_t *window = (otk_window_t *) this;

    l_list_append (window->subs, widget);
}


void
otk_window_remove (otk_widget_t * this, otk_widget_t * widget)
{
    if (!widget)
        return;
    if (!otk_widget_is_correct (this, OTK_WIDGET_WINDOW))
        return;

    otk_window_t *window = (otk_window_t *) this;

    if (window->focus_ptr == widget) {
        window->focus_ptr = NULL;
    }

    l_list_remove (window->subs, widget);
}


void
otk_window_focus_pointer_set (otk_widget_t * this, otk_widget_t * widget)
{
    if (!otk_widget_is_correct (this, OTK_WIDGET_WINDOW))
        return;

    otk_window_t *window = (otk_window_t *) this;

    window->focus_ptr = widget;
}


otk_widget_t *
otk_window_focus_pointer_get (otk_widget_t * this)
{
    if (!otk_widget_is_correct (this, OTK_WIDGET_WINDOW))
        return NULL;

    otk_window_t *window = (otk_window_t *) this;

    return window->focus_ptr;
}


/*
 * ***********************************************************************************
 * Eventhandlers
 * ***********************************************************************************
 */

static void
motion_handler (otk_t * otk, oxine_event_t * ev)
{
    if (!otk->window)
        return;

    int x = ev->data.mouse.pos.x;
    int y = ev->data.mouse.pos.y;

    otk_widget_t *new_focus = widget_find_xy (otk, x, y);
    otk_widget_t *old_focus = otk->window->focus_ptr;

    if (old_focus == new_focus) {
        /* Do nothing. */
    }

    else if (new_focus && (new_focus->selectable & OTK_SELECTABLE_MOUSE)
             && (new_focus->is_enabled) && (new_focus->is_visible)) {
        otk_widget_set_focused (new_focus, true);
        otk_draw (otk);
    }

    else if (old_focus) {
        otk_widget_set_focused (old_focus, false);
        otk_draw (otk);
    }
}


static void
button_handler (otk_t * otk, oxine_event_t * ev)
{
    if (ev->source.button == OXINE_MOUSE_BUTTON_NULL) {
        return;
    }
    if (!otk->window) {
        return;
    }

    int x = ev->data.mouse.pos.x;
    int y = ev->data.mouse.pos.y;

    otk_widget_t *cur_widget = widget_find_xy (otk, x, y);
    if (!cur_widget) {
        return;
    }
    if (!cur_widget->is_enabled) {
        return;
    }
    if (!(cur_widget->selectable & OTK_SELECTABLE_MOUSE)) {
        return;
    }
    if (cur_widget->button_handler) {
        cur_widget->button_handler (cur_widget, ev);
    }
}


static void
key_handler (otk_t * otk, oxine_event_t * ev)
{
    if (ev->source.key == OXINE_KEY_NULL) {
        return;
    }
    if (!otk->window) {
        return;
    }

    otk_widget_t *cur_widget = otk->window->focus_ptr;

    if (cur_widget && cur_widget->key_handler) {
        cur_widget->key_handler (cur_widget, ev);
    }

    if (ev->source.key == OXINE_KEY_NULL) {
        return;
    }

    otk_widget_t *new_widget = NULL;
    switch (ev->source.key) {
    case OXINE_KEY_UP:
        new_widget = otk_widget_find_neighbour (otk, OTK_DIRECTION_UP);
        ev->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_DOWN:
        new_widget = otk_widget_find_neighbour (otk, OTK_DIRECTION_DOWN);
        ev->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_LEFT:
        new_widget = otk_widget_find_neighbour (otk, OTK_DIRECTION_LEFT);
        ev->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_RIGHT:
        new_widget = otk_widget_find_neighbour (otk, OTK_DIRECTION_RIGHT);
        ev->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_PREV_WIDGET:
        new_widget = otk_widget_get_prev_focus (otk);
        ev->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_NEXT_WIDGET:
        new_widget = otk_widget_get_next_focus (otk);
        ev->source.key = OXINE_KEY_NULL;
        break;
    default:
        break;
    }

    if (new_widget) {
        otk_widget_set_focused (new_widget, true);
        otk_draw (otk);
    }
}


static void
otk_event_handler (void *this, oxine_event_t * ev)
{
    otk_t *otk = (otk_t *) this;

    switch (ev->type) {
    case OXINE_EVENT_KEY:
        key_handler (otk, ev);
        break;
    case OXINE_EVENT_MOTION:
        motion_handler (otk, ev);
        break;
    case OXINE_EVENT_BUTTON:
        button_handler (otk, ev);
        break;
    case OXINE_EVENT_OSD_FORMAT_CHANGED:
        otk_draw (otk);
        break;
    case OXINE_EVENT_OTK_UPDATE:
        otk_update (otk);
        break;
    default:
        break;
    }
}


/*
 * ***********************************************************************************
 * Global Functions
 * ***********************************************************************************
 */
void
otk_draw (otk_t * otk)
{
    otk_widget_t *widget;

    if (!otk->window)
        return;

#ifdef HAVE_OSD_IMAGE
    /* Preload all the images. This makes sense so that the user does not at
     * some point see only half the GUI because the other half is not yet
     * loaded. */
    widget = l_list_first (otk->window->subs);
    while (widget) {
        if (widget->is_visible) {
            otk_image_preload (widget);
        }
        widget = l_list_next (otk->window->subs, widget);
    }

    /* As we have no possibility to hide images, that are not in the current
     * window, we have to hide all images. But because we don't want to do
     * this on every repaint we only hide the images when the window has
     * changed. */
    static otk_window_t *last_window = NULL;
    if (otk->window != last_window) {
        odk_osd_hide_images (otk->odk);
        last_window = otk->window;
    }
#endif /* HAVE_OSD_IMAGE */

    /* We swap the two OSD's we use. */
    odk_osd_swap (otk->odk);

    /* Draw our widgets */
    widget = l_list_first (otk->window->subs);
    while (widget) {
        if (!widget->is_visible && widget->hide) {
            widget->hide (widget);
        }
        else if (widget->is_visible && widget->draw) {
            widget->draw (widget);
        }
        widget = l_list_next (otk->window->subs, widget);
    }

    /* Show the OSD drawing area */
    odk_osd_show (otk->odk);
}


void
otk_clear (otk_t * otk)
{
    if (otk->window && !otk->window->keep) {
        window_destroy ((otk_widget_t *) otk->window);
    }
    otk->window = NULL;

    odk_osd_hide (otk->odk);
    odk_osd_clear (otk->odk);
}


static void
otk_update (otk_t * otk)
{
    bool need_repaint = false;

    if (otk->window) {
        otk_widget_t *widget = l_list_first (otk->window->subs);
        while (widget) {
            if (widget->need_update && widget->update && widget->is_visible) {
                widget->update (widget);
            }
            widget = l_list_next (otk->window->subs, widget);
        }
    }

    if (otk->window) {
        otk_widget_t *widget = l_list_first (otk->window->subs);
        while (widget) {
            need_repaint |= (widget->need_repaint && widget->is_visible);
            widget = l_list_next (otk->window->subs, widget);
        }
    }

    if (need_repaint) {
        otk_draw (otk);
    }
}


/**
 * We do not update the widgets in this job directly as this makes
 * synchronization between the threads difficult. Instead we send an event
 * through odk_oxine_event_send which is protected by a mutex.
 */
static void
otk_update_job (void *p)
{
    otk_t *otk = (otk_t *) p;

    oxine_event_t ev;
    ev.type = OXINE_EVENT_OTK_UPDATE;
    odk_oxine_event_send (otk->odk, &ev);

    otk->update_job = schedule_job (250, otk_update_job, otk);
}


static int
read_color (otk_t * otk, xml_node_t * node, bool inverse)
{
    uint32_t c[] = {
        0xffffff,
        0xff0000,
        0x00ff00
    };
    uint8_t t[] = {
        0xf, 0xf, 0xf
    };

    xml_node_t *child = node->child;
    while (child) {
        if (strcasecmp (child->name, "color") == 0) {
            const char *name = xml_parser_get_property (child, "name");
            if (!name) {
                error (_("Missing property (required) '%s'!"), "name");
                break;
            }

            unsigned int cx;
            const char *value = xml_parser_get_property (child, "value");
            if (!value || (sscanf (value, "%06x", &cx) != 1)) {
                error (_("Missing property (required) '%s'!"), "value");
                cx = 0xff0000;
            }

            unsigned int tx;
            const char *trans =
                xml_parser_get_property (child, "transparency");
            if (!trans || (sscanf (trans, "%02x", &tx) != 1)) {
                tx = 0xf;
            }

            if (strcasecmp (name, "foreground") == 0) {
                if (inverse) {
                    c[1] = cx;
                    t[1] = tx;
                }
                else {
                    c[0] = cx;
                    t[0] = tx;
                }
            }

            else if (strcasecmp (name, "background") == 0) {
                if (inverse) {
                    c[0] = cx;
                    t[0] = tx;
                }
                else {
                    c[1] = cx;
                    t[1] = tx;
                }
            }

            else if (strcasecmp (name, "border") == 0) {
                c[2] = cx;
                t[2] = tx;
            }

            else {
                error (_("Unknown name '%s' for node!"), name);
            }
        }

        else {
            error (_("Unknown name '%s' for node!"), child->name);
        }

        child = child->next;
    }

    return odk_osd_alloc_text_palette (otk->odk,
                                       c[0], t[0], c[1], t[1], c[2], t[2]);
}


static int
otk_load_palette (otk_t * otk, xml_node_t * palette_node)
{
    xml_node_t *child = palette_node->child;
    while (child) {
        if (strcasecmp (child->name, "text_palette") == 0) {
            const char *name = xml_parser_get_property (child, "name");

            if (strcasecmp (name, "button") == 0) {
                otk->palettes[OTK_PALETTE_BUTTON] =
                    read_color (otk, child, false);
                otk->palettes[OTK_PALETTE_SLIDER] =
                    otk->palettes[OTK_PALETTE_BUTTON];
                otk->palettes[OTK_PALETTE_SLIDER_INVERSE] =
                    read_color (otk, child, true);
            }
            else if (strcasecmp (name, "focused_button") == 0) {
                otk->palettes[OTK_PALETTE_BUTTON_FOCUSED] =
                    read_color (otk, child, false);
                otk->palettes[OTK_PALETTE_SLIDER_FOCUSED] =
                    otk->palettes[OTK_PALETTE_BUTTON_FOCUSED];
            }
            else if (strcasecmp (name, "disabled_button") == 0) {
                otk->palettes[OTK_PALETTE_BUTTON_DISABLED] =
                    read_color (otk, child, false);
            }
            else if (strcasecmp (name, "label") == 0) {
                otk->palettes[OTK_PALETTE_LABEL] =
                    read_color (otk, child, false);
            }
            else {
                error (_("Unknown name '%s' for node!"), name);
            }
        }
        else {
            error (_("Unknown name '%s' for node!"), child->name);
        }

        child = child->next;
    }

    return true;
}


int
otk_get_palette (otk_t * otk, otk_palette_id_t id)
{
    return otk->palettes[id];
}


static bool
otk_load_skin (otk_t * otk, const char *mrl)
{
    debug (_("Loading skin file '%s'..."), mrl);

    int size;
    char *file = read_entire_file (mrl, &size);
    xml_node_t *node;

    if (!file) {
        return false;
    }

    xml_parser_init (file, strlen (file), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node) < 0) {
        error (_("Parsing '%s' failed!"), mrl);
        return false;
    }

    if (strcasecmp (node->name, "oxine_skin")) {
        error (_("Root node of '%s' must be '%s'!"), mrl, "oxine_skin");
        return 0;
    }

    int xres = 0;
    int yres = 0;

    xml_node_t *child = node->child;
    while (child) {
        if (strcasecmp (child->name, "palette") == 0) {
            otk_load_palette (otk, child);
        }
        else if (strcasecmp (child->name, "resolution") == 0) {
            const char *x = xml_parser_get_property (child, "x");
            const char *y = xml_parser_get_property (child, "y");
            if (x) {
                xres = atoi (x);
            }
            if (y) {
                yres = atoi (y);
            }
        }
        else {
            error (_("Unknown name '%s' for node!"), child->name);
        }

        child = child->next;
    }

    xml_parser_free_tree (node);
    ho_free (file);

    if (!xres || !yres) {
        error (_("Could not determine resolution of skin!"));
        return false;
    }

    debug ("Skin has resolution of %dx%d.", xres, yres);
    odk_osd_set_resolution (otk->odk, xres, yres);

    return true;
}



otk_t *
otk_init (odk_t * odk, const char *skin_xml)
{
    otk_t *otk = ho_new (otk_t);
    otk->odk = odk;

    if (!otk_load_skin (otk, skin_xml)) {
        ho_free (otk);
        return NULL;
    }

    otk->window = NULL;
    otk->windows = l_list_new ();

    otk->textpalette_label = otk->palettes[OTK_PALETTE_LABEL];
    otk->textpalette_button[0] = otk->palettes[OTK_PALETTE_BUTTON];
    otk->textpalette_button[1] = otk->palettes[OTK_PALETTE_BUTTON_FOCUSED];
    otk->textpalette_button[2] = otk->palettes[OTK_PALETTE_BUTTON_DISABLED];

    /* Register the OTK event handler. */
    odk_add_event_handler (odk, otk_event_handler, otk,
                           EVENT_HANDLER_PRIORITY_HIGH);

    /* Start the update job. */
    otk->update_job = schedule_job (250, otk_update_job, otk);

    return otk;
}


void
otk_free (otk_t * otk)
{
    /* stop the update job */
    lock_job_mutex ();
    cancel_job (otk->update_job);
    unlock_job_mutex ();

    /* unregister the otk event handler */
    odk_del_event_handler (otk->odk, otk_event_handler);

    /* Free all the windows. */
    l_list_free (otk->windows, (l_free_cb_t) window_destroy);
    otk->windows = NULL;
    otk->window = NULL;

    ho_free (otk);
}


odk_t *
otk_get_odk (otk_t * otk)
{
    return otk->odk;
}


void
otk_set_current_window (otk_t * otk, otk_widget_t * this)
{
    if (!otk_widget_is_correct (this, OTK_WIDGET_WINDOW))
        return;

    otk_window_t *window = (otk_window_t *) this;

    otk->window = window;
}


otk_widget_t *
otk_get_current_window (otk_t * otk)
{
    return (otk_widget_t *) otk->window;
}
