
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk_widget.c 2335 2007-06-26 18:26:07Z mschwerin $
 */
#include "config.h"

#include <assert.h>

#include "heap.h"
#include "otk.h"


bool
otk_widget_is_correct (otk_widget_t * widget, otk_widget_type_t expected)
{
    if (!widget)
        return false;

    if (widget->type != expected)
        return false;

    return true;
}


void
otk_widget_destructor (otk_widget_t * widget)
{
    otk_t *otk = widget->otk;
    otk_widget_t *window = otk_get_current_window (otk);

    /* Free the font. */
    ho_free (widget->font);

    /* And call the widgets own destroy function. */
    widget->destroy (widget);

    /* Remove the widget from the current window. */
    otk_window_remove (window, widget);
}


void
otk_widget_constructor (otk_widget_t * widget, otk_t * otk,
                        otk_widget_type_t type, int x, int y, int w, int h)
{
    otk_widget_t *window = otk_get_current_window (otk);

    widget->type = type;

    widget->otk = otk;
    widget->odk = otk_get_odk (otk);

    widget->x = ((otk_widget_t *) window)->x + x;
    widget->y = ((otk_widget_t *) window)->y + y;
    widget->w = w;
    widget->h = h;

    widget->selectable = OTK_SELECTABLE_MOUSE | OTK_SELECTABLE_KEY;

    widget->is_focused = false;
    widget->is_enabled = true;
    widget->is_visible = true;

    widget->need_update = false;
    widget->need_repaint = false;

    widget->font = ho_strdup ("sans");
    widget->fontsize = 20;
    widget->alignment = ODK_ALIGN_LEFT | ODK_ALIGN_VCENTER;

    widget->draw = NULL;
    widget->hide = NULL;
    widget->destroy = NULL;
    widget->update = NULL;

    widget->key_handler = NULL;
    widget->motion_handler = NULL;
    widget->button_handler = NULL;

    widget->focus_enter_cb = NULL;
    widget->focus_enter_cb_data = NULL;

    widget->focus_leave_cb = NULL;
    widget->focus_leave_cb_data = NULL;

    otk_window_append (window, widget);
}


void
otk_widget_set_font (otk_widget_t * widget, const char *font, int fontsize)
{
    assert (widget);
    assert (font);
    ho_free (widget->font);
    widget->font = ho_strdup (font);
    widget->fontsize = fontsize;
    widget->need_repaint = true;
}


void
otk_widget_set_alignment (otk_widget_t * widget, int alignment)
{
    assert (widget);
    widget->alignment = alignment;
    widget->need_repaint = true;
}


void
otk_widget_set_enabled (otk_widget_t * widget, bool enabled)
{
    assert (widget);
    widget->is_enabled = enabled;
    widget->need_repaint = true;
}


void
otk_widget_set_visible (otk_widget_t * widget, bool visible)
{
    assert (widget);
    widget->is_visible = visible;
    widget->need_repaint = true;
}


void
otk_widget_set_selectable (otk_widget_t * widget, int selectable)
{
    assert (widget);
    widget->selectable = selectable;
    widget->need_repaint = true;
}


void
otk_widget_set_updated (otk_widget_t * widget, bool update)
{
    assert (widget);
    widget->need_update = update;
}


void
otk_widget_set_focused (otk_widget_t * widget, bool focused)
{
    otk_widget_t *window = otk_get_current_window (widget->otk);
    otk_widget_t *focused_widget = otk_window_focus_pointer_get (window);

    if (!widget)
        return;

    if (focused) {
        if (!widget->is_visible)
            return;
        if (!widget->is_enabled)
            return;
        if (widget->selectable == OTK_SELECTABLE_NONE)
            return;

        if (focused_widget) {
            otk_widget_set_focused (focused_widget, false);
        }

        widget->is_focused = true;
        otk_window_focus_pointer_set (window, widget);

        if (widget->focus_enter_cb) {
            widget->focus_enter_cb (widget->focus_enter_cb_data);
        }
    }

    else {
        if (focused_widget != widget) {
            return;
        }

        widget->is_focused = false;
        otk_window_focus_pointer_set (window, NULL);

        if (widget->focus_leave_cb) {
            widget->focus_leave_cb (widget->focus_leave_cb_data);
        }
    }
}


void
otk_widget_set_focus_callbacks (otk_widget_t * widget,
                                otk_cb_t enter_cb, void *enter_cb_data,
                                otk_cb_t leave_cb, void *leave_cb_data)
{
    assert (widget);

    widget->focus_enter_cb_data = enter_cb_data;
    widget->focus_enter_cb = enter_cb;
    widget->focus_leave_cb_data = leave_cb_data;
    widget->focus_leave_cb = leave_cb;
}


const char *
otk_widget_get_type (otk_widget_t * widget)
{
    if (!widget)
        return "null";

    switch (widget->type) {
    case OTK_WIDGET_WINDOW:
        return "window";
    case OTK_WIDGET_BUTTON:
        return "button";
    case OTK_WIDGET_LABEL:
        return "label";
    case OTK_WIDGET_LIST:
        return "list";
    case OTK_WIDGET_LISTENTRY:
        return "listentry";
    case OTK_WIDGET_SLIDER:
        return "slider";
    case OTK_WIDGET_SCROLLBAR:
        return "scrollbar";
    case OTK_WIDGET_CHECKBOX:
        return "checkbox";
    case OTK_WIDGET_BORDER:
        return "border";
    case OTK_WIDGET_EDITBOX:
        return "editbox";
    case OTK_WIDGET_IMAGE:
        return "image";
    default:
        break;
    }

    return "unknown";
}
