
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 * Most of this file was copied from the xine-ui project.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: weather.c 2537 2007-07-17 13:45:38Z mschwerin $
 *
 * This downloads weather data from an FTP server at weather.noaa.gov. An
 * explanation of the METAR code can be found at
 * http://www.met.tamu.edu/class/METAR/metar.html
 *
 * Some code in this file was taken from libgweather (gnome-applets).
 *
 */
#include "config.h"

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <math.h>
#ifdef HAVE_METAR
#include <metar.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "oxine.h"
#include "download.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "scheduler.h"
#include "utils.h"
#include "weather.h"

#define PRESSURE_INCH_TO_KPA(inch)      ((inch) * 3.386)
#define PRESSURE_INCH_TO_HPA(inch)      ((inch) * 33.86)
#define PRESSURE_INCH_TO_MM(inch)       ((inch) * 25.40005)
#define PRESSURE_INCH_TO_MB(inch)       (PRESSURE_INCH_TO_HPA(inch))
#define PRESSURE_INCH_TO_ATM(inch)      ((inch) * 0.033421052)
#define PRESSURE_MBAR_TO_INCH(mbar)     ((mbar) * 0.029533373)

#define WINDSPEED_KNOTS_TO_KPH(knots)   ((knots) * 1.851965)
#define WINDSPEED_KNOTS_TO_MPH(knots)   ((knots) * 1.150779)
#define WINDSPEED_KNOTS_TO_MS(knots)    ((knots) * 0.514444)

#define TEMP_F_TO_C(f)                  (((f) - 32.0) * 0.555556)
#define TEMP_F_TO_K(f)                  (TEMP_F_TO_C(f) + 273.15)
#define TEMP_C_TO_F(c)                  (((c) * 1.8) + 32.0)

#define VISIBILITY_SM_TO_KM(sm)         ((sm) * 1.609344)
#define VISIBILITY_SM_TO_M(sm)          (VISIBILITY_SM_TO_KM(sm) * 1000)

#ifdef HAVE_WEATHER

#ifndef HAVE_CURL
#error No weather support without CURL!
#endif

extern oxine_t *oxine;

static char *station_cache_id = NULL;
static char *station_cache_name = NULL;

static pthread_t thread;
static bool is_running = false;
static l_list_t *countries = NULL;

static void *
weather_thread (void *p)
{
    is_running = true;

    info (_("Successfully started weather thread."));
    debug ("    weather thread: 0x%X", (int) pthread_self ());

    while (1) {
        pthread_testcancel ();

        /* We put a mutex around weather retrieval, so we don't get into any
         * conflicts with the weather retrieval in the menu_weather.c */
        mutex_lock (&oxine->weather_mutex);
        const char *station = config_get_string ("weather.station_id");
        oxine->weather = get_current_weather (station);
        if (oxine->weather) {
            oxine->weather->retrieval_time = time (NULL);
        }
        mutex_unlock (&oxine->weather_mutex);

        /* We sleep for half an hour before trying to retrieve the weather
         * again. */
        int i = 0;
        for (; i < 60 * 30; i++) {
            sleep (1);
            pthread_testcancel ();
        }
    }

    pthread_exit (NULL);
    return NULL;
}


static char *
get_station_name (const char *station_id)
{
    if (station_cache_id && station_cache_name
        && (strcmp (station_cache_id, station_id) == 0))
        return ho_strdup (station_cache_name);

    char *mrl = OXINE_DATADIR "/metar_stations";
    FILE *f = fopen (mrl, "r");
    if (!f) {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
        return ho_strdup (station_id);
    }

    char *station_name = NULL;
    while (!feof (f) && !station_name) {
        char line[1024];
        if (!fgets (line, 1024, f))
            break;
        if ((line[0] == station_id[0])
            && (line[1] == station_id[1])
            && (line[2] == station_id[2])
            && (line[3] == station_id[3])) {
            station_name = ho_strdup (line + 5);
            break;
        }
    }
    fclose (f);

    if (station_name) {
        char *p = index (station_name, ';');
        if (p)
            p[0] = '\0';
        ho_free (station_cache_id);
        ho_free (station_cache_name);
        station_cache_id = ho_strdup (station_id);
        station_cache_name = ho_strdup (station_name);
    }

    return station_name ? station_name : ho_strdup (station_id);
}


static char *
get_update_time (Decoded_METAR * decodedMetar)
{
    const time_t now = time (NULL);
    struct tm tm;

    localtime_r (&now, &tm);

    /* If last reading took place just before midnight UTC on the first,
     * adjust the date downward.  This ASSUMES that the reading won't be more
     * than 24 hrs old! */
    if (decodedMetar->ob_date > tm.tm_mday) {
        tm.tm_mday--;
    }
    else {
        tm.tm_mday = decodedMetar->ob_date;
    }

    long int gmtoff = 0;
#ifdef HAVE_TM_TM_GMOFF
    gmtoff = tm.tm_gmtoff;
#elif defined HAVE_TIMEZONE
    gmtoff = timezone;
#endif

    /* Correct tm to local timezone. */
    tm.tm_hour = decodedMetar->ob_hour + (gmtoff / 3600);
    gmtoff -= (gmtoff / 3600) * 3600;
    tm.tm_min = decodedMetar->ob_minute + (gmtoff / 60);
    gmtoff -= (gmtoff / 60) * 60;
    tm.tm_sec = 0;

    char str_time[64];
    strftime (str_time, 64, _("%a, %b %d / %H:%M"), &tm);

    return ho_strdup (str_time);
}


static double
get_wind_speed_kmh (Decoded_METAR * decodedMetar)
{
    if (strcasecmp (decodedMetar->winData.windUnits, "KT") == 0) {
        return WINDSPEED_KNOTS_TO_KPH (decodedMetar->winData.windSpeed);
    }
    else if (strcasecmp (decodedMetar->winData.windUnits, "MPS") == 0) {
        return (decodedMetar->winData.windSpeed * 3.600);
    }
    else {
        return decodedMetar->winData.windSpeed;
    }
}


static int
get_altimeter_hpa (Decoded_METAR * decodedMetar)
{
    if (decodedMetar->Q_altstng) {
        return decodedMetar->hectoPasc_altstng;
    }
    else if (decodedMetar->A_altstng) {
        return PRESSURE_INCH_TO_HPA (decodedMetar->inches_altstng);
    }
    else {
        return 0;
    }
}


static double
get_saturation_vapor_pressure (int temp)
{
    double a = 7.6;
    double b = 240.7;

    if (temp >= 0) {
        a = 7.5;
        b = 237.3;
    }

    return 6.1078 * pow (10, (a * temp) / (b + temp));
}


static int
get_humidity (Decoded_METAR * decodedMetar)
{
    double temp = decodedMetar->temp;
    double dew_point = decodedMetar->dew_pt_temp;
    return (int) (100.0 * get_saturation_vapor_pressure (dew_point)
                  / get_saturation_vapor_pressure (temp));
}


static double
get_wind_chill (Decoded_METAR * decodedMetar)
{
    double temp = decodedMetar->temp;
    double wind = get_wind_speed_kmh (decodedMetar);
    double chill = temp;

    if (temp == INT_MAX) {
        return INT_MAX;
    }
    if (wind > 5.0) {
        double v = pow (wind, 0.16);
        chill = (13.12 + 0.6215 * temp - 11.37 * v + 0.3965 * temp * v);
    }

    return round (chill * 10) / 10;
}


static char *
get_wind_dir (Decoded_METAR * decodedMetar)
{
    int degree = decodedMetar->winData.windDir;

    if (degree < 22.5) {
        return _("North");
    }
    else if (degree < 67.5) {
        return _("North-East");
    }
    else if (degree < 112.5) {
        return _("East");
    }
    else if (degree < 157.5) {
        return _("South-East");
    }
    else if (degree < 202.5) {
        return _("South");
    }
    else if (degree < 247.5) {
        return _("South-West");
    }
    else if (degree < 292.5) {
        return _("West");
    }
    else if (degree < 337.5) {
        return _("North-West");
    }
    else {
        return _("North");
    }
}


static char *
get_weather_icon (Decoded_METAR * decodedMetar)
{
    if (decodedMetar->CAVOK) {
        return "menu_weather_clear.png";
    }

    char *icon_name = NULL;
    char *weather_type = decodedMetar->WxObstruct[0];
    if (strstr (weather_type, "TS")) {
        icon_name = "menu_weather_storm.png";
    }
    else if (strstr (weather_type, "DZ")) {
        icon_name = "menu_weather_showers_scattered.png";
    }
    else if (strstr (weather_type, "RA")) {
        icon_name = "menu_weather_showers.png";
    }
    else if (strstr (weather_type, "SN")) {
        icon_name = "menu_weather_snow.png";
    }
    else if (strstr (weather_type, "SG")) {
        icon_name = "menu_weather_snow.png";
    }

    if (icon_name) {
        return icon_name;
    }

    struct {
        char *short_name;
        char *icon_name;
    } cloud_types[] = {
        { "SKC", "menu_weather_clear.png"       },
        { "CLR", "menu_weather_clear.png"       },
        { "FEW", "menu_weather_few_clouds.png"  },
        { "SCT", "menu_weather_few_clouds.png"  },
        { "BKN", "menu_weather_overcast.png"    },
        { "OVC", "menu_weather_overcast.png"    },
        {  NULL, NULL                           }
    };

    int i = 0;
    while ((i < 6) && (decodedMetar->cldTypHgt[i].cloud_type[0] != '\0')) {
        i += 1;
    }
    if ((i == 0) || (i == 6)) {
        return "-";
    }
    char *cloud_type = decodedMetar->cldTypHgt[i - 1].cloud_type;

    i = 0;
    for (; cloud_types[i].short_name; i++) {
        if (strcasecmp (cloud_type, cloud_types[i].short_name) == 0) {
            icon_name = cloud_types[i].icon_name;
        }
    }

    return icon_name;
}


static char *
get_weather_type (Decoded_METAR * decodedMetar)
{
    /* This struct contains all combinations of weather descriptor and
     * phenomena. */
    struct {
        char *short_name;
        char *long_name;
    } weather_types[] = {
        /* Thunderstorm, Showers */
        {   "TS", _("Thunderstorm")                     },
        {   "SH", _("Showers")                          },
        /* Drizzle */
        {   "DZ", _("Drizzle")                          },
        { "FZDZ", _("Freezing Drizzle")                 },
        /* Rain */
        {   "RA", _("Rain")                             },
        { "SHRA", _("Rainshowers")                      },
        { "TSRA", _("Thunderstorm with rain")           },
        { "FZRA", _("Freezing Rain")                    },
        /* Snow */
        {   "SN", _("Snow") },
        { "DRSN", _("Drifting snow")                    },
        { "BLSN", _("Blowing snowfall")                 },
        { "SHSN", _("Snowshowers")                      },
        { "TSSN", _("Thunderstorm with snow")           },
        /* Snow Grains */
        {   "SG", _("Snow Grains")                      },
        /* Ice Crystals */
        {   "IC", _("Ice Crystals")                     },
        /* Ice Pellets */
        {   "PL", _("Ice Pellets")                      },
        { "SHPL", _("Showers of ice pellets")           },
        { "TSPL", _("Thunderstorm with ice pellets")    },
        /* Hail */
        {   "GR", _("Hail")                             },
        { "SHGR", _("Hailshowers")                      },
        { "TSPL", _("Thunderstorm with hail")           },
        /* Small hail */
        {   "GS", _("Small Hail")                       },
        { "SHGS", _("Showers of small hail")            },
        { "TSGS", _("Thunderstorm with small hail")     },
        /* Mist */
        {   "BR", _("Mist")                             },
        /* Fog */
        {   "FG", _("Fog")                              },
        { "MIFG", _("Shallow fog")                      },
        { "PRFG", _("Partial fog")                      },
        { "BCFG", _("Patches of fog")                   },
        { "FZFG", _("Freezing fog")                     },
        /* Haze */
        {   "HZ", _("Haze")                             },
        {   "FU", _("Smoke")                            },
        {   "VA", _("Volcanic Ash")                     },
        /* Dust */
        {   "DU", _("Dust")                             },
        { "DRDU", _("Low drifting dust")                },
        { "BLDU", _("Blowing dust")                     },
        /* Sand */
        {   "SA", _("Sand")                             },
        { "DRSA", _("Low drifting sand")                },
        { "BLSA", _("Blowing sand")                     },
        /* Spray */
        { "BLPY", _("Blowing spray")                    },
        {   "PO", _("Well-developed Dust/Sand Whirls")  },
        {   "SQ", _("Squalls")                          },
        {   "FC", _("Funnel Cloud")                     },
        {   "SS", _("Sandstorm")                        },
        {   "DS", _("Duststorm")                        },
        {   NULL, NULL                                  }
    };

    if (decodedMetar->CAVOK) {
        return "-";
    }

    char *long_name = NULL;
    char *weather_type = decodedMetar->WxObstruct[0];

    if ((strlen (weather_type) % 2) == 1) {
        weather_type++;
    }

    int i = 0;
    for (; (weather_types[i].short_name && !long_name); i++) {
        if (strcasecmp (weather_types[i].short_name, weather_type) == 0) {
            long_name = weather_types[i].long_name;
        }
    }

    return long_name ? long_name : "-";
}


static char *
get_cloud_type (Decoded_METAR * decodedMetar)
{
    struct {
        char *short_name;
        char *long_name;
    } cloud_types[] = {
        { "SKC", _("Clear sky")         },
        { "CLR", _("Clear sky")         },
        { "FEW", _("Few clouds")        },
        { "SCT", _("Scattered clouds")  },
        { "BKN", _("Broken clouds")     },
        { "OVC", _("Overcast")          },
        {  NULL, NULL                   }
    };

    if (decodedMetar->CAVOK) {
        return _("Clear sky");
    }

    int i = 0;
    while ((i < 6) && (decodedMetar->cldTypHgt[i].cloud_type[0] != '\0'))
        i += 1;
    if ((i == 0) || (i == 6))
        return "-";
    char *cloud_type = decodedMetar->cldTypHgt[i - 1].cloud_type;

    i = 0;
    for (; cloud_types[i].short_name; i++) {
        if (strcasecmp (cloud_type, cloud_types[i].short_name) == 0)
            return cloud_types[i].long_name;
    }

    return "-";
}


weather_t *
get_current_weather (const char *station_id)
{
    if (strlen (station_id) != 4) {
        error (_("Invalid METAR ICAO Location Indicator: %s"), station_id);
        return NULL;
    }

    const char *base_url = config_get_string ("weather.metar_url");
    char *station_url = ho_strdup_printf ("%s/%s.TXT", base_url, station_id);

    int i = 0;
    int l = strlen (station_url);
    for (i = l - 8; i < l; i++) {
        station_url[i] = toupper (station_url[i]);
    }

    download_t *download = download_new (NULL);
    if (!download) {
        error (_("Failed to download weather information!"));
        ho_free (station_url);
        return NULL;
    }
    if (!network_download (station_url, download)) {
        error (_("Failed to download weather information!"));
        download_free (download, true);
        ho_free (station_url);
        return NULL;
    }

    char *metar = index (download->buffer, '\n') + 1;
    while (metar[strlen (metar) - 1] == '\n') {
        metar[strlen (metar) - 1] = '\0';
    }
    debug ("METAR raw data: %s", metar);
    Decoded_METAR decodedMetar;
    DcdMETAR (metar, &decodedMetar);

    weather_t *weather = ho_new (weather_t);
    weather->station_id = ho_strdup (station_id);
    weather->station_name = get_station_name (station_id);
    weather->update_time = get_update_time (&decodedMetar);
    weather->icon_name = get_weather_icon (&decodedMetar);
    weather->length = 10;
    weather->names = ho_malloc (sizeof (char *) * weather->length);
    weather->values = ho_malloc (sizeof (char *) * weather->length);

    i = 0;
    weather->names[i] = _("Conditions");
    char *weather_type = get_weather_type (&decodedMetar);
    weather->values[i] = ho_strdup (weather_type);

    i += 1;
    weather->names[i] = _("Sky cover");
    char *cloud_type = get_cloud_type (&decodedMetar);
    weather->values[i] = ho_strdup (cloud_type);

    i += 1;
    weather->names[i] = _("Temperature");
    double temp = decodedMetar.temp;
    if (temp == INT_MAX) {
        weather->values[i] = ho_strdup ("-");
    }
    else {
        weather->values[i] = ho_strdup_printf ("%.1f °C", temp);
    }

    i += 1;
    weather->names[i] = _("Feels like");
    double wind_chill = get_wind_chill (&decodedMetar);
    if (wind_chill == INT_MAX) {
        weather->values[i] = ho_strdup ("-");
    }
    else {
        weather->values[i] = ho_strdup_printf ("%.1f °C", wind_chill);
    }

    i += 1;
    weather->names[i] = _("Dew point");
    double dew_point = decodedMetar.dew_pt_temp;
    if (dew_point == INT_MAX) {
        weather->values[i] = ho_strdup ("-");
    }
    else {
        weather->values[i] = ho_strdup_printf ("%.1f °C", dew_point);
    }

    i += 1;
    weather->names[i] = _("Relative humidity");
    int humidity = get_humidity (&decodedMetar);
    weather->values[i] = ho_strdup_printf ("%d%%", humidity);

    i += 1;
    weather->names[i] = _("Altimeter");
    int altimeter = get_altimeter_hpa (&decodedMetar);
    weather->values[i] = ho_strdup_printf ("%d hPa", altimeter);

    i += 1;
    weather->names[i] = _("Wind direction");
    char *dir = get_wind_dir (&decodedMetar);
    weather->values[i] = ho_strdup (dir);

    i += 1;
    weather->names[i] = _("Wind speed");
    double wind_speed_kmh = get_wind_speed_kmh (&decodedMetar);
    weather->values[i] = ho_strdup_printf ("%.2f km/h", wind_speed_kmh);

    i += 1;
    weather->names[i] = _("Visibility");
    int visibility = (int) decodedMetar.prevail_vsbyM;
    if (decodedMetar.CAVOK || (visibility == 9999) || (visibility == INT_MAX)) {
        weather->values[i] = ho_strdup ("> 10 km");
    }
    else {
        weather->values[i] = ho_strdup_printf ("%d m", visibility);
    }

#ifdef DEBUG
    debug ("%30s: %s", _("Weather station"), weather->station_id);
    debug ("%30s: %s", _("Weather station"), weather->station_name);
    debug ("%30s: %s", _("Last update"), weather->update_time);
    debug ("%30s: %s", "Icon Name", weather->icon_name);
    for (i = 0; i < weather->length; i++) {
        debug ("%30s: %s", weather->names[i], weather->values[i]);
    }
#endif

    ho_free (station_url);
    download_free (download, true);

    return weather;
}


void
weather_free (weather_t * weather)
{
    if (!weather) {
        return;
    }

    int i = 0;
    for (; i < weather->length; i++) {
        ho_free (weather->values[i]);
    }

    ho_free (weather->names);
    ho_free (weather->values);
    ho_free (weather->station_id);
    ho_free (weather->station_name);
    ho_free (weather->update_time);
    ho_free (weather);
}


bool
start_weather_thread (void)
{
    if (pthread_create (&thread, NULL, weather_thread, NULL) != 0) {
        error (_("Could not create weather thread: %s!"), strerror (errno));
        return false;
    }

    return true;
}


void
stop_weather_thread (void)
{
    if (is_running) {
        pthread_cancel (thread);
        info (_("Successfully stopped weather thread."));
    }

    ho_free (station_cache_id);
    ho_free (station_cache_name);
}


static void
station_free_cb (void *data)
{
    weather_station_t *station = (weather_station_t *) data;

    assert (station);

    ho_free (station->id);
    ho_free (station->name);

    ho_free (station);
}


static bool
country_swap_cb (void *d1, void *d2)
{
    weather_country_t *c1 = (weather_country_t *) d1;
    weather_country_t *c2 = (weather_country_t *) d2;

    assert (c1);
    assert (c2);

    return swap_strings (c1->name, c2->name);
}


static bool
station_swap_cb (void *d1, void *d2)
{
    weather_station_t *s1 = (weather_station_t *) d1;
    weather_station_t *s2 = (weather_station_t *) d2;

    assert (s1);
    assert (s2);

    return (strcasecmp (s1->name, s2->name) > 0);
}


static void
country_free_cb (void *data)
{
    weather_country_t *country = (weather_country_t *) data;

    assert (country);
    assert (country->stations);

    ho_free (country->name);
    l_list_free (country->stations, station_free_cb);

    ho_free (country);
}


static weather_country_t *
country_get (const char *name)
{
    weather_country_t *country =
        (weather_country_t *) l_list_first (countries);
    while (country) {
        if (strcmp (country->name, name) == 0) {
            return country;
        }
        country = (weather_country_t *) l_list_next (countries, country);
    }

    country = ho_new (weather_country_t);
    country->name = ho_strdup (name);
    country->stations = l_list_new ();
    country->last_station_i = 0;
    l_list_append (countries, country);

    return country;
}


weather_country_t *
weather_country_get_first (void)
{
    return (weather_country_t *) l_list_first (countries);
}


weather_country_t *
weather_country_get_next (weather_country_t * c)
{
    return (weather_country_t *) l_list_next (countries, c);
}


void
weather_locations_init (void)
{
    countries = l_list_new ();

    char *mrl = OXINE_DATADIR "/metar_stations";
    FILE *f = fopen (mrl, "r");
    if (!f) {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
    }

    weather_country_t *country = NULL;
    while (!feof (f)) {
        char line[1024];
        if (!fgets (line, 1024, f))
            break;

        char *station_id = line;
        char *station_name = index (line, ';') + 1;
        char *country_name = rindex (line, ';') + 1;
        station_name[-1] = 0;
        country_name[-1] = 0;
        station_name = trim_whitespace (station_name);
        country_name = _(trim_whitespace (country_name));

        if (!country || (strcmp (country->name, country_name) != 0)) {
            country = country_get (country_name);
        }

        weather_station_t *station = ho_new (weather_station_t);
        station->id = ho_strdup (station_id);
        station->name = ho_strdup (station_name);
        station->country = country;

        l_list_append (country->stations, station);
    }
    fclose (f);

    l_list_sort (countries, country_swap_cb);

    country = weather_country_get_first ();
    while (country) {
        l_list_sort (country->stations, station_swap_cb);
        country = weather_country_get_next (country);
    }
}


void
weather_locations_free (void)
{
    l_list_free (countries, country_free_cb);
    countries = NULL;

    _("Afghanistan");
    _("Albania");
    _("Algeria");
    _("Angola");
    _("Anguilla");
    _("Antarctica");
    _("Antigua and Barbuda");
    _("Argentina");
    _("Armenia");
    _("Aruba");
    _("Australia");
    _("Austria");
    _("Azerbaijan");
    _("Bahamas, The");
    _("Bahrain");
    _("Bangladesh");
    _("Barbados");
    _("Belarus");
    _("Belgium");
    _("Belize");
    _("Benin");
    _("Bermuda");
    _("Bolivia");
    _("Bosnia and Herzegovina");
    _("Botswana");
    _("Brazil");
    _("British Indian Ocean Territory");
    _("British Virgin Islands");
    _("Brunei");
    _("Bulgaria");
    _("Burkina Faso");
    _("Burundi");
    _("Cambodia");
    _("Cameroon");
    _("Canada");
    _("Cape Verde");
    _("Cayman Islands");
    _("Central African Republic");
    _("Chad");
    _("Chile");
    _("China");
    _("Christmas Island");
    _("Colombia");
    _("Comoros");
    _("Congo, Democratic Republic of the");
    _("Congo, Republic of the");
    _("Cook Islands");
    _("Costa Rica");
    _("Cote d'Ivoire");
    _("Croatia");
    _("Cuba");
    _("Cyprus");
    _("Czech Republic");
    _("Denmark");
    _("Djibouti");
    _("Dominica");
    _("Dominican Republic");
    _("East Timor");
    _("Ecuador");
    _("Egypt");
    _("El Salvador");
    _("Equatorial Guinea");
    _("Estonia");
    _("Ethiopia");
    _("Falkland Islands, Islas Malvinas");
    _("Fiji");
    _("Finland");
    _("France");
    _("French Guiana");
    _("French Polynesia");
    _("Gabon");
    _("Gambia, The");
    _("Georgia");
    _("Germany");
    _("Ghana");
    _("Gibraltar");
    _("Greece");
    _("Greenland");
    _("Grenada");
    _("Guadeloupe");
    _("Guatemala");
    _("Guinea");
    _("Guinea-Bissau");
    _("Guyana");
    _("Haiti");
    _("Honduras");
    _("Hong Kong");
    _("Hungary");
    _("Iceland");
    _("India");
    _("Indonesia");
    _("Iran");
    _("Iraq");
    _("Ireland");
    _("Israel");
    _("Italy");
    _("Jamaica");
    _("Japan");
    _("Jordan");
    _("Kazakhstan");
    _("Kenya");
    _("Kiribati");
    _("Korea, North");
    _("Korea, South");
    _("Kuwait");
    _("Kyrgyzstan");
    _("Laos");
    _("Latvia");
    _("Lebanon");
    _("Lesotho");
    _("Liberia");
    _("Libya");
    _("Lithuania");
    _("Luxembourg");
    _("Macau");
    _("Macedonia, The Republic of");
    _("Madagascar");
    _("Malawi");
    _("Malaysia");
    _("Maldives");
    _("Mali");
    _("Malta");
    _("Marshall Islands");
    _("Martinique");
    _("Mauritania");
    _("Mauritius");
    _("Mexico");
    _("Micronesia, Federated States of");
    _("Moldova");
    _("Mongolia");
    _("Morocco");
    _("Mozambique");
    _("Myanmar");
    _("Namibia");
    _("Nauru");
    _("Nepal");
    _("Netherlands");
    _("Netherlands Antilles");
    _("New Caledonia");
    _("New Zealand");
    _("Nicaragua");
    _("Niger");
    _("Nigeria");
    _("Norway");
    _("Oman");
    _("Pakistan");
    _("Palau");
    _("Panama");
    _("Papua New Guinea");
    _("Paraguay");
    _("Peru");
    _("Philippines");
    _("Poland");
    _("Portugal");
    _("Qatar");
    _("Reunion");
    _("Romania");
    _("Russia");
    _("Rwanda");
    _("Saint Helena");
    _("Saint Kitts and Nevis");
    _("Saint Lucia");
    _("Saint Pierre and Miquelon");
    _("Saint Vincent and the Grenadines");
    _("Samoa");
    _("Sao Tome and Principe");
    _("Saudi Arabia");
    _("Senegal");
    _("Serbia and Montenegro");
    _("Seychelles");
    _("Sierra Leone");
    _("Singapore");
    _("Slovakia");
    _("Slovenia");
    _("Solomon Islands");
    _("Somalia");
    _("South Africa");
    _("South Georgia and the Islands");
    _("Spain");
    _("Sri Lanka");
    _("Sudan");
    _("Suriname");
    _("Swaziland");
    _("Sweden");
    _("Switzerland");
    _("Syria");
    _("Taiwan");
    _("Tajikistan");
    _("Tanzania");
    _("Thailand");
    _("Togo");
    _("Tonga");
    _("Trinidad and Tobago");
    _("Tunisia");
    _("Turkey");
    _("Turkmenistan");
    _("Turks and Caicos Islands");
    _("Tuvalu");
    _("Uganda");
    _("Ukraine");
    _("United Arab Emirates");
    _("United Kingdom");
    _("United States");
    _("United States Minor Outlying Islands");
    _("Uruguay");
    _("Uzbekistan");
    _("Vanuatu");
    _("Venezuela");
    _("Vietnam");
    _("Virgin Islands");
    _("Western Sahara");
    _("Yemen");
    _("Zambia");
    _("Zimbabwe");
}


#endif /* HAVE_WEATHER */
