// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package builds

import (
	"context"
	"net/http/httptest"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/mock"

	"github.com/go-chi/chi"
	"github.com/golang/mock/gomock"
)

func TestCancel(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockStages := []*core.Stage{
		{Status: core.StatusPassing},
		{
			Status: core.StatusPending,
			Steps: []*core.Step{
				{Status: core.StatusPassing},
				{Status: core.StatusPending},
			},
		},
	}

	mockBuildCopy := new(core.Build)
	*mockBuildCopy = *mockBuild

	repos := mock.NewMockRepositoryStore(controller)
	repos.EXPECT().FindName(gomock.Any(), mockRepo.Namespace, mockRepo.Name).Return(mockRepo, nil)

	builds := mock.NewMockBuildStore(controller)
	builds.EXPECT().FindNumber(gomock.Any(), mockRepo.ID, mockBuild.Number).Return(mockBuildCopy, nil)
	builds.EXPECT().Update(gomock.Any(), mockBuildCopy).Return(nil)

	users := mock.NewMockUserStore(controller)
	users.EXPECT().Find(gomock.Any(), mockRepo.UserID).Return(mockUser, nil)

	stages := mock.NewMockStageStore(controller)
	stages.EXPECT().ListSteps(gomock.Any(), mockBuild.ID).Return(mockStages, nil)
	stages.EXPECT().Update(gomock.Any(), mockStages[1]).Return(nil)

	steps := mock.NewMockStepStore(controller)
	steps.EXPECT().Update(gomock.Any(), mockStages[1].Steps[1]).Return(nil)

	statusService := mock.NewMockStatusService(controller)
	statusService.EXPECT().Send(gomock.Any(), mockUser, gomock.Any()).Return(nil)

	webhook := mock.NewMockWebhookSender(controller)
	webhook.EXPECT().Send(gomock.Any(), gomock.Any()).Return(nil)

	scheduler := mock.NewMockScheduler(controller)
	scheduler.EXPECT().Cancel(gomock.Any(), mockBuild.ID).Return(nil)

	c := new(chi.Context)
	c.URLParams.Add("owner", "octocat")
	c.URLParams.Add("name", "hello-world")
	c.URLParams.Add("number", "1")

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/", nil)
	r = r.WithContext(
		context.WithValue(context.Background(), chi.RouteCtxKey, c),
	)

	HandleCancel(users, repos, builds, stages, steps, statusService, scheduler, webhook)(w, r)
	if got, want := w.Code, 200; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}
}
