// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package livelog

import (
	"context"
	"sync"
	"testing"

	"github.com/drone/drone/core"

	"github.com/google/go-cmp/cmp"
)

func TestStreamer(t *testing.T) {
	s := newStreamer().(*streamer)
	err := s.Create(context.Background(), 1)
	if err != nil {
		t.Error(err)
	}
	if len(s.streams) == 0 {
		t.Errorf("Want stream registered")
	}

	w := sync.WaitGroup{}
	w.Add(4)
	go func() {
		s.Write(context.Background(), 1, &core.Line{})
		s.Write(context.Background(), 1, &core.Line{})
		s.Write(context.Background(), 1, &core.Line{})
		w.Done()
	}()

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	tail, errc := s.Tail(ctx, 1)

	go func() {
		for {
			select {
			case <-errc:
				return
			case <-ctx.Done():
				return
			case <-tail:
				w.Done()
			}
		}
	}()

	w.Wait()
}

func TestStreamerDelete(t *testing.T) {
	s := newStreamer().(*streamer)
	err := s.Create(context.Background(), 1)
	if err != nil {
		t.Error(err)
	}
	if len(s.streams) == 0 {
		t.Errorf("Want stream registered")
	}
	err = s.Delete(context.Background(), 1)
	if err != nil {
		t.Error(err)
	}
	if len(s.streams) != 0 {
		t.Errorf("Want stream unregistered")
	}
}

func TestStreamerDeleteErr(t *testing.T) {
	s := newStreamer()
	err := s.Delete(context.Background(), 1)
	if err != errStreamNotFound {
		t.Errorf("Want errStreamNotFound")
	}
}

func TestStreamerWriteErr(t *testing.T) {
	s := newStreamer()
	err := s.Write(context.Background(), 1, &core.Line{})
	if err != errStreamNotFound {
		t.Errorf("Want errStreamNotFound")
	}
}

func TestStreamTailNotFound(t *testing.T) {
	s := newStreamer()
	outc, errc := s.Tail(context.Background(), 0)
	if outc != nil && errc != nil {
		t.Errorf("Expect nil channel when stream not found")
	}
}

func TestStreamerInfo(t *testing.T) {
	s := newStreamer().(*streamer)
	s.streams[1] = &stream{list: map[*subscriber]struct{}{{}: struct{}{}, {}: struct{}{}}}
	s.streams[2] = &stream{list: map[*subscriber]struct{}{{}: struct{}{}}}
	s.streams[3] = &stream{list: map[*subscriber]struct{}{}}
	got := s.Info(context.Background())

	want := &core.LogStreamInfo{
		Streams: map[int64]int{
			1: 2,
			2: 1,
			3: 0,
		},
	}

	if diff := cmp.Diff(got, want); diff != "" {
		t.Errorf(diff)
	}
}
