/*
  --------------------------------------------------------------
  Implementation of private interface for veriT
  --------------------------------------------------------------
*/

#include "response.h"
#include "utils/dll.h"
#include "utils/general.h"
#include "utils/list.h"
#include "utils/stack.h"
#include "utils/statistics.h"
#include "utils/types.h"
#include "veriT-config.h"

#include <limits.h>
#include <time.h>
#include <stdbool.h>
#ifdef HAVE_POSIX_TIMER
#include <sys/time.h>
#endif

/* #define DEBUG */

/* HB necessary to put it again here why?? */
#define STATS_LEVEL 1

#define CC_ADD_ALL_DAGS_BEFORE
#include "arith/LA.h"
#include "bool/bool.h"
#include "complete.h"
#include "congruence/congruence.h"
#include "hint.h"
#include "pre/ite-elim.h"
#include "pre/pre.h"
#include "pre/simp-unit.h"
#include "pre/simplify.h"
#include "proof/proof.h"
#include "symbolic/DAG-print.h"
#include "symbolic/DAG-ptr.h"
#include "symbolic/DAG-tmp.h"
#include "symbolic/DAG.h"
#include "symbolic/veriT-status.h"
#include "utils/veriT-qsort.h"
#include "veriT.h"
#ifdef DEBUG
#include "bool/literal.h"
#endif
#include "arith/totality.h"
#include "instantiation/ccfv.h"
#include "instantiation/inst-del.h"
#include "instantiation/inst-man.h"
#include "instantiation/inst-trigger.h"
#include "symbolic/veriT-status.h"
#include "undo.h"
#include "utils/options.h"
#include "utils/veriT-signal.h"
#include "veriT-state.h"


/* #define DEBUG_VERIT_SOLVE */

static Tstatus status = SAT;
/* PF buffer table for clauses between theory reasoner and SAT solver */

/*
  --------------------------------------------------------------
  User configurable options
  --------------------------------------------------------------
*/

/**
   \addtogroup arguments_developer

   - --list=s1;s2

   Only print stats in list */
static char* stats_list = NULL;

#ifdef DEBUG
/**
   \addtogroup arguments_developer

   - --check-deduced

   Generates files with verification conditions in SMT format for each
   conflict clause, model and lemma found. Useful to debug consistency
   of the theory solver (only available when compiled with DEBUG
   defined). */
static bool check_deduced;

/**
   \addtogroup arguments_developer

   - --dump-abstract-models

   Generates files in DIMACS format with the propositional abstraction
   of the model, the lemmas and conflict clauses that have been given
   to the boolean engine. Useful to debug consistency (only available
   when compiled with DEBUG defined). */
static bool dump_abstract_models;

/**
   \addtogroup arguments_developer

   - --dump-CIs

   Produce SMT files for current ground model plus each conflict
   instance generated by CCFV. Useful to check whether generated
   instances are indeed conflicting (only available when compiled with
   DEBUG defined). */
static bool dump_CIs;
#endif

#if 0
/**
   \addtogroup arguments_developer

   - --sym-sat

   Find and uses symmetries in the SAT solver.  For every added unit clause,
   add all the symmetric ones. */
static bool sym_sat;
#endif

/**
   \addtogroup arguments_developer

   - --print-report

   Prints statistics report on exit.
*/
bool veriT_print_report;

#ifdef HAVE_POSIX_TIMER
int option_max_time;
int option_max_virtual_time;
#else
#ifdef HAVE_WINDOWS_H
int option_max_time;
HANDLE windows_timer_queue;
#endif
#endif

bool option_print_simp_and_exit;
char* output_format;
bool option_parse_only;

#if STATS_LEVEL > 0
static unsigned stat_total_time;
static unsigned stat_pre_time;
#endif
#if STATS_LEVEL > 1
static unsigned stat_nb_clauses;
static unsigned stat_nb_binary_clauses;
#endif
#if STATS_TRACK_STATE
static unsigned state_check_sat;
static unsigned state_solve;
static unsigned state_solve_full;
static unsigned state_inst;
static unsigned state_theory;
static unsigned state_done;
static unsigned state_init;
static unsigned stat_solver_state;
#endif
#if STATS_LEVEL > 0
#define VERIT_SOLVE_TIMER_START stats_timer_start(stat_total_time);
#define VERIT_SOLVE_TIMER_STOP stats_timer_stop(stat_total_time);
#define VERIT_PRE_TIMER_START stats_timer_start(stat_pre_time);
#define VERIT_PRE_TIMER_STOP stats_timer_stop(stat_pre_time);
#else
#define VERIT_SOLVE_TIMER_START
#define VERIT_SOLVE_TIMER_STOP
#define VERIT_PRE_TIMER_START
#define VERIT_PRE_TIMER_STOP
#endif

/** \brief types of backtracking items */
enum
{
	VERIT_UNSAT = UNDO_VERIT
};

Tstack_lit veriT_conflict = NULL;

bool Q_active = false;
bool LA_active = false;

/*
  --------------------------------------------------------------
  Post-recheck
  --------------------------------------------------------------
*/

#if defined(DEBUG) && defined(DEBUG_RECHECK_UNSAT)
Tstack_clause recheck_clauses = NULL;
Tstack_DAG recheck_formulas = NULL;

static void
veriT_boolean_abstraction_store_input(TDAG DAG)
{
	stack_push(recheck_formulas, DAG_dup(DAG));
}

static void
veriT_boolean_abstraction_store_clause(Tclause clause)
{
	stack_push(recheck_clauses, clause_dup(clause));
}

static void
veriT_boolean_abstraction_store_lemmas(Tstack_DAG lemmas)
{
	unsigned i;
	for (i = 0; i < stack_size(lemmas); i++)
		stack_push(recheck_formulas, DAG_dup(stack_get(lemmas, i)));
}

static void
veriT_boolean_abstraction_init(void)
{
	stack_INIT(recheck_clauses);
	stack_INIT(recheck_formulas);
}

static void
veriT_boolean_abstraction_out(Tstatus status)
{
	bool_recheck("boolean.cnf", status, recheck_formulas, recheck_clauses);
}

static void
veriT_boolean_abstraction_done(void)
{
	while (!stack_is_empty(recheck_clauses))
		clause_free(stack_pop(recheck_clauses));
	stack_free(recheck_clauses);
	while (!stack_is_empty(recheck_formulas))
		DAG_free(stack_pop(recheck_formulas));
	stack_free(recheck_formulas);
}
#endif /* defined(DEBUG) && defined(DEBUG_RECHECK_UNSAT) */

/*
  --------------------------------------------------------------
  Helpers for main loop
  --------------------------------------------------------------
*/

static Tstack_DAG table_lemma;

static void
veriT_add_lemma(TDAG lemma)
/* PF add the lemma without preprocessing.  For internal use */
{
	stack_push(table_lemma, lemma);
#ifdef POLARITY_FILTER
	stack_push(orig_formula, DAG_dup(lemma));
#endif
	if (proof_on)
		bool_add_proof(lemma, proof_get_lemma_id(lemma));
	else
		bool_add(lemma);
	CC_notify_formula(lemma);
	if (LA_active) {
		LA_notify_formula(lemma);
		if (LA_overflow) LA_switch_to_mp();
	}
	SAT_propagate();
}

#ifdef DEBUG

static void
veriT_print_literals(TDAG* literals, unsigned nb, char* filename, char* status)
{
	FILE* file = fopen(filename, "w");
	DAG_fprint_smt2_set(file, literals, nb, status);
	fprintf(stderr, "File %s written.\n", filename);
	fclose(file);
}

static void
veriT_dump_literals_and_CIs(char* status, Tstack_DAG conf_instances)
{
	unsigned i;
	TDAG* PDAG;
	char filename[128];
	static unsigned ci_count = 0;
	MY_MALLOC(PDAG, (SAT_literal_stack_n + 1) * sizeof(TDAG));
	for (i = 0; i < SAT_literal_stack_n; i++) {
		Tlit lit = SAT_literal_stack[i];
		TDAG DAG = var_to_DAG(lit_var(lit));
		assert(DAG);
		PDAG[i] = DAG_dup(lit_pol(lit) ? DAG : DAG_not(DAG));
	}
	for (i = 0; i < stack_size(conf_instances); ++i) {
		sprintf(filename, "ci-%02u.smt2", ++ci_count);
		TDAG CI = DAG_dup(DAG_arg1(stack_get(conf_instances, i)));
		FILE* file = fopen(filename, "w");
		DAG_fprint_smt2_set_gr_DAG(file, PDAG, SAT_literal_stack_n, CI, status);
		fprintf(stderr, "File %s written.\n", filename);
		fclose(file);
		DAG_free(CI);
	}
	for (i = 0; i < SAT_literal_stack_n; i++) DAG_free(PDAG[i]);
	free(PDAG);
}

/**
   \brief prints the stack of literals in dump-**.smt2
   \param status: the status to add to file */
static void
veriT_dump_literals(char* status)
{
	unsigned i;
	TDAG* PDAG;
	char filename[128];
	static unsigned count = 0;
	sprintf(filename, "dump-%u.smt2", ++count);
	MY_MALLOC(PDAG, SAT_literal_stack_n * sizeof(TDAG));
	for (i = 0; i < SAT_literal_stack_n; i++) {
		Tlit lit = SAT_literal_stack[i];
		TDAG DAG = var_to_DAG(lit_var(lit));
		assert(DAG);
		PDAG[i] = DAG_dup(lit_pol(lit) ? DAG : DAG_not(DAG));
	}
	veriT_print_literals(PDAG, SAT_literal_stack_n, filename, status);
	for (i = 0; i < SAT_literal_stack_n; i++) DAG_free(PDAG[i]);
	free(PDAG);
}

/**
   \brief prints the stack of literals in model-**.smt2 */
static void
veriT_output_model(void)
{
	unsigned i, j;
	TDAG* PDAG;
	char filename[128];
	static unsigned model_n = 0;
	sprintf(filename, "model-%05d.smt2", ++model_n);
	MY_MALLOC(PDAG, SAT_literal_stack_n * sizeof(*PDAG));
	for (i = 0, j = 0; i < SAT_literal_stack_n; ++i) {
		Tlit lit = SAT_literal_stack[i];
		TDAG DAG = var_to_DAG(lit_var(lit));
		if (boolean_connector(DAG_symb(DAG))) continue;
		PDAG[j++] = DAG_dup(lit_pol(lit) ? DAG : DAG_not(DAG));
	}
	MY_REALLOC(PDAG, j * sizeof(*PDAG));
	if (!j)
		veriT_print_literals(&DAG_TRUE, 1, filename, "sat");
	else
		veriT_print_literals(PDAG, j, filename, "sat");
	for (i = 0; i < j; i++) DAG_free(PDAG[i]);
	free(PDAG);
}

static void
veriT_print_lemmas(Tstack_DAG lemmas)
{
	unsigned i;
	static int lemma_no = 0;
	for (i = 0; i < stack_size(lemmas); i++) {
		FILE* file;
		char filename[128];
		TDAG DAG = stack_get(lemmas, i);
		TDAG DAG2 = DAG_dup(DAG_not(DAG));
		sprintf(filename, "lemma-%d.smt2", ++lemma_no);
		file = fopen(filename, "w");
		DAG_fprint_smt2_bench(file, DAG2, "unsat");
		fprintf(stderr, "File %s written.\n", filename);
		DAG_free(DAG2);
		fclose(file);
	}
}

#endif /* DEBUG */

/*
  --------------------------------------------------------------
  Assertion stack
  --------------------------------------------------------------
*/

/** \brief local nicknaming macro */
#define assertion_set_stack veriT_assertion_set_stack

Tstack_assertion_set assertion_set_stack = NULL;
static unsigned current_level = 0;
static unsigned last_checked_level = 0;

static void
verit_stack_push(void)
{
	if (!assertion_set_stack) veriT_error("verit_stack_push: internal error");
	assert(stack_size(assertion_set_stack) == current_level);
	stack_inc(assertion_set_stack);
	stack_top(assertion_set_stack).level = ++current_level;
	stack_INIT(stack_top(assertion_set_stack).DAG_stack);
}

static void
verit_stack_push_n(unsigned n)
{
	for (; n > 0; n--) verit_stack_push();
}

static void
verit_stack_add(TDAG DAG)
{
	if (!assertion_set_stack || !stack_size(assertion_set_stack))
		veriT_error("verit_stack_add: internal error");
	assert(stack_size(assertion_set_stack) == current_level);
	stack_push(stack_top(assertion_set_stack).DAG_stack, DAG_dup(DAG));
}

static void
verit_stack_pop(void)
{
	unsigned i;
	if (!assertion_set_stack || !stack_size(assertion_set_stack))
		veriT_error("empty stack");
	assert(stack_size(assertion_set_stack) == current_level);
	assert(current_level > 0);
	for (i = 0; i < stack_size(stack_top(assertion_set_stack).DAG_stack); i++)
		DAG_free(stack_get(stack_top(assertion_set_stack).DAG_stack, i));
	stack_free(stack_top(assertion_set_stack).DAG_stack);
	stack_dec(assertion_set_stack);
	--current_level;
}

static void
verit_stack_pop_n(unsigned n)
{
	for (; n > 0; n--) verit_stack_pop();
}

static unsigned
verit_stack_get_level(unsigned i)
{
	if (!assertion_set_stack || stack_size(assertion_set_stack) <= i)
		veriT_error("verit_stack_get_level: internal error");
	return stack_get(assertion_set_stack, i).level;
}

static Tstack_DAG
verit_stack_get_assertions(unsigned i)
{
	if (!assertion_set_stack || stack_size(assertion_set_stack) <= i)
		veriT_error("verit_stack_get_assertions: internal error");
	return stack_top(assertion_set_stack).DAG_stack;
}

static unsigned
verit_stack_length(void)
{
	if (!assertion_set_stack) veriT_error("verit_stack_length: internal error");
	return stack_size(assertion_set_stack);
}

static void
verit_stack_init(void)
{
	if (assertion_set_stack) veriT_error("verit_stack_init: internal error");
	stack_INIT(assertion_set_stack);
	verit_stack_push();
}

static void
verit_stack_done(void)
{
	unsigned i;
	if (!assertion_set_stack) veriT_error("verit_stack_done: internal error");
	for (i = stack_size(assertion_set_stack); i-- > 0;) verit_stack_pop();
	stack_free(assertion_set_stack);
}

/*
  --------------------------------------------------------------
  veriT miscellaneous
  --------------------------------------------------------------
*/

int
veriT_complete(void)
{
	return complete_check();
}

/* TODO: make sure it fails if unknown logic */
void
veriT_logic(char* logic)
{
	DAG_smtlib_logic_set(logic);
	pre_logic(logic);
	complete_logic(logic);
}

void
veriT_model(void)
{
	/* TODO: this only works partially
     Especially with arithmetics: if a variable is assigned a value by
     the arithmetic module, and if this value is not explicitely present in
     the congruence module, then this function will fail to associate
     the variable and the value */
	CC_model(veriT_out_no_newline);
}

/*
  --------------------------------------------------------------
  public old functions
  --------------------------------------------------------------
*/

#ifdef DEBUG
static Tstack_lit root_lit_stack = NULL;

static void
veriT_print_clause(Tclause clause)
{
	unsigned i, j, num;
	static int conflict_no = 0;
	FILE* file;
	char filename[128];
	TDAG* PDAG;
	sprintf(filename, "conflict-%05d.smt2", ++conflict_no);
	file = fopen(filename, "w");
	num = clause->nb_lits + stack_size(root_lit_stack);
	MY_MALLOC(PDAG, num * sizeof(TDAG));
	for (i = 0; i < clause->nb_lits; i++) {
		PDAG[i] = var_to_DAG(lit_var(clause->lits[i]));
		if (lit_pol(clause->lits[i])) PDAG[i] = DAG_not(PDAG[i]);
		DAG_dup(PDAG[i]);
	}
	/* Adds all literals that have forced arithmetic variables to a
     unique value. The reason is that these literals may have caused
     simplifications that are a cause for the current conflict. */
	for (j = 0; j < stack_size(root_lit_stack); ++j, ++i) {
		assert(lit_var(stack_get(root_lit_stack, j)));
		PDAG[i] = var_to_DAG(lit_var(stack_get(root_lit_stack, j)));
		if (lit_pol(stack_get(root_lit_stack, j))) PDAG[i] = DAG_not(PDAG[i]);
		DAG_dup(PDAG[i]);
	}

	/* remove duplicates */
	veriT_qsort(PDAG, num, sizeof(TDAG), (TFcmp)DAG_cmp_q);
	for (i = 0, j = 1; j < num; ++j)
		if (PDAG[i] != PDAG[j]) {
			++i;
			PDAG[i] = PDAG[j];
		} else
			DAG_free(PDAG[j]);
	j = i + 1; /* j is essentially the number of useful elements in PDAG */
	DAG_fprint_smt2_set(file, PDAG, j, "unsat");
	for (i = 0; i < j; i++) DAG_free(PDAG[i]);
	free(PDAG);
	fclose(file);
	fprintf(stderr, "File %s written.\n", filename);
}
#endif

/* DD+PF returns status */
Tstatus
veriT_status(void)
{
	return status;
}

/* DD+PF reset for new formulas */
void
veriT_reset(void)
{
	status = SAT;
	bool_reset();
}

void
veriT_get_proof(void)
{
	assert(status == UNSAT);
	write_proof(veriT_out_file);
}

void
veriT_get_unsat_core(void)
{
	assert(status == UNSAT);
	proof_hyp(veriT_out_file);
}

/*
  --------------------------------------------------------------
  veriT core
  --------------------------------------------------------------
*/

static enum { cc, la, la_z } conflict_dp;

static void
backtrack_unsat(void)
{
	undo_push(VERIT_UNSAT);
}

static void
backtrack_unsat_hook(void* P)
{
	status = OPEN;
#ifdef PEDANTIC
	printf("%p\n", P);
#endif
}

static Tproof* resolution_chain = NULL;
static unsigned resolution_chain_size = 0;
static unsigned resolution_chain_allocated = 0;

static void
resolution_chain_add(Tproof proof)
{
	if (resolution_chain_allocated < resolution_chain_size + 1) {
		resolution_chain_allocated *= 2;
		MY_REALLOC(resolution_chain, resolution_chain_allocated * sizeof(Tproof));
	}
	resolution_chain[resolution_chain_size++] = proof;
}

static Tproof
resolution_chain_close(void)
{
	Tproof proof;
	if (resolution_chain_size == 1)
		proof = resolution_chain[0];
	else
		proof = proof_resolve_array(resolution_chain_size, resolution_chain);
	resolution_chain_size = 0;
	return proof;
}

static void (*conflict_clause)(void) = NULL;

/**
   \brief  On conflict, build conflict clause and add to SAT solver */
static void
conflict_clause_noproof(void)
{
	unsigned i;
	Tclause clause;
	stack_INIT(veriT_conflict);
	if (conflict_dp == cc)
		CC_conflict();
	else if (conflict_dp == la) {
		LA_conflict();
		if (LA_overflow) return;
	} else if (conflict_dp == la_z) {
		LA_conflict_z();
		if (LA_overflow) return;
	}
	if (stack_size(veriT_conflict_eqs)) {
		for (i = 0; i < stack_size(veriT_conflict_eqs); i += 2)
			CC_premisses_eq(
				stack_get(veriT_conflict_eqs, i), stack_get(veriT_conflict_eqs, i + 1));
		stack_reset(veriT_conflict_eqs);
	}
	assert(stack_size(veriT_conflict));
	for (i = 0; i < stack_size(veriT_conflict); i++)
		veriT_conflict->data[i] = lit_neg(veriT_conflict->data[i]);

	clause = clause_new_stack(veriT_conflict);
	stack_free(veriT_conflict);
#ifdef DEBUG
	if (check_deduced) {
		veriT_print_clause(clause);
#ifdef DEBUG_RECHECK_UNSAT
		veriT_boolean_abstraction_store_clause(clause);
#endif
	}
#endif
#if STATS_LEVEL > 1
	stats_counter_inc(stat_nb_clauses);
	if (clause->nb - lits == 2) stack_counter_inc(stat_nb_binary_clauses);
#endif
	bool_add_c_conflict(clause);
}

static void
conflict_clause_proof(void)
{
	unsigned i;
	Tclause clause;
	Tproof proof = 0; /* = 0 fixes a spurious compiler warning */
	stack_INIT(veriT_conflict);
	if (conflict_dp == cc)
		proof = CC_conflict_proof();
	else if (conflict_dp == la)
		proof = LA_conflict_proof();
	else if (conflict_dp == la_z)
		proof = LA_conflict_proof_z();
	if (stack_size(veriT_conflict_eqs)) {
		if (proof) resolution_chain_add(proof);
		for (i = 0; i < stack_size(veriT_conflict_eqs); i += 2) {
			Tproof proof = CC_premisses_eq_proof(
				stack_get(veriT_conflict_eqs, i), stack_get(veriT_conflict_eqs, i + 1));
			if (proof) resolution_chain_add(proof);
		}
		stack_reset(veriT_conflict_eqs);
		proof = resolution_chain_close();
	}
	assert(stack_size(veriT_conflict));
	for (i = 0; i < stack_size(veriT_conflict); i++)
		veriT_conflict->data[i] = lit_neg(veriT_conflict->data[i]);

	clause = clause_new_stack(veriT_conflict);
	clause->proof_id = proof;
	stack_free(veriT_conflict);
#ifdef DEBUG
	if (check_deduced) {
		veriT_print_clause(clause);
#ifdef DEBUG_RECHECK_UNSAT
		veriT_boolean_abstraction_store_clause(clause);
#endif
	}
#endif
#if STATS_LEVEL > 1
	stats_counter_inc(stat_nb_clauses);
	if (clause->nb - lits == 2) stack_counter_inc(stat_nb_binary_clauses);
#endif
	bool_add_c_conflict(clause);
}

/* #define DEBUG_HINT */
#ifndef DEBUG_HINT
void (*hint_explain)(Tlit lit) = NULL;
#endif

/**
   \brief Generates a clause for literal propagated as hint
   \param lit a literal */
static void
hint_explain_noproof(Tlit lit)
{
	unsigned i;
	Tclause clause;
	stack_INIT(veriT_conflict);
	hint_explain_dispatch(lit);
	assert(stack_size(veriT_conflict));
	for (i = 0; i < stack_size(veriT_conflict); i++)
		veriT_conflict->data[i] = lit_neg(veriT_conflict->data[i]);
	clause = clause_new_stack(veriT_conflict);
	stack_free(veriT_conflict);
	/* IMPROVE yet another function to add a clause to SAT solver */
#ifdef DEBUG
	if (check_deduced) {
		veriT_print_clause(clause);
#ifdef DEBUG_RECHECK_UNSAT
		veriT_boolean_abstraction_store_clause(clause);
#endif
	}
#endif
#if STATS_LEVEL > 1
	stats_counter_inc(stat_nb_clauses);
	if (clause->nb - lits == 2) stack_counter_inc(stat_nb_binary_clauses);
#endif
	bool_add_c_hint(clause);
}

#ifdef DEBUG_HINT
static void
hint_explain(Tlit lit)
#else
static void
hint_explain_proof(Tlit lit)
#endif
{
	unsigned i;
	Tclause clause;
	Tproof proof;
	stack_INIT(veriT_conflict);
	proof = CC_hint_explain_proof(lit);
	assert(stack_size(veriT_conflict));
	for (i = 0; i < stack_size(veriT_conflict); i++)
		veriT_conflict->data[i] = lit_neg(veriT_conflict->data[i]);
	clause = clause_new_stack(veriT_conflict);
	stack_free(veriT_conflict);
	/* IMPROVE yet another function to add a clause to SAT solver */
#ifdef DEBUG
	if (check_deduced) {
		veriT_print_clause(clause);
#ifdef DEBUG_RECHECK_UNSAT
		veriT_boolean_abstraction_store_clause(clause);
#endif
	}
#endif
#if STATS_LEVEL > 1
	stats_counter_inc(stat_nb_clauses);
	if (clause->nb - lits == 2) stack_counter_inc(stat_nb_binary_clauses);
#endif
	clause->proof_id = proof;
	bool_add_c_hint(clause);
}

/*
  --------------------------------------------------------------
  veriT main commands
  --------------------------------------------------------------
*/

void
veriT_push(unsigned n)
{
	verit_stack_push_n(n);
}

void
veriT_pop(unsigned n)
{
	status = OPEN;
	last_checked_level = 0;
	/* TODO: RESET EVERYTHING, UNTIL WE CAN DO BETTER */
	veriT_error("unimplemented");
	if (verit_stack_length() <= n) veriT_error("pop exceeds sum of pushes");
	verit_stack_pop_n(n);
}

void
veriT_assert(TDAG DAG)
{
	if (status != UNSAT) status = OPEN;
	if (DAG_sort(DAG) != SORT_BOOLEAN)
		veriT_error("asserting a non Boolean term");
	verit_stack_add(DAG);
}

extern bool LA_enable_lemmas_totality;

Tstatus
veriT_check_sat(void)
{
	unsigned i, j;
	TDAG DAG1 = DAG_NULL, DAG2 = DAG_NULL;
	unsigned nb = 0;
	TDAG* PDAG = NULL;
#ifdef STATS_TRACK_STATE
	stats_state_switch(stat_solver_state, state_check_sat);
#endif
	Tproof* Pproof = NULL; /* = NULL fixes a spurious compiler warning */
	if (proof_on) {
		conflict_clause = conflict_clause_proof;
#ifndef DEBUG_HINT
		hint_explain = hint_explain_proof;
#endif
		SAT_proof = 1;
	} else {
		conflict_clause = conflict_clause_noproof;
#ifndef DEBUG_HINT
		hint_explain = hint_explain_noproof;
#endif
		SAT_proof = 0;
	}
	/* TODO
     Difficult things: pre process:
     where to put and how to handle non-local stuff
     what to do when veriT_pop used
  */
	if (current_level == last_checked_level) return status;
	assert(current_level > last_checked_level);
	assert(stack_size(assertion_set_stack) == current_level);

	/* PF Collect assertions */
	for (i = verit_stack_length();
			 i-- > 0 && verit_stack_get_level(i) > last_checked_level;) {
		unsigned j;
		Tstack_DAG DAGs = verit_stack_get_assertions(i);
		MY_REALLOC(PDAG, (nb + stack_size(DAGs)) * sizeof(TDAG));
		for (j = 0; j < stack_size(DAGs); j++) PDAG[nb++] = stack_get(DAGs, j);
	}
	/* PF Eliminate duplicates and find contradictions */
	DAG_tmp_reserve();
	for (i = 0, j = 0; i < nb; i++)
		if (!DAG_tmp_unsigned[DAG1 = DAG_atom(PDAG[i])]) {
			DAG_tmp_unsigned[DAG1] = DAG_polarity(PDAG[i]) | (j << 2);
			PDAG[j++] = PDAG[i];
		} else if (DAG_polarity(PDAG[i]) != (DAG_tmp_unsigned[DAG1] & 3)) {
			Tproof proof;
			Tproof proof2;
			DAG2 = PDAG[DAG_tmp_unsigned[DAG1] >> 2];
			DAG1 = PDAG[i];
			for (i = 0; i < j; i++) DAG_tmp_unsigned[DAG_atom(PDAG[i])] = 0;
			if (proof_on) {
				proof = proof_add_input(DAG1);
				proof2 = proof_add_input(DAG2);
				proof_resolve(2, proof, proof2);
				proof_unsatisfiable();
			}
			free(PDAG);
			status = UNSAT;
			DAG_tmp_release();
			return status;
		}
	nb = j;
	for (i = 0; i < nb; i++) DAG_tmp_unsigned[DAG_atom(PDAG[i])] = 0;
	DAG_tmp_release();

	if (!nb) {
		free(PDAG);
		status = SAT;
		if (proof_on) proof_satisfiable();
		return status;
	}
	VERIT_PRE_TIMER_START;

	if (proof_on) {
		MY_MALLOC(Pproof, nb * sizeof(Tproof));
		for (i = 0; i < nb; i++) Pproof[i] = proof_add_input(PDAG[i]);
		for (i = 0; i < nb; i++) DAG_dup(PDAG[i]);
		pre_process_array_proof(nb, PDAG, Pproof);
		for (i = 0; i < nb; i++) complete_add(PDAG[i]);
#ifdef POLARITY_FILTER
		for (i = 0; i < nb; i++) stack_push(orig_formula, DAG_dup(PDAG[i]));
#endif
	} else {
		TDAG* PDAG2;
		MY_MALLOC(PDAG2, nb * sizeof(TDAG));
		for (i = 0; i < nb; i++) PDAG2[i] = PDAG[i];
		DAG1 = DAG_dup(DAG_new(CONNECTOR_AND, nb, PDAG2));
#if defined(DEBUG) && defined(DEBUG_RECHECK_UNSAT)
		DAG1 = DAG_dup(DAG1);
#endif
		DAG2 = pre_process(DAG1);
		if (option_print_simp_and_exit) {
			if (strcmp(output_format, "smtlib2") == 0)
				DAG_fprint_smt2_bench(stdout, DAG2, "unknown");
			else if (strcmp(output_format, "b") == 0)
				DAG_fprint_b(stdout, DAG2);
		}
		complete_add(DAG2);
#ifdef POLARITY_FILTER
		stack_push(orig_formula, DAG_dup(DAG2));
#endif
	}
	VERIT_PRE_TIMER_STOP;
	if (proof_on) {
		for (i = 0; i < nb; i++) bool_add_proof(PDAG[i], Pproof[i]);
		free(Pproof);
		undo_level_new();
		assert(undo_level == 1);
		for (i = 0; i < nb; i++) {
#ifdef DEBUG
			if (check_deduced) veriT_boolean_abstraction_store_input(PDAG[i]);
#endif
			CC_notify_formula(PDAG[i]);
			if (!LA_active) continue;
			LA_notify_formula(PDAG[i]);
			if (LA_enable_lemmas) LA_unate_propagation();
			if (LA_overflow) {
				undo_level_del_to_level(0);
				undo_level_new();
				LA_switch_to_mp();
			}
		}
		for (i = 0; i < nb; i++) DAG_free(PDAG[i]);
	} else {
		bool_add(DAG2);
		if (option_print_simp_and_exit || option_parse_only) {
			free(PDAG);
			return status = UNDEF;
		}
		/* TODO: really here?  Not after add_watch???
         Anyway this is a dirty hack and should not exist
         There may even be dirty bugs due to hints/propagation */
		undo_level_new();
		assert(undo_level == 1);
		CC_notify_formula(DAG2);
		if (LA_active) {
			LA_notify_formula(DAG2);
			if (LA_enable_lemmas) LA_unate_propagation();
			if (LA_overflow) {
				undo_level_del_to_level(0);
				undo_level_new();
				LA_switch_to_mp();
			}
		}
#if defined(DEBUG) && defined(DEBUG_RECHECK_UNSAT)
		if (check_deduced) {
			if (DAG2 != DAG1) my_warning("Input formula changed by preprocessing.\n");
			veriT_boolean_abstraction_store_input(DAG2);
		}
		DAG_free(DAG1);
#endif /* defined(DEBUG) && defined(DEBUG_RECHECK_UNSAT) */
		DAG_free(DAG2);
	}

	if (LA_enable_lemmas_totality) {
		stack_apply(veriT_lemmas, veriT_add_lemma);
		stack_apply(veriT_lemmas, totality_process);
		stack_reset(veriT_lemmas);
	}
	last_checked_level = current_level;

	/* Initialize bitmasks for function symbols */
	/* set_symbols_masks(); */

	/* IMPROVE: set the SAT solver heuristic and phases */
	status = veriT_solve();

	/* TODO: there should be a pop corresponding to dirty push();
     above.  I let the module disposal do the work, otherwise model
     does not correspond. */
	free(PDAG);
	return status;
}

#ifdef HAVE_POSIX_TIMER
/** \brief Disarms the timeout timer if one is active */
void
veriT_disarm_timer(void) {
	if (option_max_time || option_max_virtual_time) {
		int which = ITIMER_VIRTUAL;
		if (option_max_time) {
			option_max_virtual_time = option_max_time;
			which = ITIMER_REAL;
		}
		struct itimerval its;
		its.it_value.tv_sec = 0;
		its.it_value.tv_usec = 0;
		its.it_interval.tv_sec = 0;
		its.it_interval.tv_usec = 0;
		/* We have a solution, disarm timer */
		setitimer(which, &its, NULL);
	}
}
#endif

#if defined(HAVE_WINDOWS_H) && !defined(HAVE_POSIX_TIMER)
/** \brief Disarms the timeout timer if one is active */
void
veriT_disarm_timer(void) {
	DeleteTimerQueueEx(windows_timer_queue, NULL);
}
#endif

/*
  --------------------------------------------------------------
  --------------------------------------------------------------
  Main loop
  --------------------------------------------------------------
  --------------------------------------------------------------
*/

/*
  --------------------------------------------------------------
  veriT_solve output intermediate results
  --------------------------------------------------------------
*/

#ifdef DEBUG
#define VERIT_SOLVE_OUTPUT_ABSTRACT_MODELS_SAT \
	if (status == SAT && check_deduced) veriT_output_model(); \
	if (dump_abstract_models) \
		veriT_dump_literals(status == SAT ? "sat" : "unknown");
#define VERIT_SOLVE_OUTPUT_ABSTRACT_MODELS_UNSAT \
	if (dump_abstract_models) veriT_dump_literals("unsat");
#ifdef DEBUG_VERIT_SOLVE
#define PRINT_STACK \
	{ \
		unsigned i; \
		for (i = 0; i < SAT_literal_stack_n; i++) \
			fprintf(stderr, "%d ", SAT_literal_stack[i]); \
		fprintf(stderr, "\n"); \
	}
#define PRINT_QUEUE_TOP(A) \
	my_DAG_message( \
		"%s %D %D\n", A, veriT_xqueue_get_DAG(i + 1), veriT_xqueue_get_DAG(i + 2))
#else /* DEBUG_VERIT_SOLVE */
#define PRINT_STACK
#define PRINT_QUEUE_TOP(A)
#endif /* DEBUG_VERIT_SOLVE */
#else /* DEBUG */
#define VERIT_SOLVE_OUTPUT_ABSTRACT_MODELS_SAT
#define VERIT_SOLVE_OUTPUT_ABSTRACT_MODELS_UNSAT
#define PRINT_STACK
#define PRINT_QUEUE_TOP(A)
#endif /* DEBUG */

/*
  --------------------------------------------------------------
  Main loop
  --------------------------------------------------------------
*/

/** \brief Number of complete assignments the Boolean solver needs
    to find on start-up */
#define NB_COMPLETE_PROP_MODELS_ON_START 5 /* PARAM 0..20 */

/* TODO: Ugly workaround to force addition of clauses from insts into
   manager */
extern bool inst_marking;
extern bool inst_deletion_track_vars;

/**
   \brief restarts and add lemmas if any */
static inline void
veriT_restart(void)
{
	undo_level_del_to_level(0);
	undo_level_new();
	veriT_xqueue_clear();
	SAT_restart();
	SAT_status = SAT_STATUS_UNDEF;
	SAT_level_stack_hold = 0;
	SAT_literal_stack_hold = 0;
	assert(undo_level == 1);
	if (LA_active) LA_repair();
	if (LA_overflow) LA_switch_to_mp();
	if (stack_size(veriT_lemmas)) {
#ifdef DEBUG_VERIT_SOLVE
		unsigned i;
		my_DAG_message("veriT_solve: has lemma\n");
		for (i = 0; i < stack_size(veriT_lemmas); ++i)
			my_DAG_message("lemma: %D\n", stack_get(veriT_lemmas, i));
#endif
		stack_apply(veriT_lemmas, veriT_add_lemma);
		/* Done here because needs literals, after CNF */
		if (Q_active && inst_marking && inst_deletion_track_vars)
			inst_mark_instances();
#ifdef DEBUG
		if (check_deduced) veriT_print_lemmas(veriT_lemmas);
#ifdef DEBUG_RECHECK_UNSAT
		veriT_boolean_abstraction_store_lemmas(veriT_lemmas);
#endif
#endif
		stack_reset(veriT_lemmas);
		assert(!LA_active || LA_solve_r() != UNSAT);
	}
	if (inst_marking) inst_marking = false;
}

#define goto_handle_conflict(DP) \
	do { \
		conflict_dp = DP; \
		goto handle_conflict; \
	} while (0)

#define check_LA_overflow() \
	if (LA_overflow) { \
		veriT_restart(); \
		continue; \
	}

static Tstatus
veriT_solve_full(unsigned i)
{
#ifdef STATS_TRACK_STATE
	stats_state_switch(stat_solver_state, state_solve_full);
#endif
	if (!i) return status;
	assert(SAT_literal_stack_hold == 0);
	assert(undo_level == 1 && SAT_level == 0);
	for (; i; i--) {
		if (SAT_solve() == SAT_STATUS_UNSAT) {
			backtrack_unsat();
			status = UNSAT;
			if (proof_on) proof_unsatisfiable();
			return status;
		}
		undo_level_del_to_level(SAT_literal_stack_hold + 1);
		assert(undo_level == SAT_literal_stack_hold + 1);
		if (LA_active) LA_repair();
		for (; SAT_literal_stack_hold < SAT_literal_stack_n;) {
			Tlit literal = SAT_literal_stack[SAT_literal_stack_hold++];
			undo_level_new();
			if (
				lit_to_DAG(literal) == DAG_NULL || lit_to_DAG(literal) == 1 ||
				boolean_connector(DAG_symb(lit_to_DAG(literal))) ||
				!DAG_arity(lit_to_DAG(literal)))
				continue;
			/* DD+PF here begins the new level */
			if (CC_assert(literal) == UNSAT) goto handle_conflict;
		}
		assert(undo_level == SAT_literal_stack_hold + 1);
		if (SAT_literal_stack_hold == SAT_literal_stack_n) {
			if (!LA_active && !Q_active) {
				status = complete_check() ? SAT : UNDEF;
				VERIT_SOLVE_OUTPUT_ABSTRACT_MODELS_SAT;
				if (proof_on && status == SAT) proof_satisfiable();
				return status;
			}
			break;
		}
	handle_conflict:
		VERIT_SOLVE_OUTPUT_ABSTRACT_MODELS_UNSAT;
		if (proof_on) SAT_sanitize_root_level();
		conflict_clause();
		/* WHAT IS THE RATIONALE HERE */
		/* undo_level_del(); */
	}
	if (SAT_solve() == SAT_STATUS_UNSAT) {
		backtrack_unsat();
		status = UNSAT;
		if (proof_on) proof_unsatisfiable();
		return status;
	}
	assert(status != UNSAT);
	veriT_restart();
	return status;
}

#define NB_SOLVE_FULL 4 /* MAGIC PARAM */

Tstatus
veriT_solve(void)
{
	bool model_eq = false;
	unsigned i;
#ifdef STATS_TRACK_STATE
	stats_state_switch(stat_solver_state, state_solve);
#endif
	if (status != OPEN) return status;
	assert(undo_level == 1 && SAT_level == 0);
	SAT_level_stack_hold = 0;
	VERIT_SOLVE_TIMER_START;
	if ((status = veriT_solve_full(NB_SOLVE_FULL)) != OPEN) {
		VERIT_SOLVE_TIMER_STOP;
		return status;
	}
#ifdef STATS_TRACK_STATE
	stats_state_switch(stat_solver_state, state_theory);
#endif
	assert(undo_level == 1 && SAT_level == 0);
	status = SAT;
	bool is_unknown = false;
	while (SAT_propagate() != SAT_STATUS_UNSAT) {
		PRINT_STACK;
		/* PF Theory backtrack */
		if (undo_level > SAT_level_stack_hold + 1) {
			veriT_xqueue_clear();
			undo_level_del_to_level(SAT_level_stack_hold + 1);
			if (LA_active) LA_repair();
		}
		assert(SAT_level_stack_hold == SAT_level);
		assert(undo_level == SAT_level_stack_hold + 1);
		PRINT_STACK;
		/* PF push literals */
		for (; SAT_literal_stack_hold < SAT_literal_stack_n;) {
			Tlit literal = SAT_literal_stack[SAT_literal_stack_hold++];
			if (
				lit_to_DAG(literal) == DAG_NULL || lit_to_DAG(literal) == 1 ||
				boolean_connector(DAG_symb(lit_to_DAG(literal))) ||
				!DAG_arity(lit_to_DAG(literal)))
				continue;
			/* DD+PF here begins the new level */
			if (CC_assert(literal) == UNSAT)
				goto_handle_conflict(cc);
			else if (LA_active && LA_assert(literal) == UNSAT)
				goto_handle_conflict(la);
		}
		PRINT_STACK;
		if (LA_active && (status = LA_solve_r()) == UNSAT) goto_handle_conflict(la);
		/* PF Check for equalities to share, and loop with previous line */
		/* handle_no_conflict : */
		check_LA_overflow();
		if (SAT_literal_stack_to_propagate < SAT_literal_stack_n) continue;
		if (SAT_level == 0) {
			if (!proof_on) {
#ifdef DEBUG
				unsigned i;
				stack_reset(root_lit_stack);
				for (i = 0; i < SAT_literal_stack_n; i++)
					if (bool_required[SAT_literal_stack[i]])
						stack_push(root_lit_stack, SAT_literal_stack[i]);
#endif
				LA_simp();
			}
			check_LA_overflow();
		}
		if (LA_active && (status = LA_solve_z()) == UNSAT)
			goto_handle_conflict(la_z);
		check_LA_overflow();
		if (stack_size(xqueue)) {
			for (i = 0; i < stack_size(xqueue); i += 3)
				switch (veriT_xqueue_get_type(i)) {
					case XTYPE_CC_EQ:
						PRINT_QUEUE_TOP("CC == -> LA");
						if (
							LA_assert_eq(
								veriT_xqueue_get_DAG(i + 1), veriT_xqueue_get_DAG(i + 2)) ==
							UNSAT)
							goto_handle_conflict(la);
						break;
					case XTYPE_CC_INEQ:
						PRINT_QUEUE_TOP("CC != -> LA");
						if (
							LA_assert_neq(
								veriT_xqueue_get_DAG(i + 1), veriT_xqueue_get_DAG(i + 2)) ==
							UNSAT)
							goto_handle_conflict(la);
						break;
					default: my_error("veriT_xqueue: unexpected type\n");
				}
			veriT_xqueue_clear();
			if ((status = LA_solve_r()) == UNSAT) goto_handle_conflict(la);
			if ((status = LA_solve_z()) == UNSAT) goto_handle_conflict(la_z);
			check_LA_overflow();
			if (SAT_literal_stack_to_propagate < SAT_literal_stack_n) continue;
		}
		if (stack_size(veriT_lemmas)) {
			veriT_restart();
			continue;
		}
		if (SAT_decide()) {
			undo_level_new();
			SAT_level_stack_hold++;
			assert(!stack_size(xqueue));
			continue;
		}
		/* Model is complete, now let model equalities being exchanged */
		if (LA_active && LA_model_eq()) {
			model_eq = true;
			undo_level_new();
			for (i = 0; i < stack_size(xqueue); i += 3)
				switch (veriT_xqueue_get_type(i)) {
					case XTYPE_CC_EQ:
						PRINT_QUEUE_TOP("CC == -> LA");
						if (
							LA_assert_eq(
								veriT_xqueue_get_DAG(i + 1), veriT_xqueue_get_DAG(i + 2)) ==
							UNSAT)
							goto_handle_conflict(la);
						break;
					case XTYPE_CC_INEQ:
						PRINT_QUEUE_TOP("CC != -> LA");
						if (
							LA_assert_neq(
								veriT_xqueue_get_DAG(i + 1), veriT_xqueue_get_DAG(i + 2)) ==
							UNSAT)
							goto_handle_conflict(la);
						break;
					case XTYPE_LA_MODEL_EQ: {
						TDAG DAG0 = veriT_xqueue_get_DAG(i + 1);
						TDAG DAG1 = veriT_xqueue_get_DAG(i + 2);
						PRINT_QUEUE_TOP("LA == -> CC");
						if (CC_assert_eq(DAG0, DAG1, LIT_MODEL_EQ) == UNSAT)
							goto_handle_conflict(cc);
					}
				}
			veriT_xqueue_clear();
			if ((status = LA_solve_r()) == UNSAT) goto_handle_conflict(la);
			if ((status = LA_solve_z()) == UNSAT) goto_handle_conflict(la_z);
			check_LA_overflow();
		}
		if (model_eq) {
			/* TODO: before we should store the model somewhere */
			undo_level_del();
			LA_repair();
			model_eq = false;
		}
		if (Q_active) {
#ifdef STATS_TRACK_STATE
			stats_state_switch(stat_solver_state, state_inst);
#endif
			inst(&veriT_lemmas);
		}
		if (stack_size(veriT_lemmas)) {
			veriT_restart();
			continue;
		}
		status = complete_check() ? status : UNDEF;
		VERIT_SOLVE_OUTPUT_ABSTRACT_MODELS_SAT;
		VERIT_SOLVE_TIMER_STOP;
		if (proof_on && status == SAT) proof_satisfiable();
		return status;
	handle_conflict:
		veriT_xqueue_clear();
		check_LA_overflow();
		VERIT_SOLVE_OUTPUT_ABSTRACT_MODELS_UNSAT;
		if (proof_on) SAT_sanitize_root_level();
		conflict_clause();
		check_LA_overflow();
		if (model_eq) {
			undo_level_del();
			model_eq = false;
		}
		status = SAT;
	}
	backtrack_unsat();
	status = is_unknown ? UNDEF : UNSAT;
	VERIT_SOLVE_TIMER_STOP;
	if (proof_on) proof_unsatisfiable();
	stack_apply(veriT_lemmas, DAG_free);
	stack_reset(veriT_lemmas);
	return status;
}

/*
  --------------------------------------------------------------
  Initialisation and release
  --------------------------------------------------------------
*/

void
veriT_init(void)
{
	status = SAT;
	stack_INIT(table_lemma);
	veriT_signal_init();
	list_init();
	dll_init();
	response_init();
	options_init();
	stats_init();
	DAG_init();
	veriT_state_init();
	verit_stack_init();
	literal_init();
	simplify_init();
	simplify_unit_init();
	pre_init();
	undo_init();
	CC_init();
	LA_init();
	options_new_string(
		'l', "list", "stats list (default is all)", "s1+s2", &stats_list);
	stats_list = strmake("");
	inst_init();
	hint_init();
	undo_set_hook(VERIT_UNSAT, (Tundo_hook)backtrack_unsat_hook, 0);
	resolution_chain_size = 0;
	resolution_chain_allocated = 1;
	MY_MALLOC(resolution_chain, sizeof(int));
	proof_init();
	bool_init();
	ite_elim_init();
#if STATS_LEVEL > 0
	stat_total_time =
		stats_timer_new("total_time", "Total time", "%7.2f", STATS_TIMER_ALL);
	stat_pre_time =
		stats_timer_new("pre_time", "Preprocess time", "%7.2f", STATS_TIMER_ALL);
#endif
#if STATS_LEVEL > 1
	stat_nb_clauses =
		stats_counter_new("clauses", "Number of clauses generated", "%5d");
	stat_nb_binary_clauses =
		stats_counter_new("2cl_c", "Number of binary clauses generated", "%5d");
#endif /* STATS_LEVEL */
#if STATS_TRACK_STATE
	state_check_sat = state_new("check_sat");
	state_init = state_new("init");
	state_solve = state_new("solve");
	state_solve_full = state_new("solve_full");
	state_inst = state_new("inst");
	state_theory = state_new("theory");
	stat_solver_state = stats_state_new("solver_state", "State of the solver");

#endif /* STATS_TRACK_STATE */
#ifdef DEBUG
	check_deduced = false;
	options_new(
		0, "check-deduced",
		"Produce SMT files for conflicts, lemmas, and model.  "
		"Useful for debugging",
		&check_deduced);
	dump_abstract_models = false;
	options_new(
		0, "dump-abstract-models",
		"Produce SMT files for every propositional model.  "
		"Useful for debugging",
		&dump_abstract_models);
	dump_CIs = false;
	options_new(
		0, "dump-CIs", "Produce SMT files for prop. model + each conf. instance",
		&dump_CIs);
	stack_INIT(root_lit_stack);
#ifdef DEBUG_RECHECK_UNSAT
	veriT_boolean_abstraction_init();
#endif
#endif /* DEBUG_RECHECK_UNSAT */
#if 0
  options_new(0, "sym-sat",
              "Uses symmetries to find more units in the SAT solver",
              &sym_sat);
#endif
	options_new(
		'v', "verbose", "Print statistics report on exit, ...",
		&veriT_print_report);
	undo_top_level_new();
#ifdef STATS_TRACK_STATE
	stats_state_switch(stat_solver_state, state_init);
#endif
}

void
veriT_set(void)
{
	LA_set();
}

void
veriT_done(void)
{
#if defined(DEBUG) && defined(DEBUG_RECHECK_UNSAT)
	if (check_deduced) veriT_boolean_abstraction_out(status);
	veriT_boolean_abstraction_done();
#endif
#ifdef DEBUG
	stack_free(root_lit_stack);
#endif
	proof_done();
	resolution_chain_size = 0;
	resolution_chain_allocated = 0;
	free(resolution_chain);
	ite_elim_done();
	inst_done();
	bool_done();
	undo_level_del_to_level(0);
	while (undo_top_level) undo_top_level_del();
	hint_done();
	LA_done();
	CC_done();
	undo_done();
	pre_done();
	simplify_unit_done();
	simplify_done();
	literal_done();
	while (!stack_is_empty(table_lemma)) DAG_free(stack_pop(table_lemma));
	stack_free(table_lemma);
	verit_stack_done();
	veriT_state_done();
	DAG_done();
	response_done();
	list_done();
	dll_done();
	veriT_signal_done();
}

void
veriT_exit(int status)
{
#ifdef STATS_TRACK_STATE
	stats_state_switch(stat_solver_state, state_done);
#endif
	/* DD+PF exit library */
	DAG_smtlib_logic_done();
	veriT_done();
	/* DD+PF print some statistics */
	if (veriT_print_report) {
		if (strcmp(stats_list, ""))
			stats_fprint_list(stdout, str_split(stats_list, '+'));
		else {
			stats_fprint_formats(stdout);
			stats_fprint(stdout);
		}
	}
	stats_done();
	options_done();
#ifdef DEBUG
#ifdef MACOS
	pause();
#endif
#endif
	exit(status);
}
