/*
 * Copyright (C) 2014-2024 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#include <QApplication> /* qApp */
#include <QFile>
#include <QString>

#include "src/datovka_shared/log/log.h"
#include "src/datovka_shared/settings/prefs.h"
#include "src/gui/styles.h"
#include "src/initialisation_gui.h"
#include "src/settings/prefs_specific.h"

/*!
 * @brief Get application style sheet text.
 *
 * @param[in] index Value as defined by enum PrefsSpecific::AppUiTheme.
 * @return Style sheet content.
 */
static
QString appUiStylesheet(const int index)
{
	const QString styleSheetFilePath = GuiStyles::styleSheetPath(index);

	if (styleSheetFilePath.isEmpty()) {
		return QString();
	}

	QFile file(styleSheetFilePath);
	if (Q_UNLIKELY(!file.open(QFile::ReadOnly))) {
		logWarningNL(
		    "Couldn't open UI stylesheet file '%s'. Default theme is used.",
		    styleSheetFilePath.toUtf8().constData());
		return QString();
	}

	return GuiStyles::replaceStyleSheetVariables(QLatin1String(file.readAll()));
}

void loadAppUiThemeFromIndex(const int index)
{
	QApplication *app = qApp;
	if (app != Q_NULLPTR) {
		app->setStyleSheet(appUiStylesheet(index));
	}
}

void loadAppUiTheme(const Prefs &prefs)
{
	loadAppUiThemeFromIndex(PrefsSpecific::appUiTheme(prefs));
}
