/*
 * Copyright 2015-2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j;

import static java.nio.charset.Charset.defaultCharset;
import static java.nio.charset.StandardCharsets.UTF_8;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertSame;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import org.junit.Test;

/**
 * Unit tests for {@link FileInfo}.
 *
 * @author Marc Philipp
 * @author Reinhold Degenfellner
 * @since 1.3
 */
public class FileInfoTests {

	FileInfo actual = new FileInfo("actual path", "actual contents".getBytes(UTF_8));
	FileInfo expected = new FileInfo("expected path", "expected contents".getBytes(UTF_8));
	AssertionFailedError assertionFailedError = new AssertionFailedError("message", expected, actual);

	@Test
	public void actualAndExpectedFileInfoAreAccessible() {
		assertSame(actual, assertionFailedError.getActual().getValue());
		assertSame(expected, assertionFailedError.getExpected().getValue());
	}

	@Test
	public void isSerializable() throws Exception {
		var out = new ByteArrayOutputStream();

		new ObjectOutputStream(out).writeObject(assertionFailedError);
		var deserialized = (AssertionFailedError) new ObjectInputStream(
			new ByteArrayInputStream(out.toByteArray())).readObject();

		assertEquals(actual, deserialized.getActual().getValue());
		assertEquals(expected, deserialized.getExpected().getValue());
	}

	@Test
	public void constructors() {
		var contents = "contents".getBytes(UTF_8);
		var fileInfo = new FileInfo("path", contents);

		assertSame(fileInfo.getContents(), contents);
		assertEquals(fileInfo.getContentsAsString(defaultCharset()), "contents");
		assertEquals(fileInfo.getPath(), "path");

		contents = "contents".getBytes();
		fileInfo = new FileInfo("path", contents);

		assertSame(fileInfo.getContents(), contents);
		assertEquals(fileInfo.getContentsAsString(defaultCharset()), "contents");
		assertEquals(fileInfo.getPath(), "path");
	}
}
