/* $Id: stimg.c,v 1.4 2003/05/03 13:41:05 hito Exp $ */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <assert.h>
#include "stimg.h"
#ifdef HAVE_CONFIG_H 
#include "config.h"
#endif

#ifndef MAX
#define MAX(a, b)          (((a) > (b))? (a): (b)) 
#endif
#ifndef MIN
#define MIN(a, b)          (((a) < (b))? (a): (b)) 
#endif

typedef STIMG_ANIMATION *(*STIMG_ANIMATION_FUNC) (char *filename, int check);
typedef STIMG *(*STIMG_FUNC) (char *filename, int check);


/*** Image loader ***/

#ifdef HAVE_LIBJPEG
STIMG_ANIMATION *load_animation_jpeg(char *file, int check);
STIMG *load_jpeg(char *file, int check);
#endif

#ifdef HAVE_LIBPNG
STIMG_ANIMATION *load_animation_png(char *file, int check);
STIMG *load_png(char *file, int check);
#endif

#ifdef HAVE_LIBTIFF
STIMG_ANIMATION *load_animation_tiff(char *file, int check);
STIMG *load_tiff(char *file, int check);
#endif

STIMG_ANIMATION *load_animation_gif(char *file, int check);
STIMG_ANIMATION *load_animation_xpm(char *file, int check);
STIMG_ANIMATION *load_animation_xbm(char *file, int check);
STIMG_ANIMATION *load_animation_pnm(char *file, int check);
STIMG_ANIMATION *load_animation_bmp(char *file, int check);
STIMG *load_gif(char *file, int check);
STIMG *load_xpm(char *file, int check);
STIMG *load_xbm(char *file, int check);
STIMG *load_pnm(char *file, int check);
STIMG *load_bmp(char *file, int check);

static STIMG_ANIMATION_FUNC Image_animation_func[] = {
  load_animation_gif,
#ifdef HAVE_LIBPNG
  load_animation_png,
#endif
#ifdef HAVE_LIBJPEG
  load_animation_jpeg,
#endif
  load_animation_bmp,
#ifdef HAVE_LIBTIFF
  load_animation_tiff,
#endif
  load_animation_xpm,
  load_animation_xbm,
  load_animation_pnm,
  NULL
};

static STIMG_FUNC Image_func[] = {
  load_gif,
#ifdef HAVE_LIBPNG
  load_png,
#endif
#ifdef HAVE_LIBJPEG
  load_jpeg,
#endif
  load_bmp,
#ifdef HAVE_LIBTIFF
  load_tiff,
#endif
  load_xpm,
  load_xbm,
  load_pnm,
  NULL
};
/********************/

#define STIMG_FUNC_NUM (sizeof(Image_func)/sizeof(*Image_func))
#define STIMG_ANIMATION_FUNC_NUM (sizeof(Image_animation_func)/sizeof(*Image_animation_func))

static int animation_add_frame(STIMG_ANIMATION *animation, STIMG *image);
static unsigned char *clip(const STIMG *image, int x1, int y1, int x2, int y2);
static int clip2(STIMG *image);

STIMG *
stimg_load(char *filename)
{
  int i;
  STIMG *image = NULL;

  for (i = 0; Image_func[i] != NULL; i++) {
    image = Image_func[i](filename, 0);
    if (image) break;
  }
  return image;
}

int
stimg_size(char *filename, int *w, int *h) 
{
  int i;
  STIMG *image = NULL;

  for (i = 0; Image_func[i] != NULL; i++) {
    image = Image_func[i](filename, 1);
    if (image) break;
  }

  if (image == NULL)
    return 1;

  *w = image->w;
  *h = image->h;

  stimg_delete(image);

  return 0;
}

STIMG *
stimg_new(int w, int h, int alpha)
{
  STIMG *image;
  int bpp;

  if (w < 1 || h < 1)
    return NULL;

  bpp = 3 + ((alpha)? 1: 0);

  image = malloc(sizeof(STIMG));
  if (image == NULL)
    return NULL;

  image->data = malloc(w * h * bpp);
  if (image->data == NULL) {
    free(image);
    return NULL;
  }

  image->w = w;
  image->h = h;
  image->alpha = (alpha)? 1: 0;
  image->rowstride = w * bpp;
  image->x_offset = 0;
  image->y_offset = 0;
  image->delay_time = 0;
  image->action = 0;
  image->next = NULL;
  
  return image;
}

void 
stimg_delete(STIMG *image)
{
  STIMG *tmp;
  while (image != NULL) {
    if (image->data)
      free(image->data);
    tmp = image->next;
    free(image);
    image = tmp;
  }
}

STIMG *
stimg_clip(const STIMG *image, int x1, int y1, int x2, int y2)
{
  STIMG *new_image;
  unsigned char *src;

  if (image == NULL || image->data == NULL ||
      image->w < 1 || image->h < 1 ||
      x1 < 0 || y1 < 0 || x2 <= x1 || y2 <= y1 ||
      x2 > image->w || y2 > image->h)
    return NULL;

  new_image = stimg_new(x2 - x1, y2 - y1, image->alpha);
  if(new_image == NULL)
    return NULL;

  src = clip(image, x1, y1, x2, y2);
  if (src == NULL) {
    stimg_delete(new_image);
    return NULL;
  }

  memcpy(new_image->data, src, (x2 - x1) * (y2 - y1) * (3 + image->alpha));
  free(src);
  return new_image;
}

STIMG *
stimg_resize(const STIMG *image, int w, int h, int aa)
{
  STIMG *new_image;
  int x, y, bpp, sw, sh, rowstride, alpha;
  unsigned char *data_new, *data, *tmp;
  double ratio_w, ratio_h;

  if (image == NULL || image->data == NULL || image->w < 1 || image->h < 1)
    return NULL;

  if (w < 1 || h < 1)
    return NULL;

  new_image = stimg_new(w, h, image->alpha);
  if(new_image == NULL)
    return NULL;

  data = image->data;
  bpp = 3 + image->alpha;
  ratio_w = (double) image->w / w;
  ratio_h = (double) image->h / h;

  data_new = new_image->data;
  new_image->x_offset = image->x_offset / ratio_w;
  new_image->y_offset = image->y_offset / ratio_h;
  new_image->delay_time = image->delay_time;
  new_image->action = image->action;
  new_image->next = NULL;

  sw = image->w;
  sh = image->h;
  rowstride = image->rowstride;
  alpha = image->alpha;

  if (w == sw && h == sh) {
    memcpy(new_image->data, image->data, new_image->rowstride * h);
  } else if (aa == 0 || (w >= sw && h >= sh)) {
    for (y = 0; y < h; y++) {
      tmp = data + bpp * image->w * (int)(y * ratio_h);
      for (x = 0; x < w; x++) {
	memcpy(data_new, tmp + bpp * (int)(x * ratio_w), bpp);
	data_new += bpp;
      }
    }
  } else {
    int ix, jx, iy, jy, yy, i, j, n, r, g, b, a;
    for (y = 0; y < h; y++) {
      iy = (y - 0.5) * ratio_h;
      iy = MAX(0, iy);
      jy = (y + 0.5) * ratio_h;
      jy = MIN(sh - 1, jy);
      yy = y * ratio_h;
      for (x = 0; x < w; x++) {
	ix = (x - 0.5) * ratio_w;
	ix = MAX(0, ix);
	jx = (x + 0.5) * ratio_w;
	jx = MIN(sw - 1, jx);
	r = g = b = a = n = 0;
	tmp = data + yy * rowstride + (int)(x * ratio_w) * bpp;
	a = tmp[3];
	for (i = iy; i <= jy; i++) {
	  tmp = data + i * rowstride + ix * bpp;
	  for (j = 0; j <= jx - ix; j++) {
	    r += *tmp++;
	    g += *tmp++;
	    b += *tmp++;
	    if(alpha)
	      tmp++;
	    n++;
	  }
	}

	n = MAX(1, n);
	*data_new++ = r / n;
	*data_new++ = g / n;
	*data_new++ = b / n;
	if(alpha)
	  *data_new++ = a;
      }
    }
  }
  return new_image;
}

int
stimg_rotate(STIMG *image, int angle)
{
  unsigned char *src, *dest, *tmp;
  int x, y, i, bpp, ofst, ofst2, len;

  if (image == NULL ||
      image->data == NULL ||
      image->w < 1 || image->h < 1)
    return 1;

  len = image->rowstride * image->h;
  tmp = malloc(len);
  if (tmp == NULL)
    return 1;

  src = image->data;
  dest = tmp;

  bpp = 3 + ((image->alpha)? 1: 0);
  angle %= 4;

  switch (angle) {
  case 1:
    ofst2 = image->rowstride * (image->h - 1);
    for (x = 0; x < image->rowstride; x += bpp) {
      ofst = ofst2 + x;
      for (y = image->h - 1; y >= 0; y--) {
	memcpy(dest, src + ofst, bpp);
	dest += bpp;
	ofst -= image->rowstride;
      }
    }
    memcpy(src, tmp, len);
    i = image->w;
    image->w = image->h;
    image->h = i;
    image->rowstride = image->w * bpp;
    break;
  case 2:
    src = image->data + image->rowstride * image->h - bpp;
    for (y = 0; y < image->h; y++) {
      for (x = 0; x < image->w; x++) {
	memcpy(dest, src, bpp);
	dest += bpp;
	src  -= bpp;
      }
    }
    memcpy(src, tmp, len);
    break;
  case 3:
    for (x = image->rowstride - bpp; x >= 0; x -= bpp) {
      ofst = x;
      for (y = 0; y < image->h; y++) {
	memcpy(dest, src + ofst, bpp);
	dest += bpp;
	ofst += image->rowstride;
      }
    }
    memcpy(src, tmp, len);
    i = image->w;
    image->w = image->h;
    image->h = i;
    image->rowstride = image->w * bpp;
    break;
  }
  free(tmp);
  return 0;
}

int
stimg_flip(STIMG *image)
{
  unsigned char *src, *dest, *tmp;
  int i, len;

  if (image == NULL ||
      image->data == NULL ||
      image->w < 1 || image->h < 1)
    return 1;

  len = image->rowstride * image->h;
  tmp = malloc(len);
  if (tmp == NULL)
    return 1;

  src = image->data + len;
  dest = tmp;

  for (i = 0; i < image->h; i++) {
    src  -= image->rowstride;
    memcpy(dest, src, image->rowstride);
    dest += image->rowstride;
  }
  memcpy(src, tmp, len);
  free(tmp);
  return 0;
}

int
stimg_flop(STIMG *image)
{
  unsigned char *src, *dest, *tmp;
  int x, y, bpp;

  if (image == NULL ||
      image->data == NULL ||
      image->w < 1 || image->h < 1)
	return 1;

  tmp = malloc(image->rowstride);
  if (tmp == NULL)
    return 1;

  bpp = 3 + ((image->alpha)? 1: 0);
  src = image->data;
  for (y = 0; y < image->h; y++) {
    dest = tmp;
    src += image->rowstride;
    for (x = 0; x < image->w; x++) {
      src  -= bpp;
      memcpy(dest, src, bpp);
      dest += bpp;
    }
    memcpy(src, tmp, image->rowstride);
    src += image->rowstride;
  }
  free(tmp);
  return 0;
}


STIMG_ANIMATION *
stimg_animation_load(char *filename)
{
  int i;
  STIMG_ANIMATION *animation = NULL;

  for (i = 0; Image_animation_func[i] != NULL; i++) {
    animation = Image_animation_func[i](filename, 0);
    if (animation) break;
  }
  return animation;
}

int
stimg_animation_size(char *filename, int *w, int *h) 
{
  int i;
  STIMG_ANIMATION *animation = NULL;

  for (i = 0; Image_animation_func[i] != NULL; i++) {
    animation = Image_animation_func[i](filename, 1);
    if (animation) break;
  }

  if (animation == NULL)
    return 1;

  *w = animation->w;
  *h = animation->h;

  stimg_animation_delete(animation);

  return 0;
}

STIMG_ANIMATION *
stimg_animation_new(void)
{
  STIMG_ANIMATION *animation;

  animation = malloc(sizeof(STIMG_ANIMATION));
  if (animation) {
    animation->n = 0;
    animation->image = NULL;
    animation->cur_image = NULL;
    animation->w = 0;
    animation->h = 0;
    animation->iterations = 1;
  }
  return animation;
}

void
stimg_animation_delete(STIMG_ANIMATION *animation)
{
  if (animation == NULL)
    return;

  if (animation->image)
    stimg_delete(animation->image);

  free(animation);
}

STIMG_ANIMATION *
stimg_animation_resize(const STIMG_ANIMATION *animation, int w, int h, int aa)
{
  STIMG_ANIMATION *tmp;
  STIMG *image, *resized_image;
  double ratio_w, ratio_h;

  if (animation == NULL || animation->image == NULL || animation->w < 1 || animation->h < 1)
    return NULL;

  if (w < 1 || h < 1)
    return NULL;

  tmp = stimg_animation_new();
  if (tmp == NULL)
    return NULL;

  tmp->iterations = animation->iterations;

  ratio_w = (double) w / animation->w;
  ratio_h = (double) h / animation->h;

  for (image = animation->image; image != NULL; image = image->next) {
    int new_w, new_h;
    new_w = image->w * ratio_w + 0.5;
    new_h = image->h * ratio_h + 0.5;
    resized_image = stimg_resize(image, (new_w)? new_w: 1, (new_h)? new_h: 1, aa);
    if (resized_image == NULL) {
      stimg_animation_delete(tmp);
      return NULL;
    }
    resized_image->x_offset = image->x_offset * ratio_w;
    resized_image->y_offset = image->y_offset * ratio_h;

    if (resized_image->x_offset + resized_image->w > w ||
	image->x_offset + image->w == animation->w)
      resized_image->x_offset = w - resized_image->w;

    if (resized_image->y_offset + resized_image->h > h || 
	image->y_offset + image->h == animation->h)
      resized_image->y_offset = h - resized_image->h;

    animation_add_frame(tmp, resized_image);
  }
  assert(tmp->w == w && tmp->h == h);

  return tmp;
}

int
stimg_animation_add_frame(STIMG_ANIMATION *animation, STIMG *image,
			  int x_ofst, int y_ofst, int delay, int action)
{
  if (animation == NULL || image == NULL)
    return 1;

  image->x_offset = (x_ofst > 0)? x_ofst: 0;
  image->y_offset = (y_ofst > 0)? y_ofst: 0;
  image->delay_time = (delay > 0)? delay: 0;
  image->action =  (action > 0)? action: 0;
  return animation_add_frame(animation, image);
}


int stimg_get_has_alpha(const STIMG *image)
{
  if (image)
    return image->alpha;
  return 0;
}

unsigned char *stimg_get_data(const STIMG *image)
{
  if (image)
    return image->data;
  return NULL;
}

int stimg_get_width(const STIMG *image)
{
  if (image)
    return image->w;
  return 0;
}

int stimg_get_height(const STIMG *image)
{
  if (image)
    return image->h;
  return 0;
}

int stimg_get_rowstride(const STIMG *image)
{
  if (image)
    return image->rowstride;
  return 0;
}

int stimg_animation_get_width(const STIMG_ANIMATION *animation)
{
  if (animation)
    return animation->w;
  return 0;
}

int stimg_animation_get_height(const STIMG_ANIMATION *animation)
{
  if (animation)
    return animation->h;
  return 0;
}

int stimg_animation_get_num(const STIMG_ANIMATION *animation)
{
  if (animation)
    return animation->n;
  return 0;
}

int stimg_animation_get_iterations(const STIMG_ANIMATION *animation)
{
  if (animation)
    return animation->iterations;
  return 0;
}

STIMG *stimg_animation_get_first_image(STIMG_ANIMATION *animation)
{
  if (animation)
    return animation->image;
  return 0;
}

STIMG *stimg_animation_get_next_image(STIMG_ANIMATION *animation)
{
  if (animation == NULL)
    return NULL;

  if (animation->cur_image == NULL) {
    animation->cur_image = animation->image;
  } else {
    animation->cur_image = animation->cur_image->next;
  }
      
  return animation->cur_image;
}

STIMG *stimg_animation_get_nth_image(STIMG_ANIMATION *animation, int n)
{
  int i;
  STIMG *image;

  if (animation == NULL || n > animation->n || n < 1 ||animation->image == NULL)
    return NULL;

  image = animation->image;
  for (i = 1; i < n; i++) {
    image = image->next;
    if (image == NULL)
      break;
  }
  animation->cur_image = image;
  return image;
}

int stimg_animation_get_action(const STIMG_ANIMATION *animation)
{
  if (animation && animation->cur_image)
    return animation->cur_image->action;
  return 0;
}

int stimg_animation_get_x_offset(const STIMG_ANIMATION *animation)
{
  if (animation && animation->cur_image)
    return animation->cur_image->x_offset;
  return 0;
}

int stimg_animation_get_y_offset(const STIMG_ANIMATION *animation)
{
  if (animation && animation->cur_image)
    return animation->cur_image->y_offset;
  return 0;
}

int stimg_animation_get_delay(const STIMG_ANIMATION *animation)
{
  if (animation && animation->cur_image)
    return animation->cur_image->delay_time;
  return 0;
}

/*******************************************************************
static functions
********************************************************************/

static unsigned char *
clip(const STIMG *image, int x1, int y1, int x2, int y2)
{
  int y, bpp, rowstride;
  unsigned char *dist, *src;

  if (image == NULL || image->data == NULL ||
      image->w < 1 || image->h < 1 ||
      x1 < 0 || y1 < 0 || x2 <= x1 || y2 <= y1 ||
      x2 > image->w || y2 > image->h)
    return NULL;

  src = image->data;
  bpp = 3 + image->alpha;
  rowstride = (x2 - x1) * bpp;

  dist = malloc((x2 - x1) * (y2 - y1) * bpp);

  src += x1 * bpp;
  for (y = y1; y < y2; y++) {
    memcpy(dist, src, rowstride);
    dist += rowstride;
    src  += image->rowstride;
  }
  return dist;
}

static int
animation_add_frame(STIMG_ANIMATION *animation, STIMG *image)
{
  STIMG *tmp;
  int w, h, n;

  if (animation == NULL || image == NULL)
    return 1;

  w = animation->w;
  h = animation->h;
  n = 0;

  if (animation->image == NULL) {
    animation->image = animation->cur_image = image;
    w = image->w + image->x_offset;
    h = image->h + image->y_offset;
  } else {
    for (tmp = animation->image; tmp->next != NULL; tmp = tmp->next) {
      n++;
    } 
    n++;

    if (w < image->w + image->x_offset)
      w = image->w + image->x_offset;
    if (h < image->h + image->y_offset)
      h = image->h + image->y_offset;
    tmp->next = image;
  }

  animation->w = w;
  animation->h = h;
  animation->n = n + 1;
  return 0;
}
