#!/usr/bin/perl -w
#
# Name ............: simplecrypt.pl
# Type ............: Perl Script
# Notice ..........: This program is part of the simplebackup package
#
# Version .........: 1.8.1
# Last Update .....: 12 January 2007
#
# License .........: GNU GENERAL PUBLIC LICENSE Version 2, June 1991.
#
# Authors ..........: Miguel Angelo Martins Morais Leite,
# Program E-Mail ...: migas.miguel@gmail.com
# Site .............: http://migas-sbackup.sourceforge.net
# Forum ............: http://sourceforge.net/projects/migas-sbackup
#
# Objective .......: Encrypt or Decrypt any file.
#
# Depends on ......: Perl (v5.6 or over)
#
# Runs On .........: Windows NT4, Windows 2000, Windows XP, 2003 and All Unix systems
#
#
# Revision History :
#                    1.7.0 rc1 - 27 December 2005 - Miguel Angelo Martins Leite
#                      1. First version, the version number matches the current simplebackup version.
#
#                    1.7.0 rc2 - 19 January 2006 - Miguel Angelo Martins Leite
#                      1. Bug fix and major security enhancement, in some rare combination of passwords and encryption
#                         cycles the encryption algorithm would not encrypt. This bug fix enhances the way the
#                         simplecrypt reads the encryption sed and adds a alternative encryption sed, there for
#                         encryption will now be done using two encryption keys. Warning this makes files encrypted with
#                         simplecrypt/simplebackup 1.7.0 rc1 incompatible with this version.
#                         Huge thanks to my friend Alexandre Pereira for all is testing and for reporting this.
#                      2. Bug fix, the algorithm was wrongly adding one encrypting cycle into what the user had required,
#                         has a result of this bug fix the encryption/decryption is a little bit faster.
#                         Again huge thanks to my friend Alexandre Pereira for all is testing and for reporting this.
#
#                    1.7.0 - 02 March 2006 - Miguel Angelo Martins Morais Leite
#                      1. First stable version of simplecrypt, this is a copy of 1.7.0 rc2.
#                      2. I got married ;), my name changed from Miguel Angelo Martins Leite into Miguel Angelo Martins Morais
#                         Leite.
#
#                    1.7.1 - 14 March 2006 - Miguel Angelo Martins Morais Leite
#                      1. Increase the version number to match the simplebackup version number, no changes where done in the
#                         simplecrypt program or the simplecrypt algorithm. Simplecrypt 1.7.0 and 1.7.1 are the same.
#
#                    1.8.0 rc1 - 28 November 2006 - Miguel Angelo Martins Morais Leite
#                      1. Added code to match up the changes done under the simplebackup program (adding a few debug code lines)
#                      2. The encryption algorithm remains fully compatible with the previous version.
#                      3. Updated the program site and program forum url's.
#
#                    1.8.0 rc2 - 07 December 2006 - Miguel Angelo Martins Morais Leite
#                      1. Increase the version number to match the simplebackup version number, no changes where done in the
#                         simplecrypt program or the simplecrypt algorithm.
#
#                    1.8.0 - 19 December 2006 - Miguel Angelo Martins Morais Leite
#                      1. Increase the version number to match the simplebackup version number, no changes where done in the
#                         simplecrypt program or the simplecrypt algorithm.
#
#                    1.8.1 - 12 January 2007 - Miguel Angelo Martins Morais Leite
#                      1. Increase the version number to match the simplebackup version number, no changes where done in the
#                         simplecrypt program or the simplecrypt algorithm.
#
##########################
# Modules
##########################
use strict; # implement a more strict language
use Fcntl qw(:DEFAULT :flock);  # file locking for the logs and other written files


##########################
# Global Variables
##########################
my $script_version = "1.8.1";
my $script_last_update = "12 January 2007";
my $crypting_algorithm_version = "SimpleCrypt 1.0.0";
my $script_license = "GNU GENERAL PUBLIC LICENSE Version 2, June 1991.";
my $script_authors = "Miguel Angelo Martins Morais Leite";
my $script_email = "migas.miguel\@gmail.com";
my $script_site = "http://migas-sbackup.sourceforge.net";
my $script_forum = "http://sourceforge.net/projects/migas-sbackup";
my $cycle = 2; # default number of times that the encryption/decryption is done on each byte/bit
my $encryption_level = "byte"; # default level where the encryption is applied, available levels are
                               # byte or bit. Encryption at the bit level is safer but is very
                               # (VERY !!) slow !.
                               # WARNING !!! changing this has direct
                               # implications on the encryption algorithm, meaning files
                               # encrypted at bit level cannot be decrypted at byte level and
                               # vice verse.
my $file_read_size = 16384; # number of bytes read from the file at each time,
                            # WARNING !!! changing this has direct
                            # implications on the encryption algorithm, meaning files
                            # encrypted with a given file_read_size cannot cannot be
                            # decrypted with a different file_read_size
                            # and on the write buffer size (file_read_size*32 / 16384*32 = 524288)
                            # Changing this does not increase or decrease the safety of the encryption
                            # in any way.
my %global_config = (    # script configuration hash
    "operation"    => "",    # Encrypt / Decrypt
    "pause"        => "",    # if pausing afther the output messages
    "erase_source" => "",    # Erase the file to encrypt or decrypt (yes/no)
    "passwd"       => "",    # User passwrd
    "in_file"      => "",    # File to encrypt or decrypt
    "out_file"     => "",    # Destiny file, encrypted or decrypted
    "crypt"        => "",    # Number of times that each byte/bit is crypted/decrypted
    "level"        => "",    # Level of encryption, bit or byte level
);


##########################
# Functions
##########################


#
# Show a small command line help
# Will return: 0
# Explain: 0 is sucess
sub show_commands {

    my $my_script_version = $_[0]; # script version parameter
    my $my_script_last_update = $_[1]; # script lastupdate version
    my $my_script_license = $_[2]; # script license
    my $my_script_authors = $_[3]; # script authors
    my $my_script_email = $_[4]; # script email
    my $my_script_site = $_[5]; # script site
    my $my_script_forum = $_[6]; # script forum
    my $my_cycle = $_[7]; # default encryption cycle number
    my $my_encryption_level = $_[8]; # default encryption level

    print <<EOF;

 Name ...: simplecrypt.pl v$my_script_version ($my_script_last_update)
 License : $script_license
 Author .: $my_script_authors
 E-Mail .: $my_script_email
 Forum ..: $my_script_forum
 Site ...: $my_script_site

 Encrypt or decrypt any file, using the SimpleCrypt algorithm.

 Encryption :
   --encrypt file_to_encrypt
   --encryptp file_to_encrypt

 Decryption :
   --decrypt file_to_encrypt.sc *1
   --decryptp file_to_encrypt.sc *1

 *1) Optional arguments :
   --to some_file         (Were to encrypt/decrypt),
   --passwd some_pass     (Encryption/decryption password),
   --crypt some_number    (Number of encrypting/decrypting cycles),
   --level byte/bit       (Encryption/decryption level),
   --erase_source yes/no  (Erase the file that was encrypted/decrypted).

 Examples :
   simplecrypt.pl --encrypt "myfile"
   simplecrypt.pl --decrypt "myfile.sc"
   simplecrypt.pl --erase_source yes --encrypt "myfile"
   simplecrypt.pl --erase_source no --decrypt "myfile.sc"
   simplecrypt.pl --passwd "my pass" --encrypt "myfile"
   simplecrypt.pl --passwd "my pass" --decrypt "myfile.sc"
   simplecrypt.pl --passwd "my pass" --crypt 10 --level bit --encryptp "myfile"
   simplecrypt.pl --passwd "my pass" --crypt 10 --level bit --decryptp "myfile.sc"
   simplecrypt.pl --passwd "my pass" --encrypt "myfile" --to "myfile.sc"
   simplecrypt.pl --passwd "my pass" --decrypt "myfile.sc" --to "myfile.sc"
   simplecrypt.pl --passwd "my pass" --crypt 10 --encrypt "myfile" --to "myfile.sc"
   simplecrypt.pl --passwd "my pass" --crypt 10 --decrypt "myfile.sc" --to "myfile.sc"
   simplecrypt.pl --passwd "my pass" --crypt 10 --level bit --encryptp "myfile" --to "myfile.sc"
   simplecrypt.pl --erase_source yes --passwd "my pass" --crypt 10 --level bit --decryptp "myfile.sc" --to "myfile.sc"
   simplecrypt.pl --passwd "my pass" --level bit --encrypt "myfile" --to "myfile.sc"
   simplecrypt.pl --passwd "my pass" --level bit --decrypt "myfile.sc" --to "myfile.sc"

 Notes :
    1 > The higher the encrypting/decrypting cycle number the slower.
    2 > The default encryption/decryption crypt times number is [ $my_cycle ].
    3 > The default encryption/decryption level is [ $my_encryption_level ].
    4 > Warning !! the bit level encryption safer but it's very slow !.
    5 > The files must be decrypted using the same number of the encrypting
        cycles and the same encryption level (byte/bit).
    6 > To increase security you are advice to use a "difficult password",
        using a combination of letters, numbers and symbols.
    7 > By default SimpleCrypt will NOT erase the source file
        (--erase_source = no).
    8 > SimpleCrypt will automatically add and remove the ".sc" if the --to
        argument is not defined.
    9 > If --passwd argument is not supplied SimpleCrypt will enter in
        interactive mode, asking the user (with defaults) all options.
   10 > All arguments can also be used when working in interactive mode,
        the supplied values will be consider the new defaults.
   11 > If you replace --encrypt or --decrypt with --encryptp or --decryptp,
        simpleCrypt will wait for user before ending (Press any key to end).
   12 > Notes 8 to 11 can be very useful when using SimpleCrypt in graphical
        environments such as Windows, Kde, Gnome, CDE, etc.

 The encryption/decryption algorithm was developed by me, Miguel Angelo
 Martins Morais Leite <migas.miguel\@gmail.com>, for more technical details
 check the simplebackup documentation [ readme.html/readme.txt ] file or edit
 the [ simplecrypt.pl ] file and check the source code.

EOF

    return 0; # always return sucess
}


#
# test the user cmd line parameters/commands
# Will return: 0, 1
# Explain: 0 is sucess, 1 is failure
sub test_loads_commands {

    my $parameters_p = $_[0]; # Pointer into the user arguments
    my $my_cycle = $_[1]; # default number of times that the encryption/decryption is done on each byte/bit
    my $my_encryption_level = $_[2]; # default level where the encryption is applied, available levels are
    my $global_config_p = $_[3]; # pointer into the configuration

    my $error_flag = 0; # to mark errors

    # scalar counts a array
    if ( scalar(@$parameters_p) != 2 and scalar(@$parameters_p) != 4 and scalar(@$parameters_p) != 6 and scalar(@$parameters_p) != 8 and scalar(@$parameters_p) != 10 ) {
        return 1; # return error
    }
    else {
        $$global_config_p{pause} = "no";
        # Test the user arguments
        my $pos_counter = 0;
        foreach my $user_arg ( @$parameters_p ) {

            if ( $user_arg eq "--encrypt" or $user_arg eq "--encryptp" or $user_arg eq "--decrypt" or $user_arg eq "--decryptp" ) {
                $error_flag = 1 if ( $$global_config_p{operation} ne "" ); # mark error if user pass twice the same argument
                $$global_config_p{operation} = "encrypt" if ( $user_arg eq "--encrypt" or $user_arg eq "--encryptp" );
                $$global_config_p{operation} = "decrypt" if ( $user_arg eq "--decrypt" or $user_arg eq "--decryptp" );
                $$global_config_p{in_file} = $$parameters_p[$pos_counter+1];
                if ( $user_arg eq "--encryptp" or $user_arg eq "--decryptp" ) {
                    $$global_config_p{pause} = "yes";
                }
            }
            
            if ( $user_arg eq "--erase_source" ) {
                $error_flag = 1 if ( $$global_config_p{erase_source} ne "" ); # mark error if user pass twice the same argument
                $$global_config_p{erase_source} = $$parameters_p[$pos_counter+1];
            }

            if ( $user_arg eq "--passwd" ) {
                $error_flag = 1 if ( $$global_config_p{passwd} ne "" ); # mark error if user pass twice the same argument
                $$global_config_p{passwd} = $$parameters_p[$pos_counter+1];
            }

            if ( $user_arg eq "--to" ) {
                $error_flag = 1 if ( $$global_config_p{out_file} ne "" ); # mark error if user pass twice the same argument
                $$global_config_p{out_file} = $$parameters_p[$pos_counter+1];
            }

            if ( $user_arg eq "--crypt" ) {
                $error_flag = 1 if ( $$global_config_p{crypt} ne "" ); # mark error if user pass twice the same argument
                $$global_config_p{crypt} = $$parameters_p[$pos_counter+1];
            }

            if ( $user_arg eq "--level" ) {
                $error_flag = 1 if ( $$global_config_p{level} ne "" ); # mark error if user pass twice the same argument
                $$global_config_p{level} = $$parameters_p[$pos_counter+1];
            }
            $pos_counter++;
        }

        if ( $$global_config_p{operation} eq "" ) {
            $error_flag = 1; # mark error operation not declared
        }
        if ( $$global_config_p{out_file} eq "" ) { # defined the output file if the user did not define one
            if ( $$global_config_p{operation} eq "encrypt" ) { # if encrypting add the -sc extension
                $$global_config_p{out_file} = $$global_config_p{in_file} . ".sc";
            }
            else { # if decrypting attempt to remove the .sc extension or add a .sc_decrypted extension
                $$global_config_p{out_file} = $$global_config_p{in_file};
                if ( ($$global_config_p{out_file} =~ s/.sc$//) != 1 ) {  # Delete trailing .sc extension
                    $$global_config_p{out_file} = $$global_config_p{out_file} . ".sc_decrypted"; # failed erasing the .sc extetion, so add the .sc_decrypted
                }
            }
        }
        $$global_config_p{crypt} = $my_cycle if ( $$global_config_p{crypt} eq "" );
        $$global_config_p{level} = $my_encryption_level if ( $$global_config_p{level} eq "" );
        $$global_config_p{erase_source} = "no" if ( $$global_config_p{erase_source} eq "" );
        if ( $error_flag == 0 ) {
            $error_flag = 1; # by default mark error
            if ( $$global_config_p{crypt} ne "" and ( ($$global_config_p{crypt} =~ /^\d+$/) and $$global_config_p{crypt} > 0 ) ) {
                $error_flag = 0; # unmark error;

            }
            if ( $error_flag == 0 and ($$global_config_p{level} eq "byte" or $$global_config_p{level} eq "bit" ) ) {
                $error_flag = 0; # unmark error;
            }
            if ( $error_flag == 0 and ($$global_config_p{erase_source} eq "yes" or $$global_config_p{erase_source} eq "no" ) ) {
                $error_flag = 0; # unmark error;
            }
        }
    }

    return $error_flag;
}


#
# Receive any string and return the corresponding CRC32 number
# The crc32 algorithm used is based in the Swissknife library (crc32.pm module)
# from Wolfgang Fleischmann wfl\@ebi.ac.uk
# Will return: a integer (crc32 number)
# Explain: The integer is a crc32 number
sub build_crc32 {

    my $the_string = $_[0]; # the string that we are going to convert into a CRC32 number
    my $global_config_p = $_[1]; # pointer into the configuration
    print "Running the build_crc32() function\n" if ( defined $$global_config_p{debug_level} and ($$global_config_p{debug_level} eq "half" or $$global_config_p{debug_level} eq "full") ); # print message if the debug level requires it

    $the_string=~tr/a-z/A-Z/;
    my @the_string = split //, $the_string;
    my @crc32_table=();

    # Build the Crc32 table
    my ($i,$j,$crc_table);
    my $CRC32_POLYNOMIAL = 0xEDB88320;
    for ($i=0; $i <= 255; $i++){
        $crc_table = $i;
        for ($j=8; $j>0; $j--){
            if ($crc_table & 1){
                $crc_table = (($crc_table >> 1) & 0x7FFFFFFF) ^ $CRC32_POLYNOMIAL;
            } 
            else {
                $crc_table = ($crc_table >> 1) & 0x7FFFFFFF;
            }
        }
        $crc32_table[$i]=$crc_table;
    }

    # Build the Crc32 number
    my $crc = 0xFFFFFFFF;
    my $temp1;
    my $temp2;
    while (my $c=shift @the_string){
	my $cc=ord($c);
	$temp1 = ($crc >> 8) & 0x00FFFFFF;
	$temp2 = $crc32_table[($crc ^ $cc) & 0xFF ];
	$crc = $temp1 ^ $temp2;
    }

    print "Function build_crc32() exit value is [ $crc ]\n" if ( defined $$global_config_p{debug_level} and $$global_config_p{debug_level} eq "full" ); # print message if the debug level requires it
    return $crc; # return the crc32 number
}


#
# Get the user encryption sed (password) and add more spice to it with more things and also encrypted using the same algorightm
# Will return: undef ; sequence of bytes
# Explain: undef is failure ; any other is the encrypted sequence of bytes
sub improve_encryption_sed {

    my $encrypting_sed = $_[0];
    my $new_sed = $_[1]; # this must be always a integer number
    my $encryption_times = $_[2]; # number of times that a encryption loop is done on each byte/bit
    my $encryption_level = $_[3]; # at what level is the encryption (byte / bit)
    my $crypting_file_algorithm_version = $_[4]; # The version of the crypting algoritm the encrypted file will use,
                                                 # notice this might change during the decrypt operation (if the file
                                                 # was encrypted with a older version of the algorith)
    my $global_config_p = $_[5]; # pointer into the configuration
    my $new_sed_position;

    my $new_sed_alternative = build_crc32("$new_sed", $global_config_p); # build the crc32 number for this backup directory

    if ( $crypting_file_algorithm_version eq "SimpleCrypt 1.0.0" ) { # if the file is using version 1.0.0 of the encryption algorithm
        $encrypting_sed = $encrypting_sed . ($new_sed * length($new_sed)); # complicate a little more by  user encrption sed by adding the new sed * it's lenght
        $encrypting_sed .= $new_sed; # complicate the user encrption sed by adding the new sed to the end of this
        if ( $encryption_level eq "byte" ) { # if using bytes
            $new_sed_position = (length($new_sed)-1);
        }
        else { # if using bits
            $new_sed = return_bits($new_sed, $global_config_p); # Convert the spiced user encryption sed string into bits
            $new_sed_alternative = return_bits($new_sed_alternative, $global_config_p); # Convert the spiced user encryption sed string into bits
            $new_sed_position = (length($new_sed)-8);
        }
        $encryption_times = $encryption_times * 10; # the encryption key is encrypted a lot ( configuration crypting times * 10 )
        # encript the encryption sed
        $encrypting_sed = crypt_decrypt_bytes($encrypting_sed, $new_sed, $new_sed_alternative, \$new_sed_position, $encryption_times, $encryption_level, "encrypt", $crypting_file_algorithm_version, $global_config_p); # add the read buffer into the write buffer
    }

    print "Function improve_encryption_sed() exit value is [ $encrypting_sed ]\n" if ( defined $$global_config_p{debug_level} and $$global_config_p{debug_level} eq "full" ); # print message if the debug level requires it
    return $encrypting_sed;
}


#
# Receive a sequence of bytes and return the corresponding sequence of bits
# Will return: a sequence of bits
# Explain: the sequence of bits represents the bytes entered
sub return_bits {

    my $in_bytes = $_[0]; # bytes or bits to encrypt/decrypt
    my $global_config_p = $_[1]; # pointer into the configuration
    print "Running the return_bits() function\n" if ( defined $$global_config_p{debug_level} and ($$global_config_p{debug_level} eq "half" or $$global_config_p{debug_level} eq "full") ); # print message if the debug level requires it
    my $out_bits = "";

    # Sequence of bytes into a sequence of bits
    for ( my $i=0 ; $i<length($in_bytes) ; $i++ ) { # read each char at a time
        $out_bits .= unpack("B8", substr($in_bytes,$i,1)); # B8 because, 8 bits is a byte
    }

    print "Function return_bits() exit value is [ $out_bits ]\n" if ( defined $$global_config_p{debug_level} and $$global_config_p{debug_level} eq "full" ); # print message if the debug level requires it
    return $out_bits;
}


#
# Encrypt/Decrypt sequence of bytes by using a encryption sed (password)
# Notes: The encryption is done by using a xor, bit by bit, to complicate things even further the
# encryption sed is read from end to start and from start to end and it will start being read from
# a position passed by a argument (usually the start value is equal to the last value...), the
# sequence of bytes to encrypt can be encrypted several times (encrypt_decrypt_times argumment).
# The bit by bit encrition reads each byte from end to start.
# Will return: undef ; sequence of bytes
# Explain: undef is failure ; any other is the encrypted/decrypted sequence of bytes
sub crypt_decrypt_bytes {

    my $in_bytes_bits_tmp = $_[0]; # bytes or bits to encrypt/decrypt
    my $encrypting_sed_1 = $_[1]; # the "password" to encrypt/decrypt this sequence of bytes/bits
    my $encrypting_sed_2 = $_[2]; # the alternative "password" to encrypt/decrypt this sequence of bytes/bits
    my $encrypting_sed_position_pointer = $_[3]; # the position where the encryption sed will start to be read
    my $encrypt_decrypt_times = $_[4]; # number of times that the byte string will be encrypted/decrypted
    my $encryption_level = $_[5];  # at what level is the encryption (byte / bit)
    my $operation = $_[6]; # type of operation ( encrypt / decrypt ), this is only used if the level is bit
    my $crypting_file_algorithm_version = $_[7]; # The version of the crypting algoritm the encrypted file will use,
                                                 # notice this might change during the decrypt operation (if the file
                                                 # was encrypted with a older version of the algorith)
    my $global_config_p = $_[8]; # pointer into the configuration
    print "Running the crypt_decrypt_bytes() function\n" if ( defined $$global_config_p{debug_level} and ($$global_config_p{debug_level} eq "half" or $$global_config_p{debug_level} eq "full") ); # print message if the debug level requires it

    my $out_bytes; # will contain the bytes encrypted/decrypted

    if ( $crypting_file_algorithm_version eq "SimpleCrypt 1.0.0" ) {
        my $encrypting_sed_position = $$encrypting_sed_position_pointer; # get the sed position from the previous cycle
        my $encryption_sed_max; # size of the user encryption key minus one byte/bit
        my $sed_movement = "minus"; # when starting it will go backwards
        my $in_bits_tmp; # variable declared here to try to increase bit encryption speed a bit
        my $encrypting_sed_bits; # variable declared here to try to increase bit encryption speed a bit
        my $in_byte_tmp; # variable declared here to try to increase byte encryption speed a bit
        my $encrypting_sed_byte; # variable declared here to try to increase byte encryption speed a bit
        my $encrypting_sed = $encrypting_sed_1; # encrytpion sed when starting comes from the outside
        my $byte_bit_counter = 0;
        if ( $encryption_level eq "byte" ) {
            $encryption_sed_max = (length($encrypting_sed)-1); # The encryption key arrives in bytes
        }
        else {
            $encryption_sed_max = (length($encrypting_sed)-8); # The encryption key arrives in bits
        }

        while ( $encrypt_decrypt_times > 0 ) {
            $out_bytes = (); 

            for ( my $i=0 ; $i<length($in_bytes_bits_tmp) ; $i++ ) {
                if ( $encrypting_sed_position >= $encryption_sed_max ) { # update the encryption key position
                    $sed_movement = "minus"; # sed position is larger or equal than the sed
                }
                elsif ( $encrypting_sed_position <= 0 ) {
                    $sed_movement = "plus"; # sed position is smaller or equal to zero
                }
                if ( $encryption_level eq "byte" ) { # if things came in bytes
                    $in_byte_tmp = substr($in_bytes_bits_tmp,$i,1);
                    $encrypting_sed_byte = substr($encrypting_sed, $encrypting_sed_position, 1);
                    $out_bytes .= ($in_byte_tmp) ^ ($encrypting_sed_byte); # encrypt/decrypt the byte
                }
                else { # if things came in bits
                    $in_bits_tmp = unpack("B8", substr($in_bytes_bits_tmp,$i,1));
                    $encrypting_sed_bits = substr($encrypting_sed, $encrypting_sed_position, 8);
                    my $encrypted_decrypt_bits = "";
                    if ( $operation eq "encrypt" ) { # if encripting
                        for ( my $ii = 7 ; $ii>=0 ; $ii-- ) { # encrypt/decrypt every bit of the byte, notice the byte is read from end to start
                                $encrypted_decrypt_bits .= (0+substr($in_bits_tmp,$ii,1)) ^ (0+substr($encrypting_sed_bits,$ii,1));
                        }
                    }
                    else { # if decrypting
                        my $sed_ii = 7; # sed posioion is the last
                        for ( my $ii = 0 ; $ii<8 ; $ii++ ) { # encrypt/decrypt every bit of the byte, notice the byte was encrypted from end to start, so now read start to end
                                $encrypted_decrypt_bits = ((0+substr($in_bits_tmp,$ii,1)) ^ (0+substr($encrypting_sed_bits,$sed_ii,1))) . $encrypted_decrypt_bits;
                                $sed_ii--; # encryption sed is still read from the same position
                        }
                    }
                    $out_bytes .= pack("B8", $encrypted_decrypt_bits); # reconvert the encrypt/decrypt bits into a encrypt/decrypt byte and add it into the string to crypt/decrypt
                }
                # add or decrease the position inside the encryption sed... this complicates things a bit futher
                if ( $sed_movement eq "plus" ) {
                    $encrypting_sed_position = $encrypting_sed_position + 1 if ( $encryption_level eq "byte" );
                    $encrypting_sed_position = $encrypting_sed_position + 8 if ( $encryption_level eq "bit" );
                }
                else {
                    $encrypting_sed_position = $encrypting_sed_position - 1 if ( $encryption_level eq "byte" );
                    $encrypting_sed_position = $encrypting_sed_position - 8 if ( $encryption_level eq "bit" );
                }
                $byte_bit_counter++; # count the number of bytes/bits encrypted
            }

            # play a bit around with the way the encryption sed and what is read bettwen each encryption cycle, this is a further security eachement
            if ( ($encrypt_decrypt_times)%2 == 0 ) { # if the crypting cycle number is odd
                $encrypting_sed =  $encrypting_sed_2 . $byte_bit_counter if $encryption_level eq "byte"; # if using bytes, create a new encryption sed, based on the alternative of the encryption sed and on the last byte_bit_counter
                $encrypting_sed =  $encrypting_sed_2 . return_bits($byte_bit_counter, $global_config_p) if $encryption_level eq "bit"; # if using bits create a new encryption sed, based on the alternative of the encryption sed and on the last byte_bit_counter and convert the value into bit
            }
            else {
                $encrypting_sed =  $encrypting_sed_1; # use the main encryption sed
            }

            if ( $encryption_level eq "byte" ) {
                $encryption_sed_max = (length($encrypting_sed)-1); # The encryption key arrives in bytes
            }
            else {
                $encryption_sed_max = (length($encrypting_sed)-8); # The encryption key arrives in bits
            }

            if ( $encrypting_sed_position >= $encryption_sed_max ) { # update the encryption key position if required
                $encrypting_sed_position = $encryption_sed_max;
                $sed_movement = "minus"; # sed position is larger or equal than the sed
            }
            if ( $sed_movement eq "plus" and $encrypting_sed_position > 0  and ($byte_bit_counter)%2 == 0 ) { # if the movement in the encryption sed is going up and if the sed posionton is over zero and number is odd(dividable by 2)
                $sed_movement = "minus"; # change the movement to decrease
            }
            elsif ( $sed_movement eq "plus" and ($byte_bit_counter)%2 != 0 ) { # if the movement in the encryption sed is going up and number is even(not dividable by 2)
                $encrypting_sed_position = $encryption_sed_max; # reset into the max position
                $sed_movement = "minus"; # change the movement to decrease
            }
            elsif ( $sed_movement eq "minus" and $encrypting_sed_position < $encryption_sed_max and ($byte_bit_counter)%2 == 0 ) { # if the movement in the encryption sed is going down and the number is odd(dividable by 2)
                $sed_movement = "plus";  # change the movement into going up
            }
            elsif ( $sed_movement eq "minus" and ($byte_bit_counter)%2 != 0 ) { # if the movement in the encryption sed is going down and the number is odd(dividable by 2)
                $encrypting_sed_position = 0; # reset into the zero position
                $sed_movement = "plus";  # change the movement into going up
            }

            $encrypt_decrypt_times--;
            $in_bytes_bits_tmp = $out_bytes;
        }
        $$encrypting_sed_position_pointer = $encrypting_sed_position; # update the encryption pointer posiontion, for the next sequence of bytes to encrypt
    }

    return $out_bytes;
}


#
# Encrypt/Decrypt a file, by reading the source file and write a new file.
# 
# Will return: 0 ; 1
# Explain: 0 is sucess ; 1 is failure
sub crypt_decrypt_file {

    my $in_file = $_[0]; # filename to encrypt/decrypt
    my $out_file = $_[1]; # new filename (the encrypted/decrypted output)
    my $encrypting_sed = $_[2]; # the "password" to encrypt/decrypt the file
    my $operation = $_[3]; # type of operation ( encrypt / decrypt )
    my $read_buffer_size = $_[4]; # the read buffer, write buffer is always 32 times larger
    my $encryption_times = $_[5]; # number of times that a encryption loop is done on the file
    my $encryption_level = $_[6];  # at what level is the encryption (byte / bit)
    my $crypting_file_algorithm_version = $_[7]; # The version of the crypting algoritm the encrypted file will use,
                                                 # notice this might change during the decrypt operation (if the file
                                                 # was encrypted with a older version of the algorith)
    my $erase_source = $_[8]; # erase the user input file or not after sucessefull encryption/decryption (yes/no)
    my $give_feedback = $_[9]; # if this function will print a feedback info on the screen (yes/no), during each file read operation
    my $global_config_p = $_[10]; # pointer into the configuration
    print "Running the crypt_decrypt_file() function\n" if ( defined $$global_config_p{debug_level} and ($$global_config_p{debug_level} eq "half" or $$global_config_p{debug_level} eq "full") ); # print message if the debug level requires it
    my $error_flag = 0; # flag to mark errors
    # open the source file in read mode ( < )
    open(source_file_handle, "<$in_file") or $error_flag = 1;
    # open or create the log file in append > rewrite mode
    open(destiny_file_handle, ">$out_file") or $error_flag = 1;
    #lock the file for exclusive script use
    if ( $error_flag == 0 ) {
        flock (destiny_file_handle, LOCK_EX) or $error_flag = 1;
        binmode(source_file_handle); # Hack to make Win32 work
        binmode(destiny_file_handle); # Hack to make Win32 work
    }
    if ( $error_flag == 0 ) { # if no errors occured, encript or decrypt the file

        # notice starting from here errors are marked on the error_flag using undef value
        my $counter = 0;
        my $read_buffer = "";
        my $file_position = 0;
        my $encrypting_sed_position = 0;
        my $write_buffer = "";
        my $cripted_sucess_msg = "SuCeSs"; # Marker msg to know if the file is being sucessfuly decrypted or not
        my $encrypting_sed_alternative;
        my ($source_dev,$source_ino,$source_mode,$source_nlink,$source_uid,$source_gid,$source_rdev,$source_size,$source_atime,$source_present_mtime,$source_ctime,$source_blksize,$source_blocks) = stat("$in_file");

        # write the encryption file header if creating a new encrypted file
        if ( $operation eq "encrypt" ) {
            $error_flag = syswrite destiny_file_handle,$crypting_file_algorithm_version;
            $error_flag = 0 if ( defined $error_flag );
        }
        elsif ( $operation eq "decrypt" ) { # decrypting file, remove the file header and test if
            my $crypting_algorithm_version_size = length($crypting_file_algorithm_version);
            my $file_header_identify; # will contain the file header
            # read the file identitifier
            $error_flag = sysseek source_file_handle,$file_position,0; # position (in bytes) the file pointer into the next step
            $error_flag = sysread source_file_handle,$file_header_identify,$crypting_algorithm_version_size; # read the simplecrypt algothm file header
            if ( !defined $file_header_identify or $file_header_identify ne "SimpleCrypt 1.0.0" ) { # test if the file header is valid
                $error_flag = (); # mark error, invalid file header
            }
            else {
                $crypting_file_algorithm_version = $file_header_identify; # we are decrypting a file, get the actual algorithm version from the encrypted file header.
            }
        }

        if ( defined $error_flag and $crypting_file_algorithm_version eq "SimpleCrypt 1.0.0" ) { # if the file is using version 1.0.0 of the encryption algorithm

            # write the encryption file header if creating a new encrypted file
            if ( $operation eq "encrypt" ) {
                # spice up the user encription by encripting it also, the encription sed used, is based on the
                # file size*user string(user encryption sed)
                $encrypting_sed = improve_encryption_sed($encrypting_sed, ($source_size*length($encrypting_sed)), $encryption_times, $encryption_level, $crypting_file_algorithm_version, $global_config_p );
                $encrypting_sed_alternative = build_crc32("$encrypting_sed", $global_config_p); # build the crc32 number for this backup directory
                if ( $encryption_level eq "byte" ) {
                    $encrypting_sed_position = (length($encrypting_sed)-1); # configure the sed pointer
                }
                else { # if encrypting at bit level
                    $encrypting_sed = return_bits($encrypting_sed, $global_config_p); # Convert the spiced user encryption sed string into bits
                    $encrypting_sed_alternative = return_bits($encrypting_sed_alternative, $global_config_p); # Convert the spiced user encryption sed string into bits
                    $encrypting_sed_position = (length($encrypting_sed)-8); # configure the sed pointer
                }
                $cripted_sucess_msg = crypt_decrypt_bytes($cripted_sucess_msg, $encrypting_sed, $encrypting_sed_alternative, \$encrypting_sed_position, $encryption_times, $encryption_level, $operation, $crypting_file_algorithm_version, $global_config_p); # write a encrypted sucess message into the file header to know in the future that de decompress when ok
                $error_flag = syswrite destiny_file_handle,$cripted_sucess_msg; # write the encrypted sucess msg for future decrypt tests
            }
            elsif ( $operation eq "decrypt" ) { # decrypting file, remove the file header and test if
                my $crypting_algorithm_version_size = length($crypting_file_algorithm_version);
                my $cripted_sucess_msg_size = length($cripted_sucess_msg);
                my $file_header_identify; # will contain the file header
                my $sucess_check; # will contain the sucess encrypted msg
                # spice up the user encription by encripting it also, the encription sed used, is based on the
                # file size*user string(user encryption sed), in this case we  count out the encription file header size
                $encrypting_sed = improve_encryption_sed($encrypting_sed, (($source_size-$crypting_algorithm_version_size-$cripted_sucess_msg_size)*length($encrypting_sed)), $encryption_times, $encryption_level, $crypting_file_algorithm_version, $global_config_p );
                $encrypting_sed_alternative = build_crc32("$encrypting_sed", $global_config_p); # build the crc32 number for this backup directory
                $file_position = $crypting_algorithm_version_size; # Position the file read pointer after the file header
                # read the sucess cryption code
                $error_flag = sysseek source_file_handle,$file_position,0; # position (in bytes) the file pointer into the next step
                $error_flag = sysread source_file_handle,$sucess_check,length($cripted_sucess_msg); # read the sucess msg header
                if ( $encryption_level eq "byte" ) {
                    $encrypting_sed_position = (length($encrypting_sed)-1);
                }
                else { # if decrypting at bit level
                    $encrypting_sed = return_bits($encrypting_sed, $global_config_p); # Convert the spiced user encryption sed string into bits
                    $encrypting_sed_alternative = return_bits($encrypting_sed_alternative, $global_config_p); # Convert the spiced user encryption sed string into bits
                    $encrypting_sed_position = (length($encrypting_sed)-8);
                }
                $sucess_check = crypt_decrypt_bytes($sucess_check, $encrypting_sed, $encrypting_sed_alternative, \$encrypting_sed_position, $encryption_times, $encryption_level, $operation, $crypting_file_algorithm_version, $global_config_p); # attempt to decrypt the sucess msg
                $file_position = $file_position + length($cripted_sucess_msg); # place the file pointer after all simplecrypt algorithm headers
                if ( !defined $sucess_check or $sucess_check ne "SuCeSs" ) { # test if the encrypted file versions is valid
                    $error_flag = (); # mark error, invalid file header or uncript went bad
                }
            }
            while ( defined $error_flag and $file_position < $source_size ) {
                print "." if ( $give_feedback eq "yes" ); # print a . char during each read operation (user feedback)
                $error_flag = sysseek source_file_handle,$file_position,0; # position (in bytes) the file pointer into the next step
                if ( ($read_buffer_size + $file_position) > $source_size ) { # detect if the buffer + the file position is overunning the total file size
                    $read_buffer_size = $source_size - $file_position; # fix the buffer size to only the the last file bytes
                }
                $error_flag = sysread source_file_handle,$read_buffer,$read_buffer_size; # position (in bytes) the file pointer into the next step
                $write_buffer .= crypt_decrypt_bytes($read_buffer, $encrypting_sed, $encrypting_sed_alternative, \$encrypting_sed_position, $encryption_times, $encryption_level, $operation, $crypting_file_algorithm_version, $global_config_p); # add the read buffer into the write buffer
                if ( $counter == 128 ) { # check if we need to write anything;
                    $error_flag = syswrite destiny_file_handle,$write_buffer; # write the new file using the write buffer string
                    $write_buffer = (); # clear the write buffer
                    $counter = 0; # reset the read counter
                    print "*" if ( $give_feedback eq "yes" ); # print a . char during each read operation (user feedback)
                }
                $file_position = $file_position + $read_buffer_size; # read the N buffer bytes at once
                $counter++;
                last if ( !(defined $error_flag) );
            }
            if ( defined $error_flag and $write_buffer ne ""  ) { # check if we need to write any file left overs
                print "*" if ( $give_feedback eq "yes" ); # print a . char during each read operation (user feedback)
                $error_flag = syswrite destiny_file_handle,$write_buffer; # write the new file using the write buffer string
            }
        }
        # fix the error flag since it can have undef (if a error occured)
        if ( defined $error_flag ) {
            $error_flag = 0;
        }
        else {
            $error_flag = 1; # error flag had undef so mark error
        }
    }
    close (source_file_handle); # close the source file
    close (destiny_file_handle); # close the destiny file

    if ( $error_flag == 0 and $erase_source eq "yes" ) { # erase the user input file it sucesefull and if required
        unlink $in_file;
    }
    unlink $out_file if ( $error_flag != 0 ); # delete the created file if any error occured

    print "Function crypt_decrypt_file() exit value is [ $error_flag ]\n" if ( defined $$global_config_p{debug_level} and $$global_config_p{debug_level} eq "full" ); # print message if the debug level requires it
    return $error_flag;
}


##########################
# Main
##########################


# test if the commands/parameters are correct
if ( test_loads_commands(\@ARGV, $cycle, $encryption_level, \%global_config) != 0 ) {
    show_commands($script_version, $script_last_update, $script_license, $script_authors, $script_email, $script_site, $script_forum, $cycle, $encryption_level); # show the valid commands
    exit 1; # exit from script if the arguments are wrong or there are no arguments
}
else {
    my $user_input_file = $global_config{in_file}; # user defined file
    my $user_output_file = $global_config{out_file};
    my $sed = $global_config{passwd}; # user encryption sed / password
    my $operation = $global_config{operation}; # type of operation encryption/decryption
    my $user_encryption_level = $global_config{level};
    my $erase_source = $global_config{erase_source};
    my $pause = $global_config{pause};
    my $user_cycle = $global_config{crypt};
    my $error_flag = 0;

    if ( !defined $sed or $sed eq "" ) { # ask a user password if not passed on the arguments and enter in interactive mode
        print "\nSimplecrypt v$script_version ($script_last_update),\n";
        print "$script_site\n\n";
        print "Press enter to accept the default values within the []\n";
        my $tmp_value = "";
        while ( $tmp_value eq "" or -f $tmp_value ) {
            my $my_operation = $operation;
            $my_operation =~ s/--//;
            print "\nWhere to $my_operation (path into the output file)\n";
            print "[ $user_output_file ]\npath to new file > ";
            $tmp_value = <STDIN>;
            chop $tmp_value; # remove the new line char
            $tmp_value = $user_output_file if ( $tmp_value eq "" ); # user pressed enter... meaning he want's the default value
            print "\nError : [ $tmp_value ],\nalready exists, to encrypt/decrypt you must create a new file.\n" if ( -f $tmp_value );
        }
        $user_output_file = $tmp_value;

        $tmp_value = "";
        while ( $tmp_value eq "" or !( ($tmp_value =~ /^\d+$/) and $tmp_value > 0 ) ) {
            print "\nNumber of crypting times (the higher the safer and slower)\n";
            print "[ $user_cycle ]\nnumber > ";
            $tmp_value = <STDIN>;
            chop $tmp_value; # remove the new line char
            $tmp_value = $user_cycle if ( $tmp_value eq "" ); # user pressed enter... meaning he want's the default value
        }
        $user_cycle = $tmp_value;
        
        $tmp_value = "";
        while ( $tmp_value ne "byte" and $tmp_value ne "bit" ) {
            print "\nEncrypting level (bit level is safer but very slow)\n";
            print "[ $user_encryption_level ]\nbyte/bit > ";
            $tmp_value = <STDIN>;
            chop $tmp_value; # remove the new line char
            $tmp_value = $user_encryption_level if ( $tmp_value eq "" ); # user pressed enter... meaning he want's the default value
        }
        $user_encryption_level = $tmp_value;

        $tmp_value = "";
        while ( $tmp_value ne "yes" and $tmp_value ne "no" ) {
            my $my_operation = $operation;
            $my_operation =~ s/--//;
            print "\nErase the file to $my_operation after ending (with sucess) ?\n";
            print "[ $erase_source ]\nyes/no > ";
            $tmp_value = <STDIN>;
            chop $tmp_value; # remove the new line char
            $tmp_value = $erase_source if ( $tmp_value eq "" ); # user pressed enter... meaning he want's the default value
        }
        $erase_source = $tmp_value;

        while ( !defined $sed or $sed eq "" ) {
            print "\nPlease type in the encryption password\n";
            print "password > ";
            $sed = <STDIN>;
            chop $sed; # remove the new line char
        }
    }

    if ( !-f $user_input_file ) {
        print "\nError : [ $user_input_file ],\ndoes not exist or you do not have permissions to access it.\n\n";
        $error_flag = 1;
    }
    if ( $error_flag == 0 and -f $user_output_file ) {
        print "\nError : [ $user_output_file ],\nalready exists, to encrypt/decrypt you must create a new file.\n\n";
        $error_flag = 1;
    }

    if ( $error_flag == 0 ) {
        print "\n\nDetails :\n";
        if ( $operation eq "encrypt" ) {
            print "  Encrypting file [ $user_input_file ]\n";
            print "  Encrypting file will be erased (if sucess)\n" if ( $erase_source eq "yes" );
            print "  Encrypting file will not be erased\n" if ( $erase_source ne "yes" );
        }
        if ( $operation eq "decrypt" ) {
            print "  Decrypting file [ $user_input_file ]\n";
            print "  Decrypting file will be erased (if sucess)\n" if ( $erase_source eq "yes" );
            print "  Decrypting file will not be erased\n" if ( $erase_source ne "yes" );
        }
        print "  Into file [ $user_output_file ]\n";
        print "  Crypting times [ $user_cycle ]\n";
        print "  Crypting level [ $user_encryption_level ]\n\n";
        print "Notice :\n  The . char represents a read + $operation operation and the * char\n  represents a write operation.\n\n";
        print "Encrypting\n" if ( $operation eq "encrypt" );
        print "Decrypting\n" if ( $operation eq "decrypt" );
        # encrypt or decrypt the user file
        if ( crypt_decrypt_file($user_input_file, $user_output_file, $sed, $operation, $file_read_size, $user_cycle, $user_encryption_level, $crypting_algorithm_version, $erase_source, "yes" ) == 0 ) {
            print "\n\nFile [ $user_output_file ] was created.\n";
            print "File [ $user_input_file ] was erased.\n" if ( !-f $user_input_file );
            print "\nOperation ended in sucess.\n";
            $error_flag = 0;
        }
        else {
            print "\n\nError : [ $user_input_file ],\nunable to encrypt/decrypt the file, check permissions, free space,\nif the password is correct and if the crypt times number and\nlevel are correct.\n\n";
            $error_flag = 1;
        }
    }

    if ( $pause eq "yes" ) {
        print "\n** Press any key to end **";
        my $trash = <STDIN>;
    }

    exit $error_flag;
}
