/*
 * This file is part of tiptop.
 *
 * Author: Antoine NAUDIN
 * Copyright (c) 2012 Inria
 *
 * License: GNU General Public License version 2.
 *
 */

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <papi.h>

#include "formula-parser.h"
#include "process.h"
#include "screen.h"
#include "utils-expression.h"


/* copy of the expression found in XML file. To be passed to the lexer
   (possibly in chunks) by my_yyinput. */
static char* ptr_myinput = NULL;

/* Expression built by parser, */
expression* res_expr = NULL;

void yyparse();  /* generated by yacc */


/* dedicated tools to allocate expression and son */

int my_yyinput(char* buffer, int buffer_size)
{
  int len;

  strncpy(buffer, ptr_myinput, buffer_size);
  len = strlen(buffer);
  ptr_myinput += len;
  return len;
}


unit* alloc_unit()
{
  unit* u = malloc(sizeof(unit));
  u->alias = NULL;
  u->val = -1;
  u->type = -1;
  u->delta = -1;
  return u;
}


expression* alloc_expression()
{
  expression* e = malloc(sizeof(expression));
  e->ele = NULL;
  e->op = NULL;
  e->type = -1;
  return e;
}

operation* alloc_operation()
{
  operation* o = malloc(sizeof(operation));
  o->operator = '!';
  o->exp1 = NULL;
  o->exp2 = NULL;
  return o;
}

/* dedicated destroyer to free expressions */

void free_unit(unit* u)
{
  if (u == NULL)
    return;
  if (u->alias != NULL)
    free(u->alias);
  free(u);
}


void free_expression (expression* e)
{
  if (e == NULL)
    return;
  free_unit(e->ele);
  free_operation(e->op);
  free(e);
}


void free_operation(operation* p)
{
  if (p == NULL)
    return;
  free_expression(p->exp1);
  free_expression(p->exp2);
  free(p);
}


#ifdef ENABLE_DEBUG
/* Print an expression */
void print_expression(expression* e)
{
  if (e->type == ERROR) {
    printf("ERROR");
      return ;
  }
  if (e->type == ELEM){
    if (e->ele->type == COUNT) {
      if (e->ele->delta == DELT)
        printf("delta(%s)", e->ele->alias);
      else
        printf("%s", e->ele->alias);
    }
    else if (e->ele->type == CONST) {
      printf("%2.1f", e->ele->val);
    }
  }
  else if (e->type == OPER && e->op != NULL) {
    printf("(");
    print_expression(e->op->exp1);
    printf(" %c ", e->op->operator);
    print_expression(e->op->exp2);
    printf(")");
  }
}
#endif


/* Build a representative stream of an expression */
int build_expression(expression* e, FILE* fd)
{
  if (e->type == ELEM) {
    if (e->ele->type == COUNT) {
      if (e->ele->delta == DELT)
        return fprintf(fd, "delta(%s)", e->ele->alias);
      else
        return fprintf(fd, "%s", e->ele->alias);
    }
    else if (e->ele->type == CONST)
      return fprintf(fd, "%4.2lf", e->ele->val);
  }
  else if (e->type == OPER && e->op != NULL) {
    if (fprintf(fd, "(") < 0)
      return -1;
    if (build_expression(e->op->exp1, fd) < 0)
      return -1;

    if (e->op->operator == '<') {
      if (fprintf(fd, " shl ") < 0)
        return -1;
    }
    else if (e->op->operator == '>') {
      if (fprintf(fd, " shr ") < 0)
        return -1;
    }
    else if (e->op->operator == '&') {
      if (fprintf(fd, " and ") < 0)
        return -1;
    }

    else if (e->op->operator == '|') {
      if (fprintf(fd, " or ") < 0)
        return -1;
    }
    else if (fprintf(fd, " %c ", e->op->operator) < 0)
      return -1;

    if (build_expression(e->op->exp2, fd) < 0)
      return -1;
    if (fprintf(fd, ")") < 0)
      return -1;
  }
  return 0;
}


/* Parse and return a representative tree */
expression* parser_expression(char* txt)
{
  ptr_myinput = txt;
  res_expr = NULL;  /* will be computed by yacc-generated parser */
  if (strlen(txt) > 0) {
    yyparse();
  }
  return res_expr;
}


/* Tools to find counters' ID */
static int get_counter_id(char* alias, counter_t* tab, int nbc)
{
  int i;
  for(i=0 ;i<nbc; i++)
    if (strcmp(alias, tab[i].alias) == 0)
      return i;
  return -1;
}


/* Tools to get counter value */
static double get_counter_value(unit* e, counter_t* tab, int nbc, char delta,
                                struct process* p, int* error)
{
  int id;
  /* System information: not based on performances counters */
  if (strcmp(e->alias, "CPU_TOT") == 0)
    return p->cpu_percent;

  if (strcmp(e->alias, "CPU_SYS") == 0)
    return p->cpu_percent_s;

  if (strcmp(e->alias, "CPU_USER") == 0)
    return p->cpu_percent_u;

  if (strcmp(e->alias, "NUM_THREADS") == 0)
    return p->num_threads;

  int EventCode = PAPI_NULL;
  if (PAPI_event_name_to_code(e->alias,&EventCode) == PAPI_OK) {
    double retval;
    
    int retcode;
    PAPI_option_t options;

    int i = 0;
    do {
        options.attach.eventset = i;
        retcode = PAPI_get_opt(PAPI_ATTACH, &options);
        if (retcode<0) {
            handle_error(retcode);
            *error=1;
            return 1;
        }
        if (options.attach.tid == p->tid) break;
        i++;
    } while (1);

    long long values[PAPI_num_events(options.attach.eventset)];
    for (i=0;i<PAPI_num_events(options.attach.eventset);i++)
        values[i] = (long long)0;

    int k = 0;
    for (i=0;i<MAX_EVENTS;i++) {
        if (p->papi[i]==EventCode) {
            k = i;
            break;
        }
    }

    int numPapiEvents = 0;
    for (i=0;i<MAX_EVENTS;i++) {
        if (p->papi[i]==-1) {
            numPapiEvents = i;
            break;
        }
    }    

    if (k==(numPapiEvents-1)) {
        retcode = PAPI_accum( options.attach.eventset, values );
        if (retcode!=PAPI_OK) handle_error(retcode);
    } else {
        retcode = PAPI_read( options.attach.eventset, values );
        if (retcode!=PAPI_OK) handle_error(retcode);
    }
    
    return (double)(values[k]);
  }

  if (strcmp(e->alias, "PROC_ID") == 0) {
    if (p->proc_id != -1) {
      return (double)p->proc_id;
    }
    else {
      *error=1;
      return 1;
    }
  }

  if (strcmp(e->alias, "NUM_THREADS") == 0)
    return p->num_threads;

  id = get_counter_id(e->alias, tab, nbc);

  if ((id == -1) || (p->values[id] == 0xffffffff)) {
    /* Invalid counter */
    *error = 1;
    return 1;
  }

  if (delta == DELT)
    return (double) (p->values[id] - p->prev_values[id]);

  return (double) p->values[id];
}


/* Calculous and return result of expression gived in entry
 *
 * (argument error is used for indicate and determine which kind of
 * error appear during evaulation:
 *      1 => Invalid Counter
 *      2 => Divide per zero (also if it is double)
 */


double evaluate_column_expression(expression* e, counter_t* c, int nbc,
                           struct process* p, int* error)
{
  /* Invalid Expression */
  if (e == NULL) {
    *error = 1;
    return 0;
  }
  *error = 0;
  if (e->type == ELEM) {
    /* Return Element value */
    if (e->ele->type == COUNT)
      return get_counter_value(e->ele, c, nbc, e->ele->delta, p, error);
    else if(e->ele->type == CONST)
      return e->ele->val;
  }
  else if ((e->type == OPER) && (e->op != NULL)) {
    /* Or calcul leaf value and return the result */
    switch(e->op->operator) {
    case '+':
      return evaluate_column_expression(e->op->exp1, c, nbc, p, error) +
             evaluate_column_expression(e->op->exp2, c, nbc, p, error);
      break;

    case '-':
      return evaluate_column_expression(e->op->exp1, c, nbc, p, error) -
             evaluate_column_expression(e->op->exp2, c, nbc, p, error);
      break;

    case '*':
      return evaluate_column_expression(e->op->exp1, c, nbc, p, error) *
             evaluate_column_expression(e->op->exp2, c, nbc, p, error);
      break;
    case '/': {
      double tmp = evaluate_column_expression(e->op->exp2, c, nbc, p, error);
      if (tmp == 0) {
        /* Divide by 0 */
        *error = 2;
        return 0;
      }
      return evaluate_column_expression(e->op->exp1, c, nbc, p, error) / tmp;
      break;
    }
    default:
      /* Unknown operator */
      assert(0);
    }
  }
  return 0;
}

uint64_t evaluate_counter_expression(expression* e, int* error)
{
  uint64_t val = 0;
  if (e == NULL  || e->type == ERROR) {
    (*error)++;
    return 0;
  }
  if (e->type == ELEM) {
    if (e->ele->type == COUNT) {
      if (get_counter_config(e->ele->alias, &val) < 0)
        (*error)++;
      return val;
    }
    else return (uint64_t) e->ele->val;
  }
  else if ((e->type == OPER) && (e->op != NULL)) {
    switch (e->op->operator) {
      /* Binary operator to define a counter type */
    case '<':
      return evaluate_counter_expression(e->op->exp1, error) <<
             evaluate_counter_expression(e->op->exp2, error);
      break;
    case '>':
      return evaluate_counter_expression(e->op->exp1, error) >>
             evaluate_counter_expression(e->op->exp2, error);
      break;
    case '&':
      return evaluate_counter_expression(e->op->exp1, error) &
             evaluate_counter_expression(e->op->exp2, error);
      break;
    case '|':
      return evaluate_counter_expression(e->op->exp1, error) |
             evaluate_counter_expression(e->op->exp2, error);
      break;
    default:
      assert(0);
    }
  }
  return 0;
}
