#!/usr/bin/perl

# X10 MouseRemote client software,
# Copyright 1999, Dan Wilga
# gribnif@pair.com  http://www.pair.com/gribnif/ha
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# See the COPYING file for more copyright info
#

my $DEBUG = 0;				# turn on for debugging messages
my $X10_FIFO = "/dev/x10fifo";		# name of FIFO provided by MultiMouse
my $CONFIG = "MouseRemote.conf";	# name of config file to read
my $PIDFILE = "MouseRemote.pid";	# pid of this run
my $DEBOUNCE_EXPIRE = 0.75;		# after this many seconds, a new keypress registers
my $DEFAULT_DEVICE = 'WEB';		# default device at startup (PC/CD/WEB/DVD/PHONE)

my $VERSION = 0.90;			# version number

# The X10 MouseRemote doesn't tell you when a key has been released.
# Instead, it just keeps repeating the last key pressed until it
# has been released by the user. So we have to do some special things
# to keep keypresses from getting queued and appearing multiple times.
#
# $eat_key gets set if we have launched a child process to handle
# a keystroke (and the config file says the command should auto-repeat).
# This way, we ignore any duplicate keypresses until the command has
# finished.
my $eat_key;				# do we need to eat keystrokes?
# $debounce holds the keycode last pressed, to ensure that the
# multiple keypresses coming from the remote within $DEBOUNCE_EXPIRE
# get treated as just one keypress.
my $debounce = -1;			# help track multiple keypresses
my $debounce_time;			# time of first keypress

# constant for accessing the @command array
my %dev = ( PC=>1, CD=>2, WEB=>3, DVD=>4, PHONE=>5 );
my $device = $DEFAULT_DEVICE;
my( @command,				# holds the user-defined commands
    @repeat );				# set if user wants auto-repeat for cmd

use vars qw|%key2num|;			# var defined in MouseRemoteKeys.pl
require "MouseRemoteKeys.pl";		# read keycode definitions
my %num2key;				# convert number->keyname
foreach( keys %key2num ) {		# read keyname->number hash
	$num2key{$key2num{$_}} = $_;	# and convert to number->keyname
}

use strict;
use Fcntl;				# defines for sysopen()
use Time::HiRes;			# timers with granularity < 1 sec.

while( $ARGV[0] =~ /^-/ ) {
	if( $ARGV[0] =~ /^-c/ ) {
		shift;
		$CONFIG = shift;
	}
	elsif( $ARGV[0] =~ /^-d/ ) {
		shift;
		$DEBUG = 1;
	}
	elsif( $ARGV[0] =~ /^(-h|--help)/ ) {
		&usage();
	}
	if( $ARGV[0] =~ /^-p/ ) {
		shift;
		$PIDFILE = shift;
	}
}
if( !$#ARGV ) {
	if( !$dev{uc($ARGV[0])} ) {
		print STDERR "Unknown device name $ARGV[0]\n";
		&usage();
	}
	$DEFAULT_DEVICE = uc(shift);
}
if( $#ARGV>=0 ) {
	print STDERR "Unknown parameter: $ARGV[0]\n";
	&usage();
}

&loadConfig;

sysopen( X10, $X10_FIFO, O_RDONLY ) or die "Can't open $X10_FIFO: $!\n";

open( PID, ">$PIDFILE" ) || die( "Can't create $PIDFILE: $!\n" );
print PID $$;				# print PID
close PID;

$SIG{TERM} = \&myTerm;			# intercept SIGTERM in parent
$SIG{HUP} = \&myHup;			# intercept SIGHUP so we can reload conf
$SIG{CHLD} = \&myChld;			# intercept SIGCHLD for fork()

my( $len, $data );
while( ($len = sysread( X10, $data, 3 )) != undef ) {	# read data
	my( $type, $byte, $end ) = unpack('CCC',$data);	# convert data
	if( !$eat_key &&				# eating key due to child?
			($byte != $debounce ||		# different key pressed
			Time::HiRes::gettimeofday()-$debounce_time > $DEBOUNCE_EXPIRE) ) {	# or time is up
		printf( ">> %02X Device:$device\n", $byte ) if $DEBUG;
		if( $type == 0x08 && $end == 0x7F ) {	# packet always has this wrapper
			&process($byte);
		}
	}
	else {				# eat the keypress
		printf( "Eating key %02X\n", $byte ) if $DEBUG;
	}
}
die "MouseRemote FIFO Error: $!\n";	# MultiMouse probably died

# Handle the debounced keypress
sub process {
	my $byte = $_[0];
	$debounce = $repeat[$dev{$device}]{$byte} ? -1 : $byte;	# don't debounce if user wants auto-repeat
	$debounce_time = Time::HiRes::gettimeofday();		# get time in floating point
	if( defined $num2key{$byte} ) {				# is this a valid key?
		# By testing for a command first, we allow the WEB/PC/CD/DVD keys
		# to be redefined
		if( defined $command[$dev{$device}]{$byte} ) {	# is there a user command?
			$eat_key = 1;				# eat keys until child returns
			my $pid = fork;				# fork a child
			if( !defined($pid) ) {
				die "Can't fork a child: $! $@ $?\n";
			}
			elsif( !$pid ) {			# child process
				undef $SIG{TERM};		# only handle in parent
				my $cmd = $command[$dev{$device}]{$byte};	# user's command
#				if( $debounce == $byte ) {	# cheap way to test auto-repeat
					system($cmd);		# no auto-repeat, so run cmd in
#				}				#   a separate process
#				else {				# auto-repeat: run the cmd
#					exec($cmd);		#   and never return
#				}
				exit 0;				# probably only for system()
			}					# myChld catches child exit
		}
		elsif( $dev{uc($num2key{$byte})} ) {		# is this a device key?
			$device = uc($num2key{$byte});
			print "$device device selected\n" if $DEBUG;
		}
		else {
			print "$num2key{$byte} not defined for $device\n" if $DEBUG;
		}
	}
	else {							# user didn't set this key
		printf( "undefined key:%02X for $device\n", $byte ) if $DEBUG;
	}
}

sub loadConfig {
	my $curdev = $dev{PC};					# default device in config
	@command = ();						# empty-out the current values
	@repeat = ();
	open( CONFIG, $CONFIG ) || die "Can't open config file $CONFIG: $!\n";
	while(<CONFIG>) {
		s/^\s*//;					# remove spaces at start
		s/\s*$//;					# and end
		next if !$_ || /^#/;				# skip if empty or comment
		if( /^\[(.*)\]$/ ) {				# look for [DEVICENAME]
			die "Unknown device $_ at line $. of $CONFIG\n"
					if( !defined($dev{uc($1)}) );
			$curdev = $dev{uc($1)};
		}
		else {
			/^(\S+)(\s+(.*))?$/;			# look for keyname-space-command
			my $keyname=lc($1);			# make sure lowercase
			my $command=$3;
			my $rpt;
			$rpt = 1 if $keyname =~ s/^\.//;	# if cmd starts with '.', user wants auto-repeat
			die "Syntax error '$_' at line $. of $CONFIG\n"
					if( $keyname eq "" );
			die "Unknown key name $1 at line $. of $CONFIG\n"
					if( !defined($key2num{$keyname}) );
			my $num = $key2num{$keyname};
			$command[$curdev]{$num} = $command if $command;	# ignore if cmd is blank
			$repeat[$curdev]{$num} = 1 if $rpt;
		}
	}
	close CONFIG;
}

# SIGHUP routine: re-read config
sub myHup {
	print "Reloading config file\n" if $DEBUG;
	&loadConfig;
}

# SIGCHLD routine: clear $eat_key
sub myChld {
	wait;
	$SIG{CHLD} = \&myChld;		# System V needs this
	$eat_key = 0;
}

# SIGTERM routine: remove pidfile
sub myTerm {
	unlink $PIDFILE;
	exit 0;
}

sub usage {
	printf( STDERR qq#
                      MouseRemote.pl %1.2f by Dan Wilga

  MouseRemote.pl [-c configfile] [-d] [-h] [-p pidfile] [PC|CD|WEB|DVD|PHONE]

  -d: Turn on debugging messages
  -c: Load an alternate conf file. By default, when the program runs it looks
      for the config file MouseRemote.conf in the current directory.
  -h: This help message
  -p: File where the script's pid is written. The default is MouseRemote.pid.
  Device name: The default device to use. This script has no way of knowing
      what device is currently selected by the MouseRemote when it starts, so
      you may have to press a device button before using a MouseRemote button
      for the first time after the script starts. The WEB device is the default
      if this parameter is not used.\n\n#, $VERSION );
	exit 0;
}
