/*
  Copyright (c) 2012 Christopher Howard

  This file is part of the Recursive Remove Library.
  
  the Recursive Remove Library is free software: you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.
  
  the Recursive Remove Library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public License
  along with the Recursive Remove Library.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _XOPEN_SOURCE 700

#include <ftw.h>
#include <stdio.h>
#include <error.h>
#include <string.h>
#include <stdbool.h>
#include <errno.h>

#include "rremove.h"
#include "config.h"

#define NOPENFD 128

int rr_flags;
FILE * rr_errstream;
bool failed_removal;

int ntfw_h(const char *fpath, const struct stat *sb,
	   int typeflag, struct FTW *ftwbuf)
{
  
#ifdef RRDEBUG  
  fprintf(stderr, "ntfw_h parameters:\n  fpath: %s\n  typeflag: %d\n", fpath, typeflag);
  fprintf(stderr, "  ftwbuf:\n    base: %d\n    level: %d\n", ftwbuf->base, ftwbuf->level);
#endif 

  if(rr_errstream)
    {

      if(typeflag == FTW_DNR)
	fprintf(rr_errstream, "error: could not read directory %s\n", fpath);
      
      if(typeflag == FTW_NS)
	fprintf(rr_errstream, "error: could not stat file %s\n", fpath);
    }

  if(remove(fpath))
    {
      /* remove failure */
      int e = errno;
      
      if(rr_errstream)
	{
	  fprintf(rr_errstream, "error: could not remove %s\n", fpath);
	  fprintf(rr_errstream, "%s\n", strerror(e));
	}

      if(rr_flags & RR_STOP_ON_ERR)
	{

#ifdef RRDEBUG  
	  fprintf(stderr, "ntfw_h returns %d\n", e);
#endif 
	  
      return e;

	}
      else
	{
	  
#ifdef RRDEBUG  
	  fputs("setting failed_removal to true\n", stderr);
#endif 
	  
	  failed_removal = true;
	}
    }
  
#ifdef RRDEBUG  
  fputs("ntfw_h returns 0\n", stderr);
#endif 
  
  return 0;
}

extern int rremove(const char * pathname, int flags, FILE * errstream)
{
  
#ifdef RRDEBUG  
  fprintf(stderr, "rremove parameters:\n  pathname: %s\n  flags: %x\n  errstream: %p\n",
	  pathname, flags, (void *) errstream);
  fputs("setting failed_removal to false\n", stderr);
#endif 

  failed_removal = false;
  rr_flags = flags;
  rr_errstream = errstream;
  
  int nftw_options = FTW_DEPTH | FTW_PHYS;

  if(!(rr_flags & RR_IGNORE_FS)) { nftw_options |= FTW_MOUNT; }

  {  
    int nftw_ret = nftw(pathname, ntfw_h, NOPENFD, nftw_options);
    if(nftw_ret > 0)
      {
	return nftw_ret;
      }
    else if(failed_removal)
      {
	/* at least one file removal failure */
	
#ifdef RRDEBUG  
	fputs("rremove returns -1\n", stderr);
#endif
	
	return -1;
      }
    else
      {
	/* complete success */
	
#ifdef RRDEBUG  
	fputs("rremove returns 0\n", stderr);
#endif
	
	return 0;
      }
  }
}

