/*****************************************************************************
 *MainMenu.m: MacOS X interface module
 *****************************************************************************
 *Copyright (C) 2011-2019 Felix Paul Kühne
 *
 *Authors: Felix Paul Kühne <fkuehne -at- videolan -dot- org>
 *
 *This program is free software; you can redistribute it and/or modify
 *it under the terms of the GNU General Public License as published by
 *the Free Software Foundation; either version 2 of the License, or
 *(at your option) any later version.
 *
 *This program is distributed in the hope that it will be useful,
 *but WITHOUT ANY WARRANTY; without even the implied warranty of
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *GNU General Public License for more details.
 *
 *You should have received a copy of the GNU General Public License
 *along with this program; if not, write to the Free Software
 *Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 *****************************************************************************/

#import "VLCMainMenu.h"

#import "coreinteraction/VLCVideoFilterHelper.h"

#import "extensions/NSScreen+VLCAdditions.h"
#import "extensions/NSString+Helpers.h"

#import "library/VLCLibraryController.h"
#import "library/VLCLibraryWindow.h"
#import "library/VLCLibraryWindowController.h"
#import "library/VLCLibraryWindowSplitViewController.h"

#import "main/VLCMain.h"

#import "menus/renderers/VLCRendererMenuController.h"

#import "panels/VLCAudioEffectsWindowController.h"
#import "panels/VLCTrackSynchronizationWindowController.h"
#import "panels/VLCVideoEffectsWindowController.h"
#import "panels/VLCBookmarksWindowController.h"
#import "panels/dialogs/VLCCoreDialogProvider.h"
#import "panels/dialogs/VLCCustomCropArWindowController.h"
#import "panels/VLCInformationWindowController.h"
#import "panels/VLCTimeSelectionPanelController.h"

#import "playqueue/VLCPlayQueueController.h"
#import "playqueue/VLCPlayQueueModel.h"
#import "playqueue/VLCPlayerController.h"
#import "playqueue/VLCPlayQueueSortingMenuController.h"
#import "preferences/VLCSimplePrefsController.h"

#import "windows/VLCAboutWindowController.h"
#import "windows/VLCDetachedAudioWindow.h"
#import "windows/VLCOpenWindowController.h"
#import "windows/VLCErrorWindowController.h"
#import "windows/VLCHelpWindowController.h"
#import "windows/controlsbar/VLCMainWindowControlsBar.h"
#import "windows/extensions/VLCExtensionsManager.h"
#import "windows/convertandsave/VLCConvertAndSaveWindowController.h"
#import "windows/logging/VLCLogWindowController.h"
#import "windows/addons/VLCAddonsWindowController.h"
#import "windows/video/VLCVoutView.h"
#import "windows/video/VLCVideoOutputProvider.h"

#import <vlc_configuration.h>
#import <vlc_interface.h>

#ifdef HAVE_SPARKLE
#import <Sparkle/Sparkle.h>
#endif

typedef NS_ENUM(NSInteger, VLCObjectType) {
    VLCObjectTypeInterface,
    VLCObjectTypeAout,
    VLCObjectTypeVout,
};

@interface VLCAutoGeneratedMenuContent : NSObject
{
    vlc_object_t *_vlcObject;
}
- (instancetype)initWithVariableName:(const char *)name
                            ofObject:(vlc_object_t *)object
                      withObjectType:(VLCObjectType)objectType
                            andValue:(vlc_value_t)value
                      ofVariableType:(int)type;

@property (readonly) char *variableName;
@property (readonly) vlc_value_t variableValue;
@property (readonly) vlc_object_t *vlcObject;
@property (readonly) VLCObjectType objectType;
@property (readonly) int variableType;

@end

@interface NSMenuItem (KeyEquivalentAddition)
- (void)matchKeyEquivalentsOfMenuItem:(NSMenuItem *)menuItem;
@end

@interface VLCMainMenu() <NSMenuDelegate>
{
    VLCAboutWindowController *_aboutWindowController;
    VLCHelpWindowController  *_helpWindowController;
    VLCAddonsWindowController *_addonsController;
    VLCPlayQueueController *_playQueueController;
    VLCPlayerController *_playerController;
    VLCPlayQueueSortingMenuController *_playQueueSortingController;
    VLCInformationWindowController *_infoWindowController;

    __strong VLCTimeSelectionPanelController *_timeSelectionPanel;
    __strong VLCCustomCropArWindowController *_customARController;
}
@end

@implementation VLCMainMenu

#pragma mark - Initialization

- (void)dealloc
{
    msg_Dbg(getIntf(), "Deinitializing main menu");
    [NSNotificationCenter.defaultCenter removeObserver: self];

    [self releaseRepresentedObjects:[NSApp mainMenu]];
}

- (void)awakeFromNib
{
    _playQueueController = VLCMain.sharedInstance.playQueueController;
    _playerController = _playQueueController.playerController;

    /* check whether the user runs OSX with a RTL language */
    NSArray *languages = [NSLocale preferredLanguages];
    NSString *preferredLanguage = [languages firstObject];

    if ([NSLocale characterDirectionForLanguage:preferredLanguage] == NSLocaleLanguageDirectionRightToLeft) {
        msg_Dbg(getIntf(), "adapting interface since '%s' is a RTL language", [preferredLanguage UTF8String]);
        [_rateTextField setAlignment:NSLeftTextAlignment];
    }

#ifdef HAVE_SPARKLE
    [_checkForUpdate setAction:@selector(checkForUpdates:)];
    [_checkForUpdate setTarget:[SUUpdater sharedUpdater]];
#else
    [_checkForUpdate setEnabled:NO];
#endif

    [self initStrings];
    [self setupKeyboardShortcuts];

    /* configure playback / controls menu */
    self.controlsMenu.delegate = self;
    self.subtitlesMenu.delegate = self;
    [_rendererNoneItem setState:NSOnState];
    _rendererMenuController = [[VLCRendererMenuController alloc] init];
    _rendererMenuController.rendererNoneItem = _rendererNoneItem;
    _rendererMenuController.rendererMenu = _rendererMenu;
    _playQueueSortingController = [[VLCPlayQueueSortingMenuController alloc] init];
    _sortPlayQueue.submenu = _playQueueSortingController.playQueueSortingMenu;

    [self mediaItemChanged:nil];
    [self playbackStateChanged:nil];
    [self updateTitleAndChapterMenus:nil];
    [self updateProgramMenu:nil];
    [self updateLibraryPlayQueueMode];

    NSNotificationCenter *notificationCenter = NSNotificationCenter.defaultCenter;
    [notificationCenter addObserver:self
                           selector:@selector(refreshVoutDeviceMenu:)
                               name:NSApplicationDidChangeScreenParametersNotification
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(updatePlaybackRate)
                               name:VLCPlayerRateChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(updateRecordState)
                               name:VLCPlayerRecordingChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(playbackStateChanged:)
                               name:VLCPlayerStateChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(playModeChanged:)
                               name:VLCPlaybackRepeatChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(playOrderChanged:)
                               name:VLCPlaybackOrderChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(updateTrackHandlingMenus:)
                               name:VLCPlayerTrackListChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(updateTrackHandlingMenus:)
                               name:VLCPlayerTrackSelectionChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(updateTitleAndChapterMenus:)
                               name:VLCPlayerTitleListChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(updateTitleAndChapterMenus:)
                               name:VLCPlayerTitleSelectionChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(updateTitleAndChapterMenus:)
                               name:VLCPlayerChapterSelectionChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(updateProgramMenu:)
                               name:VLCPlayerProgramListChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(updateProgramMenu:)
                               name:VLCPlayerProgramSelectionChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(mediaItemChanged:)
                               name:VLCPlayerCurrentMediaItemChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(voutListChanged:)
                               name:VLCPlayerListOfVideoOutputThreadsChanged
                             object:nil];
    [notificationCenter addObserver:self
                           selector:@selector(updateRateControls:)
                               name:VLCPlayerCapabilitiesChanged
                             object:nil];

    [self setupVarMenuItem:_add_intf
                    target:VLC_OBJECT(getIntf())
                objectType:VLCObjectTypeInterface
                       var:"intf-add"
                  selector:@selector(toggleVar:)];

    /* setup extensions menu */
    /* Let the ExtensionsManager itself build the menu */
    VLCExtensionsManager *extMgr = VLCMain.sharedInstance.extensionsManager;
    [extMgr buildMenu:_extensionsMenu];
    [_extensions setEnabled:([_extensionsMenu numberOfItems] > 0)];

    /* setup post-proc menu */
    [_postprocessingMenu removeAllItems];
    [_postprocessingMenu setAutoenablesItems: YES];
    [_postprocessingMenu addItemWithTitle: _NS("Disable") action:@selector(togglePostProcessing:) keyEquivalent:@""];
    NSMenuItem *mitem = [_postprocessingMenu itemAtIndex: 0];
    [mitem setTag: -1];
    [mitem setEnabled: YES];
    [mitem setTarget: self];
    for (NSUInteger x = 1; x < 7; x++) {
        [_postprocessingMenu addItemWithTitle:[NSString stringWithFormat:_NS("Level %i"), x]
                                       action:@selector(togglePostProcessing:)
                                keyEquivalent:@""];
        mitem = [_postprocessingMenu itemAtIndex:x];
        [mitem setEnabled:YES];
        [mitem setTag:x];
        [mitem setTarget:self];
    }
    char *psz_config = config_GetPsz("video-filter");
    if (psz_config) {
        if (!strstr(psz_config, "postproc"))
            [[_postprocessingMenu itemAtIndex:0] setState:NSOnState];
        else
            [[_postprocessingMenu itemWithTag:config_GetInt("postproc-q")] setState:NSOnState];
        free(psz_config);
    } else
        [[_postprocessingMenu itemAtIndex:0] setState:NSOnState];
    [_postprocessing setEnabled: NO];

    [self refreshAudioDeviceList];

    /* setup subtitles menu */
    [self setupMenu: _subtitle_textcolorMenu withIntList:"freetype-color" andSelector:@selector(switchSubtitleOption:)];
    [_subtitle_bgopacity_sld setIntegerValue: config_GetInt("freetype-background-opacity")];
    [self setupMenu: _subtitle_bgcolorMenu withIntList:"freetype-background-color" andSelector:@selector(switchSubtitleOption:)];
    [self setupMenu: _subtitle_outlinethicknessMenu withIntList:"freetype-outline-thickness" andSelector:@selector(switchSubtitleOption:)];

    [_voutMenuplay matchKeyEquivalentsOfMenuItem:_play];
    [_voutMenustop matchKeyEquivalentsOfMenuItem:_stop];
    [_voutMenunext matchKeyEquivalentsOfMenuItem:_next];
    [_voutMenuprev matchKeyEquivalentsOfMenuItem:_previous];
    [_voutMenuRecord matchKeyEquivalentsOfMenuItem:_record];
    [_voutMenuvolup matchKeyEquivalentsOfMenuItem:_vol_up];
    [_voutMenuvoldown matchKeyEquivalentsOfMenuItem:_vol_down];
    [_voutMenumute matchKeyEquivalentsOfMenuItem:_mute];
    [_voutMenufullscreen matchKeyEquivalentsOfMenuItem:_fullscreenItem];
    [_voutMenusnapshot matchKeyEquivalentsOfMenuItem:_snapshot];
}

- (void)setupMenu:(NSMenu *)menu withIntList:(char *)psz_name andSelector:(SEL)selector
{
    module_config_t *p_item;

    [menu removeAllItems];
    p_item = config_FindConfig(psz_name);

    if (!p_item) {
        msg_Err(getIntf(), "couldn't create menu int list for item '%s' as it does not exist", psz_name);
        return;
    }

    for (int i = 0; i < p_item->list_count; i++) {
        NSMenuItem *mi;
        if (p_item->list_text != NULL)
            mi = [[NSMenuItem alloc] initWithTitle: NSTR(p_item->list_text[i]) action:NULL keyEquivalent: @""];
        else if (p_item->list.i[i])
            mi = [[NSMenuItem alloc] initWithTitle: [NSString stringWithFormat: @"%d", p_item->list.i[i]] action:NULL keyEquivalent: @""];
        else {
            msg_Err(getIntf(), "item %d of pref %s failed to be created", i, psz_name);
            continue;
        }

        [mi setTarget:self];
        [mi setAction:selector];
        [mi setTag:p_item->list.i[i]];
        [mi setRepresentedObject:toNSStr(psz_name)];
        [menu addItem:mi];
        if (p_item->value.i == p_item->list.i[i])
            [mi setState:NSOnState];
    }
}

- (void)initStrings
{
    /* main menu */
    [_about setTitle: _NS("About VLC media player...")];
    [_checkForUpdate setTitle: _NS("Check for Updates...")];
    [_prefs setTitle: _NS("Preferences...")];
    [_extensions setTitle: _NS("Extensions")];
    [_extensionsMenu setTitle: _NS("Extensions")];
    [_addonManager setTitle: _NS("Addons Manager")];
    [_add_intf setTitle: _NS("Add Interface")];
    [_add_intfMenu setTitle: _NS("Add Interface")];
    [_services setTitle: _NS("Services")];
    [_hide setTitle: _NS("Hide VLC")];
    [_hide_others setTitle: _NS("Hide Others")];
    [_show_all setTitle: _NS("Show All")];
    [_quit setTitle: _NS("Quit VLC")];

    /* this special case is needed to due to archaic legacy translations of the File menu
     * on the Mac to the German translation which resulted in 'Ablage' instead of 'Datei'.
     * This remains until the present day and does not affect the Windows world. */
     /* xgettext: Label for the macOS main "File" menu */
    [_fileMenu setTitle: _PNS("macOS MainMenu", "File")];
    [_open_generic setTitle: _NS("Advanced Open File...")];
    [_open_file setTitle: _NS("Open File...")];
    [_open_disc setTitle: _NS("Open Disc...")];
    [_open_net setTitle: _NS("Open Stream...")];
    [_open_capture setTitle: _NS("Open Capture Device...")];
    [_open_recent setTitle: _NS("Open Recent")];
    [_close_window setTitle: _NS("Close Window")];
    [_convertandsave setTitle: _NS("Convert / Stream...")];
    [_save_playlist setTitle: _NS("Save Playlist...")];
    [_savePlayqueueToLibrary setTitle: _NS("Save Play Queue to Library...")];
    [_revealInFinder setTitle: _NS("Reveal in Finder")];

    [_editMenu setTitle: _NS("Edit")];
    [_cutItem setTitle: _NS("Cut")];
    [_mcopyItem setTitle: _NS("Copy")];
    [_pasteItem setTitle: _NS("Paste")];
    [_clearItem setTitle: _NS("Delete")];
    [_select_all setTitle: _NS("Select All")];
    [_findItem setTitle: _NS("Find")];

    [_viewMenu setTitle: _NS("View")];

    [_controlsMenu setTitle: _NS("Playback")];
    [_play setTitle: _NS("Play")];
    [_stop setTitle: _NS("Stop")];
    [_record setTitle: _NS("Record")];
    [_rate_view setAutoresizingMask:NSViewWidthSizable];
    [_rate setView: _rate_view];
    [_rateLabel setStringValue: _NS("Playback Speed")];
    [_rate_slowerLabel setStringValue: _NS("Slower")];
    [_rate_normalLabel setStringValue: _NS("Normal")];
    [_rate_fasterLabel setStringValue: _NS("Faster")];
    [_trackSynchronization setTitle: _NS("Track Synchronization")];
    [_previous setTitle: _NS("Previous")];
    [_next setTitle: _NS("Next")];
    [_random setTitle: _NS("Random")];
    [_repeat setTitle: _NS("Repeat")];
    [_AtoBloop setTitle: _NS("A→B Loop")];
    [_libraryPlayQueueMode setTitle: _NS("Library Play Queue Mode")];
    [_sortPlayQueue setTitle: _NS("Sort Play Queue")];
    [_quitAfterPB setTitle: _NS("Quit after Playback")];
    [_fwd setTitle: _NS("Step Forward")];
    [_bwd setTitle: _NS("Step Backward")];
    [_jumpToTime setTitle: _NS("Jump to Time")];
    [_rendererMenuItem setTitle:_NS("Renderer")];
    [_rendererNoneItem setTitle:_NS("No renderer")];
    [_program setTitle: _NS("Program")];
    [_programMenu setTitle: _NS("Program")];
    [_title setTitle: _NS("Title")];
    [_titleMenu setTitle: _NS("Title")];
    [_chapter setTitle: _NS("Chapter")];
    [_chapterMenu setTitle: _NS("Chapter")];

    [_audioMenu setTitle: _NS("Audio")];
    [_vol_up setTitle: _NS("Increase Volume")];
    [_vol_down setTitle: _NS("Decrease Volume")];
    [_mute setTitle: _NS("Mute")];
    [_audiotrack setTitle: _NS("Audio Track")];
    [_audiotrackMenu setTitle: _NS("Audio Track")];
    [_channels setTitle: _NS("Stereo audio mode")];
    [_channelsMenu setTitle: _NS("Stereo audio mode")];
    [_audioDevice setTitle: _NS("Audio Device")];
    [_audioDeviceMenu setTitle: _NS("Audio Device")];
    [_visual setTitle: _NS("Visualizations")];
    [_visualMenu setTitle: _NS("Visualizations")];

    [_videoMenu setTitle: _NS("Video")];
    [_half_window setTitle: _NS("Half Size")];
    [_normal_window setTitle: _NS("Normal Size")];
    [_double_window setTitle: _NS("Double Size")];
    [_fittoscreen setTitle: _NS("Fit to Screen")];
    [_fullscreenItem setTitle: _NS("Fullscreen")];
    [_floatontop setTitle: _NS("Float on Top")];
    [_snapshot setTitle: _NS("Snapshot")];
    [_videotrack setTitle: _NS("Video Track")];
    [_videotrackMenu setTitle: _NS("Video Track")];
    [_aspect_ratio setTitle: _NS("Aspect ratio")];
    [_aspect_ratioMenu setTitle: _NS("Aspect ratio")];
    [_crop setTitle: _NS("Crop")];
    [_cropMenu setTitle: _NS("Crop")];
    [_screen setTitle: _NS("Fullscreen Video Device")];
    [_screenMenu setTitle: _NS("Fullscreen Video Device")];
    [_deinterlace setTitle: _NS("Deinterlace")];
    [_deinterlaceMenu setTitle: _NS("Deinterlace")];
    [_deinterlace_mode setTitle: _NS("Deinterlace mode")];
    [_deinterlace_modeMenu setTitle: _NS("Deinterlace mode")];
    [_postprocessing setTitle: _NS("Post processing")];
    [_postprocessingMenu setTitle: _NS("Post processing")];

    [_subtitlesMenu setTitle:_NS("Subtitles")];
    [_openSubtitleFile setTitle: _NS("Add Subtitle File...")];
    [_subtitle_track setTitle: _NS("Subtitles Track")];
    [_subtitle_tracksMenu setTitle: _NS("Subtitles Track")];
    [_subtitleSizeView setAutoresizingMask: NSViewWidthSizable];
    [_subtitleSize setView: _subtitleSizeView];
    [_subtitleSizeLabel setStringValue: _NS("Subtitles Size")];
    [_subtitleSizeSmallerLabel setStringValue: _NS("Smaller")];
    [_subtitleSizeLargerLabel setStringValue: _NS("Larger")];
    [_subtitle_textcolor setTitle: _NS("Text Color")];
    [_subtitle_outlinethickness setTitle: _NS("Outline Thickness")];

    // Autoresizing with constraints does not work on 10.7,
    // translate autoresizing mask to constriaints for now
    [_subtitle_bgopacity_view setAutoresizingMask:NSViewWidthSizable];
    [_subtitle_bgopacity setView: _subtitle_bgopacity_view];
    [_subtitle_bgopacityLabel setStringValue: _NS("Background Opacity")];
    [_subtitle_bgopacityLabel_gray setStringValue: _NS("Background Opacity")];
    [_subtitle_bgcolor setTitle: _NS("Background Color")];
    [_teletext setTitle: _NS("Teletext")];
    [_teletext_transparent setTitle: _NS("Transparent")];
    [_teletext_index setTitle: _NS("Index")];
    [_teletext_red setTitle: _NS("Red")];
    [_teletext_green setTitle: _NS("Green")];
    [_teletext_yellow setTitle: _NS("Yellow")];
    [_teletext_blue setTitle: _NS("Blue")];

    [_windowMenu setTitle: _NS("Window")];
    [_minimize setTitle: _NS("Minimize")];
    [_zoom_window setTitle: _NS("Zoom")];
    [_player setTitle: _NS("Player...")];
    [_controller setTitle: _NS("Main Window...")];
    [_audioeffects setTitle: _NS("Audio Effects...")];
    [_videoeffects setTitle: _NS("Video Effects...")];
    [_bookmarks setTitle: _NS("Bookmarks...")];
    [_playQueue setTitle: _NS("Play Queue...")];
    [_detachedAudioWindow setTitle: _NS("Detached Audio Window...")];
    [_info setTitle: _NS("Media Information...")];
    [_messages setTitle: _NS("Messages...")];
    [_errorsAndWarnings setTitle: _NS("Errors and Warnings...")];

    [_bring_atf setTitle: _NS("Bring All to Front")];

    [_helpMenu setTitle: _NS("Help")];
    [_help setTitle: _NS("VLC media player Help...")];
    [_license setTitle: _NS("License")];
    [_documentation setTitle: _NS("Online Documentation...")];
    [_website setTitle: _NS("VideoLAN Website...")];
    [_donation setTitle: _NS("Make a donation...")];
    [_forum setTitle: _NS("Online Forum...")];

    /* dock menu */
    [_dockMenuplay setTitle: _NS("Play")];
    [_dockMenustop setTitle: _NS("Stop")];
    [_dockMenunext setTitle: _NS("Next")];
    [_dockMenuprevious setTitle: _NS("Previous")];
    [_dockMenumute setTitle: _NS("Mute")];

    /* vout menu */
    [_voutMenuplay setTitle: _NS("Play")];
    [_voutMenustop setTitle: _NS("Stop")];
    [_voutMenuprev setTitle: _NS("Previous")];
    [_voutMenunext setTitle: _NS("Next")];
    [_voutMenuvolup setTitle: _NS("Volume Up")];
    [_voutMenuvoldown setTitle: _NS("Volume Down")];
    [_voutMenumute setTitle: _NS("Mute")];
    [_voutMenuAudiotrack setTitle: _NS("Audio Track")];
    [_voutMenuAudiotrackMenu setTitle: _NS("Audio Track")];
    [_voutMenuVideotrack setTitle: _NS("Video Track")];
    [_voutMenuVideotrackMenu setTitle: _NS("Video Track")];
    [_voutMenuOpenSubtitleFile setTitle:_NS("Add Subtitle File...")];
    [_voutMenuSubtitlestrack setTitle: _NS("Subtitles Track")];
    [_voutMenuSubtitlestrackMenu setTitle: _NS("Subtitles Track")];
    [_voutMenufullscreen setTitle: _NS("Fullscreen")];
    [_voutMenusnapshot setTitle: _NS("Snapshot")];
}

- (void)setupKeyboardShortcuts
{
    char *key;

    key = config_GetPsz("key-quit");
    [_quit setKeyEquivalent: VLCKeyToString(key)];
    [_quit setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    // do not assign play/pause key

    key = config_GetPsz("key-stop");
    [_stop setKeyEquivalent: VLCKeyToString(key)];
    [_stop setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-prev");
    [_previous setKeyEquivalent: VLCKeyToString(key)];
    [_previous setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-next");
    [_next setKeyEquivalent: VLCKeyToString(key)];
    [_next setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-jump+short");
    [_fwd setKeyEquivalent: VLCKeyToString(key)];
    [_fwd setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-jump-short");
    [_bwd setKeyEquivalent: VLCKeyToString(key)];
    [_bwd setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-vol-up");
    [_vol_up setKeyEquivalent: VLCKeyToString(key)];
    [_vol_up setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-vol-down");
    [_vol_down setKeyEquivalent: VLCKeyToString(key)];
    [_vol_down setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-vol-mute");
    [_mute setKeyEquivalent: VLCKeyToString(key)];
    [_mute setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-toggle-fullscreen");
    [_fullscreenItem setKeyEquivalent: VLCKeyToString(key)];
    [_fullscreenItem setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-snapshot");
    [_snapshot setKeyEquivalent: VLCKeyToString(key)];
    [_snapshot setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-random");
    [_random setKeyEquivalent: VLCKeyToString(key)];
    [_random setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-loop");
    [_repeat setKeyEquivalent: VLCKeyToString(key)];
    [_repeat setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-zoom-half");
    [_half_window setKeyEquivalent: VLCKeyToString(key)];
    [_half_window setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-zoom-original");
    [_normal_window setKeyEquivalent: VLCKeyToString(key)];
    [_normal_window setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);

    key = config_GetPsz("key-zoom-double");
    [_double_window setKeyEquivalent: VLCKeyToString(key)];
    [_double_window setKeyEquivalentModifierMask: VLCModifiersToCocoa(key)];
    FREENULL(key);
}

#pragma mark - Termination

- (void)releaseRepresentedObjects:(NSMenu *)the_menu
{
    NSArray *menuitems_array = [the_menu itemArray];
    NSUInteger menuItemCount = [menuitems_array count];
    for (NSUInteger i=0; i < menuItemCount; i++) {
        NSMenuItem *one_item = [menuitems_array objectAtIndex:i];
        if ([one_item hasSubmenu])
            [self releaseRepresentedObjects: [one_item submenu]];

        [one_item setRepresentedObject:NULL];
    }
}

#pragma mark - Interface update

- (void)mediaItemChanged:(NSNotification *)notification
{
    [self updateTrackHandlingMenus:notification];
    [self updateRateControls:notification];

    if (_playerController.currentMedia != nil) {
        [self rebuildAoutMenu];
        [self rebuildVoutMenu];
    } else {
        self.windowMenu.autoenablesItems = NO;
    }
}

- (void)rebuildAoutMenu
{
    audio_output_t *p_aout = [_playerController mainAudioOutput];
    if (!p_aout) {
        return;
    }
    [self setupVarMenuItem:_channels
                    target:VLC_OBJECT(p_aout)
                objectType:VLCObjectTypeAout
                       var:"stereo-mode"
                  selector:@selector(toggleVar:)];

    [self setupVarMenuItem:_visual
                    target:VLC_OBJECT(p_aout)
                objectType:VLCObjectTypeAout
                       var:"visual"
                  selector:@selector(toggleVar:)];
    aout_Release(p_aout);
}

- (void)voutListChanged:(NSNotification *)aNotification
{
    [self rebuildVoutMenu];
}

- (void)rebuildVoutMenu
{
    vout_thread_t *p_vout = [_playerController videoOutputThreadForKeyWindow];
    if (!p_vout) {
        return;
    }

    [self setupVarMenuItem:_aspect_ratio
                    target:VLC_OBJECT(p_vout)
                objectType:VLCObjectTypeVout
                       var:"aspect-ratio"
                  selector:@selector(toggleVar:)];
    [self appendCustomizationItem:_aspect_ratio];

    [self setupVarMenuItem:_crop
                    target:VLC_OBJECT(p_vout)
                objectType:VLCObjectTypeVout
                       var:"crop"
                  selector:@selector(toggleVar:)];
    [self appendCustomizationItem:_crop];

    [self setupVarMenuItem:_deinterlace
                    target:VLC_OBJECT(p_vout)
                objectType:VLCObjectTypeVout
                       var:"deinterlace"
                  selector:@selector(toggleVar:)];

    [self setupVarMenuItem:_deinterlace_mode
                    target:VLC_OBJECT(p_vout)
                objectType:VLCObjectTypeVout
                       var:"deinterlace-mode"
                  selector:@selector(toggleVar:)];

    vout_Release(p_vout);

    [self refreshVoutDeviceMenu:nil];
}

- (void)refreshVoutDeviceMenu:(NSNotification *)notification
{
    NSMenu *submenu = _screenMenu;
    [submenu removeAllItems];

    NSArray *screens = [NSScreen screens];
    NSMenuItem *menuItem;
    NSUInteger numberOfScreens = [screens count];
    [_screen setEnabled: YES];
    [submenu addItemWithTitle: _NS("Default") action:@selector(toggleFullscreenDevice:) keyEquivalent:@""];
    menuItem = [submenu itemAtIndex: 0];
    [menuItem setTag: 0];
    [menuItem setEnabled: YES];
    [menuItem setTarget: self];
    NSRect s_rect;
    for (NSUInteger i = 0; i < numberOfScreens; i++) {
        s_rect = [[screens objectAtIndex:i] frame];
        [submenu addItemWithTitle:[NSString stringWithFormat: @"%@ %li (%ix%i)", _NS("Screen"), i+1, (int)s_rect.size.width, (int)s_rect.size.height]
                           action:@selector(toggleFullscreenDevice:)
                    keyEquivalent:@""];
        menuItem = [submenu itemAtIndex:i+1];
        [menuItem setTag:(int)[[screens objectAtIndex:i] displayID]];
        [menuItem setEnabled: YES];
        [menuItem setTarget: self];
    }
    [[submenu itemWithTag: var_InheritInteger(getIntf(), "macosx-vdev")] setState: NSOnState];
}

- (void)updateSubtitlesMenu:(NSNotification *)notification
{
    const BOOL enabled = [self validateUserInterfaceItem:self.openSubtitleFile];
    self.subtitleSizeSlider.enabled = enabled;

    const unsigned int scaleFactor = _playerController.subtitleTextScalingFactor;
    self.subtitleSizeSlider.intValue = scaleFactor;
    self.subtitleSizeTextField.stringValue =
        [NSString stringWithFormat:@"%.2fx", scaleFactor / 100.];

    NSColor * const color = enabled
        ? NSColor.controlTextColor
        : NSColor.disabledControlTextColor;
    self.subtitleSizeLabel.textColor = color;
    self.subtitleSizeSmallerLabel.textColor = color;
    self.subtitleSizeLargerLabel.textColor = color;
    self.subtitleSizeTextField.textColor = color;

    self.subtitle_bgopacityLabel_gray.hidden = enabled;
    self.subtitle_bgopacityLabel.hidden = !enabled;
    self.subtitle_bgopacity_sld.enabled = enabled;
}

- (void)updateRateControls:(NSNotification *)notification
{
    const BOOL enabled = [self validateUserInterfaceItem:self.rate];
    self.rate_sld.enabled = enabled;

    NSColor * const color = enabled
        ? NSColor.controlTextColor
        : NSColor.disabledControlTextColor;

    self.rateLabel.textColor = color;
    self.rate_slowerLabel.textColor = color;
    self.rate_normalLabel.textColor = color;
    self.rate_fasterLabel.textColor = color;
    self.rateTextField.textColor = color;
}

#pragma mark - View

#pragma mark - Playback

- (IBAction)play:(id)sender
{
    [_playerController togglePlayPause];
}

- (IBAction)stop:(id)sender
{
    [_playerController stop];
}

- (IBAction)prev:(id)sender
{
    [_playQueueController playPreviousItem];
}

- (IBAction)next:(id)sender
{
    [_playQueueController playNextItem];
}

- (IBAction)random:(id)sender
{
    if (_playQueueController.playbackOrder == VLC_PLAYLIST_PLAYBACK_ORDER_RANDOM) {
        _playQueueController.playbackOrder = VLC_PLAYLIST_PLAYBACK_ORDER_NORMAL;
    } else {
        _playQueueController.playbackOrder = VLC_PLAYLIST_PLAYBACK_ORDER_RANDOM;
    }
}

- (IBAction)repeat:(id)sender
{
    if (_playQueueController.playbackRepeat == VLC_PLAYLIST_PLAYBACK_REPEAT_CURRENT) {
        _playQueueController.playbackRepeat = VLC_PLAYLIST_PLAYBACK_REPEAT_ALL;
    } else if (_playQueueController.playbackRepeat == VLC_PLAYLIST_PLAYBACK_REPEAT_ALL) {
        _playQueueController.playbackRepeat = VLC_PLAYLIST_PLAYBACK_REPEAT_NONE;
    } else {
        _playQueueController.playbackRepeat = VLC_PLAYLIST_PLAYBACK_REPEAT_CURRENT;
    }
}

- (IBAction)forward:(id)sender
{
    [_playerController jumpForwardShort];
}

- (IBAction)backward:(id)sender
{
    [_playerController jumpBackwardShort];
}

- (IBAction)volumeUp:(id)sender
{
    [_playerController incrementVolume];
}

- (IBAction)volumeDown:(id)sender
{
    [_playerController decrementVolume];
}

- (IBAction)mute:(id)sender
{
    [_playerController toggleMute];
}

- (void)lockVideosAspectRatio:(id)sender
{
    [_playerController setAspectRatioIsLocked: ![sender state]];
    [sender setState: [_playerController aspectRatioIsLocked]];
}

- (IBAction)quitAfterPlayback:(id)sender
{
    if (_playQueueController.actionAfterStop != VLC_PLAYLIST_MEDIA_STOPPED_EXIT) {
        _playQueueController.actionAfterStop = VLC_PLAYLIST_MEDIA_STOPPED_EXIT;
    } else {
        _playQueueController.actionAfterStop = VLC_PLAYLIST_MEDIA_STOPPED_CONTINUE;
    }
}

- (IBAction)toggleRecord:(id)sender
{
    [_playerController toggleRecord];
}

- (void)updateRecordState
{
    NSControlStateValue state = _playerController.enableRecording ? NSOnState : NSOffState;
    [_record setState:state];
    [_voutMenuRecord setState:state];
}

- (IBAction)setPlaybackRate:(id)sender
{
    double speed =  pow(2, (double)[_rate_sld intValue] / 17);
    _playerController.playbackRate = speed;
    [_rateTextField setStringValue: [NSString stringWithFormat:@"%.2fx", speed]];
}

- (void)updatePlaybackRate
{
    double playbackRate = _playerController.playbackRate;
    double value = 17 * log(playbackRate) / log(2.);
    int intValue = (int) ((value > 0) ? value + .5 : value - .5);

    if (intValue < -34)
        intValue = -34;
    else if (intValue > 34)
        intValue = 34;

    [_rateTextField setStringValue: [NSString stringWithFormat:@"%.2fx", playbackRate]];
    [_rate_sld setIntValue: intValue];
}

- (IBAction)toggleAtoBloop:(id)sender
{
    [_playerController setABLoop];
}

- (IBAction)toggleLibraryPlayQueueMode:(id)sender
{
    _playQueueController.libraryPlayQueueMode = !_playQueueController.libraryPlayQueueMode;
    [self updateLibraryPlayQueueMode];
}

- (void)updateLibraryPlayQueueMode
{
    const BOOL libraryPlayQueueMode = _playQueueController.libraryPlayQueueMode;
    _libraryPlayQueueMode.state = libraryPlayQueueMode ? NSOnState : NSOffState;
}

- (IBAction)goToSpecificTime:(id)sender
{
    if (!_timeSelectionPanel) {
        _timeSelectionPanel = [[VLCTimeSelectionPanelController alloc] init];
    }
    vlc_tick_t length = _playerController.length;
    [_timeSelectionPanel setMaxTime:(int)SEC_FROM_VLC_TICK(length)];
    vlc_tick_t time = _playerController.time;
    [_timeSelectionPanel setPosition:(int)SEC_FROM_VLC_TICK(time)];
    [_timeSelectionPanel runModalForWindow:[NSApp mainWindow]
                         completionHandler:^(NSInteger returnCode, int64_t returnTime) {
                             if (returnCode != NSModalResponseOK)
                                 return;
                             [self->_playerController setTimePrecise:vlc_tick_from_sec(returnTime)];
                         }];
}

- (IBAction)selectRenderer:(id)sender
{
    [_rendererMenuController selectRenderer:sender];
}

#pragma mark - track handling
- (void)updateTrackHandlingMenus:(NSNotification *)notification
{
    NSArray * const audioTracks = _playerController.audioTracks;
    const NSUInteger numberOfAudioTracks = audioTracks.count;
    [self rebuildTracksMenu:self.audiotrackMenu
               withMetadata:audioTracks
                      count:numberOfAudioTracks
                   category:AUDIO_ES];
    [self rebuildTracksMenu:self.voutMenuAudiotrackMenu
               withMetadata:audioTracks
                      count:numberOfAudioTracks
                   category:AUDIO_ES];

    NSArray * const videoTracks = _playerController.videoTracks;
    const NSUInteger numberOfVideoTracks = videoTracks.count;
    [self rebuildTracksMenu:self.videotrackMenu
               withMetadata:videoTracks
                      count:numberOfVideoTracks
                   category:VIDEO_ES];
    [self rebuildTracksMenu:self.voutMenuVideotrackMenu
               withMetadata:videoTracks
                      count:numberOfVideoTracks
                   category:VIDEO_ES];

    NSArray * const subtitleTracks = _playerController.subtitleTracks;
    const NSUInteger numberOfSubtitleTracks = subtitleTracks.count;
    [self rebuildTracksMenu:self.subtitle_tracksMenu
               withMetadata:subtitleTracks
                      count:numberOfSubtitleTracks
                   category:SPU_ES];
    [self rebuildTracksMenu:self.voutMenuSubtitlestrackMenu
               withMetadata:subtitleTracks
                      count:numberOfSubtitleTracks
                   category:SPU_ES];

    [self updateSubtitlesMenu:notification];
}

- (void)rebuildTracksMenu:(NSMenu *)menu
             withMetadata:(NSArray *)metadataArray
                    count:(size_t)count
                 category:(enum es_format_category_e)category
{
    [menu removeAllItems];
    BOOL itemSelected = NO;

    NSMenuItem *menuItem = [[NSMenuItem alloc] initWithTitle:_NS("Disable")
                                                      action:@selector(unselectTrackCategory:)
                                               keyEquivalent:@""];
    [menuItem setTarget:self];
    [menuItem setTag:category];
    [menuItem setEnabled:YES];
    [menu addItem:menuItem];

    for (VLCTrackMetaData *metaDataItem in metadataArray) {
        menuItem = [[NSMenuItem alloc] initWithTitle:metaDataItem.name
                                              action:@selector(selectTrack:)
                                       keyEquivalent:@""];
        [menuItem setTarget:self];
        [menuItem setRepresentedObject:metaDataItem];
        [menuItem setEnabled:YES];
        if (metaDataItem.selected) {
            itemSelected = YES;
            [menuItem setState:NSOnState];
        } else {
            [menuItem setState:NSOffState];
        }
        [menu addItem:menuItem];
    }

    /* select the "Disabled" item in case no track is selected */
    if (!itemSelected) {
        [menu.itemArray.firstObject setState:NSOnState];
    }
}

- (void)selectTrack:(NSMenuItem *)sender
{
    NSEvent *currentEvent = [NSApp currentEvent];
    [_playerController selectTrack:[sender representedObject] exclusively:!(currentEvent.modifierFlags & NSAlternateKeyMask)];
}

- (void)unselectTrackCategory:(NSMenuItem *)sender
{
    [_playerController unselectTracksFromCategory:(enum es_format_category_e)sender.tag];
}

#pragma mark - title and chapter handling
- (void)updateTitleAndChapterMenus:(NSNotification *)aNotification
{
    [_titleMenu removeAllItems];
    [_chapterMenu removeAllItems];

    size_t count = [_playerController numberOfTitlesOfCurrentMedia];
    size_t selectedIndex = [_playerController selectedTitleIndex];
    for (size_t x = 0; x < count; x++) {
        const struct vlc_player_title *p_title = [_playerController titleAtIndexForCurrentMedia:x];
        if (p_title == NULL) {
            break;
        }
        NSMenuItem *menuItem = [[NSMenuItem alloc] initWithTitle:toNSStr(p_title->name)
                                                          action:@selector(selectTitle:)
                                                   keyEquivalent:@""];
        [menuItem setTarget:self];
        [menuItem setTag:x];
        [menuItem setEnabled:YES];
        [menuItem setState: x == selectedIndex ? NSOnState : NSOffState];
        [_titleMenu addItem:menuItem];
    }
    _title.enabled = count > 0 ? YES : NO;

    count = [_playerController numberOfChaptersForCurrentTitle];
    selectedIndex = [_playerController selectedChapterIndex];
    for (size_t x = 0; x < count; x++) {
        const struct vlc_player_chapter *p_chapter = [_playerController chapterAtIndexForCurrentTitle:x];
        if (p_chapter == NULL) {
            break;
        }
        NSMenuItem *menuItem = [[NSMenuItem alloc] initWithTitle:toNSStr(p_chapter->name)
                                                          action:@selector(selectChapter:)
                                                   keyEquivalent:@""];
        [menuItem setTarget:self];
        [menuItem setTag:x];
        [menuItem setEnabled:YES];
        [menuItem setState: x == selectedIndex ? NSOnState : NSOffState];
        [_chapterMenu addItem:menuItem];
    }
    _chapter.enabled = count > 0 ? YES : NO;
}

- (void)selectTitle:(NSMenuItem *)sender
{
    _playerController.selectedTitleIndex = [sender tag];
}

- (void)selectChapter:(NSMenuItem *)sender
{
    _playerController.selectedChapterIndex = [sender tag];
}

#pragma mark - program handling
- (void)updateProgramMenu:(NSNotification *)notification
{
    [_programMenu removeAllItems];

    size_t count = [_playerController numberOfPrograms];
    for (size_t x = 0; x < count; x++) {
        VLCProgramMetaData *program = [_playerController programAtIndex:x];
        if (program == nil) {
            break;
        }
        NSMenuItem *menuItem = [[NSMenuItem alloc] initWithTitle:program.name
                                                          action:@selector(selectProgram:)
                                                   keyEquivalent:@""];
        [menuItem setTarget:self];
        [menuItem setRepresentedObject:program];
        [menuItem setEnabled:YES];
        [menuItem setState:program.selected ? NSOnState : NSOffState];
        [_programMenu addItem:menuItem];
    }
    _program.enabled = count > 0 ? YES : NO;
}

- (void)selectProgram:(NSMenuItem *)sender
{
    [_playerController selectProgram:[sender representedObject]];
}

#pragma mark - audio menu

- (void)refreshAudioDeviceList
{
    char **ids, **names;
    char *currentDevice;

    [_audioDeviceMenu removeAllItems];

    audio_output_t *p_aout = [_playerController mainAudioOutput];
    if (!p_aout)
        return;

    int numberOfAudioDevices = aout_DevicesList(p_aout, &ids, &names);
    if (numberOfAudioDevices < 0) {
        aout_Release(p_aout);
        return;
    }

    currentDevice = aout_DeviceGet(p_aout);
    NSMenuItem *_tmp;

    for (NSUInteger x = 0; x < (NSUInteger)numberOfAudioDevices; x++) {
        _tmp = [_audioDeviceMenu addItemWithTitle:toNSStr(names[x]) action:@selector(toggleAudioDevice:) keyEquivalent:@""];
        [_tmp setTarget:self];
        [_tmp setTag:[[NSString stringWithFormat:@"%s", ids[x]] intValue]];
    }
    aout_Release(p_aout);

    [[_audioDeviceMenu itemWithTag:[[NSString stringWithFormat:@"%s", currentDevice] intValue]] setState:NSOnState];

    free(currentDevice);

    for (NSUInteger x = 0; x < (NSUInteger)numberOfAudioDevices; x++) {
        free(ids[x]);
        free(names[x]);
    }
    free(ids);
    free(names);

    [_audioDeviceMenu setAutoenablesItems:YES];
    [_audioDevice setEnabled:YES];
}

- (void)toggleAudioDevice:(id)sender
{
    audio_output_t *p_aout = [_playerController mainAudioOutput];
    if (!p_aout)
        return;

    int returnValue = 0;

    if ([sender tag] > 0)
        returnValue = aout_DeviceSet(p_aout, [[NSString stringWithFormat:@"%li", [sender tag]] UTF8String]);
    else
        returnValue = aout_DeviceSet(p_aout, NULL);

    if (returnValue != 0)
        msg_Warn(getIntf(), "failed to set audio device %li", [sender tag]);

    aout_Release(p_aout);
    [self refreshAudioDeviceList];
}

#pragma mark - video menu

- (IBAction)toggleFullscreen:(id)sender
{
    [_playerController toggleFullscreen];
}

- (IBAction)resizeVideoWindow:(id)sender
{
    vout_thread_t *p_vout = [_playerController videoOutputThreadForKeyWindow];
    if (p_vout) {
        if (sender == _half_window)
            var_SetFloat(p_vout, "zoom", 0.5);
        else if (sender == _normal_window)
            var_SetFloat(p_vout, "zoom", 1.0);
        else if (sender == _double_window)
            var_SetFloat(p_vout, "zoom", 2.0);
        else
        {
            [[NSApp keyWindow] performZoom:sender];
        }
        vout_Release(p_vout);
    }
}

- (IBAction)floatOnTop:(id)sender
{
    vout_thread_t *p_vout = [_playerController videoOutputThreadForKeyWindow];
    if (p_vout) {
        var_ToggleBool(p_vout, "video-on-top");
        vout_Release(p_vout);
    }
}

- (IBAction)createVideoSnapshot:(id)sender
{
    [_playerController takeSnapshot];
}

- (void)_disablePostProcessing
{
    // FIXME re-write using VLCPlayerController
    [VLCVideoFilterHelper setVideoFilter:"postproc" on:false];
}

- (void)_enablePostProcessing
{
    // FIXME re-write using VLCPlayerController
    [VLCVideoFilterHelper setVideoFilter:"postproc" on:true];
}

- (void)togglePostProcessing:(id)sender
{
    // FIXME re-write using VLCPlayerController
    NSInteger count = [_postprocessingMenu numberOfItems];
    for (NSUInteger x = 0; x < (NSUInteger)count; x++)
        [[_postprocessingMenu itemAtIndex:x] setState:NSOffState];

    if ([sender tag] == -1) {
        [self _disablePostProcessing];
        [sender setState:NSOnState];
    } else {
        [self _enablePostProcessing];
        [sender setState:NSOnState];

        [VLCVideoFilterHelper setVideoFilterProperty:"postproc-q" forFilter:"postproc" withValue:(vlc_value_t){ .i_int = [sender tag] }];
    }
}

- (void)toggleFullscreenDevice:(id)sender
{
    config_PutInt("macosx-vdev", [sender tag]);
    [self refreshVoutDeviceMenu: nil];
}

#pragma mark - Subtitles Menu

- (IBAction)addSubtitleFile:(id)sender
{
    NSInteger i_returnValue = 0;

    NSOpenPanel *openPanel = [NSOpenPanel openPanel];
    [openPanel setCanChooseFiles: YES];
    [openPanel setCanChooseDirectories: NO];
    [openPanel setAllowsMultipleSelection: YES];

    NSMutableString *subtitleExtensionsString = [toNSStr(EXTENSIONS_SUBTITLE) mutableCopy];
    [subtitleExtensionsString replaceOccurrencesOfString:@"*." withString:@"" options:NSLiteralSearch range:NSMakeRange(0, subtitleExtensionsString.length)];
    [openPanel setAllowedFileTypes:[subtitleExtensionsString componentsSeparatedByString:@";"]];

    NSURL *url = _playerController.URLOfCurrentMediaItem;
    url = [url URLByDeletingLastPathComponent];
    [openPanel setDirectoryURL: url];

    i_returnValue = [openPanel runModal];

    if (i_returnValue == NSModalResponseOK) {
        for (NSURL *url in [openPanel URLs]) {
            [_playerController addAssociatedMediaToCurrentFromURL:url
                                                       ofCategory:SPU_ES
                                                 shallSelectTrack:YES
                                                  shallDisplayOSD:YES
                                             shallVerifyExtension:NO];
        }
    }
}

- (IBAction)subtitleSize:(id)sender
{
    unsigned int scaleFactor = _subtitleSizeSlider.intValue;
    _playerController.subtitleTextScalingFactor = scaleFactor;
    [_subtitleSizeTextField setStringValue: [NSString stringWithFormat:@"%.2fx", scaleFactor / 100.]];
}

- (void)switchSubtitleOption:(id)sender
{
    NSMenuItem * const menuItem = (NSMenuItem *)sender;
    if (menuItem == nil) {
        return;
    }

    const NSInteger intValue = menuItem.tag;
    NSString * const representedObject = menuItem.representedObject;

    config_PutInt(representedObject.UTF8String, intValue);

    NSMenu * const menu = menuItem.menu;
    const NSUInteger count = (NSUInteger)menu.numberOfItems;
    for (NSUInteger x = 0; x < count; x++) {
        [menu itemAtIndex:x].state = NSOffState;
    }
    [menu itemWithTag:intValue].state = NSOnState;
}

- (IBAction)switchSubtitleBackgroundOpacity:(id)sender
{
    config_PutInt("freetype-background-opacity", [sender intValue]);
}

- (IBAction)telxTransparent:(id)sender
{
    _playerController.teletextTransparent = !_playerController.teletextTransparent;
}

- (IBAction)telxNavLink:(id)sender
{
    unsigned int page = 0;

    if ([[sender title] isEqualToString: _NS("Index")])
        page = VLC_PLAYER_TELETEXT_KEY_INDEX;
    else if ([[sender title] isEqualToString: _NS("Red")])
        page = VLC_PLAYER_TELETEXT_KEY_RED;
    else if ([[sender title] isEqualToString: _NS("Green")])
        page = VLC_PLAYER_TELETEXT_KEY_GREEN;
    else if ([[sender title] isEqualToString: _NS("Yellow")])
        page = VLC_PLAYER_TELETEXT_KEY_YELLOW;
    else if ([[sender title] isEqualToString: _NS("Blue")])
        page = VLC_PLAYER_TELETEXT_KEY_BLUE;

    _playerController.teletextPage = page;
}

#pragma mark - Panels

- (IBAction)intfOpenFile:(id)sender
{
    [[VLCMain.sharedInstance open] openFileWithAction:^(NSArray *files) {
        [self->_playQueueController addPlayQueueItems:files];
    }];
}

- (IBAction)intfOpenFileGeneric:(id)sender
{
    [[VLCMain.sharedInstance open] openFileGeneric];
}

- (IBAction)intfOpenDisc:(id)sender
{
    [[VLCMain.sharedInstance open] openDisc];
}

- (IBAction)intfOpenNet:(id)sender
{
    [[VLCMain.sharedInstance open] openNet];
}

- (IBAction)intfOpenCapture:(id)sender
{
    [[VLCMain.sharedInstance open] openCapture];
}

- (IBAction)savePlaylist:(id)sender
{
    static dispatch_once_t once;
    dispatch_once(&once, ^{
        [[NSBundle mainBundle] loadNibNamed:@"PlaylistAccessoryView" owner:self topLevelObjects:nil];
    });

    [_playlistSaveAccessoryText setStringValue: _NS("File Format:")];
    [_playlistSaveAccessoryPopup removeAllItems];

    NSArray *availableExportModules = _playQueueController.availablePlaylistExportModules;
    NSUInteger count = availableExportModules.count;
    NSMutableArray *allowedFileTypes = [NSMutableArray arrayWithCapacity:count];
    for (NSUInteger x = 0; x < count; x++) {
        VLCPlaylistExportModuleDescription *exportModule = availableExportModules[x];
        [_playlistSaveAccessoryPopup addItemWithTitle:exportModule.humanReadableName];
        [allowedFileTypes addObject:exportModule.fileExtension];
    }

    NSSavePanel *savePanel = [NSSavePanel savePanel];
    [savePanel setTitle: _NS("Save Playlist")];
    [savePanel setPrompt: _NS("Save")];
    [savePanel setAccessoryView: _playlistSaveAccessoryView];
    [savePanel setNameFieldStringValue: _NS("Untitled")];
    [savePanel setAllowedFileTypes:allowedFileTypes];
    [savePanel setCanSelectHiddenExtension:YES];

    if ([savePanel runModal] == NSFileHandlingPanelOKButton) {
        NSString *filename = [[savePanel URL] path];
        VLCPlaylistExportModuleDescription *exportModule = availableExportModules[[_playlistSaveAccessoryPopup indexOfSelectedItem]];

        if ([[filename pathExtension] caseInsensitiveCompare:exportModule.fileExtension] != NSOrderedSame) {
            filename = [filename stringByAppendingPathExtension:exportModule.fileExtension];
        }

        [_playQueueController exportPlaylistToPath:filename exportModule:exportModule];
    }
}

- (IBAction)savePlayQueueToLibrary:(id)sender
{
    // Check if there are items in the play queue
    if (_playQueueController.playQueueModel.numberOfPlayQueueItems == 0) {
        NSAlert * const alert = [[NSAlert alloc] init];
        alert.messageText = _NS("Play Queue is Empty");
        alert.informativeText = _NS("There are no items in the play queue to save.");
        alert.alertStyle = NSAlertStyleWarning;
        [alert runModal];
        return;
    }
    
    [VLCMain.sharedInstance.libraryController showCreatePlaylistDialogForPlayQueue];
}

- (IBAction)showConvertAndSave:(id)sender
{
    [[VLCMain.sharedInstance convertAndSaveWindow] showWindow:self];
}

- (IBAction)showVideoEffects:(id)sender
{
    [[VLCMain.sharedInstance videoEffectsPanel] toggleWindow:sender];
}

- (IBAction)showTrackSynchronization:(id)sender
{
    [[VLCMain.sharedInstance trackSyncPanel] toggleWindow:sender];
}

- (IBAction)showAudioEffects:(id)sender
{
    [[VLCMain.sharedInstance audioEffectsPanel] toggleWindow:sender];
}

- (IBAction)showBookmarks:(id)sender
{
    [[VLCMain.sharedInstance bookmarks] toggleWindow:sender];
}

- (IBAction)showPreferences:(id)sender
{
    VLCMain *mainInstance = VLCMain.sharedInstance;
    NSInteger i_level = [[mainInstance voutProvider] currentStatusWindowLevel];
    [[mainInstance simplePreferences] showSimplePrefsWithLevel:i_level];
}

- (IBAction)openAddonManager:(id)sender
{
    if (!_addonsController)
        _addonsController = [[VLCAddonsWindowController alloc] init];

    [_addonsController showWindow:self];
}

- (IBAction)showErrorsAndWarnings:(id)sender
{
    [[[VLCMain.sharedInstance coreDialogProvider] errorPanel] showWindow:self];
}

- (IBAction)showMessagesPanel:(id)showMessagesPanel
{
    [[VLCMain.sharedInstance debugMsgPanel] showWindow:self];
}

- (IBAction)showMainWindow:(id)sender
{
    [VLCMain.sharedInstance.libraryWindow makeKeyAndOrderFront:sender];
}

- (IBAction)showPlayQueue:(id)sender
{
    [VLCMain.sharedInstance.libraryWindowController.window makeKeyAndOrderFront:sender];
    [VLCMain.sharedInstance.libraryWindow.splitViewController toggleMultifunctionSidebar:self];
}

#pragma mark - Help and Docs

- (IBAction)showAbout:(id)sender
{
    if (!_aboutWindowController)
        _aboutWindowController = [[VLCAboutWindowController alloc] init];

    [_aboutWindowController showAbout];
}

- (IBAction)showLicense:(id)sender
{
    if (!_aboutWindowController)
        _aboutWindowController = [[VLCAboutWindowController alloc] init];

    [_aboutWindowController showGPL];
}

- (IBAction)showHelp:(id)sender
{
    if (!_helpWindowController)
        _helpWindowController = [[VLCHelpWindowController alloc] init];

    [_helpWindowController showHelp];
}

- (IBAction)openDocumentation:(id)sender
{
    NSURL *url = [NSURL URLWithString: @"https://www.videolan.org/doc/"];

    [NSWorkspace.sharedWorkspace openURL: url];
}

- (IBAction)openWebsite:(id)sender
{
    NSURL *url = [NSURL URLWithString: @"https://www.videolan.org/"];

    [NSWorkspace.sharedWorkspace openURL: url];
}

- (IBAction)openForum:(id)sender
{
    NSURL *url = [NSURL URLWithString: @"https://forum.videolan.org/"];

    [NSWorkspace.sharedWorkspace openURL: url];
}

- (IBAction)openDonate:(id)sender
{
    NSURL *url = [NSURL URLWithString: @"https://www.videolan.org/contribute.html#paypal"];

    [NSWorkspace.sharedWorkspace openURL: url];
}

- (IBAction)showDetachedAudioWindow:(id)sender
{
    [VLCMain.sharedInstance.detachedAudioWindow makeKeyAndOrderFront:self];
}

- (IBAction)showInformationPanel:(id)sender
{
    if (!_infoWindowController) {
        _infoWindowController = [[VLCInformationWindowController alloc] init];
        _infoWindowController.mainMenuInstance = YES;
    }

    _infoWindowController.representedInputItems = @[_playQueueController.currentlyPlayingInputItem];
    [_infoWindowController toggleWindow:sender];
}

#pragma mark - playback state

- (void)playbackStateChanged:(NSNotification *)aNotification
{
    switch (_playerController.playerState) {
        case VLC_PLAYER_STATE_PLAYING:
            [self setPause];
            break;
        default:
            [self setPlay];
            break;
    }
}

- (void)playModeChanged:(NSNotification *)aNotification
{
    enum vlc_playlist_playback_repeat repeatState = _playQueueController.playbackRepeat;
    switch (repeatState) {
        case VLC_PLAYLIST_PLAYBACK_REPEAT_ALL:
            [self setRepeatAll];
            break;

        case VLC_PLAYLIST_PLAYBACK_REPEAT_CURRENT:
            [self setRepeatOne];
            break;

        default:
            [self setRepeatOff];
            break;
    }
}

- (void)playOrderChanged:(NSNotification *)aNotification
{
    [_random setState:_playQueueController.playbackOrder == VLC_PLAYLIST_PLAYBACK_ORDER_RANDOM];
}

- (void)setPlay
{
    [_play setTitle: _NS("Play")];
    [_dockMenuplay setTitle: _NS("Play")];
    [_voutMenuplay setTitle: _NS("Play")];
}

- (void)setPause
{
    [_play setTitle: _NS("Pause")];
    [_dockMenuplay setTitle: _NS("Pause")];
    [_voutMenuplay setTitle: _NS("Pause")];
}

- (void)setRepeatOne
{
    [_repeat setState: NSOnState];
    [_repeat setTitle: _NS("Repeat One")];
}

- (void)setRepeatAll
{
    [_repeat setState: NSOnState];
    [_repeat setTitle: _NS("Repeat All")];
}

- (void)setRepeatOff
{
    [_repeat setState: NSOffState];
    [_repeat setTitle: _NS("Repeat")];
}

#pragma mark - Dynamic menu creation and validation

- (void)setupVarMenuItem:(NSMenuItem *)menuItem
                  target:(vlc_object_t *)p_object
              objectType:(VLCObjectType)objectType
                     var:(const char *)psz_variable
                selector:(SEL)pf_callback
{
    vlc_value_t val;
    char *text;
    int i_type = var_Type(p_object, psz_variable);

    switch(i_type & VLC_VAR_TYPE) {
        case VLC_VAR_VOID:
        case VLC_VAR_BOOL:
        case VLC_VAR_STRING:
        case VLC_VAR_INTEGER:
            break;
        default:
            /* Variable doesn't exist or isn't handled */
            msg_Warn(p_object, "variable %s doesn't exist or isn't handled", psz_variable);
            return;
    }

    /* Get the descriptive name of the variable */
    var_Change(p_object, psz_variable, VLC_VAR_GETTEXT, &text);
    [menuItem setTitle: NSTR(text ? text : psz_variable)];

    if (i_type & VLC_VAR_HASCHOICE) {
        NSMenu *menu = [menuItem submenu];

        [self setupVarMenu:menu
               forMenuItem:menuItem
                    target:p_object
                objectType:objectType
                       var:psz_variable
                  selector:pf_callback];

        free(text);
        return;
    }

    if (var_Get(p_object, psz_variable, &val) < 0) {
        return;
    }

    VLCAutoGeneratedMenuContent *data;
    switch(i_type & VLC_VAR_TYPE) {
        case VLC_VAR_VOID:
            data = [[VLCAutoGeneratedMenuContent alloc] initWithVariableName:psz_variable
                                                                    ofObject:p_object
                                                              withObjectType:objectType
                                                                    andValue:val
                                                              ofVariableType:i_type];
            [menuItem setRepresentedObject:data];
            break;

        case VLC_VAR_BOOL:
            data = [[VLCAutoGeneratedMenuContent alloc] initWithVariableName:psz_variable
                                                                    ofObject:p_object
                                                              withObjectType:objectType
                                                                    andValue:val
                                                              ofVariableType:i_type];
            [menuItem setRepresentedObject:data];
            if (!(i_type & VLC_VAR_ISCOMMAND))
                [menuItem setState:val.b_bool ? NSOnState : NSOffState];
            break;

        default:
            break;
    }

    if ((i_type & VLC_VAR_TYPE) == VLC_VAR_STRING)
        free(val.psz_string);
    free(text);
}

- (void)setupVarMenu:(NSMenu *)menu
         forMenuItem:(NSMenuItem *)parent
              target:(vlc_object_t *)p_object
          objectType:(VLCObjectType)objectType
                 var:(const char *)psz_variable
            selector:(SEL)pf_callback
{
    vlc_value_t val;
    vlc_value_t *val_list;
    char **text_list;
    size_t count, i;
    int i_type;

    /* remove previous items */
    [menu removeAllItems];

    /* we disable everything here, and enable it again when needed, below */
    [parent setEnabled:NO];

    /* Aspect Ratio */
    if ([[parent title] isEqualToString:_NS("Aspect ratio")] == YES) {
        NSMenuItem *lmi_tmp2;
        lmi_tmp2 = [menu addItemWithTitle:_NS("Lock Aspect Ratio")
                                   action:@selector(lockVideosAspectRatio:)
                            keyEquivalent:@""];
        [lmi_tmp2 setTarget: self];
        [lmi_tmp2 setEnabled: YES];
        [lmi_tmp2 setState: [_playerController aspectRatioIsLocked]];
        [parent setEnabled: YES];
        [menu addItem: [NSMenuItem separatorItem]];
    }

    /* Check the type of the object variable */
    i_type = var_Type(p_object, psz_variable);

    /* Make sure we want to display the variable */
    if (i_type & VLC_VAR_HASCHOICE) {
        size_t count;

        var_Change(p_object, psz_variable, VLC_VAR_CHOICESCOUNT, &count);
        if (count <= 1)
            return;
    }
    else
        return;

    switch(i_type & VLC_VAR_TYPE) {
        case VLC_VAR_VOID:
        case VLC_VAR_BOOL:
        case VLC_VAR_STRING:
        case VLC_VAR_INTEGER:
            break;
        default:
            /* Variable doesn't exist or isn't handled */
            return;
    }

    if (var_Get(p_object, psz_variable, &val) < 0) {
        return;
    }

    if (var_Change(p_object, psz_variable, VLC_VAR_GETCHOICES,
                   &count, &val_list, &text_list) < 0) {
        if ((i_type & VLC_VAR_TYPE) == VLC_VAR_STRING)
            free(val.psz_string);
        return;
    }

    /* make (un)sensitive */
    [parent setEnabled:(count > 1)];

    for (i = 0; i < count; i++) {
        NSMenuItem *lmi;
        NSString *title = @"";
        VLCAutoGeneratedMenuContent *data;

        switch(i_type & VLC_VAR_TYPE) {
            case VLC_VAR_STRING:

                title = NSTR(text_list[i] ? text_list[i] : val_list[i].psz_string);

                lmi = [menu addItemWithTitle:title action:pf_callback keyEquivalent:@""];
                data = [[VLCAutoGeneratedMenuContent alloc] initWithVariableName:psz_variable
                                                                        ofObject:p_object
                                                                  withObjectType:objectType
                                                                        andValue:val_list[i]
                                                                  ofVariableType:i_type];
                [lmi setRepresentedObject:data];
                [lmi setTarget:self];

                if (!strcmp(val.psz_string, val_list[i].psz_string) && !(i_type & VLC_VAR_ISCOMMAND))
                    [lmi setState:NSOnState];

                free(text_list[i]);
                free(val_list[i].psz_string);
                break;

            case VLC_VAR_INTEGER:

                title = text_list[i] ?
                NSTR(text_list[i]) : [NSString stringWithFormat:@"%"PRId64, val_list[i].i_int];

                lmi = [menu addItemWithTitle: title action: pf_callback keyEquivalent: @""];
                data = [[VLCAutoGeneratedMenuContent alloc] initWithVariableName:psz_variable
                                                                        ofObject:p_object
                                                                  withObjectType:objectType
                                                                        andValue:val_list[i]
                                                                  ofVariableType:i_type];
                [lmi setRepresentedObject:data];
                [lmi setTarget:self];

                if (val_list[i].i_int == val.i_int && !(i_type & VLC_VAR_ISCOMMAND))
                    [lmi setState:NSOnState];

                free(text_list[i]);
                break;

            default:
                break;
        }
    }

    /* clean up everything */
    if ((i_type & VLC_VAR_TYPE) == VLC_VAR_STRING) free(val.psz_string);
    free(text_list);
    free(val_list);
}

- (void)toggleVar:(id)sender
{
    NSMenuItem *mi = (NSMenuItem *)sender;
    VLCAutoGeneratedMenuContent *data = [mi representedObject];
    [NSThread detachNewThreadSelector:@selector(toggleVarThread:)
                             toTarget:self
                           withObject:data];

    return;
}

- (void)toggleVarThread:(id)data
{
    @autoreleasepool {
        vlc_object_t *p_object;

        assert([data isKindOfClass:[VLCAutoGeneratedMenuContent class]]);
        VLCAutoGeneratedMenuContent *menuContent = (VLCAutoGeneratedMenuContent *)data;

        p_object = [menuContent vlcObject];
        var_Set(p_object, [menuContent variableName], [menuContent variableValue]);

        if ([menuContent objectType] == VLCObjectTypeAout) {
            dispatch_async(dispatch_get_main_queue(), ^{
                [self rebuildAoutMenu];
            });
        }
    }
}

#pragma mark - crop and AR customization

- (void)appendCustomizationItem:(NSMenuItem *)menuItem
{
    NSMenu *submenu = menuItem.submenu;
    NSMenuItem *customizationItem = [[NSMenuItem alloc] initWithTitle:_NS("Custom")
                                                               action:menuItem == _aspect_ratio ? @selector(performCustomAspectRatio:) : @selector(performCustomCrop:)
                                                        keyEquivalent:@""];
    [customizationItem setTarget:self];
    [submenu addItem:[NSMenuItem separatorItem]];
    [submenu addItem:customizationItem];
}

- (void)performCustomCropOrAspectRatioActionWithVariable:(const char *)variable andl10nString:(NSString *)l10nString
{
    if (!_customARController) {
        _customARController = [[VLCCustomCropArWindowController alloc] init];
    }
    _customARController.title = l10nString;
    [_customARController runModalForWindow:[NSApp mainWindow]
                         completionHandler:^(NSInteger returnCode, NSString * _Nonnull geometry) {
                             if (returnCode != NSModalResponseOK) {
                                 return;
                             }

                             vout_thread_t *p_vout = [self->_playerController videoOutputThreadForKeyWindow];
                             if (p_vout) {
                                 var_SetString(p_vout, variable, [geometry UTF8String]);
                                 vout_Release(p_vout);
                             }
                         }];
}

- (void)performCustomAspectRatio:(id)sender
{
    [self performCustomCropOrAspectRatioActionWithVariable:"aspect-ratio" andl10nString:_NS("Aspect Ratio")];
}

- (void)performCustomCrop:(id)sender
{
    [self performCustomCropOrAspectRatioActionWithVariable:"crop" andl10nString:_NS("Crop")];
}

#pragma mark - menu delegation

- (void)menuWillOpen:(NSMenu *)menu
{
    [_rendererMenuController startRendererDiscoveries];

    if (@available(macOS 10.16, *)) {

        const int menuItemOffset = 14;
        const int menuItemOffsetWithActiveState = 24;

        if (menu == _controlsMenu) {
            BOOL controlsMenuHasActiveState = NO;
            for (NSMenuItem * const viewMenuItem in menu.itemArray) {
                if (viewMenuItem.state == NSControlStateValueOn) {
                    controlsMenuHasActiveState = YES;
                }
            }

            if (controlsMenuHasActiveState) {
                _rate_view_offset_constraint.constant = menuItemOffsetWithActiveState;
            } else {
                _rate_view_offset_constraint.constant = menuItemOffset;
            }
        }

        if (menu == _subtitlesMenu) {
            _subtitle_bgopacity_view_offset_constraint.constant = menuItemOffset;
            _subtitleSizeViewOffsetConstraint.constant = menuItemOffset;
        }
    }
}

- (BOOL)validateUserInterfaceItem:(id<NSValidatedUserInterfaceItem>)item
{
    NSMenuItem * const mi = (NSMenuItem *)item;

    if (mi == nil) {
        return YES;
    } else if (mi == self.savePlayqueueToLibrary) {
        return _playQueueController.playQueueModel.numberOfPlayQueueItems > 0;
    } else if (mi == self.play) {
        return _playerController.playerState == VLC_PLAYER_STATE_PLAYING
            ? _playerController.currentMedia != nil && _playerController.pausable
            : _playQueueController.playQueueModel.numberOfPlayQueueItems > 0;
    } else if (mi == self.stop ||
               mi == self.voutMenustop ||
               mi == self.dockMenustop ||
               mi == self.visual ||
               mi == self.channels) {
        return _playerController.currentMedia &&
               _playerController.playerState != VLC_PLAYER_STATE_STOPPED;
    } else if (mi == self.previous || mi == self.voutMenuprev || mi == self.dockMenuprevious) {
        return _playQueueController.hasPreviousPlayQueueItem;
    } else if (mi == self.next || mi == self.voutMenunext || mi == self.dockMenunext) {
        return _playQueueController.hasNextPlayQueueItem;
    } else if (mi == self.record || mi == self.voutMenuRecord) {
        return _playerController.currentMedia != nil && _playerController.recordable;
    } else if (mi == self.random) {
        const enum vlc_playlist_playback_order playbackOrder = _playQueueController.playbackOrder;
        mi.state = playbackOrder == VLC_PLAYLIST_PLAYBACK_ORDER_RANDOM ? NSOnState : NSOffState;
    } else if (mi == self.quitAfterPB) {
        const BOOL state = _playQueueController.actionAfterStop == VLC_PLAYLIST_MEDIA_STOPPED_EXIT;
        mi.state = state ? NSOnState : NSOffState;
    } else if (mi == self.fwd || mi == self.bwd || mi == self.jumpToTime) {
        return _playerController.currentMedia != nil && _playerController.seekable;
    } else if (mi == self.mute || mi == self.dockMenumute || mi == self.voutMenumute) {
        mi.state = _playerController.mute ? NSOnState : NSOffState;
        [self refreshAudioDeviceList];
    } else if (mi == self.half_window               ||
               mi == self.normal_window             ||
               mi == self.double_window             ||
               mi == self.fittoscreen               ||
               mi == self.snapshot                  ||
               mi == self.voutMenusnapshot          ||
               mi == self.deinterlace               ||
               mi == self.deinterlace_mode          ||
               mi == self.screen                    ||
               mi == self.aspect_ratio              ||
               mi == self.crop                      ||
               mi == self.postprocessing            ||
               mi == self.fullscreenItem            ||
               mi == self.voutMenufullscreen        ||
               mi == self.floatontop) {
        vout_thread_t * const p_vout = _playerController.videoOutputThreadForKeyWindow;
        if (p_vout != NULL) {
            if (mi == self.floatontop) {
                mi.state = var_GetBool(p_vout, "video-on-top") ? NSOnState : NSOffState;
            } else if (mi == self.fullscreenItem || mi == self.voutMenufullscreen) {
                mi.state = _playerController.fullscreen ? NSOnState : NSOffState;
            }
            vout_Release(p_vout);
        }
        return p_vout != NULL &&
               _playerController.currentMedia != nil &&
               _playerController.activeVideoPlayback;
    } else if (mi == self.rate) {
        return _playerController.currentMedia && _playerController.rateChangable;
    } else if (mi == self.info) {
        return _playerController.currentMedia != nil;
    } else if (mi == self.openSubtitleFile          ||
               mi == self.voutMenuOpenSubtitleFile  ||
               mi == self.subtitleSize) {
        return _playerController.currentMedia != nil && _playerController.activeVideoPlayback;
    } else if (mi == self.teletext) {
        return _playerController.currentMedia != nil && _playerController.teletextMenuAvailable;
    } else if (mi == self.voutMenuAudiotrack) {
        return _playerController.audioTracks.count > 0;
    } else if (mi == self.voutMenuVideotrack) {
        return _playerController.videoTracks.count > 0;
    } else if (mi == self.voutMenuSubtitlestrack) {
        return _playerController.subtitleTracks.count > 0;
    } else {
        NSMenuItem * const parent = mi.parentItem;
        if (parent == self.subtitle_textcolor || mi == self.subtitle_textcolor ||
            parent == self.subtitle_bgcolor || mi == self.subtitle_bgcolor     ||
            parent == self.subtitle_bgopacity || mi == self.subtitle_bgopacity ||
            parent == self.subtitle_outlinethickness || mi == self.subtitle_outlinethickness) {

            return [self validateUserInterfaceItem:self.openSubtitleFile];
        } else if (parent == self.teletext || mi == self.teletext) {
            return _playerController.teletextMenuAvailable;
        }
    }

    return YES;
}

@end

/*****************************************************************************
 *VLCAutoGeneratedMenuContent implementation
 *****************************************************************************
 *Object connected to a playlistitem which remembers the data belonging to
 *the variable of the autogenerated menu
 *****************************************************************************/

@implementation VLCAutoGeneratedMenuContent

- (instancetype)initWithVariableName:(const char *)name
                            ofObject:(vlc_object_t *)object
                      withObjectType:(VLCObjectType)objectType
                            andValue:(vlc_value_t)value
                      ofVariableType:(int)type
{
    self = [super init];

    if (self != nil) {
        switch (objectType) {
            case VLCObjectTypeAout:
                aout_Hold((audio_output_t *)object);
                break;
            case VLCObjectTypeVout:
                vout_Hold((vout_thread_t *)object);
                break;

            default:
                // No need to retain the interface because it will always be there
                break;
        }
        _vlcObject = object;
        _objectType = objectType;
        _variableName = strdup(name);
        _variableType = type;
        _variableValue = value;
        if ((type & VLC_VAR_TYPE) == VLC_VAR_STRING) {
            _variableValue.psz_string = strdup(value.psz_string);
        }
    }

    return(self);
}

- (void)dealloc
{
    if (_vlcObject) {
        switch (_objectType) {
            case VLCObjectTypeAout:
                aout_Release((audio_output_t *)_vlcObject);
                break;
            case VLCObjectTypeVout:
                vout_Release((vout_thread_t *)_vlcObject);
                break;

            default:
                // the interface will be released by the core shortly
                break;
        }
    }
    if ((_variableType & VLC_VAR_TYPE) == VLC_VAR_STRING) {
        free(_variableValue.psz_string);
    }
    free(_variableName);
}

@end

@implementation NSMenuItem (KeyEquivalentAddition)

- (void)matchKeyEquivalentsOfMenuItem:(NSMenuItem *)menuItem
{
    self.keyEquivalent = menuItem.keyEquivalent;
    self.keyEquivalentModifierMask = menuItem.keyEquivalentModifierMask;
}

@end
