/*
 *  CUPS add-on module for Canon LIPSLX/UFR2/NCAP printer.
 *  Copyright CANON INC. 2018
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdarg.h>
#include "cnrasterproc.h"
#include "buftool.h"

static int WritePipe( int fdPipe, char *pData, int nDataSize);

int cnprocCreateProcess( int *pPId, CnRasterPipeFds *pFds , char *argv[] )
{
	int ret = DEF_NO_ERR;
	int command_pipe[2];
	int data_pipe[2];

	debuglog(INDENT_RIGHT, ">> cnprocCreateProcess\n");

	if( pipe(command_pipe) == DEF_ERR )
	{
		ret = DEF_ERR;
	}
	if( pipe(data_pipe) == DEF_ERR )
	{
		ret = DEF_ERR;
	}

	if(ret == DEF_NO_ERR)
	{
		int pid = fork();
		if( pid == -1 )
		{
			ret = DEF_ERR;
		}
		else if( pid == 0 )
		{
			char *module_arg[7];
			char *module_buf = NULL;
			char cmdpipe_opt[32];
			char datapipe_opt[32];

			close( command_pipe[1] );
			close( data_pipe[1] );

			snprintf( cmdpipe_opt,  sizeof(cmdpipe_opt),  "%d", command_pipe[0] );
			snprintf( datapipe_opt, sizeof(datapipe_opt), "%d", data_pipe[0] );
			int nKeyLen = strlen(MODULE_PATH) + strlen(MODULE_NAME)+2;

			module_buf = calloc(1, nKeyLen);
			if( module_buf == NULL )
			{
				ret = DEF_ERR;
			}
			else
			{
				strncpy(module_buf, MODULE_PATH, nKeyLen -1);
				strncat(module_buf, "/", nKeyLen - strlen(module_buf) - 1);
				strncat(module_buf, MODULE_NAME, nKeyLen - strlen(module_buf) - 1);

				module_arg[0] = module_buf;
				module_arg[1] = cmdpipe_opt;
				module_arg[2] = datapipe_opt;
				module_arg[3] = argv[5];
				module_arg[4] = argv[2];
				module_arg[5] = argv[3];
				module_arg[6] = NULL;
				ret = execv(module_buf, module_arg);
				if( ret == DEF_ERR )
				{
					exit(1);
				}
			}
		}
		else
		{
			close( command_pipe[0] );
			close( data_pipe[0] );
		}

		*pPId = pid;
		pFds->cmd_fd = command_pipe[1];
		pFds->dat_fd = data_pipe[1];
	}

	debuglog(INDENT_LEFT, "<< cnprocCreateProcess\n\n");
	return ret;
}

int cnprocWriteCommand( CnRasterPipeFds tFds, int nCmdID, char *pData, int nDataSize)
{
	int result = DEF_ERR;
	int write_size = CMD_HEADER_SIZE + nDataSize;

	BufTool *buftool = buftool_new( write_size, BUFTOOL_LITTLE_ENDIAN );
	if( buftool )
	{
		buftool_write_short(buftool, nCmdID);
		buftool_write_short(buftool, nDataSize);
		buftool_write(buftool, pData, nDataSize);

		result = WritePipe(tFds.cmd_fd, (char *)buftool_data(buftool),buftool_pos(buftool));
		buftool_destroy( buftool );

	}
	return result;
}


int cnprocWriteData(CnRasterPipeFds tFds, char *pData, int nDataSize)
{
	int result = DEF_ERR;

	result = WritePipe(tFds.dat_fd, pData, nDataSize);

	return result;
}


static int WritePipe(int fdPipe, char *pData, int nDataSize)
{
	int result = DEF_NO_ERR;
	int write_size = 0;
	int total_size = nDataSize;

	while( total_size > 0 ){
		write_size = write( fdPipe, pData, total_size );
		if( write_size == -1 ){
			result = DEF_ERR;
			break;
		}
		total_size -= write_size;
		pData += write_size;
	}
	return result;
}

