/* SPDX-License-Identifier: BSD-3-Clause */
/* Copyright 2014-2023, Intel Corporation */

/*
 * mmap.h -- internal definitions for mmap module
 */

#ifndef PMDK_MMAP_H
#define PMDK_MMAP_H 1

#include <stddef.h>
#include <stdint.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>

#include "out.h"
#include "queue.h"
#include "os.h"

#ifdef __cplusplus
extern "C" {
#endif

extern int Mmap_no_random;
extern void *Mmap_hint;
extern char *Mmap_mapfile;

void *util_map_sync(void *addr, size_t len, int proto, int flags, int fd,
	os_off_t offset, int *map_sync);
void *util_map(int fd, os_off_t off, size_t len, int flags, int rdonly,
		size_t req_align, int *map_sync);
int util_unmap(void *addr, size_t len);

#define OS_MAPFILE "/proc/self/maps"

#ifndef MAP_SYNC
#define MAP_SYNC 0x80000
#endif

#ifndef MAP_SHARED_VALIDATE
#define MAP_SHARED_VALIDATE 0x03
#endif

/*
 * macros for micromanaging range protections for the debug version
 */
#ifdef DEBUG

#define RANGE(addr, len, is_dev_dax, type) do {\
	if (!is_dev_dax) ASSERT(util_range_##type(addr, len) >= 0);\
} while (0)

#else

#define RANGE(addr, len, is_dev_dax, type) do {} while (0)

#endif

#define RANGE_RO(addr, len, is_dev_dax) RANGE(addr, len, is_dev_dax, ro)
#define RANGE_RW(addr, len, is_dev_dax) RANGE(addr, len, is_dev_dax, rw)
#define RANGE_NONE(addr, len, is_dev_dax) RANGE(addr, len, is_dev_dax, none)

/* pmem mapping type */
enum pmem_map_type {
	PMEM_DEV_DAX,	/* device dax */
	PMEM_MAP_SYNC,	/* mapping with MAP_SYNC flag on dax fs */

	MAX_PMEM_TYPE
};

/*
 * this structure tracks the file mappings outstanding per file handle
 */
struct map_tracker {
	PMDK_SORTEDQ_ENTRY(map_tracker) entry;
	uintptr_t base_addr;
	uintptr_t end_addr;
	unsigned region_id;
	enum pmem_map_type type;
};

void util_mmap_init(void);
void util_mmap_fini(void);

int util_range_ro(void *addr, size_t len);
int util_range_rw(void *addr, size_t len);
int util_range_none(void *addr, size_t len);

char *util_map_hint_unused(void *minaddr, size_t len, size_t align);
char *util_map_hint(size_t len, size_t req_align);

#define KILOBYTE ((uintptr_t)1 << 10)
#define MEGABYTE ((uintptr_t)1 << 20)
#define GIGABYTE ((uintptr_t)1 << 30)

/*
 * util_map_hint_align -- choose the desired mapping alignment
 *
 * The smallest supported alignment is 2 megabytes because of the object
 * alignment requirements. Changing this value to 4 kilobytes constitutes a
 * layout change.
 *
 * Use 1GB page alignment only if the mapping length is at least
 * twice as big as the page size.
 */
static inline size_t
util_map_hint_align(size_t len, size_t req_align)
{
	size_t align = 2 * MEGABYTE;
	if (req_align)
		align = req_align;
	else if (len >= 2 * GIGABYTE)
		align = GIGABYTE;

	return align;
}

int util_range_register(const void *addr, size_t len, const char *path,
		enum pmem_map_type type);
int util_range_unregister(const void *addr, size_t len);
struct map_tracker *util_range_find(uintptr_t addr, size_t len);
int util_range_is_pmem(const void *addr, size_t len);

#ifdef __cplusplus
}
#endif
#endif
