/*
 * Copyright (c) 2021-2026 Symas Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above
 *   copyright notice, this list of conditions and the following disclaimer
 *   in the documentation and/or other materials provided with the
 *   distribution.
 * * Neither the name of the Symas Corporation nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef _CBLDIAG_H
#pragma message __FILE__ " included twice"
#else
#define _CBLDIAG_H

#if GCOBOL_GETENV
#define gcobol_getenv(x) getenv(x)
#else
#define gcobol_getenv(x) ((char *)nullptr)
#endif

const char * cobol_filename();

/*
 *  These are user-facing messages.  They go through the gcc
 *  diagnostic framework and use text that can be localized.
 */
void yyerror( const char fmt[], ... ) ATTRIBUTE_GCOBOL_DIAG(1, 2);

/* Location type.  Borrowed from parse.h as generated by Bison. */
#if ! defined YYLTYPE && ! defined YYLTYPE_IS_DECLARED
typedef struct YYLTYPE YYLTYPE;
struct YYLTYPE
{
  int first_line;
  int first_column;
  int last_line;
  int last_column;
};
# define YYLTYPE_IS_DECLARED 1
# define YYLTYPE_IS_TRIVIAL 1

const YYLTYPE& cobol_location();
#endif

#if ! defined YDFLTYPE && ! defined YDFLTYPE_IS_DECLARED
typedef struct YDFLTYPE YDFLTYPE;
struct YDFLTYPE
{
  int first_line;
  int first_column;
  int last_line;
  int last_column;
};
# define YDFLTYPE_IS_DECLARED 1
# define YDFLTYPE_IS_TRIVIAL 1

#endif

struct cbl_loc_t {
  int first_line;
  int first_column;
  int last_line;
  int last_column;

  cbl_loc_t() 
    : first_line(0)
    , first_column(0)
    , last_line(0)
    , last_column(0)
  {}
  cbl_loc_t( const YYLTYPE& loc ) 
    : first_line(loc.first_line)
    , first_column(loc.first_column)
    , last_line(loc.last_line)
    , last_column(loc.last_column)
  {}

  operator YYLTYPE() const {  //  cppcheck-suppress syntaxError
    return { first_line, first_column, last_line, last_column };
  } 
};

/*
 * Naming Convention: Names end with a letter that indicates 
 * their kind:
 * F  fatal, "fatal error: "
 * I  ice, "internal compiler error: "
 * E  error, "error: "
 * S  sorry, "sorry, unimplemented: "
 * W  warning, "warning: "
 * A  anachronism, "anachronism: "
 * N  note, "note: "
 * D  debug, "debug: "
 */
enum cbl_diag_id_t : uint64_t {
  CdfNotFoundW, 
  CdfParameterW,
  
  EcUnknownW, 

  LexIncludeE,
  LexIncludeOkN,
  LexIndicatorE,
  LexInputN,
  LexLineE,
  LexPreprocessE,
  LexReplaceE,
  LexSeparatorE,

  IbmEjectE,
  IbmEqualAssignE,
  IbmLengthOf, 
  IbmProcedurePointer, 
  IbmSectionNegE,
  IbmSectionRangeE,
  IbmSectionSegmentW,
  IbmStopNumber,
  IbmVolatileE,  
  IbmVolatileW,  // dialect warning for ignored syntax

  IsoResume,

  MfBinaryLongLong,
  MfCallGiving,
  MfCallLiteral,
  MfCdfDollar, 
  MfComp6,
  MfCompX,
  MfLevel_1_Occurs, 
  MfLevel78,
  MfAnyLength, 
  MfMoveIndex, 
  MfMovePointer, 
  MfReturningNum,
  MfUsageTypename,
  MfTrailing,
  
  Par78CdfDefinedW,
  ParIconvE, 
  ParInfoI,
  ParLangInfoW,
  ParLiteral2W, 
  ParLocaleW,
  ParNoCorrespondingW,
  ParNumstrW, 
  ParUnresolvedProcE, 

  SynApplyCommit,
  SynFileCodeSet,
  SynHighOrderBit,
  SynRecordingMode,
  SynSetLocaleTo,
  SynSetToLocale,

  DiagDiagDiag // always last
};

bool cbl_message( cbl_diag_id_t id, const char msg[], ... )
  ATTRIBUTE_GCOBOL_DIAG(2, 3);

bool cbl_message( cbl_loc_t loc, cbl_diag_id_t id, const char msg[], ... )
  ATTRIBUTE_GCOBOL_DIAG(3, 4);

bool
dialect_ok( const cbl_loc_t& loc, cbl_diag_id_t id, const char term[], bool ok = true );

static inline bool
dialect_not_ok( const cbl_loc_t& loc, cbl_diag_id_t id, const char term[] ) {
  return dialect_ok(loc, id, term, false);
}

// Diagnostic format specifiers are documented in gcc/pretty-print.cc
// an error at a location, called from the parser for semantic errors
void error_msg( const YYLTYPE& loc, const char gmsgid[], ... )
  ATTRIBUTE_GCOBOL_DIAG(2, 3);

bool
warn_msg( const YYLTYPE& loc, const char gmsgid[], ... )
  ATTRIBUTE_GCOBOL_DIAG(2, 3);

// an error that uses token_location, not yylloc
void error_msg_direct( const char gmsgid[], ... )
  ATTRIBUTE_GCOBOL_DIAG(1, 2);

// for CDF and other warnings that refer back to an earlier line
// (not in diagnostic framework yet)
void yyerrorvl( int line, const char *filename, const char fmt[], ... )
  ATTRIBUTE_PRINTF_3;

void cbl_unimplementedw(cbl_diag_id_t id, const char *gmsgid, ...)
  ATTRIBUTE_GCOBOL_DIAG(2, 3); // warning
void cbl_unimplemented(const char *gmsgid, ...)
  ATTRIBUTE_GCOBOL_DIAG(1, 2);  // error
void cbl_unimplemented_at( const  YYLTYPE& loc, const char *gmsgid, ... )
  ATTRIBUTE_GCOBOL_DIAG(2, 3);

/*
 * dbgmsg produce messages not intended for the user.  They cannot be localized
 * and fwrite directly to standard error.  dbgmsg is activated by -fflex-debug
 * or -fyacc-debug.
 */
void dbgmsg( const char fmt[], ... ) ATTRIBUTE_PRINTF_1;

void gcc_location_set( const YYLTYPE& loc );

void gcc_location_dump();

// tree.h defines yy_flex_debug as a macro because options.h
#if ! defined(yy_flex_debug)
template <typename LOC>
static void
location_dump( const char func[], int line, const char tag[], const LOC& loc) {
  extern int yy_flex_debug; // cppcheck-suppress shadowVariable
  if( yy_flex_debug ) {
    const char *detail = gcobol_getenv("update_location");
    if( detail ) { // cppcheck-suppress knownConditionTrueFalse
      fprintf(stderr, "%s:%d: %s location (%d,%d) to (%d,%d)\n",
              func, line, tag,
              loc.first_line, loc.first_column, loc.last_line, loc.last_column);
      if( *detail == '2' ) gcc_location_dump();
    }
  }
}
#endif // defined(yy_flex_debug)

#endif
