######################### BEGIN COPYRIGHT MESSAGE #########################
# GBNP - computing Gröbner bases of noncommutative polynomials
# Copyright 2001-2010 by Arjeh M. Cohen, Dié A.H. Gijsbers, Jan Willem
# Knopper, Chris Krook. Address: Discrete Algebra and Geometry (DAM) group
# at the Department of Mathematics and Computer Science of Eindhoven
# University of Technology.
#
# For acknowledgements see the manual. The manual can be found in several
# formats in the doc subdirectory of the GBNP distribution. The
# acknowledgements formatted as text can be found in the file chap0.txt.
#
# GBNP is free software; you can redistribute it and/or modify it under
# the terms of the Lesser GNU General Public License as published by the
# Free Software Foundation (FSF); either version 2.1 of the License, or
# (at your option) any later version. For details, see the file 'LGPL' in
# the doc subdirectory of the GBNP distribution or see the FSF's own site:
# https://www.gnu.org/licenses/lgpl.html
########################## END COPYRIGHT MESSAGE ##########################

### filename = "example11.g"
### authors Cohen & Gijsbers

### THIS IS A GAP PACKAGE FOR COMPUTING NON-COMMUTATIVE GROBNER BASES

# <#GAPDoc Label="Example11">
# <Section Label="Example11"><Heading>The truncated variant on two weighted homogeneous polynomials</Heading>
# Here we exhibit a truncated non-commutative homogeneous weighted Gröbner
# basis computation. This example uses the functions from Section <Ref
# Sect="truncfun"/>, the truncation variants (see also Section <Ref
# Sect="trunc"/>).
# <P/>
# The input is a set of polynomials in <M>x</M> and <M>y</M>, which
# are homogeneous when the weight of <M>x</M> is 2 and of <M>y</M> is 3.
# The input is <M>\{x^3y^2-x^6+y^4,y^2x^3+xyxyx+x^2yxy\}</M>.
# We truncate the computation at degree 16.
# The truncated Gröbner basis is
# <M>\{y^2x^3+xyxyx+x^2yxy,x^6-x^3y^2-y^4,x^3y^2x-x^4y^2-xy^4\}</M>
# and the dimension of the quotient algebra is 134.
# <P/>
# First load the package and set the standard infolevel <Ref
# InfoClass="InfoGBNP" Style="Text"/> to 1 and the time infolevel <Ref
# Func="InfoGBNPTime" Style="Text"/> to 1 (for more information about the info
# level, see Chapter <Ref Chap="Info"/>).


# <L>
LoadPackage("gbnp", false);
SetInfoLevel(InfoGBNP,1);
SetInfoLevel(InfoGBNPTime,0);
# </L>

# The variables will be printed as <M>x</M> and <M>y</M>.
# <L>
GBNP.ConfigPrint("x","y");
# </L>

# The level to truncate at is assigned to <M>n</M>.

# <L>
n := 16;;
# </L>

# Now enter the relations in NP form (see Section <Ref Sect="NP"/>) and the
# weights.

# <L>
s1 :=[[[1,1,1,2,2],[1,1,1,1,1,1],[2,2,2,2]],[1,-1,1]];;
s2 :=[[[2,2,1,1,1],[1,2,1,2,1],[1,1,2,1,2]],[1,1,1]];;
K := [s1,s2];;
weights:=[2,3];;
# </L>

# The input can be printed with <Ref Func="PrintNPList" Style="Text"/>

# <L>
PrintNPList(K);
# </L>

# Verify whether the list <C>K</C> consists only of polynomials that are
# homogeneous with respect to <C>weights</C>
# by means of  <Ref Func="CheckHomogeneousNPs" Style="Text"/>.

# <L>
CheckHomogeneousNPs(K,weights);
# </L>


# Now calculate the truncated Gröbner basis with <Ref Func="SGrobnerTrunc"
# Style="Text"/>. The output will only contain homogeneous polynomials
# of degree at most <C>n</C>.

# <L>
G := SGrobnerTrunc(K,n,weights);;
# </L>

# The Gröbner basis of the truncated quotient algebra can be printed with <Ref
# Func="PrintNPList" Style="Text"/>:

# <L>
PrintNPList(G);
# </L>

# The standard basis of the quotient of the free noncommutative algebra
# on <M>n</M> variables, where
# <M>n</M> is the length of the vector <C>weights</C>,
# by the homogeneous ideal generated by <C>K</C>
# up to degree <M>n</M>
# is obtained by means of the function
# <Ref Func="BaseQATrunc" Style="Text"/>
# applied to <C>K</C>, <C>n</C>, and <C>weights</C>.

# <L>
B := BaseQATrunc(K,n,weights);;
i := Length(B);
Print("at level ",i-1," the standard monomials are:\n");
PrintNPList(List(B[i], qq -> [[qq],[1]]));
# </L>


# The same result can be obtained by using the truncated Gröbner basis
# found for <C>G</C> instead of <C>K</C>.

# <L>
B2 := BaseQATrunc(G,n,weights);;
B = B2;
# </L>

# Also, the same result can be obtained by using the
# leading terms of the truncated Gröbner basis
# found for <C>G</C> instead of <C>K</C>.

# <L>
B3 := BaseQATrunc(List( LMonsNP(G), qq -> [[qq],[1]]),n,weights);;
B = B3;
# </L>



# A list of dimensions of the homogeneous parts of the quotient algebra
# up to degree <M>n</M> is obtained by means of
# <Ref Func="DimsQATrunc" Style="Text"/>
# with arguments <C>G</C>, <C>n</C>, and <C>weights</C>.

# <L>
DimsQATrunc(G,n,weights);
# </L>



# Even more detailed information is given by the list of
# frequencies up to degree <C>n</C>.
# This is obtained by means of
# <Ref Func="FreqsQATrunc" Style="Text"/>
# with arguments <C>G</C>, <C>n</C>, and <C>weights</C>.

# <L>
FreqsQATrunc(G,n,weights);
# </L>


# </Section>
# <#/GAPDoc>
