// ----------------------------------------------------------------------
//
// ParticleName.cc
// Author: Lynn Garren and Walter Brown
//
//  Create a map that gives a standard name for each pre-defined 
//  particle ID number.   Also create a map for the reverse lookup of 
//  the ID number from a string.  These maps are initialized if and only if 
//  the public functions are called. Because the maps are static, 
//  the initialization happens only once.
//
//  The user NEVER calls ParticleNameInit()
//  We use a data table (struct Snames) so that compile time is not impacted.
//
//  public functions:
//     PartcleIdMap const &  getPartcleIdMap()
//     std::string              ParticleName( const int pid )
//     void                     listParticleNames( std::ostream & os  )
//
// ----------------------------------------------------------------------

#include <string>
#include <map>
#include <iostream>
#include <sstream>
#include <iomanip>	// width
#include <utility>	// make_pair

#include "HepPID/ParticleName.hh"
#include "HepPID/ParticleIDMethods.hh"
#include "HepPID/Version.hh"

namespace HepPID {

typedef  std::map< int, std::string >  PartcleIdMap;
typedef  std::map< std::string, int >  ParticleLookupMap;

///
/// \class ParticleNameMap
/// \author Lynn Garren
///
/// Used internally to store the static maps
///
class ParticleNameMap{

public:

   typedef PartcleIdMap::const_iterator      idIterator;
   typedef ParticleLookupMap::const_iterator nameIterator;
   
   ParticleNameMap(PartcleIdMap m1,ParticleLookupMap m2)
   : itsNameMap(m1), itsLookupMap(m2) {}
   ~ParticleNameMap() {}
   
   PartcleIdMap       nameMap()    const { return itsNameMap; }
   ParticleLookupMap lookupMap()  const { return itsLookupMap; }
   idIterator   begin()               const { return itsNameMap.begin(); }
   idIterator   end()                 const { return itsNameMap.end(); }
   idIterator   find( const int & id) const { return itsNameMap.find(id); }
   nameIterator beginLookupMap()      const { return itsLookupMap.begin(); }
   nameIterator endLookupMap()        const { return itsLookupMap.end(); }
   nameIterator findString( const std::string & s) const { return itsLookupMap.find(s); }

private:
   
   PartcleIdMap       itsNameMap;
   ParticleLookupMap itsLookupMap;
   
   // copies are not allowed
   ParticleNameMap( const ParticleNameMap & );
   ParticleNameMap & operator = ( const ParticleNameMap & );
   
};

namespace {	// ParticleNameInit and ParticleNameMap are private

ParticleNameMap const &  ParticleNameInit()
{

  PartcleIdMap  m;
  ParticleLookupMap nameMap;

  static const struct {
      int pid;
      const char* pname;
  } SNames[] = {
      {          0, "" },
      {          1, "d" },
      {         -1, "d~" },
      {          2, "u" },
      {         -2, "u~" },
      {          3, "s" },
      {         -3, "s~" },
      {          4, "c" },
      {         -4, "c~" },
      {          5, "b" },
      {         -5, "b~" },
      {          6, "t" },
      {         -6, "t~" },
      {          7, "b'" },
      {         -7, "b'~" },
      {          8, "t'" },
      {         -8, "t'~" },
      {         11, "e^-" },
      {        -11, "e^+" },
      {         12, "nu_e" },
      {        -12, "nu_e~" },
      {         13, "mu^-" },
      {        -13, "mu^+" },
      {         14, "nu_mu" },
      {        -14, "nu_mu~" },
      {         15, "tau^-" },
      {        -15, "tau^+" },
      {         16, "nu_tau" },
      {        -16, "nu_tau~" },
      {         17, "tau'^-" },
      {        -17, "tau'^+" },
      {         18, "nu_tau'" },
      {        -18, "nu_tau'~" },
      {         21, "g" },
      {         22, "gamma" },
      {      10022, "virtual-photon" },
      {      20022, "Cerenkov-radiation" },
      {         23, "Z^0" },
      {         24, "W^+" },
      {        -24, "W^-" },
      {         25, "H_1^0" },
      {         32, "Z_2^0" },
      {         33, "Z_3^0" },
      {         34, "W_2^+" },
      {        -34, "W_2^-" },
      {         35, "H_2^0" },
      {         36, "H_3^0" },
      {         37, "H^+" },
      {        -37, "H^-" },
      {         39, "G"  },
      {         41, "R^0" },
      {        -41, "R~^0" },
      {         42, "LQ_c" },
      {        -42, "LQ_c~" },
      {         43, "Xu^0" },
      {         44, "Xu^+" },
      {        -44, "Xu^-" },
      {         51, "H_L^0" },
      {         52, "H_1^++" },
      {        -52, "H_1^--" },
      {         53, "H_2^+" },
      {        -53, "H_2^-" },
      {         54, "H_2^++" },
      {        -54, "H_2^--" },
      {         55, "H_4^0" },
      {        -55, "H_4~^0" },
      {         81, "generator-specific+81" },
      {         82, "generator-specific+82" },
      {         83, "generator-specific+83" },
      {         84, "generator-specific+84" },
      {         85, "generator-specific+85" },
      {         86, "generator-specific+86" },
      {         87, "generator-specific+87" },
      {         88, "generator-specific+88" },
      {         89, "generator-specific+89" },
      {         90, "generator-specific+90" },
      {         91, "generator-specific+91" },
      {         92, "generator-specific+92" },
      {         93, "generator-specific+93" },
      {         94, "generator-specific+94" },
      {         95, "generator-specific+95" },
      {         96, "generator-specific+96" },
      {         97, "generator-specific+97" },
      {         98, "generator-specific+98" },
      {         99, "generator-specific+99" },
      {        -81, "generator-specific-81" },
      {        -82, "generator-specific-82" },
      {        -83, "generator-specific-83" },
      {        -84, "generator-specific-84" },
      {        -85, "generator-specific-85" },
      {        -86, "generator-specific-86" },
      {        -87, "generator-specific-87" },
      {        -88, "generator-specific-88" },
      {        -89, "generator-specific-89" },
      {        -90, "generator-specific-90" },
      {        -91, "generator-specific-91" },
      {        -92, "generator-specific-92" },
      {        -93, "generator-specific-93" },
      {        -94, "generator-specific-94" },
      {        -95, "generator-specific-95" },
      {        -96, "generator-specific-96" },
      {        -97, "generator-specific-97" },
      {        -98, "generator-specific-98" },
      {        -99, "generator-specific-99" },
      {        100, "generator-specific+100" },
      {       -100, "generator-specific-100" },
      {        101, "geantino" },
      {        102, "charged-geantino" },
      {        110, "reggeon" },
      {        130, "K_L^0" },
      {        310, "K_S^0" },
      {        990, "pomeron" },
      {       9990, "odderon" },
      {    1000001, "susy-d_L" },
      {   -1000001, "susy-d_L~" },
      {    1000002, "susy-u_L" },
      {   -1000002, "susy-u_L~" },
      {    1000003, "susy-s_L" },
      {   -1000003, "susy-s_L~" },
      {    1000004, "susy-c_L" },
      {   -1000004, "susy-c_L~" },
      {    1000005, "susy-b_1" },
      {   -1000005, "susy-b_1~" },
      {    1000006, "susy-t_1" },
      {   -1000006, "susy-t_1~" },
      {    1000011, "susy-e_L^-" },
      {   -1000011, "susy-e_L^+" },
      {    1000012, "susy-nu_eL" },
      {   -1000012, "susy-nu_eL~" },
      {    1000013, "susy-mu_L^-" },
      {   -1000013, "susy-mu_L^+" },
      {    1000014, "susy-nu_muL" },
      {   -1000014, "susy-nu_muL~" },
      {    1000015, "susy-tau_L^-" },
      {   -1000015, "susy-tau_L^+" },
      {    1000016, "susy-nu_tauL" },
      {   -1000016, "susy-nu_tauL~" },
      {    1000021, "gluino" },
      {    1000022, "susy-chi_1^0" },
      {    1000023, "susy-chi_2^0" },
      {    1000024, "susy-chi_1^+" },
      {   -1000024, "susy-chi_1^-" },
      {    1000025, "susy-chi_3^0" },
      {    1000035, "susy-chi_4^0" },
      {    1000037, "susy-chi_2^+" },
      {   -1000037, "susy-chi_2^-" },
      {    1000039, "gravitino" },
      {    2000001, "susy-d_R" },
      {   -2000001, "susy-d_R~" },
      {    2000002, "susy-u_R" },
      {   -2000002, "susy-u_R~" },
      {    2000003, "susy-s_R" },
      {   -2000003, "susy-s_R~" },
      {    2000004, "susy-c_R" },
      {   -2000004, "susy-c_R~" },
      {    2000005, "susy-b_R" },
      {   -2000005, "susy-b_R~" },
      {    2000006, "susy-t_R" },
      {   -2000006, "susy-t_R~" },
      {    2000011, "susy-e_R^-" },
      {   -2000011, "susy-e_R^+" },
      {    2000012, "susy-nu_eR" },
      {   -2000012, "susy-nu_eR~" },
      {    2000013, "susy-mu_R^-" },
      {   -2000013, "susy-mu_R^+" },
      {    2000014, "susy-nu_muR" },
      {   -2000014, "susy-nu_muR~" },
      {    2000015, "susy-tau_R^-" },
      {   -2000015, "susy-tau_R^+" },
      {    2000016, "susy-nu_tauR" },
      {   -2000016, "susy-nu_tauR~" },
      {    3100021, "V8_tech" },
      {   -3100021, "V8_tech~" },
      {    3000111, "pi_tech^0" },
      {    3000115, "a_tech^0" },
      {    3060111, "pi_tech_22_1" },
      {    3160111, "pi_tech_22_8" },
      {    3000113, "rho_tech^0" },
      {    3130113, "rho_tech_11" },
      {    3140113, "rho_tech_12" },
      {    3150113, "rho_tech_21" },
      {    3160113, "rho_tech_22" },
      {    3000211, "pi_tech^+" },
      {   -3000211, "pi_tech^-" },
      {    3000213, "rho_tech^+" },
      {   -3000213, "rho_tech^-" },
      {    3000215, "a_tech^+" },
      {   -3000215, "a_tech^-" },
      {    3000221, "pi'_tech" },
      {    3100221, "eta_tech" },
      {    3000223, "omega_tech" },
      {    4000001, "d*" },
      {   -4000001, "d*~" },
      {    4000002, "u*" },
      {   -4000002, "u*~" },
      {    4000011, "e*^-" },
      {   -4000011, "e*^+" },
      {    4000012, "nu*_e" },
      {   -4000012, "nu*_e~" },
      {    4000039, "G*" },
      {   -4000039, "G*~" },
      {    5000040, "black_hole" },
      {    5100001, "d_L^(1)" },
      {   -5100001, "d~_L^(1)" },
      {    5100002, "u_L^(1)" },
      {   -5100002, "u~_L^(1)" },
      {    5100003, "s_L^(1)" },
      {   -5100003, "s~_L^(1)" },
      {    5100004, "c_L^(1)" },
      {   -5100004, "c~_L^(1)" },
      {    5100005, "b_L^(1)" },
      {   -5100005, "b~_L^(1)" },
      {    5100006, "t_L^(1)" },
      {   -5100006, "t~_L^(1)" },
      {    5100011, "e_L^(1)-" },
      {   -5100011, "e_L^(1)+" },
      {    5100012, "nu_eL^(1)" },
      {   -5100012, "nu_eL~^(1)" },
      {    5100013, "mu_L^(1)-" },
      {   -5100013, "mu_L^(1)+" },
      {    5100014, "nu_muL^(1)" },
      {   -5100014, "nu_muL~^(1)" },
      {    5100015, "tau_L^(1)-" },
      {   -5100015, "tau_L^(1)+" },
      {    5100016, "nu_tauL^(1)" },
      {   -5100016, "nu_tauL~^(1)" },
      {    6100001, "d_R^(1)" },
      {   -6100001, "d~_R^(1)" },
      {    6100002, "u_R^(1)" },
      {   -6100002, "u~_R^(1)" },
      {    6100003, "s_R^(1)" },
      {   -6100003, "s~_R^(1)" },
      {    6100004, "c_R^(1)" },
      {   -6100004, "c~_R^(1)" },
      {    6100005, "b_R^(1)" },
      {   -6100005, "b~_R^(1)" },
      {    6100006, "t_R^(1)" },
      {   -6100006, "t~_R^(1)" },
      {    6100011, "e_R^(1)-" },
      {   -6100011, "e_R^(1)+" },
      {    6100012, "nu_eR^(1)" },
      {   -6100012, "nu_eR~^(1)" },
      {    6100013, "mu_R^(1)-" },
      {   -6100013, "mu_R^(1)+" },
      {    6100014, "nu_muR^(1)" },
      {   -6100014, "nu_muR~^(1)" },
      {    6100015, "tau_R^(1)-" },
      {   -6100015, "tau_R^(1)+" },
      {    6100016, "nu_tauR^(1)" },
      {   -6100016, "nu_tauR~^(1)" },
      {    5100021, "g^(1)" },
      {    5100022, "gamma^(1)" },
      {    5100023, "Z^(1)0" },
      {    5100024, "W^(1)+" },
      {   -5100024, "W^(1)-" },
      {    5100025, "h^(1)0" },
      {    5100039, "G^(1)" },
      {    9900012, "nu_Re" },
      {   -9900012, "nu_Re~" },
      {    9900014, "nu_Rmu" },
      {   -9900014, "nu_Rmu~" },
      {    9900016, "nu_Rtau" },
      {   -9900016, "nu_Rtau~" },
      {    9900023, "Z_R^0" },
      {   -9900023, "Z_R~^0" },
      {    9900024, "W_R^+" },
      {   -9900024, "W_R^-" },
      {    9900041, "H_L^++" },
      {   -9900041, "H_L^--" },
      {    9900042, "H_R^++" },
      {   -9900042, "H_R^--" },
      {    9910113, "rho_diffr^0" },
      {    9910211, "pi_diffr^+" },
      {   -9910211, "pi_diffr^-" },
      {    9910223, "omega_diffr" },
      {    9910333, "phi_diffr" },
      {    9910443, "psi_diffr" },
      {    9912112, "n_diffr^0" },
      {   -9912112, "n_diffr~^0" },
      {    9912212, "p_diffr^+" },
      {   -9912212, "p_diffr~^-" },
      {    9920022, "remnant photon" },
      {    9922212, "remnant nucleon" },
      {   -9922212, "remnant nucleon~" },
      {    9900441, "cc~[1S08]" },     
      {    9910441, "cc~[3P08]" },     
      {    9900443, "cc~[3S18]" },     
      {    9900551, "bb~[1S08]" },     
      {    9910551, "bb~[3P08]" },     
      {    9900553, "bb~[3S18]" },    
      {       1103, "dd_1" },
      {      -1103, "dd_1~" },
      {       2101, "ud_0" },
      {      -2101, "ud_0~" },
      {       2103, "ud_1" },
      {      -2103, "ud_1~" },
      {       2203, "uu_1" },
      {      -2203, "uu_1~" },
      {       3101, "sd_0" },
      {      -3101, "sd_0~" },
      {       3103, "sd_1" },
      {      -3103, "sd_1~" },
      {       3201, "su_0" },
      {      -3201, "su_0~" },
      {       3203, "su_1" },
      {      -3203, "su_1~" },
      {       3303, "ss_1" },
      {      -3303, "ss_1~" },
      {       4101, "cd_0" },
      {      -4101, "cd_0~" },
      {       4103, "cd_1" },
      {      -4103, "cd_1~" },
      {       4201, "cu_0" },
      {      -4201, "cu_0~" },
      {       4203, "cu_1" },
      {      -4203, "cu_1~" },
      {       4301, "cs_0" },
      {      -4301, "cs_0~" },
      {       4303, "cs_1" },
      {      -4303, "cs_1~" },
      {       4403, "cc_1" },
      {      -4403, "cc_1~" },
      {       5101, "bd_0" },
      {      -5101, "bd_0~" },
      {       5103, "bd_1" },
      {      -5103, "bd_1~" },
      {       5201, "bu_0" },
      {      -5201, "bu_0~" },
      {       5203, "bu_1" },
      {      -5203, "bu_1~" },
      {       5301, "bs_0" },
      {      -5301, "bs_0~" },
      {       5303, "bs_1" },
      {      -5303, "bs_1~" },
      {       5401, "bc_0" },
      {      -5401, "bc_0~" },
      {       5403, "bc_1" },
      {      -5403, "bc_1~" },
      {       5503, "bb_1" },
      {      -5503, "bb_1~" },
      {       6101, "td_0" },
      {      -6101, "td_0~" },
      {       6103, "td_1" },
      {      -6103, "td_1~" },
      {       6201, "tu_0" },
      {      -6201, "tu_0~" },
      {       6203, "tu_1" },
      {      -6203, "tu_1~" },
      {       6301, "ts_0" },
      {      -6301, "ts_0~" },
      {       6303, "ts_1" },
      {      -6303, "ts_1~" },
      {       6401, "tc_0" },
      {      -6401, "tc_0~" },
      {       6403, "tc_1" },
      {      -6403, "tc_1~" },
      {       6501, "tb_0" },
      {      -6501, "tb_0~" },
      {       6503, "tb_1" },
      {      -6503, "tb_1~" },
      {       6603, "tt_1" },
      {      -6603, "tt_1~" },
      {       7101, "b'd_0" },
      {      -7101, "b'd_0~" },
      {       7103, "b'd_1" },
      {      -7103, "b'd_1~" },
      {       7201, "b'u_0" },
      {      -7201, "b'u_0~" },
      {       7203, "b'u_1" },
      {      -7203, "b'u_1~" },
      {       7301, "b's_0" },
      {      -7301, "b's_0~" },
      {       7303, "b's_1" },
      {      -7303, "b's_1~" },
      {       7401, "b'c_0" },
      {      -7401, "b'c_0~" },
      {       7403, "b'c_1" },
      {      -7403, "b'c_1~" },
      {       7501, "b'b_0" },
      {      -7501, "b'b_0~" },
      {       7503, "b'b_1" },
      {      -7503, "b'b_1~" },
      {       7601, "b't_0" },
      {      -7601, "b't_0~" },
      {       7603, "b't_1" },
      {      -7603, "b't_1~" },
      {       7703, "b'b'_1" },
      {      -7703, "b'b'_1~" },
      {       8101, "t'd_0" },
      {      -8101, "t'd_0~" },
      {       8103, "t'd_1" },
      {      -8103, "t'd_1~" },
      {       8201, "t'u_0" },
      {      -8201, "t'u_0~" },
      {       8203, "t'u_1" },
      {      -8203, "t'u_1~" },
      {       8301, "t's_0" },
      {      -8301, "t's_0~" },
      {       8303, "t's_1" },
      {      -8303, "t's_1~" },
      {       8401, "t'c_0" },
      {      -8401, "t'c_0~" },
      {       8403, "t'c_1" },
      {      -8403, "t'c_1~" },
      {       8501, "t'b_0" },
      {      -8501, "t'b_0~" },
      {       8503, "t'b_1" },
      {      -8503, "t'b_1~" },
      {       8601, "t't_0" },
      {      -8601, "t't_0~" },
      {       8603, "t't_1" },
      {      -8603, "t't_1~" },
      {       8701, "t'b'_0" },
      {      -8701, "t'b'_0~" },
      {       8703, "t'b'_1" },
      {      -8703, "t'b'_1~" },
      {       8803, "t't'_1" },
      {      -8803, "t't'_1~" },
      {        111, "pi^0" },
      {    9000111, "a_0(980)^0" },
      {      10111, "a_0(1450)^0" },
      {     100111, "pi(1300)^0" },
      {    9010111, "pi(1800)^0" },
      {        113, "rho(770)^0" },
      {      10113, "b_1(1235)^0" },
      {      20113, "a_1(1260)^0" },
      {    9000113, "pi_1(1400)^0" },
      {     100113, "rho(1450)^0" },
      {    9010113, "pi_1(1600)^0" },
      {    9020113, "a_1(1640)^0" },
      {      30113, "rho(1700)^0" },
      {    9030113, "rho(1900)^0" },
      {    9040113, "rho(2150)^0" },
      {        115, "a_2(1320)^0" },
      {      10115, "pi_2(1670)^0" },
      {    9000115, "a_2(1700)^0" },
      {    9010115, "pi_2(2100)^0" },
      {        117, "rho_3(1690)^0" },
      {    9000117, "rho_3(1990)^0" },
      {    9010117, "rho_3(2250)^0" },
      {        119, "a_4(2040)^0" },
      {        211, "pi^+" },
      {       -211, "pi^-" },
      {    9000211, "a_0(980)^+" },
      {   -9000211, "a_0(980)^-" },
      {      10211, "a_0(1450)^+" },
      {     -10211, "a_0(1450)^-" },
      {     100211, "pi(1300)^+" },
      {    -100211, "pi(1300)^-" },
      {    9010211, "pi(1800)^+" },
      {   -9010211, "pi(1800)^-" },
      {        213, "rho(770)^+" },
      {       -213, "rho(770)^-" },
      {      10213, "b_1(1235)^+" },
      {     -10213, "b_1(1235)^-" },
      {      20213, "a_1(1260)^+" },
      {     -20213, "a_1(1260)^-" },
      {    9000213, "pi_1(1400)^+" },
      {   -9000213, "pi_1(1400)^-" },
      {     100213, "rho(1450)^+" },
      {    -100213, "rho(1450)^-" },
      {    9010213, "pi_1(1600)^+" },
      {   -9010213, "pi_1(1600)^-" },
      {    9020213, "a_1(1640)^+" },
      {   -9020213, "a_1(1640)^-" },
      {      30213, "rho(1700)^+" },
      {     -30213, "rho(1700)^-" },
      {    9030213, "rho(1900)^+" },
      {   -9030213, "rho(1900)^-" },
      {    9040213, "rho(2150)^+" },
      {   -9040213, "rho(2150)^-" },
      {        215, "a_2(1320)^+" },
      {       -215, "a_2(1320)^-" },
      {      10215, "pi_2(1670)^+" },
      {     -10215, "pi_2(1670)^-" },
      {    9000215, "a_2(1700)^+" },
      {   -9000215, "a_2(1700)^-" },
      {    9010215, "pi_2(2100)^+" },
      {   -9010215, "pi_2(2100)^-" },
      {        217, "rho_3(1690)^+" },
      {       -217, "rho_3(1690)^-" },
      {    9000217, "rho_3(1990)^+" },
      {   -9000217, "rho_3(1990)^-" },
      {    9010217, "rho_3(2250)^+" },
      {   -9010217, "rho_3(2250)^-" },
      {        219, "a_4(2040)^+" },
      {       -219, "a_4(2040)^-" },
      {        221, "eta" },
      {    9000221, "f_0(600)" },
      {      10221, "f_0(1370)" },
      {    9010221, "f_0(980)" },
      {    9020221, "eta(1405)" },
      {    9030221, "f_0(1500)" },
      {    9040221, "eta(1760)" },
      {    9050221, "f_0(2020)" },
      {    9060221, "f_0(2100)" },
      {    9070221, "f_0(2200)" },
      {    9080221, "eta(2225)" },
      {    9090221, "sigma_0" },
      {     100221, "eta(1295)" },
      {        331, "eta'(958)" },
      {      10331, "f_0(1710)" },
      {     100331, "eta(1475)" },
      {        223, "omega(782)" },
      {    9000223, "f_1(1510)" },
      {    9010223, "h_1(1595)" },
      {      10223, "h_1(1170)" },
      {      20223, "f_1(1285)" },
      {      30223, "omega(1650)" },
      {     100223, "omega(1420)" },
      {        333, "phi(1020)" },
      {      10333, "h_1(1380)" },
      {      20333, "f_1(1420)" },
      {     100333, "phi(1680)" },
      {        225, "f_2(1270)" },
      {    9000225, "f_2(1430)" },
      {      10225, "eta_2(1645)" },
      {    9010225, "f_2(1565)" },
      {    9020225, "f_2(1640)" },
      {    9030225, "f_2(1810)" },
      {    9040225, "f_2(1910)" },
      {    9050225, "f_2(1950)" },
      {    9060225, "f_2(2010)" },
      {    9070225, "f_2(2150)" },
      {    9080225, "f_2(2300)" },
      {    9090225, "f_2(2340)" },
      {        335, "f'_2(1525)" },
      {      10335, "eta_2(1870)" },
      {        227, "omega_3(1670)" },
      {        337, "phi_3(1850)" },
      {        229, "f_4(2050)" },
      {    9000229, "f_J(2220)" },
      {    9010229, "f_4(2300)" },
      {        311, "K^0" },
      {       -311, "K~^0" },
      {    9000311, "K*_0(800)^0" },
      {   -9000311, "K*_0(800)~^0" },
      {      10311, "K*_0(1430)^0" },
      {     -10311, "K*_0(1430)~^0" },
      {     100311, "K(1460)^0" },
      {    -100311, "K(1460)~^0" },
      {    9010311, "K(1830)^0" },
      {   -9010311, "K(1830)~^0" },
      {    9020311, "K*_0(1950)^0" },
      {   -9020311, "K*_0(1950)~^0" },
      {        321, "K^+" },
      {       -321, "K^-" },
      {    9000321, "K*_0(800)^+" },
      {   -9000321, "K*_0(800)^-" },
      {      10321, "K*_0(1430)^+" },
      {     -10321, "K*_0(1430)^-" },
      {     100321, "K(1460)^+" },
      {    -100321, "K(1460)^-" },
      {    9010321, "K(1830)^+" },
      {   -9010321, "K(1830)^-" },
      {    9020321, "K*_0(1950)^+" },
      {   -9020321, "K*_0(1950)^-" },
      {        313, "K*(892)^0" },
      {       -313, "K*(892)~^0" },
      {      10313, "K_1(1270)^0" },
      {     -10313, "K_1(1270)~^0" },
      {      20313, "K_1(1400)^0" },
      {     -20313, "K_1(1400)~^0" },
      {      30313, "K*(1680)^0" },
      {     -30313, "K*(1680)~^0" },
      {     100313, "K*(1410)^0" },
      {    -100313, "K*(1410)~^0" },
      {    9000313, "K_1(1650)^0" },
      {   -9000313, "K_1(1650)~^0" },
      {        323, "K*(892)^+" },
      {       -323, "K*(892)^-" },
      {      10323, "K_1(1270)^+" },
      {     -10323, "K_1(1270)^-" },
      {      20323, "K_1(1400)^+" },
      {     -20323, "K_1(1400)^-" },
      {      30323, "K*(1680)^+" },
      {     -30323, "K*(1680)^-" },
      {     100323, "K*(1410)^+" },
      {    -100323, "K*(1410)^-" },
      {    9000323, "K_1(1650)^+" },
      {   -9000323, "K_1(1650)^-" },
      {        315, "K*_2(1430)^0" },
      {       -315, "K*_2(1430)~^0" },
      {    9000315, "K_2(1580)^0" },
      {   -9000315, "K_2(1580)~^0" },
      {      10315, "K_2(1770)^0" },
      {     -10315, "K_2(1770)~^0" },
      {    9010315, "K*_2(1980)^0" },
      {   -9010315, "K*_2(1980)~^0" },
      {    9020315, "K_2(2250)^0" },
      {   -9020315, "K_2(2250)~^0" },
      {      20315, "K_2(1820)^0" },
      {     -20315, "K_2(1820)~^0" },
      {        325, "K*_2(1430)^+" },
      {       -325, "K*_2(1430)^-" },
      {    9000325, "K_2(1580)^+" },
      {   -9000325, "K_2(1580)^-" },
      {      10325, "K_2(1770)^+" },
      {     -10325, "K_2(1770)^-" },
      {    9010325, "K*_2(1980)^+" },
      {   -9010325, "K*_2(1980)^-" },
      {    9020325, "K_2(2250)^+" },
      {   -9020325, "K_2(2250)^-" },
      {      20325, "K_2(1820)^+" },
      {     -20325, "K_2(1820)^-" },
      {     100325, "K_2(1980)^+" },
      {    -100325, "K_2(1980)^-" },
      {        317, "K*_3(1780)^0" },
      {       -317, "K*_3(1780)~^0" },
      {    9010317, "K_3(2320)^0" },
      {   -9010317, "K_3(2320)~^0" },
      {        327, "K*_3(1780)^+" },
      {       -327, "K*_3(1780)^-" },
      {    9010327, "K_3(2320)^+" },
      {   -9010327, "K_3(2320)^-" },
      {        319, "K*_4(2045)^0" },
      {       -319, "K*_4(2045)~^0" },
      {    9000319, "K_4(2500)^0" },
      {   -9000319, "K_4(2500)~^0" },
      {        329, "K*_4(2045)^+" },
      {       -329, "K*_4(2045)^-" },
      {    9000329, "K_4(2500)^+" },
      {   -9000329, "K_4(2500)^-" },
      {        411, "D^+" },
      {       -411, "D^-" },
      {      10411, "D*_0(2400)^+" },
      {     -10411, "D*_0(2400)^-" },
      {     100411, "D(2S)^+" },
      {    -100411, "D(2S)^-" },
      {        413, "D*(2010)^+" },
      {       -413, "D*(2010)^-" },
      {      10413, "D_1(2420)^+" },
      {     -10413, "D_1(2420)^-" },
      {      20413, "D_1(H)^+" },
      {     -20413, "D_1(H)^-" },
      {     100413, "D*(2S)^+" },
      {    -100413, "D*(2S)^-" },
      {        415, "D*_2(2460)^+" },
      {       -415, "D*_2(2460)^-" },
      {        421, "D^0" },
      {       -421, "D~^0" },
      {      10421, "D*_0(2400)^0" },
      {     -10421, "D*_0(2400)~^0" },
      {     100421, "D(2S)^0" },
      {    -100421, "D(2S)~^0" },
      {        423, "D*(2007)^0" },
      {       -423, "D*(2007)~^0" },
      {      10423, "D_1(2420)^0" },
      {     -10423, "D_1(2420)~^0" },
      {      20423, "D_1(2430)^0" },
      {     -20423, "D_1(2430)~^0" },
      {     100423, "D*(2S)^0" },
      {    -100423, "D*(2S)~^0" },
      {        425, "D*_2(2460)^0" },
      {       -425, "D*_2(2460)~^0" },
      {        431, "D_s^+" },
      {       -431, "D_s^-" },
      {      10431, "D*_s0(2317)^+" },
      {     -10431, "D*_s0(2317)^-" },
      {        433, "D*_s^+" },
      {       -433, "D*_s^-" },
      {      10433, "D_s1(2536)^+" },
      {     -10433, "D_s1(2536)^-" },
      {      20433, "D_s1(2460)^+" },
      {     -20433, "D_s1(2460)^-" },
      {        435, "D*_s2(2573)^+" },
      {       -435, "D*_s2(2573)^-" },
      {        441, "eta_c(1S)" },
      {      10441, "chi_c0(1P)" },
      {     100441, "eta_c(2S)" },
      {        443, "J/psi(1S)" },
      {    9000443, "psi(4040)" },
      {      10443, "hc(1P)" },
      {    9010443, "psi(4160)" },
      {      20443, "chi_c1(1P)" },
      {    9020443, "psi(4415)" },
      {      30443, "psi(3770)" },
      {     100443, "psi(2S)" },
      {        445, "chi_c2(1P)" },
      {     100445, "chi_c2(2P)" },
      {        511, "B^0" },
      {       -511, "B~^0" },
      {      10511, "B*_0^0" },
      {     -10511, "B*_0~^0" },
      {        513, "B*^0" },
      {       -513, "B*~^0" },
      {      10513, "B_1(L)^0" },
      {     -10513, "B_1(L)~^0" },
      {      20513, "B_1(H)^0" },
      {     -20513, "B_1(H)~^0" },
      {        515, "B*_2^0" },
      {       -515, "B*_2~^0" },
      {        521, "B^+" },
      {       -521, "B^-" },
      {      10521, "B*_0^+" },
      {     -10521, "B*_0^-" },
      {        523, "B*^+" },
      {       -523, "B*^-" },
      {      10523, "B_1(L)^+" },
      {     -10523, "B_1(L)^-" },
      {      20523, "B_1(H)^+" },
      {     -20523, "B_1(H)^-" },
      {        525, "B*_2^+" },
      {       -525, "B*_2^-" },
      {        531, "B_s^0" },
      {       -531, "B_s~^0" },
      {      10531, "B*_s0^0" },
      {     -10531, "B*_s0~^0" },
      {        533, "B*_s^0" },
      {       -533, "B*_s~^0" },
      {      10533, "B_s1(L)^0" },
      {     -10533, "B_s1(L)~^0" },
      {      20533, "B_s1(H)^0" },
      {     -20533, "B_s1(H)~^0" },
      {        535, "B*_s2^0" },
      {       -535, "B*_s2~^0" },
      {        541, "B_c^+" },
      {       -541, "B_c^-" },
      {      10541, "B*_c0^+" },
      {     -10541, "B*_c0^-" },
      {        543, "B*_c^+" },
      {       -543, "B*_c^-" },
      {      10543, "B_c1(L)^+" },
      {     -10543, "B_c1(L)^-" },
      {      20543, "B_c1(H)^+" },
      {     -20543, "B_c1(H)^-" },
      {        545, "B*_c2^+" },
      {       -545, "B*_c2^-" },
      {        551, "eta_b(1S)" },
      {      10551, "chi_b0(1P)" },
      {     100551, "eta_b(2S)" },
      {     110551, "chi_b0(2P)" },
      {     200551, "eta_b(3S)" },
      {     210551, "chi_b0(3P)" },
      {        553, "Upsilon(1S)" },
      {    9000553, "Upsilon(10860)" },
      {      10553, "h_b(1P)" },
      {    9010553, "Upsilon(11020)" },
      {      20553, "chi_b1(1P)" },
      {    9020553, "Upsilon(7S)" },
      {      30553, "Upsilon_1(1D)" },
      {     100553, "Upsilon(2S)" },
      {     110553, "h_b(2P)" },
      {     120553, "chi_b1(2P)" },
      {     130553, "Upsilon_1(2D)" },
      {     200553, "Upsilon(3S)" },
      {     210553, "h_b(3P)" },
      {     220553, "chi_b1(3P)" },
      {     300553, "Upsilon(4S)" },
      {        555, "chi_b2(1P)" },
      {      10555, "eta_b2(1D)" },
      {      20555, "Upsilon_2(1D)" },
      {     100555, "chi_b2(2P)" },
      {     110555, "eta_b2(2D)" },
      {     120555, "Upsilon_2(2D)" },
      {     200555, "chi_b2(3P)" },
      {        557, "Upsilon_3(1D)" },
      {     100557, "Upsilon_3(2D)" },
      {        611, "T^+" },
      {       -611, "T^-" },
      {        613, "T*^+" },
      {       -613, "T*^-" },
      {        621, "T^0" },
      {       -621, "T~^0" },
      {        623, "T*^0" },
      {       -623, "T*~^0" },
      {        631, "T_s^+" },
      {       -631, "T_s^-" },
      {        633, "T*_s^+" },
      {       -633, "T*_s^-" },
      {        641, "T_c^0" },
      {       -641, "T_c~^0" },
      {        643, "T*_c^0" },
      {       -643, "T*_c~^0" },
      {        651, "T_b^+" },
      {       -651, "T_b^-" },
      {        653, "T*_b^+" },
      {       -653, "T*_b^-" },
      {        661, "eta_t" },
      {        663, "theta" },
      {        711, "L^0" },
      {       -711, "L~^0" },
      {        713, "L*^0" },
      {       -713, "L*~^0" },
      {        721, "L^-" },
      {       -721, "L^+" },
      {        723, "L*^-" },
      {       -723, "L*^+" },
      {        731, "L_s^0" },
      {       -731, "L_s~^0" },
      {        733, "L*_s^0" },
      {       -733, "L*_s~^0" },
      {        741, "L_c^-" },
      {       -741, "L_c^+" },
      {        743, "L*_c^-" },
      {       -743, "L*_c^+" },
      {        751, "L_b^0" },
      {       -751, "L_b~^0" },
      {        753, "L*_b^0" },
      {       -753, "L*_b~^0" },
      {        761, "L_t^-" },
      {       -761, "L_t^+" },
      {        763, "L*_t^-" },
      {       -763, "L*_t^+" },
      {        771, "eta_l" },
      {        773, "theta_l" },
      {        811, "X^+" },
      {       -811, "X^-" },
      {        813, "X*^+" },
      {       -813, "X*^-" },
      {        821, "X^0" },
      {       -821, "X~^0" },
      {        823, "X*^0" },
      {       -823, "X*~^0" },
      {        831, "X_s^+" },
      {       -831, "X_s^-" },
      {        833, "X*_s^+" },
      {       -833, "X*_s^-" },
      {        841, "X_c^0" },
      {       -841, "X_c~^0" },
      {        843, "X*_c^0" },
      {       -843, "X*_c~^0" },
      {        851, "X_b^+" },
      {       -851, "X_b^-" },
      {        853, "X*_b^+" },
      {       -853, "X*_b^-" },
      {        861, "X_t^0" },
      {       -861, "X_t~^0" },
      {        863, "X*_t^0" },
      {       -863, "X*_t~^0" },
      {        871, "X_l^+" },
      {       -871, "X_l^-" },
      {        873, "X*_l^+" },
      {       -873, "X*_l^-" },
      {        881, "eta_h" },
      {        883, "theta_H" },
      {      30343, "Xsd" },
      {     -30343, "anti-Xsd" },
      {      30353, "Xsu" },
      {     -30353, "anti-Xsu" },
      {      30363, "Xss" },
      {     -30363, "anti-Xss" },
      {      30373, "Xdd" },
      {     -30373, "anti-Xdd" },
      {      30383, "Xdu" },
      {     -30383, "anti-Xdu" },
      {       2112, "n^0" },
      {      -2112, "n~^0" },
      {       2212, "p^+" },
      {      -2212, "p~^-" },
    {        12212,        "N(1440)^+"},
    {        12112,        "N(1440)^0"},
    {        22212,        "N(1535)^+"},
    {        22112,        "N(1535)^0"},
    {        32212,        "N(1650)^+"},
    {        32112,        "N(1650)^0"},
    {        42212,        "N(1710)^+"},
    {        42112,        "N(1710)^0"},
    {         1214,         "N(1520)^0"},
    {         2124,         "N(1520)^+"},
    {        21214,        "N(1700)^0"},
    {        22124,        "N(1700)^+"},
    {        31214,        "N(1720)^0"},
    {        32124,        "N(1720)^+"},
    {         2116,         "N(1675)^0"},
    {         2216,         "N(1675)^+"},
    {        12116,        "N(1680)^0"},
    {        12216,        "N(1680)^+"},
    {         1218,         "N(2190)^0"},
    {         2128,        "N(2190)^+" },
      {       1114, "Delta^-" },
      {      -1114, "Delta~^+" },
      {       2114, "Delta^0" },
      {      -2114, "Delta~^0" },
      {       2214, "Delta^+" },
      {      -2214, "Delta~^-" },
      {       2224, "Delta^++" },
      {      -2224, "Delta~^--" },
    {        31114,   "Delta(1600)^-"      },
    {        32114,   "Delta(1600)^0"      },
    {        32214,    "Delta(1600)^+"     },
    {        32224,     "Delta(1600)^++"    },
    {         1112,    "Delta(1620)^-"     },
    {         1212,    "Delta(1620)^0"     },
    {         2122,    "Delta(1620)^+"     },
    {         2222,     "Delta(1620)^++"    },
    {        11114,     "Delta(1700)^-"    },
    {        12114,     "Delta(1700)^0"    },
    {        12214,     "Delta(1700)^+"    },
    {        12224,      "Delta(1700)^++"   },
    {         1116,     "Delta(1905)^-"    },
    {         1216,     "Delta(1905)^0"    },
    {         2126,     "Delta(1905)^+"    },
    {         2226,      "Delta(1905)^++"   },
    {        21112,    "Delta(1910)^-"    },
    {        21212,     "Delta(1910)^0"   },
    {        22122,     "Delta(1910)^+"   },
    {        22222,     "Delta(1910)^++"   },
    {        21114,    "Delta(1920)^-"    },
    {        22114,     "Delta(1920)^0"   },
    {        22214,     "Delta(1920)^+"   },
    {        22224,    "Delta(1920)^++"    },
    {        11116,    "Delta(1930)^-"    },
    {        11216,     "Delta(1930)^0"   },
    {        12126,     "Delta(1930)^+"   },
    {        12226,     "Delta(1930)^++"   },
    {         1118,     "Delta(1950)^-"    },
    {         2118,      "Delta(1950)^0"   },
    {         2218,      "Delta(1950)^+"   },
    {         2228,     "Delta(1950)^++"    },
      {       3122, "Lambda^0" },
      {      -3122, "Lambda~^0" },
      {      13122, "Lambda(1405)^0" },
      {     -13122, "Lambda~(1405)^0" },
      {      23122, "Lambda(1600)^0" },
      {     -23122, "Lambda~(1600)^0" },
      {      33122, "Lambda(1670)^0" },
      {     -33122, "Lambda~(1670)^0" },
      {      43122, "Lambda(1800)^0" },
      {     -43122, "Lambda~(1800)^0" },
      {      53122, "Lambda(1810)^0" },
      {     -53122, "Lambda~(1810)^0" },
      {       3124, "Lambda(1520)^0" },
      {      -3124, "Lambda~(1520)^0" },
      {      13124, "Lambda(1690)^0" },
      {     -13124, "Lambda~(1690)^0" },
      {      23124, "Lambda(1890)^0" },
      {     -23124, "Lambda~(1890)^0" },
      {       3126, "Lambda(1820)^0" },
      {      -3126, "Lambda~(1820)^0" },
      {      13126, "Lambda(1830)^0" },
      {     -13126, "Lambda~(1830)^0" },
      {      23126, "Lambda(2110)^0" },
      {     -23126, "Lambda~(2110)^0" },
      {       3128, "Lambda(2100)^0" },
      {      -3128, "Lambda~(2100)^0" },
      {       3112, "Sigma^-" },
      {      -3112, "Sigma~^+" },
      {       3212, "Sigma^0" },
      {      -3212, "Sigma~^0" },
      {       3222, "Sigma^+" },
      {      -3222, "Sigma~^-" },
      {      13222, "Sigma(1660)^+" },
      {     -13222, "Sigma~(1660)^+" },
      {      13212, "Sigma(1660)^0" },
      {     -13212, "Sigma~(1660)^0" },
      {      13112, "Sigma(1660)^-" },
      {     -13112, "Sigma~(1660)^-" },
      {      23112, "Sigma(1750)^-" },
      {     -23112, "Sigma~(1750)^-" },
      {      23212, "Sigma(1750)^0" },
      {     -23212, "Sigma~(1750)^0" },
      {      23222, "Sigma(1750)^+" },
      {     -23222, "Sigma~(1750)^+" },
      {       3114, "Sigma*^-" },
      {      -3114, "Sigma*~^+" },
      {       3214, "Sigma*^0" },
      {      -3214, "Sigma*~^0" },
      {       3224, "Sigma*^+" },
      {      -3224, "Sigma*~^-" },
      {      13224, "Sigma(1670)^+" },
      {     -13224, "Sigma~(1670)^+" },
      {      13214, "Sigma(1670)^0" },
      {     -13214, "Sigma~(1670)^0" },
      {      13114, "Sigma(1670)^-" },
      {     -13114, "Sigma~(1670)^-" },
      {      23224, "Sigma(1940)^+" },
      {     -23224, "Sigma~(1940)^+" },
      {      23214, "Sigma(1940)^0" },
      {     -23214, "Sigma~(1940)^0" },
      {      23114, "Sigma(1940)^-" },
      {     -23114, "Sigma~(1940)^-" },
      {       3226, "Sigma(1775)^+" },
      {      -3226, "Sigma~(1775)^+" },
      {       3216, "Sigma(1775)^0" },
      {      -3216, "Sigma~(1775)^0" },
      {       3116, "Sigma(1775)^-" },
      {      -3116, "Sigma~(1775)^-" },
      {      13226, "Sigma(1915)^+" },
      {     -13226, "Sigma~(1915)^+" },
      {      13216, "Sigma(1915)^0" },
      {     -13216, "Sigma~(1915)^0" },
      {      13116, "Sigma(1915)^-" },
      {     -13116, "Sigma~(1915)^-" },
      {       3228, "Sigma(2030)^+" },
      {      -3228, "Sigma~(2030)^+" },
      {       3218, "Sigma(2030)^0" },
      {      -3218, "Sigma~(2030)^0" },
      {       3118, "Sigma(2030)^-" },
      {      -3118, "Sigma~(2030)^-" },
      {       3312, "Xi^-" },
      {      -3312, "Xi~^+" },
      {       3322, "Xi^0" },
      {      -3322, "Xi~^0" },
      {       3314, "Xi*^-" },
      {      -3314, "Xi*~^+" },
      {       3324, "Xi*^0" },
      {      -3324, "Xi*~^0" },
      {      13314, "Xi(1820)^-" },
      {     -13314, "Xi(1820)~^+" },
      {      13324, "Xi(1820)^0" },
      {     -13324, "Xi(1820)~^0" },
      {       3334, "Omega^-" },
      {      -3334, "Omega~^+" },
      {       4112, "Sigma_c^0" },
      {      -4112, "Sigma_c~^0" },
      {       4114, "Sigma*_c^0" },
      {      -4114, "Sigma*_c~^0" },
      {       4122, "Lambda_c^+" },
      {      -4122, "Lambda_c~^-" },
      {      14122, "Lambda_c(2593)^+" },
      {     -14122, "Lambda_c~(2593)^-" },
      {      14124, "Lambda_c(2625)^+" },
      {     -14124, "Lambda_c~(2625)^-" },
      {       4132, "Xi_c^0" },
      {      -4132, "Xi_c~^0" },
      {       4212, "Sigma_c^+" },
      {      -4212, "Sigma_c~^-" },
      {       4214, "Sigma*_c^+" },
      {      -4214, "Sigma*_c~^-" },
      {       4222, "Sigma_c^++" },
      {      -4222, "Sigma_c~^--" },
      {       4224, "Sigma*_c^++" },
      {      -4224, "Sigma*_c~^--" },
      {       4232, "Xi_c^+" },
      {      -4232, "Xi_c~^-" },
      {       4312, "Xi'_c^0" },
      {      -4312, "Xi'_c~^0" },
      {       4314, "Xi*_c^0" },
      {      -4314, "Xi*_c~^0" },
      {       4322, "Xi'_c^+" },
      {      -4322, "Xi'_c~^-" },
      {       4324, "Xi*_c^+" },
      {      -4324, "Xi*_c~^-" },
      {       4332, "Omega_c^0" },
      {      -4332, "Omega_c~^0" },
      {       4334, "Omega*_c^0" },
      {      -4334, "Omega*_c~^0" },
      {       4412, "Xi_cc^+" },
      {      -4412, "Xi_cc~^-" },
      {       4414, "Xi*_cc^+" },
      {      -4414, "Xi*_cc~^-" },
      {       4422, "Xi_cc^++" },
      {      -4422, "Xi_cc~^--" },
      {       4424, "Xi*_cc^++" },
      {      -4424, "Xi*_cc~^--" },
      {       4432, "Omega_cc^+" },
      {      -4432, "Omega_cc~^-" },
      {       4434, "Omega*_cc^+" },
      {      -4434, "Omega*_cc~^-" },
      {       4444, "Omega*_ccc^++" },
      {      -4444, "Omega*_ccc~^--" },
      {       5112, "Sigma_b^-" },
      {      -5112, "Sigma_b~^+" },
      {       5114, "Sigma*_b^-" },
      {      -5114, "Sigma*_b~^+" },
      {       5122, "Lambda_b^0" },
      {      -5122, "Lambda_b~^0" },
      {       5132, "Xi_b^-" },
      {      -5132, "Xi_b~^+" },
      {       5142, "Xi_bc^0" },
      {      -5142, "Xi_bc~^0" },
      {       5212, "Sigma_b^0" },
      {      -5212, "Sigma_b~^0" },
      {       5214, "Sigma*_b^0" },
      {      -5214, "Sigma*_b~^0" },
      {       5222, "Sigma_b^+" },
      {      -5222, "Sigma_b~^-" },
      {       5224, "Sigma*_b^+" },
      {      -5224, "Sigma*_b~^-" },
      {       5232, "Xi_b^0" },
      {      -5232, "Xi_b~^0" },
      {       5242, "Xi_bc^+" },
      {      -5242, "Xi_bc~^-" },
      {       5312, "Xi'_b^-" },
      {      -5312, "Xi'_b~^+" },
      {       5314, "Xi*_b^-" },
      {      -5314, "Xi*_b~^+" },
      {       5322, "Xi'_b^0" },
      {      -5322, "Xi'_b~^0" },
      {       5324, "Xi*_b^0" },
      {      -5324, "Xi*_b~^0" },
      {       5332, "Omega_b^-" },
      {      -5332, "Omega_b~^+" },
      {       5334, "Omega*_b^-" },
      {      -5334, "Omega*_b~^+" },
      {       5342, "Omega_bc^0" },
      {      -5342, "Omega_bc~^0" },
      {       5412, "Xi'_bc^0" },
      {      -5412, "Xi'_bc~^0" },
      {       5414, "Xi*_bc^0" },
      {      -5414, "Xi*_bc~^0" },
      {       5422, "Xi'_bc^+" },
      {      -5422, "Xi'_bc~^-" },
      {       5424, "Xi*_bc^+" },
      {      -5424, "Xi*_bc~^-" },
      {       5432, "Omega'_bc^0" },
      {      -5432, "Omega'_bc~^0" },
      {       5434, "Omega*_bc^0" },
      {      -5434, "Omega*_bc~^0" },
      {       5442, "Omega_bcc^+" },
      {      -5442, "Omega_bcc~^-" },
      {       5444, "Omega*_bcc^+" },
      {      -5444, "Omega*_bcc~^-" },
      {       5512, "Xi_bb^-" },
      {      -5512, "Xi_bb~^+" },
      {       5514, "Xi*_bb^-" },
      {      -5514, "Xi*_bb~^+" },
      {       5522, "Xi_bb^0" },
      {      -5522, "Xi_bb~^0" },
      {       5524, "Xi*_bb^0" },
      {      -5524, "Xi*_bb~^0" },
      {       5532, "Omega_bb^-" },
      {      -5532, "Omega_bb~^+" },
      {       5534, "Omega*_bb^-" },
      {      -5534, "Omega*_bb~^+" },
      {       5542, "Omega_bbc^0" },
      {      -5542, "Omega_bbc~^0" },
      {       5544, "Omega*_bbc^0" },
      {      -5544, "Omega*_bbc~^0" },
      {       5554, "Omega*_bbb^-" },
      {      -5554, "Omega*_bbb~^+" },
      {       6112, "Sigma_t^0" },
      {      -6112, "Sigma_t~^0" },
      {       6114, "Sigma*_t^0" },
      {      -6114, "Sigma*_t~^0" },
      {       6122, "Lambda_t^+" },
      {      -6122, "Lambda_t~^-" },
      {       6132, "Xi_t^0" },
      {      -6132, "Xi_t~^0" },
      {       6142, "Xi_tc^+" },
      {      -6142, "Xi_tc~^-" },
      {       6152, "Xi_tb^0" },
      {      -6152, "Xi_tb~^0" },
      {       6212, "Sigma_t^+" },
      {      -6212, "Sigma_t~^-" },
      {       6214, "Sigma*_t^+" },
      {      -6214, "Sigma*_t~^-" },
      {       6222, "Sigma_t^++" },
      {      -6222, "Sigma_t~^--" },
      {       6224, "Sigma*_t^++" },
      {      -6224, "Sigma*_t~^--" },
      {       6232, "Xi_t^+" },
      {      -6232, "Xi_t~^-" },
      {       6242, "Xi_tc^++" },
      {      -6242, "Xi_tc~^--" },
      {       6252, "Xi_tb^+" },
      {      -6252, "Xi_tb~^-" },
      {       6312, "Xi'_t^0" },
      {      -6312, "Xi'_t~^0" },
      {       6314, "Xi*_t^0" },
      {      -6314, "Xi*_t~^0" },
      {       6322, "Xi'_t^+" },
      {      -6322, "Xi'_t~^-" },
      {       6324, "Xi*_t^+" },
      {      -6324, "Xi*_t~^-" },
      {       6332, "Omega_t^0" },
      {      -6332, "Omega_t~^0" },
      {       6334, "Omega*_t^0" },
      {      -6334, "Omega*_t~^0" },
      {       6342, "Omega_tc^+" },
      {      -6342, "Omega_tc~^-" },
      {       6352, "Omega_tb^0" },
      {      -6352, "Omega_tb~^0" },
      {       6412, "Xi'_tc^+" },
      {      -6412, "Xi'_tc~^-" },
      {       6414, "Xi*_tc^+" },
      {      -6414, "Xi*_tc~^-" },
      {       6422, "Xi'_tc^++" },
      {      -6422, "Xi'_tc~^--" },
      {       6424, "Xi*_tc^++" },
      {      -6424, "Xi*_tc~^--" },
      {       6432, "Omega'_tc^+" },
      {      -6432, "Omega'_tc~^-" },
      {       6434, "Omega*_tc^+" },
      {      -6434, "Omega*_tc~^-" },
      {       6442, "Omega_tcc^++" },
      {      -6442, "Omega_tcc~^--" },
      {       6444, "Omega*_tcc^++" },
      {      -6444, "Omega*_tcc~^--" },
      {       6452, "Omega_tbc^+" },
      {      -6452, "Omega_tbc~^-" },
      {       6512, "Xi'_tb^0" },
      {      -6512, "Xi'_tb~^0" },
      {       6514, "Xi*_tb^0" },
      {      -6514, "Xi*_tb~^0" },
      {       6522, "Xi'_tb^+" },
      {      -6522, "Xi'_tb~^-" },
      {       6524, "Xi*_tb^+" },
      {      -6524, "Xi*_tb~^-" },
      {       6532, "Omega'_tb^0" },
      {      -6532, "Omega'_tb~^0" },
      {       6534, "Omega*_tb^0" },
      {      -6534, "Omega*_tb~^0" },
      {       6542, "Omega'_tbc^+" },
      {      -6542, "Omega'_tbc~^-" },
      {       6544, "Omega*_tbc^+" },
      {      -6544, "Omega*_tbc~^-" },
      {       6552, "Omega_tbb^0" },
      {      -6552, "Omega_tbb~^0" },
      {       6554, "Omega*_tbb^0" },
      {      -6554, "Omega*_tbb~^0" },
      {       6612, "Xi_tt^+" },
      {      -6612, "Xi_tt~^-" },
      {       6614, "Xi*_tt^+" },
      {      -6614, "Xi*_tt~^-" },
      {       6622, "Xi_tt^++" },
      {      -6622, "Xi_tt~^--" },
      {       6624, "Xi*_tt^++" },
      {      -6624, "Xi*_tt~^--" },
      {       6632, "Omega_tt^+" },
      {      -6632, "Omega_tt~^-" },
      {       6634, "Omega*_tt^+" },
      {      -6634, "Omega*_tt~^-" },
      {       6642, "Omega_ttc^++" },
      {      -6642, "Omega_ttc~^--" },
      {       6644, "Omega*_ttc^++" },
      {      -6644, "Omega*_ttc~^--" },
      {       6652, "Omega_ttb^+" },
      {      -6652, "Omega_ttb~^-" },
      {       6654, "Omega*_ttb^+" },
      {      -6654, "Omega*_ttb~^-" },
      {       6664, "Omega*_ttt^++" },
      {      -6664, "Omega*_ttt~^--" },
      {       7112, "Sigma_b'^-" },
      {      -7112, "Sigma_b'~^+" },
      {       7114, "Sigma*_b'^-" },
      {      -7114, "Sigma*_b'~^+" },
      {       7122, "Lambda_b'^0" },
      {      -7122, "Lambda_b'~^0" },
      {       7132, "Xi_b'^-" },
      {      -7132, "Xi_b'~^+" },
      {       7142, "Xi_b'c^0" },
      {      -7142, "Xi_b'c~^0" },
      {       7152, "Xi_b'b^-" },
      {      -7152, "Xi_b'b~^+" },
      {       7162, "Xi_b't^0" },
      {      -7162, "Xi_b't~^0" },
      {       7212, "Sigma_b'^0" },
      {      -7212, "Sigma_b'~^0" },
      {       7214, "Sigma*_b'^0" },
      {      -7214, "Sigma*_b'~^0" },
      {       7222, "Sigma_b'^+" },
      {      -7222, "Sigma_b'~^-" },
      {       7224, "Sigma*_b'^+" },
      {      -7224, "Sigma*_b'~^-" },
      {       7232, "Xi_b'^0" },
      {      -7232, "Xi_b'~^0" },
      {       7242, "Xi_b'c^+" },
      {      -7242, "Xi_b'c~^-" },
      {       7252, "Xi_b'b^0" },
      {      -7252, "Xi_b'b~^0" },
      {       7262, "Xi_b't^+" },
      {      -7262, "Xi_b't~^-" },
      {       7312, "Xi'_b'^-" },
      {      -7312, "Xi'_b'~^+" },
      {       7314, "Xi*_b'^-" },
      {      -7314, "Xi*_b'~^+" },
      {       7322, "Xi'_b'^0" },
      {      -7322, "Xi'_b'~^0" },
      {       7324, "Xi*_b'^0" },
      {      -7324, "Xi*_b'~^0" },
      {       7332, "Omega'_b'^-" },
      {      -7332, "Omega'_b'~^+" },
      {       7334, "Omega*_b'^-" },
      {      -7334, "Omega*_b'~^+" },
      {       7342, "Omega_b'c^0" },
      {      -7342, "Omega_b'c~^0" },
      {       7352, "Omega_b'b^-" },
      {      -7352, "Omega_b'b~^+" },
      {       7362, "Omega_b't^0" },
      {      -7362, "Omega_b't~^0" },
      {       7412, "Xi'_b'c^0" },
      {      -7412, "Xi'_b'c~^0" },
      {       7414, "Xi*_b'c^0" },
      {      -7414, "Xi*_b'c~^0" },
      {       7422, "Xi'_b'c^+" },
      {      -7422, "Xi'_b'c~^-" },
      {       7424, "Xi*_b'c^+" },
      {      -7424, "Xi*_b'c~^-" },
      {       7432, "Omega'_b'c^0" },
      {      -7432, "Omega'_b'c~^0" },
      {       7434, "Omega*_b'c^0" },
      {      -7434, "Omega*_b'c~^0" },
      {       7442, "Omega'_b'cc^+" },
      {      -7442, "Omega'_b'cc~^-" },
      {       7444, "Omega*_b'cc^+" },
      {      -7444, "Omega*_b'cc~^-" },
      {       7452, "Omega_b'bc^0" },
      {      -7452, "Omega_b'bc~^0" },
      {       7462, "Omega_b'tc^+" },
      {      -7462, "Omega_b'tc~^-" },
      {       7512, "Xi'_b'b^-" },
      {      -7512, "Xi'_b'b~^+" },
      {       7514, "Xi*_b'b^-" },
      {      -7514, "Xi*_b'b~^+" },
      {       7522, "Xi'_b'b^0" },
      {      -7522, "Xi'_b'b~^0" },
      {       7524, "Xi*_b'b^0" },
      {      -7524, "Xi*_b'b~^0" },
      {       7532, "Omega'_b'b^-" },
      {      -7532, "Omega'_b'b~^+" },
      {       7534, "Omega*_b'b^-" },
      {      -7534, "Omega*_b'b~^+" },
      {       7542, "Omega'_b'bc^0" },
      {      -7542, "Omega'_b'bc~^0" },
      {       7544, "Omega*_b'bc^0" },
      {      -7544, "Omega*_b'bc~^0" },
      {       7552, "Omega'_b'bb^-" },
      {      -7552, "Omega'_b'bb~^+" },
      {       7554, "Omega*_b'bb^-" },
      {      -7554, "Omega*_b'bb~^+" },
      {       7562, "Omega_b'tb^0" },
      {      -7562, "Omega_b'tb~^0" },
      {       7612, "Xi'_b't^0" },
      {      -7612, "Xi'_b't~^0" },
      {       7614, "Xi*_b't^0" },
      {      -7614, "Xi*_b't~^0" },
      {       7622, "Xi'_b't^+" },
      {      -7622, "Xi'_b't~^-" },
      {       7624, "Xi*_b't^+" },
      {      -7624, "Xi*_b't~^-" },
      {       7632, "Omega'_b't^0" },
      {      -7632, "Omega'_b't~^0" },
      {       7634, "Omega*_b't^0" },
      {      -7634, "Omega*_b't~^0" },
      {       7642, "Omega'_b'tc^+" },
      {      -7642, "Omega'_b'tc~^-" },
      {       7644, "Omega*_b'tc^+" },
      {      -7644, "Omega*_b'tc~^-" },
      {       7652, "Omega'_b'tb^0" },
      {      -7652, "Omega'_b'tb~^0" },
      {       7654, "Omega*_b'tb^0" },
      {      -7654, "Omega*_b'tb~^0" },
      {       7662, "Omega'_b'tt^+" },
      {      -7662, "Omega'_b'tt~^-" },
      {       7664, "Omega*_b'tt^+" },
      {      -7664, "Omega*_b'tt~^-" },
      {       7712, "Xi'_b'b'^-" },
      {      -7712, "Xi'_b'b'~^+" },
      {       7714, "Xi*_b'b'^-" },
      {      -7714, "Xi*_b'b'~^+" },
      {       7722, "Xi'_b'b'^0" },
      {      -7722, "Xi'_b'b'~^0" },
      {       7724, "Xi*_b'b'^0" },
      {      -7724, "Xi*_b'b'~^0" },
      {       7732, "Omega'_b'b'^-" },
      {      -7732, "Omega'_b'b'~^+" },
      {       7734, "Omega*_b'b'^-" },
      {      -7734, "Omega*_b'b'~^+" },
      {       7742, "Omega'_b'b'c^0" },
      {      -7742, "Omega'_b'b'c~^0" },
      {       7744, "Omega*_b'b'c^0" },
      {      -7744, "Omega*_b'b'c~^0" },
      {       7752, "Omega'_b'b'b^-" },
      {      -7752, "Omega'_b'b'b~^+" },
      {       7754, "Omega*_b'b'b^-" },
      {      -7754, "Omega*_b'b'b~^+" },
      {       7762, "Omega'_b'b't^0" },
      {      -7762, "Omega'_b'b't~^0" },
      {       7764, "Omega*_b'b't^0" },
      {      -7764, "Omega*_b'b't~^0" },
      {       7774, "Omega*_b'b'b'^-" },
      {      -7774, "Omega*_b'b'b'~^+" },
      {       8112, "Sigma_t'^0" },
      {      -8112, "Sigma_t'~^0" },
      {       8114, "Sigma*_t'^0" },
      {      -8114, "Sigma*_t'~^0" },
      {       8122, "Lambda_t'^+" },
      {      -8122, "Lambda_t'~^-" },
      {       8132, "Xi_t'^0" },
      {      -8132, "Xi_t'~^0" },
      {       8142, "Xi_t'c^+" },
      {      -8142, "Xi_t'c~^-" },
      {       8152, "Xi_t'b^0" },
      {      -8152, "Xi_t'b~^0" },
      {       8162, "Xi_t't^+" },
      {      -8162, "Xi_t't~^-" },
      {       8172, "Xi_t'b'^0" },
      {      -8172, "Xi_t'b'~^0" },
      {       8212, "Sigma_t'^+" },
      {      -8212, "Sigma_t'~^-" },
      {       8214, "Sigma*_t'^+" },
      {      -8214, "Sigma*_t'~^-" },
      {       8222, "Sigma_t'^++" },
      {      -8222, "Sigma_t'~^--" },
      {       8224, "Sigma*_t'^++" },
      {      -8224, "Sigma*_t'~^--" },
      {       8232, "Xi_t'^+" },
      {      -8232, "Xi_t'~^-" },
      {       8242, "Xi_t'c^++" },
      {      -8242, "Xi_t'c~^--" },
      {       8252, "Xi_t'b^+" },
      {      -8252, "Xi_t'b~^-" },
      {       8262, "Xi_t't^++" },
      {      -8262, "Xi_t't~^--" },
      {       8272, "Xi_t'b'^+" },
      {      -8272, "Xi_t'b'~^-" },
      {       8312, "Xi'_t'^0" },
      {      -8312, "Xi'_t'~^0" },
      {       8314, "Xi*_t'^0" },
      {      -8314, "Xi*_t'~^0" },
      {       8322, "Xi'_t'^+" },
      {      -8322, "Xi'_t'~^-" },
      {       8324, "Xi*_t'^+" },
      {      -8324, "Xi*_t'~^-" },
      {       8332, "Omega'_t'^0" },
      {      -8332, "Omega'_t'~^0" },
      {       8334, "Omega*_t'^0" },
      {      -8334, "Omega*_t'~^0" },
      {       8342, "Omega_t'c^+" },
      {      -8342, "Omega_t'c~^-" },
      {       8352, "Omega_t'b^0" },
      {      -8352, "Omega_t'b~^0" },
      {       8362, "Omega_t't^+" },
      {      -8362, "Omega_t't~^-" },
      {       8372, "Omega_t'b'^0" },
      {      -8372, "Omega_t'b'~^0" },
      {       8412, "Xi'_t'c^+" },
      {      -8412, "Xi'_t'c~^-" },
      {       8414, "Xi*_t'c^+" },
      {      -8414, "Xi*_t'c~^-" },
      {       8422, "Xi'_t'c^++" },
      {      -8422, "Xi'_t'c~^--" },
      {       8424, "Xi*_t'c^++" },
      {      -8424, "Xi*_t'c~^--" },
      {       8432, "Omega'_t'c^+" },
      {      -8432, "Omega'_t'c~^-" },
      {       8434, "Omega*_t'c^+" },
      {      -8434, "Omega*_t'c~^-" },
      {       8442, "Omega'_t'cc^++" },
      {      -8442, "Omega'_t'cc~^--" },
      {       8444, "Omega*_t'cc^++" },
      {      -8444, "Omega*_t'cc~^--" },
      {       8452, "Omega_t'bc^+" },
      {      -8452, "Omega_t'bc~^-" },
      {       8462, "Omega_t'tc^++" },
      {      -8462, "Omega_t'tc~^--" },
      {       8472, "Omega_t'b'c ^+" },
      {      -8472, "Omega_t'b'c ~^-" },
      {       8512, "Xi'_t'b^0" },
      {      -8512, "Xi'_t'b~^0" },
      {       8514, "Xi*_t'b^0" },
      {      -8514, "Xi*_t'b~^0" },
      {       8522, "Xi'_t'b^+" },
      {      -8522, "Xi'_t'b~^-" },
      {       8524, "Xi*_t'b^+" },
      {      -8524, "Xi*_t'b~^-" },
      {       8532, "Omega'_t'b^0" },
      {      -8532, "Omega'_t'b~^0" },
      {       8534, "Omega*_t'b^0" },
      {      -8534, "Omega*_t'b~^0" },
      {       8542, "Omega'_t'bc^+" },
      {      -8542, "Omega'_t'bc~^-" },
      {       8544, "Omega*_t'bc^+" },
      {      -8544, "Omega*_t'bc~^-" },
      {       8552, "Omega'_t'bb^0" },
      {      -8552, "Omega'_t'bb~^0" },
      {       8554, "Omega*_t'bb^0" },
      {      -8554, "Omega*_t'bb~^0" },
      {       8562, "Omega_t'tb^+" },
      {      -8562, "Omega_t'tb~^-" },
      {       8572, "Omega_t'b'b ^0" },
      {      -8572, "Omega_t'b'b ~^0" },
      {       8612, "Xi'_t't^+" },
      {      -8612, "Xi'_t't~^-" },
      {       8614, "Xi*_t't^+" },
      {      -8614, "Xi*_t't~^-" },
      {       8622, "Xi'_t't^++" },
      {      -8622, "Xi'_t't~^--" },
      {       8624, "Xi*_t't^++" },
      {      -8624, "Xi*_t't~^--" },
      {       8632, "Omega'_t't^+" },
      {      -8632, "Omega'_t't~^-" },
      {       8634, "Omega*_t't^+" },
      {      -8634, "Omega*_t't~^-" },
      {       8642, "Omega'_t'tc^++" },
      {      -8642, "Omega'_t'tc~^--" },
      {       8644, "Omega*_t'tc^++" },
      {      -8644, "Omega*_t'tc~^--" },
      {       8652, "Omega'_t'tb^+" },
      {      -8652, "Omega'_t'tb~^-" },
      {       8654, "Omega*_t'tb^+" },
      {      -8654, "Omega*_t'tb~^-" },
      {       8662, "Omega'_t'tt^++" },
      {      -8662, "Omega'_t'tt~^--" },
      {       8664, "Omega*_t'tt^++" },
      {      -8664, "Omega*_t'tt~^--" },
      {       8672, "Omega_t'b't ^+" },
      {      -8672, "Omega_t'b't ~^-" },
      {       8712, "Xi'_t'b'^0" },
      {      -8712, "Xi'_t'b'~^0" },
      {       8714, "Xi*_t'b'^0" },
      {      -8714, "Xi*_t'b'~^0" },
      {       8722, "Xi'_t'b'^+" },
      {      -8722, "Xi'_t'b'~^-" },
      {       8724, "Xi*_t'b'^+" },
      {      -8724, "Xi*_t'b'~^-" },
      {       8732, "Omega'_t'b'^0" },
      {      -8732, "Omega'_t'b'~^0" },
      {       8734, "Omega*_t'b'^0" },
      {      -8734, "Omega*_t'b'~^0" },
      {       8742, "Omega'_t'b'c^+" },
      {      -8742, "Omega'_t'b'c~^-" },
      {       8744, "Omega*_t'b'c^+" },
      {      -8744, "Omega*_t'b'c~^-" },
      {       8752, "Omega'_t'b'b^0" },
      {      -8752, "Omega'_t'b'b~^0" },
      {       8754, "Omega*_t'b'b^0" },
      {      -8754, "Omega*_t'b'b~^0" },
      {       8762, "Omega'_t'b't^+" },
      {      -8762, "Omega'_t'b't~^-" },
      {       8764, "Omega*_t'b't^+" },
      {      -8764, "Omega*_t'b't~^-" },
      {       8772, "Omega'_t'b'b'^0" },
      {      -8772, "Omega'_t'b'b'~^0" },
      {       8774, "Omega*_t'b'b'^0" },
      {      -8774, "Omega*_t'b'b'~^0" },
      {       8812, "Xi'_t't'^+" },
      {      -8812, "Xi'_t't'~^-" },
      {       8814, "Xi*_t't'^+" },
      {      -8814, "Xi*_t't'~^-" },
      {       8822, "Xi'_t't'^++" },
      {      -8822, "Xi'_t't'~^--" },
      {       8824, "Xi*_t't'^++" },
      {      -8824, "Xi*_t't'~^--" },
      {       8832, "Omega'_t't'^+" },
      {      -8832, "Omega'_t't'~^-" },
      {       8834, "Omega*_t't'^+" },
      {      -8834, "Omega*_t't'~^-" },
      {       8842, "Omega'_t't'c^++" },
      {      -8842, "Omega'_t't'c~^--" },
      {       8844, "Omega*_t't'c^++" },
      {      -8844, "Omega*_t't'c~^--" },
      {       8852, "Omega'_t't'b^+" },
      {      -8852, "Omega'_t't'b~^-" },
      {       8854, "Omega*_t't'b^+" },
      {      -8854, "Omega*_t't'b~^-" },
      {       8862, "Omega'_t't't^++" },
      {      -8862, "Omega'_t't't~^--" },
      {       8864, "Omega*_t't't^++" },
      {      -8864, "Omega*_t't't~^--" },
      {       8872, "Omega'_t't'b'^+" },
      {      -8872, "Omega'_t't'b'~^-" },
      {       8874, "Omega*_t't'b'^+" },
      {      -8874, "Omega*_t't'b'~^-" },
      {       8884, "Omega*_t't't'^++" },
      {      -8884, "Omega*_t't't'~^--" },
      {    9221132, "Theta^+" },
      {    9331122, "Phi^--" },
      {    1000993, "R_~gg^0" },
      {    1009113, "R_~gd~d^0" },
      {    1009213, "R_~gu~d^+" },
      {    1009223, "R_~gu~u^0" },
      {    1009313, "R_~gd~s^0" },
      {    1009323, "R_~gu~s^+" },
      {    1009333, "R_~gs~s^0" },
      {    1091114, "R_~gddd^-" },
      {    1092114, "R_~gudd^0" },
      {    1092214, "R_~guud^+" },
      {    1092224, "R_~guuu^++" },
      {    1093114, "R_~gsdd^-" },
      {    1093214, "R_~gsud^0" },
      {    1093224, "R_~gsuu^+" },
      {    1093314, "R_~gssd^-" },
      {    1093324, "R_~gssu^0" },
      {    1093334, "R_~gsss^-" },
      {    1000612, "R_~t_1~d^+" },
      {    1000622, "R_~t_1~u^0" },
      {    1000632, "R_~t_1~s^+" },
      {    1000642, "R_~t_1~c^0" },
      {    1000652, "R_~t_1~b^+" },
      {    1006113, "R_~t_1dd_1^0" },
      {    1006211, "R_~t_1ud_0^+" },
      {    1006213, "R_~t_1ud_1^+" },
      {    1006223, "R_~t_1uu_1^++" },
      {    1006311, "R_~t_1sd_0^0" },
      {    1006313, "R_~t_1sd_1^0" },
      {    1006321, "R_~t_1su_0^+" },
      {    1006323, "R_~t_1su_1^+" },
      {    1006333, "R_~t_1ss_1^0" },
      { 1000010010, "Hydrogen" },
      { 1000010020, "Deuterium" },
      {-1000010020, "Anti-Deuterium" },
      { 1000010030, "Tritium" },
      {-1000010030, "Anti-Tritium" },
      { 1000020030, "He3" },
      {-1000020030, "Anti-He3" },
      { 1000020040, "Alpha-(He4)" },
      {-1000020040, "Anti-Alpha-(He4)" }
  };

  int lnames = sizeof(SNames)/sizeof(SNames[0]);
  for( int k=0; k!=lnames; ++k) {
      m.insert( std::make_pair( SNames[k].pid, std::string(SNames[k].pname)) );
      nameMap.insert( std::make_pair( std::string(SNames[k].pname), SNames[k].pid ) );
  }
  static ParticleNameMap mymaps(m,nameMap);

  return mymaps;
}  // ParticleNameInit()

void writeParticleNameLine( int i, std::ostream & os  )
{
    if ( validParticleName( i ) ) {
        std::string pn = particleName( i );
	int pid = particleName( pn );
	os << " PDT number: " ;
	os.width(12);
	os << i << " PDT name: " << pn << std::endl;
	// verify reverse lookup
	if( pid != i ) {
	    os << "HepPID::writeParticleNameLine ERROR: "
	       << " got " << pid << " instead of " << i << std::endl;
	}
    }
    return;
}  // writeParticleNameLine()

std::string dyonName( const int & pid )
{
    std::ostringstream pn;
    pn << "Dyon^" << digit(nq1,pid) << digit(nq2,pid) << digit(nq3,pid);
    if ( digit(nl,pid) == 1 ) {
       if ( pid > 0 ) {
	  pn << "++";
       } else {
	  pn << "--";
       }
    } else if ( digit(nl,pid) == 2 ) {
       if ( pid > 0 ) {
	  pn << "+-";
       } else {
	  pn << "-+";
       }
    }
    return pn.str();
}

std::string qballName( const int & pid )
{
    std::ostringstream pn;
    pn << "QBall^" << ((abspid(pid)/100)%1000) << "." << digit(nq3,pid);
    if ( pid > 0 ) {
       pn << "+";
    } else {
       pn << "-";
    }
    return pn.str();
}

int  checkForSpecialParticle( const std::string & s )
{
    int chg, chg2, id;
    int m = 1;
    int len = s.length();
    if( s.substr(0,4) == "Dyon" ) {
       std::istringstream var1(s.substr(5,3).c_str());
       var1 >> chg;
       if( s.substr(len-2,1) == "+" && s.substr(len-1,1) == "-") m = 2;
       if( s.substr(len-2,1) == "-" && s.substr(len-1,1) == "+") m = 2;
       id = 4100000 + m*10000 + chg*10;
       if( s.substr(len-2,1) == "-" ) id = -id;
       return id;
    }
    if( s.substr(0,5) == "QBall" ) {
       int rem = len - 9;
       std::istringstream var2(s.substr(6,rem).c_str());
       var2 >> chg;
       std::istringstream var3(s.substr(7+rem,1).c_str());
       var3 >> chg2;
       id = 10000000 + chg*100+chg2*10;
       if( s.substr(len-1,1) == "-" ) id = -id;
       return id;
    }
    return 0;
}

} // unnamed namespace

//
// getPartcleIdMap is the ONLY function allowed to call ParticleNameInit
//
ParticleNameMap const &  getParticleNameMap()
{
  static  ParticleNameMap const &  pmap = ParticleNameInit();
  return pmap;
}  // getPartcleIdMap()

bool validParticleName( const int & pid )
{
    // check for the special cases first
    if ( isDyon(pid) ) return true;
    if ( isQBall(pid) ) return true;
    
    static  ParticleNameMap const &  pmap = getParticleNameMap();

    ParticleNameMap::idIterator const cit = pmap.find( pid );
    return ( cit == pmap.end() )
         ? false
	 : true;
}  // validParticleName()

bool validParticleName( const std::string & s )
{
    static  ParticleNameMap const &  pmap = getParticleNameMap();
    ParticleNameMap::nameIterator const cit = pmap.findString( s );
    return ( cit == pmap.endLookupMap() )
         ? false
	 : true;
}  // validParticleName()

std::string  particleName( const int & pid )
{
    // check for the special cases first
    if ( isDyon(pid) ) return dyonName(pid);
    if ( isQBall(pid) ) return qballName(pid);
    
    static  ParticleNameMap const &  pmap = getParticleNameMap();

    ParticleNameMap::idIterator const cit = pmap.find( pid );
    return ( cit == pmap.end() )
         ? std::string("not defined")
	 : cit->second;
}  // particleName()

int  particleName( const std::string & s )
{
    static  ParticleNameMap const &  pmap = getParticleNameMap();
    ParticleNameMap::nameIterator const cit = pmap.findString( s );
    return ( cit == pmap.endLookupMap() )
         ? checkForSpecialParticle(s)
	 : cit->second;
}  // particleName()

//
// list all the defined names
//
void  listParticleNames( std::ostream & os  )
{
    writeVersion( os );
    os << "     HepPID Particle List" << std::endl;
    os << std::endl;

    // simple: static  PartcleIdMap const &  pmap = getPartcleIdMap();
    // simple: for( PartcleIdMap::const_iterator cit = pmap.begin(), mend = pmap.end(); 
    // simple:                                 cit != mend;
	// simple: 			  ++cit ) {
	// simple: os << "  PDT number: " ;
	// simple: os.width(12);
	// simple: os << cit->first << "  PDT name: " << cit->second << std::endl;
    // simple: }
    int id, i, j, q1, q2, q3, l, m, n;
    // special cases
    for( id=1; id<101; ++id) {
        writeParticleNameLine(  id, os );
        writeParticleNameLine( -id, os );
    }
    for( i=11; i<1000; ++i) {
        id = i*10;
        writeParticleNameLine(  id, os );
        writeParticleNameLine( -id, os );
    }
    // SUSY
    for( n=1; n<3; ++n) {
	for( q1=0; q1<10; ++q1) {
	    for( j=0; j<10; ++j) {
		id = 1000000*n+10*q1+j;
        	writeParticleNameLine(  id, os );
        	writeParticleNameLine( -id, os );
	    }
	}
    }
    // technicolor, etc.
    for( n=3; n<7; ++n) {
        for( q2=0; q2<10; ++q2) {
	    for( q1=0; q1<10; ++q1) {
	        for( j=0; j<10; ++j) {
		    for( m=0; m<10; ++m) {
		        for( l=0; l<7; ++l) {
			    id = 1000000*n+100000*m+10000*l+100*q2+10*q1+j;
			    // save dyons for later
			    if( !(n == 4 && m == 1) ) {
        		    writeParticleNameLine(  id, os );
        		    writeParticleNameLine( -id, os );
			    }
			}
		    }
		}
	    }
	}
    }
    // R-hadrons
    for( q3=0; q3<10; ++q3) {
	for( q2=1; q2<10; ++q2) {
	    for( q1=1; q1<10; ++q1) {
		for( j=1; j<5; ++j) {
		    id = 1000000+1000*q3+100*q2+10*q1+j;
        	    writeParticleNameLine( id, os );
		    if(q3 > 0 ) id = 1000000+90000+1000*q3+100*q2+10*q1+j;
        	    writeParticleNameLine( id, os );
		}
	    }
	}
    }
    // miscellaneous generator particles
    for( l=0; l<9; ++l) {
	for( i=1; i<100; ++i) {
	    id = 9900000+10000*l+i;
            writeParticleNameLine(  id, os );
            writeParticleNameLine( -id, os );
	}
        for( q3=0; q3<10; ++q3) {
            for( q2=1; q2<10; ++q2) {
		for( q1=1; q1<10; ++q1) {
	            for( j=0; j<10; ++j) {
			id = 9900000+10000*l+1000*q3+100*q2+10*q1+j;
        		writeParticleNameLine(  id, os );
        		writeParticleNameLine( -id, os );
		    }
		}
	    }
	}
    }
    // diquark
    for( i=11; i<100; ++i) {
        for( j=0; j<10; ++j) {
            id = 100*i+j;
            writeParticleNameLine(  id, os );
            writeParticleNameLine( -id, os );
	}
    }
    // mesons
    for( q2=1; q2<10; ++q2) {
	for( q1=1; q1<10; ++q1) {
	    for( j=1; j<10; ++j) {
		for( m=0; m<9; ++m) {
		    for( l=0; l<10; ++l) {
			id = 100000*m+10000*l+100*q2+10*q1+j;
        		writeParticleNameLine(  id, os );
        		writeParticleNameLine( -id, os );
			id = 9000000+100000*m+10000*l+100*q2+10*q1+j;
        		writeParticleNameLine(  id, os );
        		writeParticleNameLine( -id, os );
		    }
		}
	    }
	}
    }
    // baryons
    for( q3=1; q3<10; ++q3) {
	for( q2=1; q2<10; ++q2) {
	    for( q1=1; q1<10; ++q1) {
		for( j=1; j<10; ++j) {
		    for( m=0; m<9; ++m) {
			id = 10000*m+1000*q3+100*q2+10*q1+j;
        		writeParticleNameLine(  id, os );
        		writeParticleNameLine( -id, os );
		    }
		}
	    }
	}
    }
    // pentaquarks
    for( l=1; l<9; ++l ) {
        for ( m=1; m<9; ++m ) {
	    for( q3=1; q3<9; ++q3) {
		for( q2=1; q2<9; ++q2) {
		    for( q1=1; q1<9; ++q1) {
			id = 9*1000000+l*100000+m*10000+1000*q3+100*q2+10*q1+2;
        		writeParticleNameLine(  id, os );
        		writeParticleNameLine( -id, os );
		    }
		}
	    }
        }
    }
    // ions
    for( i=1; i<3; ++i) {
	for( m=1; m<5; ++m) {
		id = 1000000000+10*m+10000*i;
        	writeParticleNameLine(  id, os );
        	writeParticleNameLine( -id, os );
	}
    }
    // some Dyons
    for( q3=0; q3<2; ++q3) {
	for( q2=0; q2<4; ++q2) {
	    for( q1=0; q1<10; ++q1) {
	        ++q1;
        	id = 4110000+1000*q3+100*q2+10*q1;
		writeParticleNameLine(  id, os );
		writeParticleNameLine( -id, os );
        	id = 4120000+1000*q3+100*q2+10*q1;
		writeParticleNameLine(  id, os );
		writeParticleNameLine( -id, os );
	    }
	}
    }
    // a few QBalls
    for( i=1; i<199; ++i ) {
	for( m=1; m<10; ) {
		id = 10000000+10*m+100*i;
        	writeParticleNameLine(  id, os );
        	writeParticleNameLine( -id, os );
		m += 3;
	}
	i += 11;
    }
    return;
}  // listParticleNames()

}	// HepPID
