/*  color.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "color.h"
#include "functions.h"
#include "ginacutils.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {
//
//
//
GINAC_IMPLEMENT_REGISTERED_CLASS_OPT(idx2, idx,
		print_func<print_context>(&idx2::do_print))

idx2::idx2() {
}

idx2::idx2(const GiNaC::ex& v, const GiNaC::ex& dim1, const GiNaC::ex& dim2) :
	inherited(v, dim1), dim2_(dim2) {
}

idx2::idx2(const GiNaC::idx& i, const GiNaC::ex& dim2) :
	inherited(i.get_value(), i.get_dim()), dim2_(dim2) {
}

// taken from idx::subs(...)
GiNaC::ex idx2::subs(const GiNaC::exmap & m, unsigned options) const {
	// First look for index substitutions
	exmap::const_iterator it = m.find(*this);
	if (it != m.end()) {

		// really substitute the index by something else
		if (is_a<idx2> (it->second) || options & subs_options::really_subs_idx)
			return it->second;

		// Substitution idx2->idx, we want an idx2 again
		if (is_a<idx> (it->second))
			return idx2(ex_to<idx> (it->second), dim2_);

		// Otherwise substitute value
		idx2 *i_copy = duplicate();
		i_copy->value = it->second;
		i_copy->clearflag(status_flags::hash_calculated);
		return i_copy->setflag(status_flags::dynallocated);
	}

	// None, substitute objects in value (not in dimension)
	const ex &subsed_value = value.subs(m, options);
	if (are_ex_trivially_equal(value, subsed_value))
		return *this;

	idx2 *i_copy = duplicate();
	i_copy->value = subsed_value;
	i_copy->clearflag(status_flags::hash_calculated);
	return i_copy->setflag(status_flags::dynallocated);
}

bool idx2::is_dummy_pair_same_type(const GiNaC::basic& other) const {
	ASSERT(is_a<idx2>(other));
	const idx2 &o = static_cast<const idx2&> (other);

	if (!dim2_.is_equal(o.dim2_))
		return false;

	return inherited::is_dummy_pair_same_type(other);
}

int idx2::compare_same_type(const GiNaC::basic& other) const {
	ASSERT(GiNaC::is_a<idx2>(other));
	const idx2& o = static_cast<const idx2&> (other);
	int comp = inherited::compare_same_type(o);
	if (comp)
		return comp;
	return dim2_.compare(o.dim2_);
}

bool idx2::match_same_type(const GiNaC::basic& other) const {
	ASSERT(is_a<idx2>(other));
	const idx2 &o = static_cast<const idx2&> (other);

	if (!dim2_.is_equal(o.dim2_))
		return false;

	return inherited::match_same_type(other);
}

void idx2::do_print(const GiNaC::print_context& c, unsigned level) const {
	c.s << ".[";
	print_index(c, level);
	c.s << "," << dim << "," << dim2_ << "]";
}

void idx2::archive(GiNaC::archive_node &n) const {
	inherited::archive(n);
	n.add_ex("dim2", dim2_);
}

void idx2::read_archive(const GiNaC::archive_node& n, GiNaC::lst& sym_lst) {
	inherited::read_archive(n, sym_lst);
	n.find_ex("dim2", dim2_, sym_lst);
}

GINAC_BIND_UNARCHIVER(idx2)
;

// copied from GiNaC's color.cpp
// sign is determined by antisymmetric commutations of index positions
// relative to iv3 == { free_index, iv2[0], iv2[1] }
static GiNaC::ex permute_free_index_to_front(const GiNaC::exvector& iv3,
		const GiNaC::exvector& iv2, int& sig) {

	ASSERT(iv3.size() == 3);
	ASSERT(iv2.size() == 2);

	sig = 1;

#define TEST_PERMUTATION(A,B,C,P) \
     if (iv3[B].is_equal(iv2[0]) && iv3[C].is_equal(iv2[1])) { \
         sig = P; \
         return iv3[A]; \
     }
	TEST_PERMUTATION(0,1,2, 1);
	TEST_PERMUTATION(0,2,1, -1);
	TEST_PERMUTATION(1,0,2, -1);
	TEST_PERMUTATION(1,2,0, 1);
	TEST_PERMUTATION(2,0,1, 1);
	TEST_PERMUTATION(2,1,0, -1);

	ABORT("permute_free_index_to_front(): no valid permutation found");
	return 0;
}
//
//
//
GINAC_IMPLEMENT_REGISTERED_CLASS_OPT(sunf_tensor, tensor,
		print_func<print_dflt>(&sunf_tensor::do_print))

sunf_tensor::sunf_tensor() {
	setflag(status_flags::evaluated | status_flags::expanded);
}

GINAC_BIND_UNARCHIVER(sunf_tensor)
;

int sunf_tensor::compare_same_type(const GiNaC::basic& other) const {
	/* by default, the objects are always identical */
	return 0;
}

void sunf_tensor::do_print(const GiNaC::print_context& c, unsigned level) const {
	c.s << "sunf";
}

/** Automatic symbolic evaluation of an indexed sunf. */
GiNaC::ex sunf_tensor::eval_indexed(const GiNaC::basic& i) const {
	ASSERT(is_a<indexed> (i));
	ASSERT(i.nops() == 4);
	ASSERT(is_a<sunf_tensor> (i.op(0)));
	ASSERT(is_a<idx2>(i.op(1)));
	ASSERT(is_a<idx2>(i.op(2)));
	ASSERT(is_a<idx2>(i.op(3)));

	if (is_dummy_pair(i.op(1), i.op(2)))
		return 0;
	if (is_dummy_pair(i.op(1), i.op(3)))
		return 0;
	if (is_dummy_pair(i.op(2), i.op(3)))
		return 0;
	return i.hold();
}

// returns an iterator to the first occurence of an T^a_ij with i=i1
static GiNaC::exvector::iterator iterator_to_sunt_with_dummy_for_i1(
		GiNaC::exvector& v, const GiNaC::ex& i1) {
	ASSERT(is_a<idx> (i1));
	for (exvector::iterator it = v.begin(); it != v.end(); ++it) {
		if (is_a<indexed> (*it) //
				&& is_exactly_a<sunt_tensor> (it->op(0)) //
				&& is_dummy_pair(i1, it->op(2))) // first fundamental index
			return it;
		else
			continue;
	}
	return v.end();
}

// returns the permutation sign between v and w
static int permutation_sign(const GiNaC::exvector& v, const GiNaC::exvector& w) {
	ASSERT(v.size() == 3);
	ASSERT(w.size() == 3);

#define PERMUTATION_SIGN(A,B,C,P) \
     if (w[A].is_equal(v[0]) && w[B].is_equal(v[1]) && w[C].is_equal(v[2])) { \
         return P; \
     }

	PERMUTATION_SIGN(0,1,2, 1);
	PERMUTATION_SIGN(0,2,1, -1);
	PERMUTATION_SIGN(1,0,2, -1);
	PERMUTATION_SIGN(1,2,0, 1);
	PERMUTATION_SIGN(2,0,1, 1);
	PERMUTATION_SIGN(2,1,0, -1);

	ABORT("Logic error");
	return 0;
}

static int permutation_sign(const GiNaC::exvector& v, const GiNaC::ex& a,
		const GiNaC::ex& b, const GiNaC::ex& c) {
	exvector w;
	w.reserve(3);
	w.push_back(a);
	w.push_back(b);
	w.push_back(c);

	return permutation_sign(v, w);
}

// returns the index of v3 that is not equal to b and c
static GiNaC::ex free_index(const GiNaC::exvector& v3, const GiNaC::ex& b,
		const GiNaC::ex& c) {
	ASSERT(v3.size() == 3);

#define FREE_INDEX3(A,B,C) \
     if (!A.is_equal(b) && !A.is_equal(c) && \
    		( (B.is_equal(b) && C.is_equal(c)) \
		  ||  (B.is_equal(c) && C.is_equal(b)) ) ) { \
         return A; \
     }

	FREE_INDEX3(v3[0],v3[1],v3[2]);
	FREE_INDEX3(v3[1],v3[2],v3[0]);
	FREE_INDEX3(v3[2],v3[0],v3[1]);

	ABORT("Logic error");
	return 0;
}

///** Contractions of an indexed sunf tensor with something else. */
bool sunf_tensor::contract_with(GiNaC::exvector::iterator self,
		GiNaC::exvector::iterator other, GiNaC::exvector& v) const {
	ASSERT(is_a<indexed> (*self));
	ASSERT(is_a<indexed> (*other));
	ASSERT(self->nops() == 4);
	ASSERT(is_a<sunf_tensor> (self->op(0)));
	ASSERT(is_a<idx2> (self->op(1)));
	ASSERT(is_a<idx2>(self->op(2)));
	ASSERT(is_a<idx2>(self->op(3)));

	const ex Da = ex_to<idx> (self->op(1)).get_dim();
	const ex Df = ex_to<idx2> (self->op(1)).get_dim2();

	if (is_exactly_a<sunf_tensor> (other->op(0))) {
		// Find the dummy indices of the contraction
		exvector dummy_indices;
		dummy_indices = ex_to<indexed> (*self).get_dummy_indices(
				ex_to<indexed> (*other));

		// f.abc f.abc = Da Df
		if (dummy_indices.size() == 3) {
			*self = permutation_sign( //
					ex_to<indexed> (*self).get_indices(), //
					ex_to<indexed> (*other).get_indices() //
					) * Da * Df;
			*other = ex(1);
			return true;

			// f.akl f.bkl = Df delta.ab
		} else if (dummy_indices.size() == 2) {
			int sign1, sign2;
			ex a = permute_free_index_to_front(
					ex_to<indexed> (*self).get_indices(), dummy_indices, sign1);
			ex b =
					permute_free_index_to_front(
							ex_to<indexed> (*other).get_indices(),
							dummy_indices, sign2);
			*self = sign1 * sign2 * Df * delta_tensor(a, b);
			*other = ex(1);
			return true;
		}

	} // sunf

	else if (is_exactly_a<sunt_tensor> (other->op(0))
			&& ex_to<indexed> (*self).has_dummy_index_for(other->op(1))) {

		exvector::iterator itnew;
		itnew = iterator_to_sunt_with_dummy_for_i1(v, other->op(3));
		if (itnew != v.end() && itnew != other
				&& ex_to<indexed> (*self).has_dummy_index_for(itnew->op(1))) {

			exvector indices = ex_to<indexed> (*self).get_indices();
			ex b = other->op(1);
			ex c = itnew->op(1);
			ex a = free_index(indices, b, c);
			int sign = permutation_sign(indices, a, b, c);

			// F^abc * T^b_ij * T^c_jk = I/2 * Df *T^a_ik
			*self = sunt(a, other->op(2), itnew->op(3));
			*other = numeric(sign, 2) * I;
			*itnew = Df;
			return true;
		}
	} // sunt


	symbol d1, d2, d3;
	idx i1(d1, Df), i2(d2, Df), i3(d3, Df);
	*self = 2 * I * sunt(self->op(1), i2, i3) * sunt(self->op(2), i1, i2)
			* sunt(self->op(3), i3, i1) - 2 * I * sunt(self->op(1), i1, i2)
			* sunt(self->op(2), i2, i3) * sunt(self->op(3), i3, i1);
	return true;

}
//
//
//
GINAC_IMPLEMENT_REGISTERED_CLASS_OPT(sunt_tensor, tensor,
		print_func<print_dflt>(&sunt_tensor::do_print))

sunt_tensor::sunt_tensor() {
	setflag(status_flags::evaluated | status_flags::expanded);
}

GINAC_BIND_UNARCHIVER(sunt_tensor)
;

int sunt_tensor::compare_same_type(const GiNaC::basic& other) const {
	/* by default, the objects are always identical */
	return 0;
}

void sunt_tensor::do_print(const GiNaC::print_context& c, unsigned level) const {
	c.s << "sunt";
}

/** Automatic symbolic evaluation of an indexed sunt tensor. */
GiNaC::ex sunt_tensor::eval_indexed(const GiNaC::basic& i) const {
	ASSERT(is_a<indexed> (i));
	ASSERT(i.nops() == 4);
	ASSERT(is_a<sunt_tensor> (i.op(0)));
	ASSERT(is_a<idx2> (i.op(1)));
	ASSERT(is_a<idx> (i.op(2)));
	ASSERT(is_a<idx> (i.op(3)));

	// sunt.a.i.i = 0
	if (is_dummy_pair(i.op(2), i.op(3)))
		return ex(0);
	else
		return i.hold();
}

///** Contraction of an indexed sunt tensor with something else. */
bool sunt_tensor::contract_with(GiNaC::exvector::iterator self,
		GiNaC::exvector::iterator other, GiNaC::exvector& v) const {
	ASSERT(is_a<indexed> (*self));
	ASSERT(is_a<indexed> (*other));
	ASSERT(self->nops() == 4);
	ASSERT(is_a<sunt_tensor> (self->op(0)));

	//	static ex Df = sqrt(1 + Da); cannot be ordered with Df ...
	const ex Da = ex_to<idx> (self->op(1)).get_dim();
	const ex Df = ex_to<idx> (self->op(2)).get_dim();

	if (is_exactly_a<sunt_tensor> (other->op(0))) {
		ASSERT(other->nops() == 4);

		// case T^a T^a
		if (is_dummy_pair(self->op(1), other->op(1))) {
			// Taij^2 = 1/2 * (Df-1)
			if (is_dummy_pair(self->op(2), other->op(2)) && is_dummy_pair(
					self->op(3), other->op(3))) {
				*self = Ident(Df - 1);
				*other = numeric(1, 2);
				return true;
			}

			// T^a_ij T^a_ji = 1/2 * Da
			else if (is_dummy_pair(self->op(2), other->op(3)) && is_dummy_pair(
					self->op(3), other->op(2))) {
				*self = Da;
				*other = numeric(1, 2);
				return true;
			}

			// Taij * Tajk = Da/(2*Df) * delta(i,k)
			else if (is_dummy_pair(self->op(3), other->op(2))) {
				*self = delta_tensor(self->op(2), other->op(3));
				*other = numeric(1, 2) * Da / Df;
				return true;
			}
			// Tajk * Taij = Da/(2*Df) * delta(i,k)
			else if (is_dummy_pair(self->op(2), other->op(3))) {
				*self = delta_tensor(self->op(3), other->op(2));
				*other = numeric(1, 2) * Da / Df;
				return true;
			}
			// Ta_ij * Ta_ik = (Df-1)/(2 Df) * delta(j,k)
			else if (is_dummy_pair(self->op(2), other->op(2))) {
				*self = delta_tensor(self->op(3), other->op(3));
				*other = numeric(1, 2) * Ident(Df - 1) / Df;
				return true;
			}
			// Ta_ij * Ta_kj = (Df-1)/(2 Df) * delta(i,k)
			else if (is_dummy_pair(self->op(3), other->op(3))) {
				*self = delta_tensor(self->op(2), other->op(2));
				*other = numeric(1, 2) * Ident(Df - 1) / Df;
				return true;
			}

			// CASE T^a_ij * T^a_kl = (1/2)(delta(i,l)*delta(j,k) - 1/Nc * delta(i,j)*delta(k,l)) : SEE BELOW


		} else {

			// T^a_ij * T^b_ji == 1/2 delta(a,b)
			if (is_dummy_pair(self->op(3), other->op(2)) //
					&& is_dummy_pair(self->op(2), other->op(3))) {
				ASSERT(!is_dummy_pair(self->op(1), other->op(1)));
				*self = delta_tensor(self->op(1), other->op(1));
				*other = numeric(1, 2);
				return true;
			}
		}

		// case T^a T^b T^c

		if (is_dummy_pair(self->op(3), other->op(2))) {
			ASSERT(!is_dummy_pair(self->op(2), other->op(3)));

			exvector::iterator itnew;
			itnew = iterator_to_sunt_with_dummy_for_i1(v, other->op(3));
			if (itnew != v.end() && itnew != self && itnew != other) {

				// T^a_ij * T^b_jk * T^a_kl = ( (N^2-1)/(2N) - N/2 ) * Tbil = -1/(2*Df) Tbil
				if (is_dummy_pair(self->op(1), itnew->op(1))) {
					if (is_dummy_pair(self->op(2), itnew->op(3))) { // T^b_ll = 0
						*self = 0;
						*other = 0;
						*itnew = 0;
					} else {
						*self = sunt(other->op(1), self->op(2), itnew->op(3));
						*other = numeric(-1, 2) / Df;
						*itnew = 1;
					}
					return true;
				}

				// T^a_ij * T^b_jk * T^b_kl =  Da / (2 Df) * Tail
				if (is_dummy_pair(other->op(1), itnew->op(1))) {
					if (is_dummy_pair(self->op(2), itnew->op(3))) { // T^a_ll = 0
						*self = 0;
						*other = 0;
						*itnew = 0;
					} else {
						*self = sunt(self->op(1), self->op(2), itnew->op(3));
						*other = numeric(1, 2) * Da / Df;
						*itnew = 1;
					}
					return true;
				}

			}
		} // T^a T^b T^c


		if (is_dummy_pair(self->op(1), other->op(1))) {

			// this rule should come at the end because it enlarges the number of terms
			//
			// T^a_ij * T^a_kl = (1/2)(delta(i,l)*delta(j,k) - 1/Nc * delta(i,j)*delta(k,l))
			ASSERT(!is_dummy_pair(self->op(2), other->op(2)));
			ASSERT(!is_dummy_pair(self->op(3), other->op(3)));
			ASSERT(!is_dummy_pair(self->op(2), other->op(3)));
			ASSERT(!is_dummy_pair(self->op(3), other->op(2)));
			*self = delta_tensor(self->op(2), other->op(3)) //
					* delta_tensor(self->op(3), other->op(2)) //
					- (1 / Df) * (delta_tensor(self->op(2), self->op(3)) //
							* delta_tensor(other->op(3), other->op(2)));
			*other = numeric(1, 2);
			return true;
		}

	} // other = T

	return false;
}

GiNaC::ex sunt_tensor::conjugate() const {
	return (new suntc_tensor)->setflag(status_flags::dynallocated);
}
//
// suntc
//
GINAC_IMPLEMENT_REGISTERED_CLASS_OPT(suntc_tensor, tensor,
		print_func<print_dflt>(&suntc_tensor::do_print))

suntc_tensor::suntc_tensor() {
	setflag(status_flags::evaluated | status_flags::expanded);
}

GINAC_BIND_UNARCHIVER(suntc_tensor)
;

int suntc_tensor::compare_same_type(const GiNaC::basic& other) const {
	return 0; // object are all identical
}

void suntc_tensor::do_print(const GiNaC::print_context& c, unsigned level) const {
	c.s << "suntc";
}

/** Automatic conversion to sunt */
GiNaC::ex suntc_tensor::eval_indexed(const GiNaC::basic& i) const {
	ASSERT(is_a<indexed> (i));
	ASSERT(i.nops() == 4);
	ASSERT(is_a<suntc_tensor> (i.op(0)));
	return sunt(i.op(1), i.op(3), i.op(2));
}

/** Contraction of an indexed suntc tensor with something else. */
bool suntc_tensor::contract_with(GiNaC::exvector::iterator self,
		GiNaC::exvector::iterator other, GiNaC::exvector & v) const {
	return false;
}

GiNaC::ex suntc_tensor::conjugate() const {
	return (new sunt_tensor)->setflag(status_flags::dynallocated);
}

//
// wrapper functions
//


GiNaC::ex sunt(const GiNaC::ex& a, const GiNaC::ex& i1, const GiNaC::ex& i2) {
	if (!is_a<idx2> (a))
		throw(std::invalid_argument("first index of sunt must be of type idx2 "
				+ to_string(a)));
	if (!is_a<idx> (i1) || !is_a<idx> (i2))
		throw(std::invalid_argument(
				"second and third index of sunt must be of type idx"));

	ex f = (new sunt_tensor)->setflag(status_flags::dynallocated);
	return indexed(f, a, i1, i2);
}

GiNaC::ex sunf(const GiNaC::ex& a1, const GiNaC::ex& a2, const GiNaC::ex& a3) {
	if (!is_a<idx2> (a1) || !is_a<idx2> (a2) || !is_a<idx2> (a3))
		throw(std::invalid_argument("indices of sunf must be of type idx2"));

	ex f = (new sunf_tensor)->setflag(status_flags::dynallocated);
	return indexed(f, a1, a2, a3);
	//return indexed(f, antisymmetric3(), a1, a2, a3);
}

GiNaC::ex sunf(const GiNaC::ex& a1, const GiNaC::ex& a2, const GiNaC::ex& a3,
		const GiNaC::ex& a4) {
	symbol d;
	VERIFY(is_a<idx2>(a1));
	idx2 dummy(d, ex_to<idx2> (a1).get_dim(), ex_to<idx2> (a1).get_dim2());
	return sunf(a1, a2, dummy) * sunf(a3, a4, dummy);
}

}
// namespace Reduze


