/*  equationlist.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef REDUZE_EQUATIONLIST_H
#define REDUZE_EQUATIONLIST_H

#include "equation.h"

namespace Reduze {

// predeclarations
class EquationLightList;
template<class TE> class EqBasicList;
template<class TE>
std::ostream& operator<<(std::ostream &, const EqBasicList<TE> &);

/// template for a system of general equations
template<class TEquation>
class EqBasicList {
public:
	// basic list functionality
	typedef TEquation eq_type;
	typedef typename TEquation::integral_type integral_type;
	typedef typename TEquation::coeff_type coeff_type;
	typedef std::list<eq_type> rep_type;
	typedef typename rep_type::const_iterator const_iterator;
	typedef typename rep_type::const_reverse_iterator const_reverse_iterator;

	const_iterator begin() const {
		return l.begin();
	}

	const_iterator end() const {
		return l.end();
	}

	const_reverse_iterator rbegin() const {
		return l.rbegin();
	}

	const_reverse_iterator rend() const {
		return l.rend();
	}

	/// returns the number of equations
	size_t size() const {
		return l.size();
	}

	/// returns true if the system of equation is empty
	bool empty() const {
		return l.empty();
	}
	/// removes all equations
	void clear() {
		l.clear();
	}
	/// sort the equations
	void sort() {
		l.sort();
	}
	/// remove consecutive duplicated equations
	void unique() {
		l.unique();
	}

	/// inserts an equation to the list
	void push_back(const eq_type& eq) {
		l.push_back(eq);
	}

	/// get reference to last element
	const eq_type& back() const {
		return l.back();
	}

	/// get reference to last element
	eq_type& back() {
		return l.back();
	}

	/// moves the equations of eqlist to the end of this list
	void splice(EqBasicList<TEquation> & from) {
		l.splice(l.end(), from.l);
	}

	/// swaps the equations
	void swap(EqBasicList<TEquation> & other) {
		l.swap(other.l);
	}

	// functionality for equations with ordered integrals:

	/// removes empty equations, returns the number of removed equations
	int remove_empty_equations();

	/// removes multiples of equations with same leading INT
	/** for multiple equations with same leading INT the one which compares
	 ** less (operator<) than the others is kept */
	void remove_same_leading_INT_multiples();

	/// inserts all integrals from the list of equations in the set or multiset
	template<class TSet>
	void find_INT(TSet & myset) const;

	/// inserts every most complicated integral of the equations in the set or multiset
	template<class TSet>
	void find_max_INT(TSet & myset) const;

	/// adds subleading integrals of each equation to a set or multiset
	template<class TSet>
	void find_subleading_INT(TSet & myset) const;

	/// adds integrals which are not max_INT() of any equation to a set or multiset
	template<class TSet>
	void find_pure_subleading_INT(TSet & myset) const;

	/** \todo: make non-const iterator methods protected */
	typedef typename rep_type::iterator iterator;

	/// returns an iterator to the most reduced equation that contains the integral max as the most complicated one
	iterator it_to_most_reduced(const integral_type & max);

	/// distributes the equations to list of equations each containing the same most complicated integral, returns the number of lists \todo test
	/** result is sorted in ascending order of most complicated integrals,
	 ** coefficient of leading INT may be zero */
	template<class TEquationList>
	int split(std::list<TEquationList> & eqlist);

	template<typename IntegralToIntegralOperation>
	void transform_integrals(IntegralToIntegralOperation mapping,
			bool normalize = false);

protected:
	std::list<eq_type> l;

	/// returns an iterator to the shortest equation that contains the integral max as the most complicated one
	iterator it_to_shortest(const integral_type & max);

	/// moves the equation at position 'it' of the list a_l to the EquationList
	void splice(std::list<eq_type>& a_l, iterator it) {
		l.splice(l.end(), a_l, it);
	}

private:
	friend std::ostream & operator<<<> (std::ostream & strm, const EqBasicList<
			TEquation> & in);
};

/// template for a system of equations with algebraic coefficients
template<class TEquation>
class EqAlgebraicList: public EqBasicList<TEquation> {
public:
	typedef EqBasicList<TEquation> parent_type;
	typedef typename parent_type::eq_type eq_type;
	typedef typename parent_type::integral_type integral_type;
	typedef typename parent_type::coeff_type coeff_type;
	typedef typename parent_type::const_iterator const_iterator;
	typedef typename parent_type::const_reverse_iterator const_reverse_iterator;

	/// returns true if a coefficient in the system of equations contains the symbol 'symb'
	bool has_symbol(const GiNaC::symbol & symb) const;

	/// returns the number of symbols from ReduzeSymbols::Symblist that appears in the coefficient of the equations
	int number_of_symbols() const;

	/// returns the number of symbols from 'alist' that appears in the coefficient of the equations
	int number_of_symbols(const GiNaC::lst & alist) const;

	/// applies the map m on each prefactor, no normalization is performed
	void apply(const GiNaC::exmap& m, unsigned options = 0);

	/// replaces equation 'toreplace' in all equations
	void replace(const eq_type& toreplace, bool normalize = true);

	/// replaces all equations of 'toreplace' in the equations
	void replace(const EqAlgebraicList<TEquation>& toreplace, bool normalize =
			true);

	/// returns true if integral 'in' is found to be zero, false if not found at all or not found to be zero
	bool gives_reduction_zero(const integral_type & in) const;

	/// returns false if an integral in 'in' is not found to be zero, true otherwise
	bool gives_reduction_zero(const std::set<integral_type> & in) const;

	/// normalizes and solves all equations for their most complicated integral
	/** removes empty equations **/
	void solve();

	/// forward eliminate the equations, result is in ascending order
	void forward_eliminate();

	/// forward eliminate current leading integrals
	/** equations with new leading integrals are just inserted in lower_eqs,
	 ** both resulting equation lists are solved and normalized */
	void forward_eliminate_leading(EqAlgebraicList<TEquation>& lower_eqs);

	/// replaces for all 'i' the i-th equation into equation 'j', for all j > i
	/** calling this method after triangularize() diagonalizes the equations */
	void back_substitute();

	/// reduces the equations
	/** this member function tries to diagonalize the system of equations */
	void reduze();

protected:
	typedef typename parent_type::iterator iterator;

};

/// Class to represent a system of Equations
class EquationList: public EqAlgebraicList<Equation> {
public:
	/// constructs an empty EquationList
	EquationList();
	/// converts by using the symbols from the default Kinematics
	EquationList(const EquationLightList& other);
	EquationList(const EquationLightList& other, const GiNaC::lst& symbols);
};

/// list of instances of IdentityGeneric
class IdentityGenericList: public EqAlgebraicList<IdentityGeneric> {
public:
	IdentityGenericList() {
	}
};

/// list of instances of IdentityGeneric
class LinearCombinationGenericList: public EqBasicList<LinearCombinationGeneric> {
public:
	LinearCombinationGenericList() {
	}
};

/// Class to represent a system of Identities
class IdentityList: public EqAlgebraicList<Identity> {

public:
	/// constructs an empty IdentityList
	IdentityList();

	/// returns the average length of the equations
	double average_length();

	/// returns the average size of the coefficients
	double average_coeff_length();

	/// inserts all integrals (integers) from the list of equations in the set or multiset 'myset'
	template<class TSet>
	void find_INT(TSet & myset) const {
		EqBasicList<Identity>::find_INT(myset);
	}

	/// finds the different integrals with t smaller than "t" in the Identity and inserts them in the set
	template<class TSet>
	void find_subsector_INT(int t, TSet& myset) const;

public:
	/// applies Identity::remove_zero_integral() to all Identities in the the list and deletes empty equations
	void remove_zero_integral();

	/// substitutes each integral I(t,r,s) by (-1)^(r+s)*I(t,r,s)
	void toggle_metric_convention();

	/// calls Identity::reconstruct_symbol_replaced_by_one() for all identities
	void reconstruct_symbol_replaced_by_one();
};

/// list of EquationXLight
class EquationXLightList: public EqBasicList<EquationXLight> {
public:
	EquationXLightList() {
	}
};

/// list of EquationLight
class EquationLightList: public EqBasicList<EquationLight> {
public:
	EquationLightList() {
	}
	EquationLightList(const EquationList& other) {
		EquationList::const_iterator it;
		for (it = other.begin(); it != other.end(); ++it)
			push_back(EquationLight(*it));
	}
};

/// list of EquationHLight
class EquationHLightList: public EqBasicList<EquationHLight> {
public:
	EquationHLightList() {
	}
};

// public members where needed template instances are not known yet

template<class TE>
template<typename IntegralToIntegralOperation>
void EqBasicList<TE>::transform_integrals(IntegralToIntegralOperation mapping,
		bool normalize) {
	for (iterator e = l.begin(); e != l.end(); ++e)
		e->transform_integrals(mapping, normalize);
}

template<class TE> template<class TSet>
void EqBasicList<TE>::find_INT(TSet & myset) const {
	const_iterator it;
	for (it = l.begin(); it != l.end(); ++it)
		it->find_INT(myset);
}

template<class TE> template<class TSet>
void EqBasicList<TE>::find_max_INT(TSet & myset) const {
	const_iterator it;
	for (it = l.begin(); it != l.end(); ++it)
		if (!it->empty())
			myset.insert(it->max_INT());
}

template<class TE> template<class TSet>
void EqBasicList<TE>::find_subleading_INT(TSet & myset) const {
	const_iterator it;
	for (it = l.begin(); it != l.end(); ++it)
		if (!it->empty())
			it->find_subleading_INT(myset);
}

template<class TE> template<class TSet>
void EqBasicList<TE>::find_pure_subleading_INT(TSet & myset) const {
	TSet sub;
	const_iterator it = l.begin();
	while (it != l.end())
		(it++)->find_INT(sub);
	for (it = l.begin(); it != l.end(); ++it)
		if (!it->empty())
			sub.erase(it->max_INT());
	myset.insert(sub.begin(), sub.end());
}

template<class TE> template<class TEquationList>
int EqBasicList<TE>::split(std::list<TEquationList> & eqlist) {
	sort();
	unique();
	remove_empty_equations(); // assert well defined max_INT()
	iterator it = l.begin();
	while (it != l.end()) {
		INTIndex max(it->max_INT());
		TEquationList tmplist;
		tmplist.splice(l, it++);
		while (it != l.end() && it->max_INT() == max)
			tmplist.splice(l, it++);
		eqlist.push_back(TEquationList());
		eqlist.back().swap(tmplist);
	}
	return static_cast<int> (eqlist.size());
}

template<class TSet>
void IdentityList::find_subsector_INT(int t, TSet& myset) const {
	std::list<Identity>::const_iterator it;
	for (it = l.begin(); it != l.end(); ++it)
		it->find_subsector_INT(t, myset);
}

// friend template operator

template<class TE>
std::ostream & operator<<(std::ostream & strm, const EqBasicList<TE> & in) {
	typename EqBasicList<TE>::const_iterator it = in.l.begin();
	for (; it != in.l.end(); ++it)
		strm << *it << "\n";
	return strm;
}

}
// namespace Reduze

#endif

