/*  graph.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include <ginac/ginac.h>
#include "functions.h"
#include "graph.h"
#include "integralfamily.h"
#include "sectormappings.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "files.h"
#include "triconnected.h"
#include "kinematics.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

std::ostream& operator<<(std::ostream& os, const GraphEdge& edge) {
	os << "(" << edge.momentum_ << ", " << edge.squaredmass_ << ")";
	return os;
}

//
// graph
//

Graph::Graph() :
		undefined(GiNaC::symbol("*")) {
}

Graph::Graph(const Topology& t) :
		Topology(t), undefined(GiNaC::symbol("*")) {
}

Graph::~Graph() {
}

Graph* Graph::clone() const {
	return new Graph(*this);
}

void Graph::swap(Graph& other) {
	// swap topologies, enforce no call back which would clear Graph members
	static_cast<Topology&>(*this).swap(static_cast<Topology&>(other), false);
	subgraph_to_sector_.swap(other.subgraph_to_sector_);
	graph_to_sector_.swap(other.graph_to_sector_);
	graphedges_.swap(other.graphedges_);
	std::swap(undefined, other.undefined);
	std::swap(loop_momenta_, other.loop_momenta_);
}

const GraphEdge& Graph::graphedge(int edge_id) const {
	map<int, GraphEdge>::const_iterator it = graphedges_.find(edge_id);
	if (it == graphedges_.end())
		ABORT("Edge " << edge_id << " does not exist");
	return it->second;
}

const std::map<int, GraphEdge>& Graph::graphedges() const {
	return graphedges_;
}

Edge Graph::insert_degree_2_node(int edge_id) {
	GraphEdge ge = graphedge(edge_id); // copy;
	Edge edge = Topology::insert_degree_2_node(edge_id);
	VERIFY(contains_edge(edge_id));
	set_momentum(edge_id, ge.momentum_);
	set_squaredmass(edge_id, ge.squaredmass_);
	set_momentum(edge.id, ge.momentum_);
	set_squaredmass(edge.id, ge.squaredmass_);
	VERIFY(get_edges_undefined_momentum().empty());
	return edge;
}

// momenta

void Graph::substitute_momenta(const GiNaC::ex& rules) {
	map<int, GraphEdge>::iterator it;
	for (it = graphedges_.begin(); it != graphedges_.end(); ++it)
		it->second.momentum_ = it->second.momentum_.subs(rules);
}

void Graph::substitute_momenta(const GiNaC::exmap& rules) {
	map<int, GraphEdge>::iterator it;
	for (it = graphedges_.begin(); it != graphedges_.end(); ++it)
		it->second.momentum_ = it->second.momentum_.subs(rules);
}

void Graph::set_loop_momenta(const GiNaC::lst& symbols) {
	loop_momenta_ = symbols;
}

const GiNaC::lst& Graph::loop_momenta() const {
	return loop_momenta_;
}

void Graph::clear_momentum(int edge) {
	set_momentum(edge, undefined);
}

void Graph::clear_internal_momenta() {
	for (map<int, GraphEdge>::iterator it = graphedges_.begin();
			it != graphedges_.end(); ++it)
		if (!is_external_edge(it->first))
			clear_momentum(it->first);
}

void Graph::set_momentum(int edge_id, const GiNaC::ex& mom) {
	if (!contains_edge(edge_id))
		ABORT("Invalid edge " << edge_id);
	map<int, GraphEdge>::iterator it = graphedges_.find(edge_id);
	if (it == graphedges_.end()) {
		GraphEdge ge(undefined, undefined);
		it = graphedges_.insert(make_pair(edge_id, ge)).first;
	}
	GraphEdge& et = it->second;

	// set the momentum but restore old value if momentum conservation violated
	GiNaC::ex old_momentum = et.momentum_;
	et.momentum_ = mom;

	const Edge& e = edge(edge_id);
	//const Edge& e = edges_[edge_id];
	// no restrictions on external nodes, but internal nodes must conserve mom.
	/// \todo: ensure total external momentum is zero for each connected comp.
	if (!is_external_node(e.from) && !is_momentum_conserved(e.from)) {
		stringstream str;
		str << "Momentum sum at node '" << e.from << "': "
				<< momentum_sum(e.from);
		et.momentum_ = old_momentum;
		throw runtime_error(
				"Momentum violation in graph " + name() + " . " + str.str());
	} else if (!is_external_node(e.to) && !is_momentum_conserved(e.to)) {
		stringstream str;
		str << "Momentum sum at node '" << e.to << "': " << momentum_sum(e.to);
		et.momentum_ = old_momentum;
		throw runtime_error(
				"Momentum violation in graph " + name() + " . " + str.str());
	}
}

void Graph::complete_momentum(int node_id, bool recurse) {
	if (is_external_node(node_id))
		return;
	GiNaC::ex sum = 0;
	list<int> undef_edges;
	const list<Edge>& edges = edges_of_node(node_id);
	for (list<Edge>::const_iterator e = edges.begin(); e != edges.end(); ++e) {
		std::map<int, GraphEdge>::iterator ge = graphedges_.find(e->id);
		if (e->from == e->to) { // tadpole edge
			continue; // just ignore
		} else if (ge == graphedges_.end()
				|| ge->second.momentum_ == undefined) {
			if (!undef_edges.empty()) // more than one undefined momentum
				return;
			undef_edges.push_back(e->id);
		} else {
			if (e->from == node_id) // outgoing
				sum -= ge->second.momentum_;
			if (e->to == node_id) // incoming
				sum += ge->second.momentum_;

		}
	}
	if (undef_edges.empty())
		return; // nothing to do
	Edge undef = edge(undef_edges.front());
	// undef is either incoming or outgoing, no tadpole at this point
	bool outgoing = (undef.from == node_id);
	set_momentum(undef.id, (outgoing ? 1 : -1) * sum);
	if (recurse) {
		if (undef.from != node_id)
			complete_momentum(undef.from, true);
		if (undef.to != node_id)
			complete_momentum(undef.to, true);
	}
}

void Graph::complete_momenta() {
	for (set<int>::const_iterator n = nodes().begin(); n != nodes().end(); ++n)
		complete_momentum(*n, true);
}

void Graph::set_internal_momenta() {
	clear_internal_momenta();
	set<int> cycles = find_cycles();
	VERIFY(cycles.size() == loop_momenta_.nops());
	unsigned int lauf = 0;
	for (set<int>::const_iterator c = cycles.begin(); c != cycles.end(); ++c)
		set_momentum(*c, loop_momenta_.op(lauf++));
	complete_momenta();
}

void Graph::set_internal_masses(const GiNaC::ex& sqmass) {
	list<Edge> edges = internal_edges();
	for (list<Edge>::const_iterator e = edges.begin(); e != edges.end(); ++e)
		set_squaredmass(e->id, sqmass);
}

GiNaC::ex Graph::get_momentum(int edge_id) const {
	if (!contains_edge(edge_id))
		ABORT("Invalid edge " << edge_id);
	map<int, GraphEdge>::const_iterator it = graphedges_.find(edge_id);
	return it == graphedges_.end() ? undefined : it->second.momentum_;
}

std::set<int> Graph::get_edges_undefined_momentum() const {
	set<int> result;
	const map<int, Edge>& es = this->edges();
	for (map<int, Edge>::const_iterator e = es.begin(); e != es.end(); ++e)
		if (graphedges_.find(e->first) == graphedges_.end()
				|| graphedges_.find(e->first)->second.momentum_ == undefined)
			result.insert(e->first);
	return result;
}

std::vector<Propagator> Graph::get_propagators_with_loop_momenta() const {
	vector<Propagator> props;
	props.reserve(edges().size());
	for (map<int, GraphEdge>::const_iterator e = graphedges_.begin();
			e != graphedges_.end(); ++e) {
		if (freeof(e->second.momentum_, loop_momenta_))
			continue;
		Propagator p(e->second.momentum_, e->second.squaredmass_);
		props.push_back(p);
	}
	return props;
}

GiNaC::ex Graph::momentum_sum(int node) const {
	GiNaC::ex sum = 0;
	list<Edge> edges = edges_of_node(node);
	for (list<Edge>::const_iterator e = edges.begin(); e != edges.end(); ++e) {
		map<int, GraphEdge>::const_iterator f = graphedges_.find(e->id);
		if (f == graphedges_.end() || f->second.momentum_ == undefined)
			return 0;
		const GraphEdge& et = f->second;
		// note: an edge may be both, incoming and outgoing
		if (e->from == node) // outgoing edge
			sum -= et.momentum_;
		if (e->to == node) // incoming edge
			sum += et.momentum_;
	}
	return sum.expand();
}

bool Graph::is_momentum_conserved(int node) const {
	return momentum_sum(node).is_zero();
}

// squared mass

void Graph::set_squaredmass(int edge_id, const GiNaC::ex& squaredmass) {
	if (!contains_edge(edge_id))
		ABORT("Invalid edge " << edge_id);
	map<int, GraphEdge>::iterator it = graphedges_.find(edge_id);
	if (it == graphedges_.end()) {
		GraphEdge ge(undefined, undefined);
		it = graphedges_.insert(make_pair(edge_id, ge)).first;
	}
	it->second.squaredmass_ = squaredmass;
}

void Graph::clear_squaredmasses() {
	for (map<int, GraphEdge>::iterator it = graphedges_.begin();
			it != graphedges_.end(); ++it)
		if (!is_external_edge(it->first))
			it->second.squaredmass_ = undefined;
}

GiNaC::ex Graph::get_squaredmass(int edge_id) const {
	if (!contains_edge(edge_id))
		ABORT("Invalid edge " << edge_id);
	map<int, GraphEdge>::const_iterator it = graphedges_.find(edge_id);
	return it == graphedges_.end() ? undefined : it->second.squaredmass_;
}

bool Graph::is_undefined_square_mass(int edge_id) const {
	ex ms = get_squaredmass(edge_id);
	if (ms.is_equal(undefined))
		return true;
	return false;
}

GiNaC::lst Graph::get_different_masses() const {
	// verify all masses set
	set<ex, mass_is_less> ms;
	map<int, GraphEdge>::const_iterator e;
	for (e = graphedges_.begin(); e != graphedges_.end(); ++e) {
		if (e->second.squaredmass_.is_equal(undefined))
			throw runtime_error(
					"graph " + name() + " has undefined square masses");
		ms.insert(e->second.squaredmass_);
	}
	lst masses;
	for (set<ex, mass_is_less>::const_iterator m = ms.begin(); m != ms.end();
			++m)
		masses.append(*m);
	return masses;
}

// graph construction

void Graph::insert_graph_edge_to_node(int edge_id, const GraphEdge& graphedge,
		int node, std::list<Graph>& graphlist) const {

	/* The algorithm loops over all partitions (into two sets with minimum block
	 * size of two) of the adjacent momenta of the node and creates a new
	 * graph if the momentum of the graphedge is equal to the sum or difference
	 * of the summed momenta of the two blocks of the partition, eg. if the node
	 * can be split into two nodes with the graphedge inserted in between respecting
	 * momentum conservation.
	 *
	 * Self-loops at node N are required to be replaced by pseudo self-loops which
	 * consist of two anti-parallel edges between N and a dummy node.
	 * If there is a partition p1=(P,Q) of the edges of a node which splits these edges
	 * into two parts P and Q such that in one part, eg. P,  there are only edges of
	 * pseudo self-loops then there is a second partition p2=(P',Q) where P' contains
	 * the edges of the same pseudo self-loops but not contained in P. These two
	 * partitions lead to two isomorphic graphs but they can become non-isomorphic
	 * if further edges are inserted. Therefore we keep both partitions unless
	 * the partition p1=(P,Q) has in both P and Q only pseudo self-loops. In this
	 * case the second partition would generate the same graph but with reversed
	 * momenta, eg. ki->-ki. If in one case a edge with momentum k can be inserted then
	 * it can be inserted in the case as with -k at the same place. So we can omit
	 * some partitions in the case where only pseudo self-loops are present and can
	 * save memory and runtime.
	 */

	// identify pseudo self-loops

	// a pair of an edge and its momentum
	typedef pair<Edge, ex> EMpair;
	// edges and momenta adjacent to the node
	set<EMpair> momentum_of_edge;
	// for each edge of a pseudo self-loop the parallel edge
	map<Edge, Edge> pseudo_loop_edges;
	// loop over the edges adjacent to the node 'node'
	const list<Edge>& edges = edges_of_node(node);
	// split vertex only if there are more than 3 edges attached
	if (edges.size() < 4)
		return;
	for (list<Edge>::const_iterator e = edges.begin(); e != edges.end(); ++e) {
		if (e->is_self_loop())
			ABORT("self loop encountered in graph " << name());
		momentum_of_edge.insert(make_pair(*e, get_momentum(e->id)));
		int opp_node = e->opposite(node);
		const list<Edge>& oppnode_edges = edges_of_node(opp_node);
		if (oppnode_edges.size() == 2
				&& oppnode_edges.front().is_between(node, opp_node)
				&& oppnode_edges.back().is_between(node, opp_node)) {
			pseudo_loop_edges[oppnode_edges.front()] = oppnode_edges.back();
			pseudo_loop_edges[oppnode_edges.back()] = oppnode_edges.front();
		}
	}

	// main loop over partitions of the legs (edge/momentum pairs) adjacent to the node

	set<set<Edge> > omitting;
	for (SetPartition<EMpair> sp(momentum_of_edge, 2, 2); sp.good(); ++sp) {
		const vector<set<EMpair> >& partition = *sp;
		ASSERT(partition.size() == 2);
		ASSERT(partition[0].size() >= 2 && partition[1].size() >= 2);
		ASSERT(partition[0].size() + partition[1].size() == edges.size());
		ex mom_right = 0, mom_left = 0;
		set<Edge> right, left;
		set<EMpair>::const_iterator e;
		for (e = partition[0].begin(); e != partition[0].end(); ++e) {
			mom_right += ((e->first.to == node) ? 1 : -1) * e->second;
			right.insert(e->first);
		}
		if (omitting.find(right) != omitting.end())
			continue;
		if (mom_right.expand().is_zero())
			continue;
		for (e = partition[1].begin(); e != partition[1].end(); ++e) {
			mom_left += ((e->first.to == node) ? 1 : -1) * e->second;
			left.insert(e->first);
		}
		ASSERT(omitting.find(left) == omitting.end());
		VERIFY((mom_right + mom_left).expand().is_zero());
		int sign = 0;
		if ((graphedge.momentum_ + mom_left).expand().is_zero())
			sign = 1;
		else if ((-graphedge.momentum_ + mom_left).expand().is_zero())
			sign = -1;
		else
			continue;

		// test if only pseudo self-loops are present, then omit the partition with the parallel edges

		map<Edge, Edge>::const_iterator find;
		set<Edge> parallel1, parallel2;
		for (set<Edge>::const_iterator m = right.begin(); m != right.end(); ++m)
			if ((find = pseudo_loop_edges.find(*m)) != pseudo_loop_edges.end())
				parallel1.insert(find->second);
		for (set<Edge>::const_iterator m = left.begin(); m != left.end(); ++m)
			if ((find = pseudo_loop_edges.find(*m)) != pseudo_loop_edges.end())
				parallel2.insert(find->second);
		if (parallel1.size() == right.size() && parallel2.size() == left.size()) {
			omitting.insert(parallel1);
			omitting.insert(parallel2);
		}

		// insert the graphedge
		Graph graph(*this);
		pair<int, int> node_edge = graph.cleave(node, right, sign, edge_id);
		graph.set_momentum(node_edge.second, graphedge.momentum_);
		graph.set_squaredmass(node_edge.second, graphedge.squaredmass_);
		graphlist.push_back(Graph());
		graphlist.back().swap(graph);
	}
}

void Graph::insert_graph_edge(int edge_id, const GraphEdge& graphedge,
		std::list<Graph>& graphlist) const {
	const map<int, list<Edge> >& eon = this->edges_of_nodes();
	map<int, list<Edge> >::const_iterator n;
	for (n = eon.begin(); n != eon.end(); ++n)
		if (n->second.size() > 3)
			insert_graph_edge_to_node(edge_id, graphedge, n->first, graphlist);
}

void Graph::insert_graph_edge(const std::map<int, GraphEdge>& gedges,
		std::list<Graph>& graphlist) const {
	list<Graph> graphs;
	graphs.push_back(*this);
	map<int, GraphEdge>::const_iterator ge;
	for (ge = gedges.begin(); ge != gedges.end(); ++ge) {
		list<Graph> updated;
		list<Graph>::const_iterator g;
		for (g = graphs.begin(); g != graphs.end(); ++g)
			g->insert_graph_edge(ge->first, ge->second, updated);
		updated.swap(graphs);
		if (graphs.empty())
			break;
	}
	graphlist.swap(graphs);
}

// graph manipulation

void Graph::remove_external_legs() {
	list<Edge> ext = external_edges();
	// remember independent external momenta
	exmap extmom2zero;
	for (list<Edge>::const_iterator e = ext.begin(); e != ext.end(); ++e) {
		const ex& mom = graphedge(e->id).momentum_;
		if (is_a<symbol>(mom))
			extmom2zero[mom] = 0;
	}
	if (extmom2zero.size() + 1 != ext.size())
		throw runtime_error(
				"Mismatch: " + to_string(ext.size() - 1)
						+ " independent external momenta with "
						+ to_string(extmom2zero.size())
						+ " different symbols.");
	// remove external legs
	Topology::remove_external_legs();
	// set external momenta to zero
	map<int, GraphEdge>::iterator g;
	for (g = graphedges_.begin(); g != graphedges_.end(); ++g)
		g->second.momentum_ = g->second.momentum_.subs(extmom2zero,
				GiNaC::subs_options::no_pattern);
	remove_degree_2_nodes();
	ASSERT(graphedges_.size() == num_edges());
}

void Graph::merge_external_legs() {
	map<int, pair<Edge, GiNaC::ex> > merged; // indexed by internal node the leg is attached to
	list<Edge> ext = external_edges();
	for (list<Edge>::iterator e = ext.begin() ; e != ext.end() ; ++e) {
		GraphEdge ge = graphedge(e->id);
		// external node -> internal node
		if (!is_external_node(e->from)) {
			e->reverse();
			ge.momentum_ = -ge.momentum_;
		}
		if (merged.find(e->to) == merged.end())
			merged[e->to] = make_pair(*e, ge.momentum_);
		else
			merged[e->to].second += ge.momentum_;
	}
	Topology::remove_external_legs(); // also deletes isolated external nodes
	map<int, pair<Edge, GiNaC::ex> >::const_iterator m;
	const Kinematics *kin = Files::instance()->kinematics();
	for (m = merged.begin() ; m != merged.end(); ++m) {
		GiNaC::ex p = m->second.second;
		exmap s2i = kin->rules_sp_to_invariants();
		GiNaC::ex m2 = ScalarProduct(p, p).expand().subs(s2i).expand();
		int id = insert_edge(m->second.first);
		graphedges_.insert(make_pair(id, GraphEdge(p, m2)));
	}
}

void Graph::join_external_nodes() {
	list<Edge> ext = external_edges();
	if (ext.empty())
		return;
	// remember external masses and momenta
	map<int, GraphEdge> gedges;
	for (list<Edge>::const_iterator e = ext.begin(); e != ext.end(); ++e)
		gedges.insert(make_pair(e->id, graphedge(e->id)));
	// join external nodes
	Topology::join_external_nodes();
	// set masses and momenta on the new edges
	exset newlm;
	for (map<int, GraphEdge>::const_iterator e = gedges.begin();
			e != gedges.end(); ++e) {
		set_momentum(e->first, e->second.momentum_);
		set_squaredmass(e->first, e->second.squaredmass_);
		if (is_a<symbol>(e->second.momentum_))
			newlm.insert(e->second.momentum_);
	}
	remove_degree_2_nodes();
	// set independent external momenta as new loop momenta
	lst lm = add_lst(loop_momenta(), Reduze::set2lst(newlm));
	set_loop_momenta(lm);
	ASSERT(newlm.size() + 1 == ext.size());
	ASSERT(graphedges_.size() == num_edges());
}

void Graph::contract_multiple_internal_propagators() {
	exset props;
	const map<int, GraphEdge> ges = graphedges_; // copy
	for (map<int, GraphEdge>::const_iterator e = ges.begin(); e != ges.end();
			++e) {
		if (is_external_edge(e->first))
			continue;
		ex mom = e->second.momentum_;
		ex sqmass = e->second.squaredmass_;
		if (mom.is_equal(undefined) || sqmass.is_equal(undefined))
			continue;
		ex p = Propagator(mom, sqmass);
		if (!props.insert(p).second)
			contract_edge(e->first);
	}
}

void Graph::disconnect_vacuum_components() {
	list<Topology> other;
	list<Topology> vac = biconnected_vacuum_components(other);
	other.splice(other.end(), vac);

	Graph graph;
	graph.set_name(name());
	graph.set_loop_momenta(loop_momenta());
	set<int> inserted;
	for (list<Topology>::const_iterator t = other.begin(); t != other.end();
			++t) {
		const map<int, Edge>& es = t->edges();
		for (map<int, Edge>::const_iterator e = es.begin(); e != es.end(); ++e)
			inserted.insert(graph.insert_edge(e->second));
	}
	for (set<int>::const_iterator e = inserted.begin(); e != inserted.end();
			++e) {
		graph.set_momentum(*e, get_momentum(*e));
		graph.set_squaredmass(*e, get_squaredmass(*e));
	}
	graph.swap(*this);
}


/// splits identified external nodes into external nodes again, returns true on success
/** function succeeds only if the edges of 'g' with the same edge ids as
 *  the external edges from 'nonvac' are adjacent to one node with the same
 *  directions.
 */
bool split_identified_external_nodes(Graph& g, const Graph& nonvac) {
	int new_node = -1;
	list<Edge> ext_edges = nonvac.external_edges();
	int num_ext_edges = ext_edges.size();
	const map<int, list<Edge> >& edges_of_node = g.edges_of_nodes();
	map<int, list<Edge> >::const_reverse_iterator n;
	for (n = edges_of_node.rbegin(); n != edges_of_node.rend(); ++n) {
		const list<Edge>& ge = n->second;
		int count = 0;
		set<int> new_nodes;
		for (list<Edge>::const_iterator it = ge.begin(); it != ge.end(); ++it) {
			if (nonvac.is_external_edge(it->id)) {
				if (nonvac.is_external_node(nonvac.edge(it->id).to)) // it->id has been an outgoing edge
					new_nodes.insert(it->to);
				else
					new_nodes.insert(it->from);
				++count;
				if (new_nodes.size() > 1)
					break;
			}
		}
		if (count == num_ext_edges && new_nodes.size() == 1) {
			new_node = *new_nodes.begin();
			break;
		}
	}
	if (n == edges_of_node.rend())
		return false;

	for (list<Edge>::const_iterator e = ext_edges.begin(); e != ext_edges.end();
			++e) {
		int id = e->id;
		Edge edge = g.edge(id);
		GraphEdge gedge = g.graphedge(id);
		g.remove_edge(id);
		if (nonvac.is_external_node(e->to)) // outgoing
			g.insert_edge(Edge(edge.from, e->to, id));
		else
			g.insert_edge(Edge(e->from, edge.to, id));
		g.set_momentum(id, gedge.momentum_);
		g.set_squaredmass(id, gedge.squaredmass_);
	}
	g.remove_isolated_node(new_node);
	return true;
}

void Graph::minimize_by_twists() {
	string origname = name();
	list<Graph*> twists = find_twists();
	list<Graph*> minimals = select_minimals(twists);
	ASSERT(!minimals.empty());
	*this = *(*minimals.begin());
	set_name(origname);
	for (; !twists.empty(); twists.pop_front())
		delete twists.front();
}

std::list<Graph*> Graph::find_twists() const {
	list<Graph*> result;
#ifdef DEBUG
	static int count = 0;
	++count;
	print_dot("g_" + to_string(count) + "_a.dot");
#endif
	const list<Edge> ext_edges = external_edges();
	const int num_ext_edges = ext_edges.size();
	Graph gvac(*this);
	if (num_ext_edges > 0)
		identify_external_nodes(gvac);
#ifdef DEBUG
	gvac.print_dot("g_" + to_string(count) + "_b.dot");
#endif
	string origname = name();
	string twistname = name();
	try {
		//bool equiv_edges = (gvac.get_different_masses().nops() == 1);
		bool equiv_edges = (max_num_edge_colors() < 2);
		if (num_ext_edges > 0)
			equiv_edges = false;
		TriConnectedComponents tcc(gvac, equiv_edges);
		list<pair<Topology, set<int> > > twists =
				tcc.find_topologies_of_matroid();
		if (twists.empty())
			throw "empty twists";
		list<pair<Topology, set<int> > >::const_iterator t;
		bool first = true;
		for (t = twists.begin(); t != twists.end(); ++t) {
			twistname = t->first.name();
			Topology tp(t->first); // twisted topo with corrected orientations
			const set<int>& rev = t->second; // reverse edges to match momenta
			for (set<int>::const_iterator r = rev.begin(); r != rev.end(); ++r)
				tp.reverse_edge(*r);
			Graph g(gvac);
			static_cast<Topology&>(g) = tp;
#ifdef DEBUG
			static int count_twist = 0;
			++count_twist;
			g.print_dot("g_" + to_string(count) + "_twisted_"
					+ to_string(count_twist) + ".dot");
#endif
			// sanity check: momentum conserved (leave in production code for now!)
			for (set<int>::const_iterator n = g.nodes().begin();
					n != g.nodes().end(); ++n)
				if (/*!g.is_external_node(*n) &&*/!g.is_momentum_conserved(*n))
					throw "momentum conservation";
			// split external node
			if (num_ext_edges > 0) {
				if (!split_identified_external_nodes(g, *this)) {
					reverse_all_edges(tp);
					static_cast<Topology&>(g) = tp;
					if (!split_identified_external_nodes(g, *this))
						continue;
				}
			}
#ifdef DEBUG
			// currently used only for graphs with all momenta defined,
			// additional check: all momenta still defined after twisting
			map<int, Edge>::const_iterator e;
			for (e = g.edges().begin(); e != g.edges().end(); ++e)
			if (g.graphedges_.find(e->first) == g.graphedges_.end())
			throw "no momentum defined for edge";
#endif
			first = false;
			Graph* ptr = clone();
			static_cast<Graph&>(*ptr) = g;
			ptr->set_name(origname + "_" + twistname);
			result.push_back(ptr);
		}
		if (first)
			throw "original graph not contained in the twisted graphs";
#ifdef DEBUG
		print_dot("g_" + to_string(count) + "_c.dot");
#endif
	} catch (const char* reason) {
		ERROR("Twisting graph '" + origname + "' / '" + twistname + "' " + ""
		"failed for an internal reason\n(" + reason + ")." + ""
		"\nYou could try disabling twisting/matroid related features"
		"\ncompletely (or just for components with external momenta)"
		"\nand using the option construct_minimal_graphs in the jobs"
		"\n(e.g. in setup_sector_mappings)."
		"\nPlease let us know about the problem.");
	}
	return result;
}

std::list<Graph*> Graph::find_non_isomorphic_twists() const {
	LOGX("Finding non-isomorphic twists of graph " << name() << ":");
	list<Graph*> result, twists = find_twists();
	set<CanonicalLabel> labels;
	for (; !twists.empty(); twists.pop_front())
		if (labels.insert(twists.front()->find_canonical_label().first).second)
			result.push_back(twists.front());
		else
			delete twists.front();
	LOGX("Found " << result.size() << " non-isomorphic twists");
	return result;
}

//void Graph::minimize_by_twists(/*bool tweak_external_nodes*/) {
//#ifdef DEBUG
//	static int count = 0;
//	++count;
//	print_postscript("g_" + to_string(count) + "_a.ps");
//#endif
//	string origname = name();
//	string twistname = name();
//	try {
//		bool equiv_edges = (get_different_masses().nops() == 1);
//		TriConnectedComponents tcc(*this, equiv_edges);
//		pair<GraphLabel, Graph> best;
//		list<pair<Topology, set<int> > > twists =
//				tcc.find_topologies_of_matroid();
//		list<pair<Topology, set<int> > >::const_iterator t;
//		for (t = twists.begin(); t != twists.end(); ++t) {
//			twistname = t->first.name();
//			Graph g(*this);
//			static_cast<Topology&> (g) = t->first;
//			map<int, GraphEdge>::iterator ge;
//			for (ge = g.graphedges_.begin(); ge != g.graphedges_.end(); ++ge)
//				// reverse direction of some momenta
//				if (ge->second.momentum_ == undefined)
//					continue; //throw "edge with undefined momentum";
//				else if (t->second.find(ge->first) != t->second.end())
//					ge->second.momentum_ = -ge->second.momentum_;
//			// sanity check: momentum conserved (leave in production code for now!)
//			for (set<int>::const_iterator n = g.nodes().begin(); n
//					!= g.nodes().end(); ++n)
//				if (!g.is_external_node(*n) && !g.is_momentum_conserved(*n))
//					throw "momentum conservation";
//#ifdef DEBUG
//			// currently used only for graphs with all momenta defined,
//			// additional check: all momenta still defined after twisting
//			map<int, Edge>::const_iterator e;
//			for (e = edges().begin(); e != edges().end(); ++e)
//			if (graphedges_.find(e->first) == graphedges_.end()) {
//				throw "no momentum defined for edge";
//			}
//#endif
//			GraphLabel label = g.find_canonical_graph_label().first;
//			if (t == twists.begin() || label < best.first)
//				best = make_pair(label, g);
//		}
//		*this = best.second;
//		set_name(origname);
//#ifdef DEBUG
//		print_postscript("g_" + to_string(count) + "_b.ps");
//#endif
//	} catch (const char* reason) {
//		ERROR("Twisting graph '" + origname + "' / '" + twistname + "' " +
//				"failed for an internal reason\n(" + reason + ")." +
//				"\nYou could try disabling twisting/matroid related features"
//				"\ncompletely (or just for components with external momenta)."
//				"\nPlease let us know about the problem.");
//	}
//}

std::pair<std::list<std::map<int, int> >, edge_attributes> Graph::get_edge_coloring() const {
	const Topology topo = static_cast<const Topology&>(*this); // copy
	pair<list<map<int, int> >, edge_attributes> edge_coloring = topo.get_edge_coloring();
	if (num_edges() == 0)
		return edge_coloring;
	GiNaC::lst masses = get_different_masses();
	map<ex, int, mass_is_less> color_by_mass;
	int lauf = 0; // start with zero
	for (lst::const_iterator m = masses.begin(); m != masses.end(); ++m)
		color_by_mass[*m] = lauf++;
	map<int, int> color_of_edge;
	map<int, GraphEdge>::const_iterator e;
	for (e = graphedges_.begin(); e != graphedges_.end(); ++e)
		color_of_edge[e->first] = color_by_mass.at(e->second.squaredmass_);
	edge_coloring.first.push_back(color_of_edge);
	edge_coloring.second.masses_ = masses;
	// ensure graph labels can match with sectorgraph labels
	edge_coloring.second.propagator_types_.insert(IntegralFamily::PropagatorType::standard);
	return edge_coloring;
}

// get the crossing from graph 'g1' to 'g2' with given node permutations
Crossing get_crossing(const std::map<int, int>& node_perm, const Graph& g1,
		const Graph& g2, const Kinematics* kin) {
	map<int, int> ext_edge_perm;
	map<int, int>::const_iterator it;
	for (it = node_perm.begin(); it != node_perm.end(); ++it) {
		VERIFY(g1.contains_node(it->first) && g2.contains_node(it->second));
		bool is_external = g1.is_external_node(it->first);
		bool is_external2 = g2.is_external_node(it->second);
		VERIFY(is_external == is_external2);
		if (is_external) {
			const list<Edge>& edges1 = g1.edges_of_node(it->first);
			const list<Edge>& edges2 = g2.edges_of_node(it->second);
			VERIFY(edges1.size() == 1 && edges2.size() == 1);
			ext_edge_perm[edges1.begin()->id] = edges2.begin()->id;
		}
	}
	VERIFY(kin->external_momenta().nops() == ext_edge_perm.size());
	Crossing c(kin, Permutation(ext_edge_perm));
	if (!Files::instance()->crossings(kin->name())->has(c))
		throw runtime_error(
				"Crossing " + to_string(c.permutation()) + " of kinematics '"
						+ c.kinematics()->name()
						+ "' not defined in crossings file.");
	return c;
}

// finds the node permutation and the minimal crossing needed for finding the shift
// from 'graph1' to 'graph2' (graphs must be isomorphic)
// with the symmetry group of 'graph2' the node permutation is transformed
// such that the crossing gets minimal
std::pair<std::map<int, int>, Crossing> get_minimal_crossing(
		const Graph& graph1, const std::vector<int>& np1, //
		const Graph& graph2, const std::vector<int>& np2, //
		const std::list<std::map<int, int> >& sym2, const Kinematics* kin) {

	map<int, int> node_perm;
	Crossing crossing(kin);
	VERIFY(!sym2.empty());
	bool first = true;
	for (list<map<int, int> >::const_iterator s2 = sym2.begin();
			s2 != sym2.end(); ++s2) {
		vector<int> new_np2(graph2.num_nodes());
		for (unsigned i = 0; i < new_np2.size(); ++i)
			new_np2[i] = s2->at(np2[i]);
		map<int, int> tmp;
		tmp = CanonicalRelabeling::find_node_permutation(np1, new_np2);
		Crossing c(kin);
		try {
			c = get_crossing(tmp, graph1, graph2, kin);
		} catch (exception& exc) {
			LOGX("discarding an invalid crossing: " + to_string(exc.what()));
			continue;
		}
		if (first || c.inverse() < crossing.inverse()) {
			first = false;
			crossing = c;
			node_perm = tmp;
			if (crossing.is_identity())
				break;
		}
	}
	if (first)
		throw runtime_error(
				"failed to find minimal crossing b.c. they are invalid.");
	return make_pair(node_perm, crossing);
}

/*
// removes nodes not contained in the graph
vector<int> clean_node_permutation(const vector<int>& node_perm,
		const Graph& graph) {
	vector<int> result;
	result.reserve(graph.nodes().size());
	for (unsigned i = 0; i < node_perm.size(); ++i)
		if (graph.contains_node(node_perm[i]))
			result.push_back(node_perm[i]);
	VERIFY(result.size() == graph.num_nodes());
	return result;
}
*/

GiNaC::ex Graph::find_shift(const Graph& graph1,
		const std::pair<CanonicalLabel, CanonicalRelabeling>& label1,
		const Graph& graph2,
		const std::pair<CanonicalLabel, CanonicalRelabeling>& label2,
		GiNaC::exmap& shift, Crossing& crossing,
		const std::list<std::map<int, int> >& sym2) {
#ifdef DEBUG
	graph1.print_postscript("shift_g1.ps");
	graph2.print_postscript("shift_g2.ps");
#endif
	const Kinematics* kin = crossing.kinematics();
	VERIFY(kin != 0);

	// assert the graphs are isomorphic
	string gname1 = "'" + graph1.name() + "'";
	string gname2 = "'" + graph2.name() + "'";
	if (label1.first != label2.first)
		throw runtime_error(
				"graphs " + gname1 + " and " + gname2
						+ " don't have the same canonical labeling.");

	// assert the loop momenta are set
	const lst& srcloops = graph1.loop_momenta();
	const lst& destloops = graph2.loop_momenta();
	if (srcloops.nops() != destloops.nops()) {
		ostringstream srcl, destl;
		srcl << srcloops;
		destl << destloops;
		throw runtime_error("not the same number of loop momenta set:\n " //
		+ gname1 + ": " + srcl.str() + "\n " //
				+ gname2 + ": " + destl.str());
	}
	// assert the propagator momenta are set
	if (!graph1.get_edges_undefined_momentum().empty())
		throw runtime_error("graph " + gname1 + " has undefined momenta.");
	if (!graph2.get_edges_undefined_momentum().empty())
		throw runtime_error("graph " + gname2 + " has undefined momenta.");

	// find the node permutation and the crossing

	map<int, int> node_perm;
	vector<int> np1 = label1.second.node_permutation();
	vector<int> np2 = label2.second.node_permutation();
	//np1 = clean_node_permutation(np1, graph1);
	//np2 = clean_node_permutation(np2, graph2);

	try {
		if (!sym2.empty() && graph2.num_external_nodes() > 0) {
			pair<map<int, int>, Crossing> c = get_minimal_crossing(graph1, np1,
					graph2, np2, sym2, kin);
			node_perm = c.first;
			crossing = c.second;
		} else {
			node_perm = CanonicalRelabeling::find_node_permutation(np1, np2);
			crossing = get_crossing(node_perm, graph1, graph2, kin);
		}
	} catch (exception& exc) {
		throw runtime_error(
				"Failed to find a shift between isomorphic graphs "
						+ graph1.name() + " and " + graph2.name() + ": "
						+ exc.what());
	}

	exmap mom_rule = crossing.rules_momenta();

	// find the mapping of the cycle edges (compare masses, stop on first match)

	set<int> cycles = graph1.find_cycles(), visited;
	if (cycles.size() != srcloops.nops()) {
		stringstream ss;
		ss << "\nsrc loop momenta:\n" << srcloops << "\n";
		ss << "cycles:\n";
		for (set<int>::const_iterator c = cycles.begin(); c != cycles.end();
				++c)
			ss << *c << " ";
		ss << "\n";
		ss << "printing graphs.\n";
		graph1.print_dot("graph_error_occurred1.dot");
		graph2.print_dot("graph_error_occurred2.dot");
		throw runtime_error(
				"wrong number of loop momenta set in graph " + gname1
						+ ss.str());
	}
	map<int, pair<int, int> > cycle_edge_perm;
	for (set<int>::const_iterator c = cycles.begin(); c != cycles.end(); ++c) {
		const Edge& edge = graph1.edge(*c);
		int from = node_perm.at(edge.from);
		int to = node_perm.at(edge.to);
		ex sqmass = graph1.get_squaredmass(edge.id);
		const list<Edge>& oes = graph2.edges_of_node(from);
		int sign = 0;
		list<Edge>::const_iterator e2 = oes.begin();
		for (; e2 != oes.end(); ++e2) {
			if (visited.find(e2->id) != visited.end())
				continue;
			ex sqmass2 = graph2.get_squaredmass(e2->id);
			if (!sqmass2.is_equal(sqmass))
				continue;
			if (e2->to == to && e2->from == from)
				sign = 1;
			else if (e2->from == to && e2->to == from)
				sign = -1;
			if (sign != 0) { // for multiple edges any edge will do the job
				visited.insert(e2->id);
				break;
			}
		}
		if (e2 == oes.end()) {
			stringstream msg;
			msg << "\nFailed mapping cycle edge of graph1: " << edge;
			msg << " to edge " << from << " -> " << to << " of graph2\n";
			msg << "Printing graphs ..." << endl;
			graph1.print_dot("graph_error_occurred1.dot");
			graph2.print_dot("graph_error_occurred2.dot");
			graph1.print_postscript("error_graph1.ps");
			graph2.print_postscript("error_graph2.ps");
			msg << "Label graph1: " << endl << label1.first << endl;
			msg << "Label graph2: " << endl << label2.first << endl;
			ERROR("Internal error: " << msg.str());
		}
		//VERIFY(e2 != oes.end());
		cycle_edge_perm[edge.id] = make_pair(sign, e2->id);
	}

	// determine the shift of the loop momenta which transforms graph1 to graph2

	lst tmpsrcloops, equations;
	exmap loop2tmp, tmp2loop;
	provide_alternative_symbols(srcloops, tmpsrcloops, loop2tmp, tmp2loop);
	map<int, pair<int, int> >::const_iterator p;
	for (p = cycle_edge_perm.begin(); p != cycle_edge_perm.end(); ++p) {
		const GraphEdge& g1 = graph1.graphedge(p->first);
		const GraphEdge& g2 = graph2.graphedge(p->second.second);
		ex lhs = g1.momentum_;
		ex rhs = p->second.first * g2.momentum_;
		lhs = lhs.subs(mom_rule, subs_options::no_pattern);
		lhs = lhs.subs(loop2tmp, subs_options::no_pattern);
		equations.append(lhs == rhs);
	}
	equations = ex_to<lst>(lsolve(equations, tmpsrcloops));

	VERIFY(equations.nops() == destloops.nops());
	ex det = abs(jacobi_determinant(equations, destloops));
	shift = equations_to_substitutions(equations, tmp2loop, true);

	Crossing inverse_cross = crossing.inverse();
	shift = substitute_rhs(shift, inverse_cross.rules_momenta());
	VERIFY(!det.is_zero());
	return det;
}

// helper function for finding symmetry shifts via node permutation

namespace {

/** find all possible ways to map edges between nodes n1 and n2 of the graph to
 *  edges between nodes mapped by the node_perm
 */

std::list<std::map<int, std::pair<int, int> > > get_multi_edge_mappings(
		const Graph& graph, //
		const std::map<int, int>& node_perm, //
		int n1, int n2) {

	// mappings of edge id1 to pair (orientation, edge id2)
	list<map<int, pair<int, int> > > result;

	int n1_to = node_perm.at(n1);
	int n2_to = node_perm.at(n2);
	list<Edge> edges_from = graph.find_edges_between(n1, n2);
	list<Edge> edges_to = graph.find_edges_between(n1_to, n2_to);

	// edge ids by their mass
	map<ex, vector<int>, ex_is_less> edges_from_by_mass, edges_to_by_mass;

	// sort edges by their masses and find orientation
	list<Edge>::const_iterator e;
	for (e = edges_from.begin(); e != edges_from.end(); ++e) {
		ex sqm = graph.get_squaredmass(e->id);
		edges_from_by_mass[sqm].push_back(e->id);
	}
	for (e = edges_to.begin(); e != edges_to.end(); ++e) {
		ex sqm = graph.get_squaredmass(e->id);
		edges_to_by_mass[sqm].push_back(e->id);
	}

	// find permutations of the multi-edges with the same mass
	// each entry of the vector corresponds to a mass
	vector<list<map<int, pair<int, int> > > > trafos_by_mass(
			edges_to_by_mass.size());

	VERIFY(edges_from_by_mass.size() == edges_to_by_mass.size());
	map<ex, vector<int>, ex_is_less>::iterator it;
	int mass_number = 0;
	for (it = edges_to_by_mass.begin(); it != edges_to_by_mass.end(); ++it) {
		vector<int>& to = it->second;
		const vector<int>& from = edges_from_by_mass.at(it->first);
		VERIFY(to.size() == from.size());
		std::sort(to.begin(), to.end());
		do {
			map<int, pair<int, int> > m;
			for (unsigned i = 0; i < to.size(); ++i) {
				int sign = (
						(node_perm.at(graph.edge(from[i]).from)
								== graph.edge(to[i]).from) ? 1 : -1);
				m[from[i]] = make_pair(sign, to[i]);
			}
			trafos_by_mass[mass_number].push_back(map<int, pair<int, int> >());
			trafos_by_mass[mass_number].back().swap(m);
		} while (std::next_permutation(to.begin(), to.end()));
		++mass_number;
	}

	// find combinations of the edge-transformations with different masses
	combinations<map<int, pair<int, int> > > combs(trafos_by_mass);
	vector<map<int, pair<int, int> > > r;
	while (combs.get_next(r)) {
		map<int, pair<int, int> > merged;
		for (unsigned i = 0; i < r.size(); ++i)
			merged.insert(r[i].begin(), r[i].end());
		result.push_back(map<int, pair<int, int> >());
		result.back().swap(merged);
	}
	return result;
}

// return permutation of internal edges
map<int, int> internal_edge_perm(const Graph& graph,
		const map<int, pair<int, int> >& cycle_perm,
		const map<int, int>& node_perm) {

	map<int, int> res;
	map<int, Edge>::const_iterator e;
	const map<int, Edge>& edges = graph.edges();
	for (e = edges.begin(); e != edges.end(); ++e) {
		if (graph.is_external_edge(e->first))
			continue;
		else if (cycle_perm.find(e->first) != cycle_perm.end())
			res[e->first] = cycle_perm.at(e->first).second;
		else {
			int ep1 = node_perm.at(e->second.from);
			int ep2 = node_perm.at(e->second.to);
			list<Edge> between = graph.find_edges_between(ep1, ep2);
			VERIFY(between.size() == 1);
			res[e->first] = between.front().id;
		}
	}
	return res;
}

map<int, int> identity_edge_perm(const Graph& graph) {
	map<int, int> identity;
	const map<int, Edge>& edges = graph.edges();
	map<int, Edge>::const_iterator e;
	for (e = edges.begin(); e != edges.end(); ++e)
		identity[e->first] = e->first;
	return identity;
}

} // namespace

void Graph::find_symmetry_shifts(std::list<GiNaC::exmap>& shifts,
		const Kinematics* kin) const {
	LOGX("Finding symmetry shifts of graph " << name() << ":");

	// find the node symmetry group of the graph with permutation of external
	// nodes partially allowed in order to find crossings equivalent to the identity
	bool suppress_free_ext_node_perms = true;
	list<map<int, int> > perms = find_node_symmetry_group(suppress_free_ext_node_perms);
	LOGX("  Found " << perms.size() //
			<< " symmetries (from node permutations) in the graph");
	vector<Crossing> the_crossings;
	set<int> cycles = find_cycles();
	VERIFY(cycles.size() == loop_momenta().nops());
	list<map<int, pair<int, int> > > cycle_mappings;

	LOGX("  performing permutations of multi edges");
	set<map<int, int> > testset;
	testset.insert(identity_edge_perm(*this));

	int num_omitted_equal = 0;
	for (list<map<int, int> >::const_iterator p = perms.begin();
			p != perms.end(); ++p) {

		// crossing
		const map<int, int>& node_perm = *p;
		Crossing crossing(kin);
		try {
			crossing = get_crossing(node_perm, *this, *this, kin);
		} catch (exception& exc) {
			LOGX("  omitting invalid crossing: " << exc.what());
			continue;
		}
		if (!crossing.is_equivalent_to_identity()) {
			LOGX("  omitting crossing: " << crossing.rules_momenta() //
					<< " (not equivalent to identity crossing)");
			continue;
		}

		// all permutations of edges from multiedges
		set<pair<int, int> > visited;
		vector<list<map<int, pair<int, int> > > > multi_edge_mappings;
		for (set<int>::const_iterator c = cycles.begin(); c != cycles.end();
				++c) {
			const Edge& e = edge(*c); // filter multi-edges, take only one
			int n1 = std::min(e.from, e.to);
			int n2 = std::max(e.from, e.to);
			if (!visited.insert(make_pair(n1, n2)).second)
				continue;
			multi_edge_mappings.push_back(
					get_multi_edge_mappings(*this, node_perm, n1, n2));
		}

		// all combinations of the edge mappings of the cycles

		combinations<map<int, pair<int, int> > > combs(multi_edge_mappings);
		vector<map<int, pair<int, int> > > r;
		while (combs.get_next(r)) {
			map<int, pair<int, int> > merged;
			for (unsigned i = 0; i < r.size(); ++i)
				merged.insert(r[i].begin(), r[i].end());

			// keep only permutations which transform internal edges differently
			map<int, int> iperm = internal_edge_perm(*this, merged, *p);
			if (testset.insert(iperm).second) {
				cycle_mappings.push_back(merged);
				the_crossings.push_back(crossing);
			} else {
				++num_omitted_equal;
			}
		}
	}
	if (num_omitted_equal > 0) {
		LOGX("  omitted " << num_omitted_equal //
				<< " multiple occurring permutations (including identity)");
	}
	LOGX("  found " << cycle_mappings.size() //
			<< " permutations of fundamental cycles");

	list<map<int, pair<int, int> > >::const_iterator ml;
	unsigned lauf = 0;
	VERIFY(the_crossings.size() == cycle_mappings.size());
	for (ml = cycle_mappings.begin(); ml != cycle_mappings.end();
			++ml, ++lauf) {
		const Crossing& crossing = the_crossings[lauf];
		exmap mom_rule = crossing.rules_momenta();

		// determine symmetry shift

		exmap shift;
		lst srcloops = loop_momenta_;
		lst tmpsrcloops, equations;
		exmap loop2tmp, tmp2loop;
		provide_alternative_symbols(srcloops, tmpsrcloops, loop2tmp, tmp2loop);
		map<int, pair<int, int> >::const_iterator p;
		for (p = ml->begin(); p != ml->end(); ++p) {
			const GraphEdge& g1 = graphedge(p->first);
			const GraphEdge& g2 = graphedge(p->second.second);
			ex lhs = g1.momentum_;
			ex rhs = p->second.first * g2.momentum_;
			lhs = lhs.subs(mom_rule, subs_options::no_pattern);
			lhs = lhs.subs(loop2tmp, subs_options::no_pattern);
			equations.append(lhs == rhs);
		}
		equations = ex_to<lst>(lsolve(equations, tmpsrcloops));
		VERIFY(equations.nops() == srcloops.nops());
		ex det = abs(jacobi_determinant(equations, srcloops));
		VERIFY(!det.is_zero());
		shift = equations_to_substitutions(equations, tmp2loop, true);
		shift.insert(mom_rule.begin(), mom_rule.end());
		// add the permutation of the external momenta to the shift
		if (det.is_equal(1)) {
			if (!shift.empty()) {
				shifts.push_back(shift);
			} else {
				LOGX("  omitting empty shift (identity)");
			}
		} else {
			LOG("  reject symmetry shift with |det| " //
					<< det << ", shift: " << shift);
		}
	}
	LOGX("  Found " << shifts.size() //
			<< " symmetry shifts (permutations of multi-edges performed)");
}

namespace {
// get the crossing in graph 'g1' with given edge permutations
Crossing get_crossing_from_edge_perm(const std::map<int, pair<int, int> >& edge_perm,
		const Graph& g1, const Kinematics* kin) {

	// filter external edge permutations
	map<int, int> ext_edge_perm;
	map<int, pair<int, int> >::const_iterator it;
	for (it = edge_perm.begin(); it != edge_perm.end(); ++it) {
		bool is_external = g1.is_external_edge(it->first);
		bool is_external2 = g1.is_external_edge(it->second.second);
		VERIFY(is_external == is_external2);
		if (is_external) {
			ext_edge_perm[it->first] = it->second.second;
			VERIFY(it->second.first > 0);
		}
	}
	VERIFY(kin->external_momenta().nops() == ext_edge_perm.size());
	Crossing c(kin, Permutation(ext_edge_perm));
	if (!Files::instance()->crossings(kin->name())->has(c))
		throw runtime_error(
				"Crossing " + to_string(c.permutation()) + " of kinematics '"
						+ c.kinematics()->name()
						+ "' not defined in crossings file.");
	return c;
}
}

void Graph::find_edge_symmetry_shifts(std::list<GiNaC::exmap>& shifts,
		const Kinematics* kin) const {
	LOGX("Finding symmetry shifts (via line graph) of graph " << name() << ":");
	list<map<int, pair<int, int> > > edge_perms;
	// allow no strict pruning to generate less unwanted permutations
	bool suppress_free_ext_node_perms = true;
	find_edge_symmetry_group(edge_perms, suppress_free_ext_node_perms);

	LOGX("  Found " << edge_perms.size() //
			<< " symmetries (from edge permutations) in the graph");

	set<int> cycles = find_cycles();
	VERIFY(cycles.size() == loop_momenta_.nops());
	set<map<int, int> > testset;
	testset.insert(identity_edge_perm(*this));

	list<map<int, pair<int, int> > >::const_iterator perm;
	for (perm = edge_perms.begin(); perm != edge_perms.end(); ++perm) {

		// permutation of fundamental cycles and internal edges
		map<int, pair<int, int> > cperm;
		map<int, int> iperm;
		map<int, pair<int, int> >::const_iterator p;
		for (p = perm->begin(); p != perm->end(); ++p) {
			if (cycles.find(p->first) != cycles.end())
				cperm[p->first] = p->second;
			if (!is_external_edge(p->first))
				iperm[p->first] = p->second.second;
		}
		VERIFY(cperm.size() == loop_momenta().nops());

		// crossing
		Crossing crossing(kin);
		try {
			crossing = get_crossing_from_edge_perm(*perm, *this, kin);
		} catch (exception& exc) {
			LOGX("  omitting invalid crossing: " << exc.what());
			continue;
		}
		if (!crossing.is_equivalent_to_identity()) {
			LOGX("  omitting crossing: " << crossing.rules_momenta() //
					<< " (not equivalent to identity crossing)");
			continue;
		}

		// keep only permutations which transform internal edges differently
		if (!testset.insert(iperm).second) {
			LOGX("  omitting permutation already used or identity");
			continue;
		}

		exmap mom_rule = crossing.rules_momenta();
		exmap shift;
		lst srcloops = loop_momenta_;
		lst tmpsrcloops, equations;
		exmap loop2tmp, tmp2loop;
		provide_alternative_symbols(srcloops, tmpsrcloops, loop2tmp, tmp2loop);
		for (p = cperm.begin(); p != cperm.end(); ++p) {
			int sign = p->second.first;
			ex lhs = graphedge(p->first).momentum_;
			ex rhs = sign * graphedge(p->second.second).momentum_;
			lhs = lhs.subs(mom_rule, subs_options::no_pattern);
			lhs = lhs.subs(loop2tmp, subs_options::no_pattern);
			equations.append(lhs == rhs);
		}
		equations = ex_to<lst>(lsolve(equations, tmpsrcloops));
		ex det = abs(jacobi_determinant(equations, srcloops));
		if (det.is_zero()) {
			LOGX("  omitting shift with determinant zero:");
			continue;
		}
		VERIFY(equations.nops() == srcloops.nops());
		shift = equations_to_substitutions(equations, tmp2loop, true);
		// add the permutation of the external momenta to the shift
		shift.insert(mom_rule.begin(), mom_rule.end());
		if (det.is_equal(1)) {
			if (!shift.empty()) {
				shifts.push_back(shift);
			} else {
				LOGX("  omitting empty shift (identity)");
			}
		} else {
			LOG("  reject symmetry shift with |det| " //
					<< det << ", shift: " << shift);
		}
	}
	LOGX("  Found " << shifts.size() << " symmetry shifts");
}

bool Graph::match_to_integralfamily(const IntegralFamily* ic) {
	using namespace GiNaC;
	const bool analyze_subtopos = false;
	const vector<Propagator>& propagators = ic->propagators();
	if (ic->loop_momenta().nops() != loop_momenta_.nops()) {
		LOGXX("mismatch in number of loops");
		return false;
	}
	// temp names for our loop momenta in case they coincide with those from ic
	lst tmp_moms;
	exmap mom2tmp, tmp2mom;
	provide_alternative_symbols(loop_momenta_, tmp_moms, mom2tmp, tmp2mom);
	// try to find shifts from of loop momenta such that propagators match
	list<pair<int, pair<int, lst> > > shifts; // {(subtopo,(sector,eqs)),..}
	shifts.push_back(make_pair(0, make_pair(0, lst()))); // one empty element
	map<int, GraphEdge>::const_iterator eit;
	int loop_edge = 0; // loop edge numbered like 0,1,...
	for (eit = graphedges_.begin(); eit != graphedges_.end(); ++eit) {
		const GraphEdge& e = eit->second;
		if (e.momentum_ == undefined || e.squaredmass_ == undefined) {
			WARNING("Graph " << name() << " has undefined kinematics");
			return false;
		} else if (freeof(e.momentum_, loop_momenta_)) { // edge not part of a loop
			continue;
		}
		list<pair<int, pair<int, lst> > > new_shifts;
		while (!shifts.empty()) {
			// consider next edge and all possible propagators
			lst shift = ex_to<lst>(shifts.front().second.second);
			int sub = shifts.front().first;
			int sec = shifts.front().second.first;
			if (analyze_subtopos) // incl. subtopos (current edge unmatched)
				new_shifts.splice(new_shifts.end(), shifts, shifts.begin());
			else
				// no subtopologies
				shifts.pop_front();
			int new_sub = sub | (1 << loop_edge); // new subtopology id
			// try to match current edge
			for (int i = 0; i < (int) propagators.size(); ++i) {
				try {
					if (e.squaredmass_ != propagators[i].squaredmass())
						continue;
					int new_sec = sec | (1 << i); // new sector id
					ex o = e.momentum_.subs(mom2tmp); // old momentum
					ex n = propagators[i].momentum(); // new momentum
					lst s1 = shift, s2 = shift;
					s1 = ex_to<lst>(lsolve(s1.append(o == n), tmp_moms));
					s2 = ex_to<lst>(lsolve(s2.append(o == -n), tmp_moms));
					if (s1.nops() > 0)
						new_shifts.push_back(
								make_pair(new_sub, make_pair(new_sec, s1)));
					if (s2.nops() > 0)
						new_shifts.push_back(
								make_pair(new_sub, make_pair(new_sec, s2)));
				} catch (exception& e) {
					// ignore propagators without well-defined momentum
				}
			}
		}
		shifts.swap(new_shifts);
		++loop_edge;
	}
	// select shifts with jacobian 1 and provide substitution maps
	bool found = false;
	list<pair<int, pair<int, lst> > >::iterator l;
	for (l = shifts.begin(); l != shifts.end(); ++l) {
		int sub = l->first;
		Sector sec(ic, l->second.first);
		lst sol = l->second.second;
		ex det = abs(jacobi_determinant(sol, ic->loop_momenta()).expand());
		if (det == 0)
			continue;
		if (det != 1)
			ABORT("Don't know how to handle Jacobian determinant " << det);
		exmap subst = equations_to_substitutions(sol, tmp2mom);
		if (sub == (1 << loop_edge) - 1) { // match for full graph
			found = true;
			graph_to_sector_[sec] = subst;
		}
		subgraph_to_sector_[sub][sec] = subst;
	}
	return found;
}

void Graph::print_sector_matches() const {
	map<int, map<Sector, GiNaC::exmap> >::const_iterator i;
	for (i = subgraph_to_sector_.begin(); i != subgraph_to_sector_.end(); ++i) {
		map<Sector, GiNaC::exmap>::const_iterator j;
		LOGNX("     sectors for subtopo " << i->first << ": ");
		for (j = i->second.begin(); j != i->second.end(); ++j) {
			LOGNX(j->first << " ");
		}
		LOGX("");
	}
}

const std::pair<const Sector, GiNaC::exmap>* Graph::shift_to_sector() const {
	if (graph_to_sector_.empty())
		return 0;
	return &(*graph_to_sector_.begin());
}

// output

YAML::Emitter& operator<<(YAML::Emitter& ye, const Graph& g) {
	using namespace YAML;
	ye << BeginMap;
	ye << Key << "name" << Value << g.name();
	ye << Key << "edges" << Value << Flow << BeginSeq;
	const map<int, Edge>& es = g.edges();
	for (map<int, Edge>::const_iterator e = es.begin(); e != es.end(); ++e) {
		ye << BeginSeq;
		ye << e->second.from << e->second.to << e->second.id
				<< g.get_momentum(e->second.id)
				<< g.get_squaredmass(e->second.id);
		ye << EndSeq;
	}
	ye << EndSeq;

	ye << Key << "nodes" << Value << Flow << BeginSeq;
	const set<int>& node = g.nodes();
	for (set<int>::const_iterator n = node.begin(); n != node.end(); ++n)
		ye << *n;
	ye << EndSeq;

	ye << EndMap;

	return ye;
}

std::ostream& operator<<(std::ostream& os, const Graph& g) {
	YAML::Emitter ye;
	ye << g;
	os << ye.c_str();
	return os;
}

void Graph::read(const YAML::Node& n, const GiNaC::lst& loopmoms,
		const GiNaC::lst& ext_kin) {
	using namespace YAML;
	if (n.Type() != NodeType::Map || n.size() != 3)
		throw runtime_error("node is not a 3-element map " + position_info(n));
	string name;
	n["name"] >> name;
	set_name(name);
	set_loop_momenta(loopmoms);
	const Node& edge_node = n["edges"];
	map<int, pair<ex, ex> > mom_mass;
	for (Iterator i = edge_node.begin(); i != edge_node.end(); ++i) {
		if (i->Type() != NodeType::Sequence || i->size() != 5)
			throw runtime_error(
					"node is not a 5-element sequence " + position_info(*i));
		Edge edge;
		(*i)[0] >> edge.from;
		(*i)[1] >> edge.to;
		(*i)[2] >> edge.id;
		insert_edge(edge);
		ex mom, mass;
		Reduze::read((*i)[3], mom, add_lst(loopmoms, ext_kin));
		Reduze::read((*i)[4], mass, ext_kin);
		mom_mass[edge.id] = make_pair(mom, mass);
	}
	for (map<int, pair<ex, ex> >::const_iterator i = mom_mass.begin();
			i != mom_mass.end(); ++i) {
		set_momentum(i->first, i->second.first);
		set_squaredmass(i->first, i->second.second);
	}

	const Node& node_node = n["nodes"];
	if (node_node.Type() != NodeType::Sequence)
		throw runtime_error(
				"node is not a sequence " + position_info(node_node));
	for (Iterator i = node_node.begin(); i != node_node.end(); ++i) {
		int n;
		*i >> n;
		insert_node(n);
	}
}

void Graph::print_mma(std::ostream& of) const {
  bool directed = true;
  of << "MultiGraph[{";
  const map<int, Edge>& es = edges();
  for (map<int, Edge>::const_iterator ep = es.begin(); ep != es.end(); ++ep) {
    const Edge& e = ep->second;
    if (ep != es.begin())
    	of << ", ";
    of << "Labeled[" << e.from << (directed ? "->" : "<->") << e.to << ", \"";
    if (graphedges_.find(e.id) != graphedges_.end()) {
	const GraphEdge& ge = graphedges_.find(e.id)->second;
	if (ge.momentum_ != undefined || ge.squaredmass_ != undefined)
		of << "(" << ge.momentum_ << "," << ge.squaredmass_
		   << ";" << ep->first << ")";
    }
    of << "\"]";
  }
  of << "}]";
}

void Graph::print_dot(const std::string& filename) const {
	ofstream of(filename.c_str());
	print_dot(of);
	of.close();
}

void Graph::print_dot(std::ostream& of) const {
	bool directed = true;
	of << (directed ? "digraph" : "graph") << " ";
	of << (name() != "graph" ? name() : "graph0") << " {\n";

	of << "  label=\"" << name() << "\";\n";
	of << "  rankdir=\"LR\";" << "\n";

	set<int> innodes, outnodes;
	of << "  subgraph edges {\n";
	const map<int, Edge>& es = edges();
	for (map<int, Edge>::const_iterator ep = es.begin(); ep != es.end(); ++ep) {
		const Edge& e = ep->second;
		if (e.from < 0)
			innodes.insert(e.from);
		if (e.to < 0)
			outnodes.insert(e.to);
		of << "    " << e.from << (directed ? " -> " : " -- ") << e.to;
		if (graphedges_.find(e.id) != graphedges_.end()) {
			const GraphEdge& ge = graphedges_.find(e.id)->second;
			if (ge.momentum_ != undefined || ge.squaredmass_ != undefined)
				of << " [label=\"(" << ge.momentum_ << "," << ge.squaredmass_
						<< ";" << ep->first << ")\"]";
		}
		of << ";\n";
	}
	of << "  }\n";

	set<int>::const_reverse_iterator n;
	if (!innodes.empty()) {
		of << "  subgraph incoming { rank=\"source\"; ";
		for (n = innodes.rbegin(); n != innodes.rend(); ++n)
			of << *n << "; ";
		of << "}\n";
	}
	if (!outnodes.empty()) {
		of << "  subgraph outgoing { rank=\"sink\"; ";
		for (n = outnodes.rbegin(); n != outnodes.rend(); ++n)
			of << *n << "; ";
		of << "}\n";
	}

	// nodes: small filled circles without label
	for (set<int>::const_iterator n = nodes().begin(); n != nodes().end(); ++n)
		of << *n << " [shape=point];\n";

	of << "}\n";

	of << "\n";
}

void Graph::removed_edge(int id) {
	graphedges_.erase(id);
}

void Graph::swapped() {
	subgraph_to_sector_.clear();
	graph_to_sector_.clear();
	graphedges_.clear();
	loop_momenta_.remove_all();
}

//
// SectorGraph
//

SectorGraph::SectorGraph() :
		Graph() {
}

SectorGraph::~SectorGraph() {
}

SectorGraph* SectorGraph::clone() const {
	return new SectorGraph(*this);
}

SectorGraph::SectorGraph(const Sector& sec, const Graph& graph) :
		Graph(graph) {
	sector_.push_back(sec);
	// set map prop pos --> edge id
	try {
		set_edge_id_by_prop_pos();
	} catch (exception& exc) {
		throw runtime_error(
				"Failed to construct a SectorGraph: " + to_string(exc.what()));
	}
}

SectorGraph::SectorGraph(const Sector& sec, const Graph& graph,
		const std::map<int, int>& edge_by_prop_pos) :
		Graph(graph), edge_id_by_prop_pos_(edge_by_prop_pos) {
	sector_.push_back(sec);
	// verify map prop pos --> edge id
	try {
		verify_edge_id_by_prop_pos();
	} catch (exception& exc) {
		throw runtime_error(
				"Failed to construct a SectorGraph: " + to_string(exc.what()));
	}
}

std::pair<std::list<std::map<int, int> >, edge_attributes> SectorGraph::get_edge_coloring() const {
	// mass colors
	const Graph graph = static_cast<const Graph&>(*this); // copy
	pair<list<map<int, int> >, edge_attributes> edge_coloring = graph.get_edge_coloring();
	if (num_edges() == 0)
		return edge_coloring;
	// colors from different types of propagators
	map<int, int> color_of_edge;
	int cutmask;
	const map<int, Propagator> props = sector().get_propagators_by_pos(cutmask);
	const IntegralFamily* ifam = sector().integralfamily();
	map<int, Propagator>::const_iterator p;
	for (p = props.begin(); p != props.end(); ++p) {
		int edge_id = edge_id_by_prop_pos_.at(p->first);
		color_of_edge[edge_id] = ifam->propagator_type(p->first);
	}
	const list<Edge> ext = external_edges();
	for (list<Edge>::const_iterator e = ext.begin(); e != ext.end(); ++e)
		color_of_edge[e->id] = IntegralFamily::PropagatorType::standard;

	edge_coloring.second.propagator_types_ = get_different_propagator_types();
	edge_coloring.first.push_back(color_of_edge);
	return edge_coloring;
}

const Sector& SectorGraph::sector() const {
	ASSERT(!sector_.empty());
	return sector_.front();
}
const std::map<int, int>& SectorGraph::edge_id_by_prop_pos() const {
	return edge_id_by_prop_pos_;
}

std::set<int> SectorGraph::get_different_propagator_types() const {
	set<int> prop_types;
	int cutmask;
	map<int, Propagator> props = sector().get_propagators_by_pos(cutmask);
	map<int, Propagator>::const_iterator p;
	for (p = props.begin(); p != props.end(); ++p)
		prop_types.insert(sector().integralfamily()->propagator_type(p->first));
	if (external_edges().size() > 0)
		prop_types.insert(IntegralFamily::PropagatorType::standard);
	return prop_types;
}

void SectorGraph::swap(SectorGraph& other) {
	Graph::swap(other);
	sector_.swap(other.sector_);
	edge_id_by_prop_pos_.swap(other.edge_id_by_prop_pos_);
}

void SectorGraph::find_symmetry_shifts(std::list<GiNaC::exmap>& shifts, bool examine_twists) const {
	const Sector& sec = sector();
	const Kinematics* kin = sec.integralfamily()->kinematics();
	LOGX("\nFinding symmetry shifts for sector " << sec //
			<< " (examine twists: " << (examine_twists ? "on" : "off") << ")");
	shifts.clear();
	list<Graph*> secgs;
	if (examine_twists) {
		secgs = find_non_isomorphic_twists();
	} else {
		Graph* ptr = new SectorGraph(sec, *this, edge_id_by_prop_pos_);
		secgs.push_back(ptr);
	}
	set<vector<int> > propagator_perms;
	for (; !secgs.empty(); delete secgs.front(), secgs.pop_front()) {
		const Graph* sg = secgs.front();
		ASSERT(dynamic_cast<const SectorGraph*>(sg) != 0);
		list<exmap> part_shifts;
		sg->Graph::find_symmetry_shifts(part_shifts, kin);
		for (list<exmap>::iterator i = part_shifts.begin();
				i != part_shifts.end(); ++i) {
			vector<int> prop_perm;
			bool b = SectorMappings::find_permutations_for_shift(sec, sec, *i,
					prop_perm);
			if (!b) {
				for (; !secgs.empty(); secgs.pop_front())
					delete secgs.front();
				throw runtime_error("invalid shift found");
			}
			if (propagator_perms.insert(prop_perm).second)
				shifts.push_back(*i);
		}
	}
	LOGX("Found " << shifts.size() << " symmetry shifts");
}

void SectorGraph::find_edge_symmetry_shifts(std::list<GiNaC::exmap>& shifts, bool examine_twists) const {
	const Sector& sec = sector();
	const Kinematics* kin = sec.integralfamily()->kinematics();
	LOGX("\nFinding symmetry shifts (via line graph) for sector " << sec //
			<< " (examine twists: " << (examine_twists ? "on" : "off") << ")");
	shifts.clear();
	list<Graph*> secgs;
	if (examine_twists) {
		secgs = find_non_isomorphic_twists();
	} else {
		Graph* ptr = new SectorGraph(sec, *this, edge_id_by_prop_pos_);
		secgs.push_back(ptr);
	}
	set<vector<int> > propagator_perms;
	for (; !secgs.empty(); delete secgs.front(), secgs.pop_front()) {
		const Graph* sg = secgs.front();
		ASSERT(dynamic_cast<const SectorGraph*>(sg) != 0);
		list<exmap> part_shifts;
		sg->Graph::find_edge_symmetry_shifts(part_shifts, kin);
		for (list<exmap>::iterator i = part_shifts.begin();
				i != part_shifts.end(); ++i) {
			vector<int> prop_perm;
			bool b = SectorMappings::find_permutations_for_shift(sec, sec, *i,
					prop_perm);
			if (!b) {
				for (; !secgs.empty(); secgs.pop_front())
					delete secgs.front();
				throw runtime_error("invalid shift found");
			}
			if (propagator_perms.insert(prop_perm).second)
				shifts.push_back(*i);
		}
	}
	LOGX("Found " << shifts.size() << " symmetry shifts");
}

void SectorGraph::set_edge_id_by_prop_pos() {
	// find the map of propagator positions and edge ids
	map<int, GraphEdge> graph_edges = graphedges(); // copy
	int cutmask;
	map<int, Propagator> props = sector().get_propagators_by_pos(cutmask);
	map<int, Propagator>::const_iterator p;
	for (p = props.begin(); p != props.end(); ++p) {
		ex pmom = p->second.momentum();
		map<int, GraphEdge>::iterator g = graph_edges.begin();
		for (; g != graph_edges.end(); ++g) {
			ex gmom = g->second.momentum_;
			if (pmom.is_equal(gmom) || pmom.is_equal(-gmom))
				break;
		}
		if (g == graph_edges.end())
			throw runtime_error(
					"graph has no momentum +/-(" + to_string(pmom) + ")");
		if (!(p->second.squaredmass().is_equal( //
				g->second.squaredmass_)))
			throw runtime_error(
					"mass of edge " + to_string(g->first)
							+ " does not match mass of propagator "
							+ to_string(p->first));
		edge_id_by_prop_pos_[p->first] = g->first;
		graph_edges.erase(g);
	}
}

void SectorGraph::verify_edge_id_by_prop_pos() const {
	int cutmask;
	const map<int, Propagator> props = sector_.front().get_propagators_by_pos(
			cutmask);
	if (sector().t() != (int) edge_id_by_prop_pos_.size())
		throw runtime_error(
				"mismatch in number of propagators and internal edges");
	map<int, int>::const_iterator m = edge_id_by_prop_pos_.begin();
	for (; m != edge_id_by_prop_pos_.end(); ++m) {
		const ex pmom = props.at(m->first).momentum();
		const ex gmom = get_momentum(m->second);
		if (!gmom.is_equal(pmom) && !gmom.is_equal(-pmom))
			throw runtime_error("momenta of edge and propagator do not match.");
	}
}

SectorGraph::SectorGraph(const YAML::Node& node) {
	Sector sec(node["sector"]);
	const IntegralFamily* ic = sec.integralfamily();
	Graph graph;
	graph.read(node["graph"], ic->loop_momenta(),
			ic->kinematics()->get_external_momenta_and_kinematics());
	map<int, int> edge_id_by_prop_pos;
	node["edge_ids_by_prop_pos"] >> edge_id_by_prop_pos;
	*this = SectorGraph(sec, graph, edge_id_by_prop_pos);
}

YAML::Emitter& operator<<(YAML::Emitter& ye, const SectorGraph& s) {
	using namespace YAML;
	ye << BeginMap;
	Graph graph = s;
	ye << Key << "graph" << Value << graph;
	ye << Key << "sector" << Value << s.sector();
	ye << Key << "edge_ids_by_prop_pos" << Value << Flow
			<< s.edge_id_by_prop_pos_;
	ye << EndMap;
	return ye;
}

std::ostream& operator<<(std::ostream& os, const SectorGraph& sg) {
	YAML::Emitter ye;
	ye << sg;
	os << ye.c_str();
	return os;
}

SectorGL::SectorGL(const YAML::Node& node) {
	sectorgraph_ = SectorGraph(node["sectorgraph"]);
	const IntegralFamily* ic = sector().integralfamily();
	CanonicalLabel l;
	l.read(node["label"], ic->kinematics()->kinematic_invariants());
	CanonicalRelabeling r;
	node["relabeling"] >> r;
	label_pair_ = make_pair(l, r);
}

YAML::Emitter& operator<<(YAML::Emitter& ye, const SectorGL& s) {
	using namespace YAML;
	ye << BeginMap;
	ye << Key << "sectorgraph" << Value << s.sectorgraph_;
	ye << Key << "label" << Value << s.label_pair_.first;
	ye << Key << "relabeling" << Value << s.label_pair_.second;
	ye << EndMap;
	return ye;
}

std::ostream& operator<<(std::ostream& os, const SectorGL& sg) {
	YAML::Emitter ye;
	ye << sg;
	os << ye.c_str();
	return os;
}


// global functions

void contract_internal_edges(const Graph& g,
		std::map<CanonicalLabel, Graph>& contracted) {
	const list<Edge> edges = g.internal_edges();
	for (list<Edge>::const_iterator e = edges.begin(); e != edges.end(); ++e) {
		if (e->is_self_loop())
			continue;
		Graph sub(g);
		sub.contract_edge(e->id);
		sub.disconnect_vacuum_components();
		CanonicalLabel l = sub.find_canonical_label().first;
		if (contracted.find(l) == contracted.end())
			contracted[l] = sub;
		VERIFY(sub.num_internal_edges() + 1 == edges.size());
	}
}

void contract_internal_edges(const std::map<CanonicalLabel, Graph>& g,
		std::map<CanonicalLabel, Graph>& contracted) {
	for (map<CanonicalLabel, Graph>::const_iterator l = g.begin(); l != g.end();
			++l)
		contract_internal_edges(l->second, contracted);
}

} // namespace Reduze

