/*  integralfamily.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef INTEGRALFAMILY_H_
#define INTEGRALFAMILY_H_

#include <set>
#include <ginac/ginac.h>
#include "yamlconfigurable.h"
#include "propagator.h"
#include "functions.h"
#include "crossing.h"

namespace Reduze {

class INT;
class Identity;
class IdentityList;
class Sector;
class Kinematics;

class IntegralFamily: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("integralfamily");
		s.set_short_description("A family of integrals.");
		s.set_long_description(""//
					"A family of integrals. For given kinematics and number"
					" of loop momenta, it defines a list of propagators"
					" allowing to index Feynman integrals.");
		s.add_option("name", true, "string", ""//
					"Name of the integral family. It must consist of"
					" ordinary letters and digits only, i.e. [a-zA-Z0-9],"
					" no whitespaces.");
		s.add_option("loop_momenta", true, "sequence of strings", ""//
					"Symbols for the loop momenta.");
		s.add_option("propagators", false, "sequence of propagators", ""//
					"The list of propagators defining the integral family."
					" This list must be complete and unique in the sense, that"
					" all scalar products involving the loop momenta are"
					" uniquely expressable as a linear combination of inverse "
					" propagators. Besides propagators which occur in actual"
					" Feynman diagrams it is typically necessary to define"
					" additional auxiliary propagators to complete the set.");
		s.add_option("cut_propagators", false, "sequence of positive integers",//
					"Propagators to be treated as on-shell cut propagators,"
				    " specified by their position in 'propagators'. Sectors "
				    " where any of the cut propagators is missing"
				    " are set to zero. In presence of cut propagators, it "
				    " might be useful to run the job setup_sector_mappings_alt "
				    " to make sure all sector relations and sector symmetries "
				    " are found.");
		s.add_option("permutation_symmetries", false, ""//
					"sequence of sequence of 2-element sequences", ""//
					"Permutations of propagators which leave any scalar"
					" Feynman integral of the family invariant."
					" Specifying such permutation symmetries may save a lot"
					" of computation time for the reductions, such that it can"
					" be very useful to choose appropriate auxiliary propagators"
					" and define such permutation symmetries."
					" Each element"
					" in the sequence defines such a permutation symmetry."
					" Each permutation symmetry is a sequence of 2-element"
					" sequences, which consist of the two integers specifying"
					" two propagators to be permuted (according to their"
					" position in 'propagators', first propagator has index 1)."
					" For two permutation"
					" symmetries, their product is considered to be another"
					" valid permutation symmetry. Those product symmetries"
					" will be supplemented automatically.");
		s.add_option("kinematics", false, "string", ""//
					"Name of the kinematics to be used. Must only be declared"
					" if more than one kinematics are defined");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	struct PropagatorType {
		enum {
			standard, standard_cut, bilinear, bilinear_cut
		};
	};

	/// creates the tree level integral family with empty name
	/** there are no propagators in the tree family, the only integral
	 ** in this family is defined to be 1 and lies in sector 0
	 ** (in contrast, the zero sector of any n-loop family for n >= 1 is 0) */
	IntegralFamily(const Kinematics*);

	/// creates the IntegralFamily of a product of two integral families
	/** note: should only be used by Files::integralfamilies(ic1, ic2)
	 ** for proper handling of CrossedIntegralFamily and special treatment
	 ** of n-loop times tree families */
	//IntegralFamily(const IntegralFamily* ic1, const IntegralFamily* ic2);

	/// creates IntegralFamily from scratch
	IntegralFamily(const Kinematics* kin, const std::string& name,
			const GiNaC::lst& loop_momenta,
			const std::vector<Propagator>& propagators,
			int cut_propagators_mask,
			const std::list<Permutation>* perms = 0);

	/// create derived IntegralFamily using suffix rules
    static IntegralFamily* create_derived_integralfamily(const std::string& name);

    virtual ~IntegralFamily();

	virtual void read(const YAML::Node&);
	virtual void print(YAML::Emitter&) const;

	const Kinematics* kinematics() const;

	virtual GiNaC::lst get_all_momenta() const;

	const GiNaC::lst& loop_momenta() const {
		return loop_momenta_;
	}

	const std::string& name() const {
		return name_;
	}

	const std::vector<Propagator>& propagators() const {
		return propagators_;
	}

	size_t num_propagators() const {
		return propagators_.size();
	}

	int max_sector_id() const {
		return (1 << num_propagators()) - 1; // 2^num_propagators - 1
	}

	/// symbols a1, ... , an for general exponents of propagators
	const GiNaC::exvector& propagator_exponents() const {
		return propagator_exponents_;
	}
	/// symbols a1, ... , an for general exponents of propagators
	const GiNaC::lst& propagator_exponent_symbols() const {
		return propagator_exponent_symbols_;
	}

	const GiNaC::exmap& rules_sp_to_prop() const {
		return rules_sp_to_prop_;
	}

	/// identifier for standard, bilinear and cut propagators
	int propagator_type(int i) const;

	/// returns the id of the sector with exactly all cut propagators
	int cut_propagators_mask() const {
		return cut_propagators_mask_;
	}

	/// returns whether propagator with index i is a cut propagator
    bool is_cut_propagator(int i /* first propagator == 0 */) const {
      return ((1 << i) & cut_propagators_mask_) != 0;
    }

	/// returns all subsectors with one propagator less
	std::set<int> get_immediate_subsectors(int sector_id) const;

	/// returns all supersectors with one propagator less
	std::set<int> get_immediate_supersectors(int sector_id) const;

	/// returns subsector equivalents
	/** sector relations are not taken into account */
	std::set<int> get_subsector_equivalents(int sector_id, bool recurse) const;

	int get_t(int sector_id) const;

	unsigned num_scalar_products() const;
	static unsigned num_scalar_products(int num_loop_mom, int num_ext_mom);

	/// returns permutation symmetries (all group elements except for identity)
	/** numbering of the propagators starts with 0, user input starts with 1 */
	virtual const std::list<Permutation>& permutation_symmetries() const {
		return permutation_symmetries_;
	}
	/// access to the permutation symmetries given as user-input
	const std::list<Permutation>& incomplete_permutation_symmetries() const {
		return incomplete_permutation_symmetries_;
	}

	int get_equivalent(int sector_id) const;

	/// returns all sectors which are zero anyway
	std::set<int> get_trivial_zero_sector_equivalents() const;

	/// tries to integrate an integrand and match the integral to this family
	/** integrand must be a power product of our Propagators and constants,
	 ** returns true on success */
	bool match_integrand(const GiNaC::ex& integrand, INT& integral,
			GiNaC::ex& coeff) const;

	bool operator<(const IntegralFamily& other) const {
		if (id_ != other.id_)
			return id_ < other.id_;
		return name_ < other.name_;
	}

	bool operator==(const IntegralFamily& other) const {
		return !((*this) < other || other < (*this));
	}

	bool operator!=(const IntegralFamily& other) const {
		return !((*this) == other);
	}

	void set_id(int id);
	void set_name(const std::string& name);

	int id() const {
		return id_;
	}

	/// whether this family is derived from another family by a crossing
	virtual bool is_crossed() const {
		return false;
	}
	/// source integral family from which this is derived, this if not derived
	// for crossed families the source is the uncrossed family
	// for uncrossed dimension shifted families the source is the unshifted one
	virtual const IntegralFamily* source_integralfamily() const {
		return source_;
	}

	/// prints a nice string of the sectors listed according to their t
	std::string
	t_tree_nice_string(const std::set<int>& sector_ids) const;

	/// test whether the permutation symmetry is valid; if not valid error message is returned
	std::pair<bool, std::string> is_valid_perm_sym(const Permutation& p,
			bool no_perm_sym_shift_verification = false) const;

	void print_info() const;

private:
	/// initializes internal structures after primary members have been set
	void init(bool verbose = true);
	/// checks the propagators: their number, if they contain loop momenta, etc.
	void check_propagators() const;
	bool have_non_standard_propagators() const;

	const IntegralFamily* source_;

	std::string name_;

	int id_;

	const Kinematics* kinematics_;

	/// the loop momenta
	GiNaC::lst loop_momenta_;

	/// all propagators
	std::vector<Propagator> propagators_;

	/// rules: PropInternal(0) -> PropList[0], PropInternal(1) -> PropList[1], ...
	//GiNaC::exmap PropInternal_to_Prop;

	/// rules: scalar products -> propagators, pi*pj on rhs expressed via kin. invariants
	GiNaC::exmap rules_sp_to_prop_;

	/// symbols a1, ... , an for general exponents of propagators
	GiNaC::exvector propagator_exponents_;
	/// symbols a1, ... , an for general exponents of propagators
	GiNaC::lst propagator_exponent_symbols_;

	/// ID of sector with all cut propagators and no others
	int cut_propagators_mask_;

	/// complete transformation rules that leave integrals invariant
	std::list<Permutation> permutation_symmetries_;
	/// incomplete transformation rules that leave integrals invariant (user-input)
	std::list<Permutation> incomplete_permutation_symmetries_;

private:
	/// private default constructor
	IntegralFamily();
	friend class YAMLProxy<IntegralFamily> ;
};

/// IntegralFamily obtained from a source family by applying a Crossing
/** Specifications for instances of this class are not intended to be written
 ** to disk or directly used for normal reductions. Instead, they serve as
 ** markers to translate reductions to crossed kinematics. */
class CrossedIntegralFamily: public IntegralFamily {
public:
	CrossedIntegralFamily(const IntegralFamily*, const Crossing&);
	virtual ~CrossedIntegralFamily();

	/// whether this family is derived from another family by a crossing
	virtual bool is_crossed() const {
		return true;
	}
	/// source integral family from which this is derived
	virtual const IntegralFamily* source_integralfamily() const {
		return source_integralfamily_;
	}
	/// Crossing by which this family can be obtained from source family
	const Crossing& crossing() const {
		return crossing_;
	}

	/// returns permutation symmetries of the source integral family
	virtual const std::list<Permutation>& permutation_symmetries() const {
		return source_integralfamily()->permutation_symmetries();
	}

private:
	/// source integral family from which this is derived
	const IntegralFamily* source_integralfamily_;

	/// rules by which this integral family can be obtained from source
	Crossing crossing_;
};

/// computes the next vector from the given on
/** transforms the vector 'vec' to the next vector for which the
 *  sum of all elements is also equal to 'sum'. 'min' is the minimal
 *  value that each entry of 'vec' must have ( 0 or 1 );
 *  the function works recursively: it starts increasing the element at
 *  position 'level' {0, ... ,vec.size() - 1} in 'vec' by one and
 *  calculates the entry on the first position, if the first entry
 *  has already reached 'min' the function calls itself at 'level-1'. */
void sequel(std::vector<int8> & vec, int8 sum, int8 min, int level);

/// computes all possible vectors in 'array' with the sum of the entries equal to 'sum'
/** applies void sequel( vec, sum, min, length-1 ) to get all possible
 *  vectors, all with sum 'sum';
 *  bool bu = true for propagators in the denominator ('min'=1);
 *  bool bu = false for propagators in the numerator ('min'=0);
 *  'lim' is the number of possible vectors, the number how often void sequel(...)
 *  will be called;
 *  'length' is the length of a vector */
void partition(std::vector<std::vector<int8> > & array, size_t lim, int length,
		int sum, bool bu);

} // namespace Reduze

#endif /* INTEGRALFAMILY_H_ */
