/*  job_findfamily.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_FINDFAMILY_H_
#define JOB_FINDFAMILY_H_

#include "job.h"
#include <ginac/ginac.h>
#include <vector>
#include <list>
#include "propagator.h"

namespace Reduze {

/// find a family
class FindFamily: public Reduze::Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("find_family");
		s.set_short_description("Finds integral family for propagators.");
		s.set_long_description("Job to determine an integral family for"
				" a given initial set of propagators. Auxiliary propagators"
				" are added such that the family has a maximal number of"
				" permutation symmetries. No graph algorithms are used."
				" This is _experimental_ code which might not work"
				" for arbitrary scenarios.");
		s.add_option("loop_momenta", true, "sequence of strings", ""//
					"Symbols for the loop momenta.");
		s.add_option("propagators", true, "sequence of propagators",
				"The list of propagators which should be completed to "
				"an integral family.");
		s.add_option("output_file", true, "string", "Output file name.");
		s.add_option("use_alt_trafo_finder", false, "boolean",
				"Whether to use an alternative strategy for construction"
				" of symmetry candidates. Enabling this option should"
				" improve the performance in cases where most propagators"
				" are specified already.");
		s.add_option("force_natural_momenta", false, "boolean",
				"Whether to require prefactors 1 or -1 for loop momenta in the"
				" propagators. This is reasonable since for standard loop "
				" momentum assignments in graphs there won't be propagators"
				" with momentum 2*k if k is a loop momentum.");
		s.add_option("force_no_external_momenta", false, "boolean",
				"Whether to require auxiliary propagators to be independent"
				" of external momenta.");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}
	FindFamily() : use_alt_trafo_finder_(false), force_natural_momenta_(true) {
		add_auto_options();
	}
	virtual ~FindFamily() {
	}
	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&, //
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);
	virtual std::string get_description() const;
protected:
	virtual void add_auto_options() {
		add_auto_io("output_file", output_file_);
		add_auto_io("use_alt_trafo_finder", use_alt_trafo_finder_);
		add_auto_io("force_natural_momenta", force_natural_momenta_);
		add_auto_io("force_no_external_momenta", force_no_external_momenta_);
	}
	virtual void read_manual_options(const YAML::Node&);
	virtual void print_manual_options(YAML::Emitter&) const;
	virtual void init();
private:
	std::vector<Propagator> input_propagators_;
	GiNaC::lst loop_momenta_;
	std::string output_file_;
	bool use_alt_trafo_finder_, force_natural_momenta_, force_no_external_momenta_;
};

}

#endif /* JOB_FINDFAMILY_H_ */
