/*  undirectedgraph.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef REDUZE_UNDIRECTEDGRAPH_H_
#define REDUZE_UNDIRECTEDGRAPH_H_

#include <set>
#include <map>
#include <vector>
#include <list>
#include "edge.h"

namespace YAML {
class Node;
class Emitter;
}

namespace Reduze {

/// An adjacency matrix over colored_multi_edge with the implementation of graph isomorphism algorithms
class UndirectedGraph {

public:

	// construction, destruction
	UndirectedGraph();
	virtual ~UndirectedGraph();

	UndirectedGraph(const std::set<int>& nodes,
			const std::map<int, Edge>& edges,
			const std::list<std::map<int, int> >& edge_coloring);

	/// returns a pair: first: the canonical adjacency list, second: the corresponding permutation of the nodes
	/** The canonical label is the canonical adjacency list which corresponds to the minimal
	 ** adjacency list with respect to lexicographic ordering.
	 ** The permutation is the renaming of the original node numbers in the vector v as v_i -> i with 0 <= i < N
	 ** the returned vector is actually the inverse transformation i -> v_i
	 ** The algorithm determines the automorphism groups of the graph and chooses the minimal representation.
	 ** Parameter node_coloring:
	 **   a coloring number for each node, nodes with different color are not permuted
	 **   eg. for fixing external nodes completely. **/
	std::pair<std::vector<std::map<int, colored_multi_edge> >, std::vector<int> > find_canonical_label(
			const std::map<int, int>& node_coloring) const;

	/// returns the node symmetry group (permutations which leave the canonical label invariant)
	/** To every node one can assign a color. In the procedure of finding the symmetry permutations
	 ** the nodes with different color are suppressed of being permuted.
	 ** Parameter no_free_permutations:
	 **   A set of nodes can be given that are not allowed to be permuted freely,
	 **   but only when forced to by other nodes not in that set.
	 **   Used for external nodes which we want to permute only when an symmetry of internal nodes
	 **   needs a permutation of external nodes. Anyway, if some of these nodes are forced to
	 **   be permuted then the remaining ones are allowed to be permutetd freely.
	 **   This is useful in case a symmetry
	 **   involves an "not allowed" crossing which could be turned into an allowed one
	 **   if the remaining external edges are permuted freely. **/
	void find_node_symmetry_group(std::list<std::map<int, int> >& symmetries,
			const std::map<int, int>& node_coloring,
			const std::set<int>& no_free_permutations) const;

private:

	/// A generalized degree of the nodes which respects multiple edges and self-loops
	struct NodeSignature {
		NodeSignature(int num_colors);
		colored_multi_edge degree_;
		colored_multi_edge loops_;
		std::multiset<colored_multi_edge> multiplicities_;
		bool operator<(const NodeSignature& other) const;
		bool operator==(const NodeSignature& other) const;
		bool operator!=(const NodeSignature& other) const;
	};

	/// returns the generalized degree of 'node' considering only the neighbors which are in test_set
	NodeSignature node_signature(int node, const std::set<int>& test_set) const;

	/// finds the ordered partition of the 'nodes'. returns an iterator to the first cell with maximum size
	/** The nodes are sorted according to their NodeSignature (with respect to the nodes in test_cell)
	 ** and nodes with equal NodeSignature are put in the same cell **/
	std::list<std::set<int> >::iterator set_ordered_partition(
			std::list<std::set<int> >& partition, const std::set<int>& nodes,
			const std::set<int>& test_set) const;

	/// the canonical label with the node coloring in the internal representation
	std::pair<std::vector<std::map<int, colored_multi_edge> >, std::vector<int> > find_canonical_label(
			const std::vector<int>& node_coloring) const;

	/// the node symmetry group in internal representation
	void find_node_symmetry_group(std::list<std::vector<int> >& symmetries,
			const std::vector<int>& node_coloring,
			const std::set<int>& no_free_permutation) const;

	/// calculates all allowed permutations for the calculation of the canonical label
	/** The number of allowed permutations is a multiple of the number of permutations
	 ** that leave the canonical label, e.g. adjacency list, invariant (automorphism
	 ** group).
	 ** Graph isomorphism algorithms are taken from McKay
	 ** ref: PRACTICAL GRAPH ISOMORPHISM, Brendan D. McKay, algorithm 2.12
	 ** A similar implementation by M. Czakon can be found at
	 ** http://www-zeuthen.desy.de/theory/capp2005/
	 ** The vector {v_1, ... v_n} is interpreted as the permutation P: v_i -> (i-1)
	 ** for all integers i with 0 <= i < n **/
	void set_allowed_node_permutations(
			std::list<std::vector<int> >& allowed_permutations,
			const std::vector<int>& node_colors,
			const std::set<int>& no_free_permutations) const;

	/// refines a partition, returns true if it is the discrete partition
	/** There are two cases in which the 'partition' is turned into the unique
	 ** coarsest equitable partition finer than 'partition'
	 ** a) sequence == partition
	 ** b) there exist an equitable partition pi' coarser than 'partition'.
	 **    Choose 'sequence' as a subset of 'partition' such that for any cell
	 **    W of pi' we have X is subset of W for at most one cell X from
	 **    'partition' \ 'sequence'.
	 ** ref: PRACTICAL GRAPH ISOMORPHISM, Brendan D. McKay, algorithm 2.5
	 ** ref: Practical graph isomorphism II, Brendan D. McKay and Adolfo Piperno,
	 **	     http://arxiv.org/abs/1301.1493
	 **
	 ** A discrete partition like {{v_1},{v_2}, ... {v_n}} is interpreted as
	 ** the permutation P: v_i -> (i-1) for all integers i with 0 <= i < n */
	bool refine_partition(std::list<std::set<int> >& partition,
			const std::list<std::set<int> >& sequence) const;

	/// sets the partition of all nodes according to the node_coloring
	/// the node_coloring assigns to each node a color such that
	/// permutations of nodes with different colors will be suppressed
	void set_colored_node_partition(std::list<std::set<int> >& partition,
			const std::vector<int>& node_color,
			const std::set<int>& no_free_permute) const;

	/// compares lexicographically the adjacency matrices obtained by the two permutations
	/// return value r: r <,=,> 0  iff  M(v1) <,=,> M(v2)
	int compare_adjacency_matrix(const std::vector<int>& v1,
			const std::vector<int>& v2) const;
	/// compares lexicographically the adjacency lists obtained by the two permutations
	/// return value r: r <,=,> 0  iff  L(v1) <,=,> L(v2)
	int compare_adjacency_list(const std::vector<int>& v1,
			const std::vector<int>& v2) const;
	/// sets the adjacency list obtained from the adjacency matrix by a transformation 'v'
	void set_adjacency_list(
			std::vector<std::map<int, colored_multi_edge> >& aj_list,
			const std::vector<int>& v) const;

private:
	/// the symmetric adjacency matrix
	std::vector<std::vector<colored_multi_edge> > adjm_;
	/// for each node the original node number
	std::vector<int> original_nodes_;
	/// the number of edge colors
	int num_edge_colors_;
};

} // namespace Reduze

#endif /* REDUZE_UNDIRECTEDGRAPH_H_ */
