/*  yamlutils.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "yamlutils.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

void verify_is_valid_map_or_non_empty_sequence(const YAML::Node& n) {
	if (n.Type() == YAML::NodeType::Map
			|| (n.Type() == YAML::NodeType::Sequence && n.size() > 0))
		return;
	throw std::runtime_error("Expected a map or a non-empty sequence.");
}

// often used YAML operators

std::string position_info(const YAML::Node& n) {
	stringstream ss;
	ss << "at line " << n.GetMark().line + 1 << " ";
	ss << "column " << n.GetMark().column + 1;
	return ss.str();
}

void operator >>(const YAML::Node& node, GiNaC::symbol& s) {
	string str;
	node >> str;
	s = GiNaC::symbol(str);
}

void read(const YAML::Node& node, GiNaC::symbol& s, const GiNaC::lst& symbols) {
	if (node.Type() != YAML::NodeType::Scalar)
		throw runtime_error("node is not a scalar " + position_info(node));
	string str;
	node >> str;
	try {
		ex res(str, symbols);
		if (!is_a<symbol> (res))
			throw runtime_error("not a plain symbol " + position_info(node));
		s = ex_to<symbol> (res);
	} catch (exception& e) {
		stringstream ss;
		ss << "Failed to parse: " << str << " " << position_info(node) << "\n";
		ss << e.what() << "\n";
		ss << "Expected a plain symbol out of: " << symbols;
		throw runtime_error(ss.str());
	}
}

void read(const YAML::Node& node, GiNaC::ex& e, const GiNaC::lst& symbols) {
	if (node.Type() != YAML::NodeType::Scalar)
		throw runtime_error("node is not a scalar " + position_info(node));
	string str;
	node >> str;
	try {
		e = ex(str, symbols);
	} catch (exception& e) {
		stringstream ss;
		ss << "Failed to parse: " << str << " " << position_info(node) << "\n";
		ss << e.what() << "\n";
		ss << "Valid symbols usable in the expression: " << symbols;
		throw runtime_error(ss.str());
	}
}

YAML::Emitter& operator <<(YAML::Emitter& os, const ex& e) {
	ostringstream strm;
	strm << e;
	os << strm.str();
	return os;
}

YAML::Emitter& operator <<(YAML::Emitter& os, const lst& e) {
	lst::const_iterator it;
	os << YAML::BeginSeq;
	for (it = e.begin(); it != e.end(); ++it)
		os << *it;
	os << YAML::EndSeq;
	return os;
}

void read(const YAML::Node& node, GiNaC::exmap& m, const GiNaC::lst& symbols) {
	if (node.Type() != YAML::NodeType::Sequence)
		throw runtime_error("Node is not a sequence " + position_info(node));
	for (YAML::Iterator it = node.begin(); it != node.end(); ++it) {
		if (it->Type() != YAML::NodeType::Sequence || it->size() != 2)
			throw runtime_error(
					"node is not a two-element sequence "
							+ position_info(node));
		ex lhs, rhs;
		read((*it)[0], lhs, symbols);
		read((*it)[1], rhs, symbols);
		m[lhs] = rhs;
	}
}

template<class T>
void read(const YAML::Node& node, GiNaC::lst& l, const GiNaC::lst& symbols) {
	if (node.Type() != YAML::NodeType::Sequence)
		throw runtime_error("Node is not a sequence " + position_info(node));
	for (YAML::Iterator it = node.begin(); it != node.end(); ++it) {
		if (it->Type() != YAML::NodeType::Scalar)
			throw runtime_error("node is not a scalar " + position_info(node));
		T e;
		read(*it, e, symbols);
		l.append(e);
	}
}

void read(const YAML::Node& node, unsigned& n) {
	if (node.Type() != YAML::NodeType::Scalar)
		throw runtime_error("Node is not a scalar " + position_info(node));
	int ns;
	node >> ns;
	if (ns < 0)
		throw runtime_error(
				"Node does not contain an unsigned integer "
						+ position_info(node));
	n = ns;
}

void read(const YAML::Node& node, std::vector<int>& v) {
	using namespace YAML;
	if (node.Type() != NodeType::Sequence)
		throw runtime_error("Node is not a sequence " + position_info(node));
	vector<int> result;
	result.reserve(node.size());
	for (Iterator n = node.begin(); n != node.end(); ++n) {
		if (n->Type() != NodeType::Scalar)
			throw runtime_error("Node is not a scalar " + position_info(*n));
		int i;
		*n >> i;
		result.push_back(i);
	}
	v.swap(result);
}

void read(const YAML::Node& node, std::vector<std::vector<int> >& v) {
	using namespace YAML;
	vector<vector<int> > result;
	if (node.Type() != NodeType::Sequence)
		throw runtime_error("Node is not a sequence " + position_info(node));
	result.reserve(node.size());
	for (Iterator n = node.begin(); n != node.end(); ++n) {
		result.push_back(vector<int> ());
		read(*n, result.back());
	}
	v.swap(result);
}

YAML::Emitter& operator <<(YAML::Emitter& os, const exmap& m) {
	os << YAML::BeginSeq;
	for (exmap::const_iterator r = m.begin(); r != m.end(); ++r)
		os << YAML::BeginSeq << r->first << r->second << YAML::EndSeq;
	os << YAML::EndSeq;
	return os;
}

void operator >>(const YAML::Node& node, map<int, string>& l) {
	if (node.Type() != YAML::NodeType::Sequence)
		throw runtime_error("node is not a sequence " + position_info(node));
	l.clear();
	for (YAML::Iterator it = node.begin(); it != node.end(); ++it) {
		int length = l.size();
		string value;
		*it >> value;
		l.insert(make_pair(length, value));
	}
}
/*
 void operator >>(const YAML::Node& node, set<string>& l) {
 if (node.Type() != YAML::NodeType::Sequence)
 throw runtime_error("Node is not a sequence " + position_info(node));
 l.clear();
 for (YAML::Iterator it = node.begin(); it != node.end(); ++it) {
 string s;
 *it >> s;
 l.insert(s);
 }
 }
 */
/*
 YAML::Emitter& operator <<(YAML::Emitter& os, const set<string>& l) {
 set<string>::const_iterator it;
 os << YAML::BeginSeq;
 for (it = l.begin(); it != l.end(); ++it)
 os << *it;
 os << YAML::EndSeq;
 return os;
 }
 */

template
void read<GiNaC::symbol> (const YAML::Node& node, GiNaC::lst& l,
		const GiNaC::lst& symbols);
template
void read<GiNaC::ex> (const YAML::Node& node, GiNaC::lst& l,
		const GiNaC::lst& symbols);

} // namespace Reduze
