(*  reduze-mma.m
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 *)


(* this file contains some snippets which might be convenient to
 * interface Mathematica with Reduze 2 *)


(*********************)
(* usage information *)
(*********************)

DecoratedINT::usage = "\
DecoratedINT[fam, props] returns INT[fam, t, id, r, s, props] where \
't' is the number of positive propagator exponents in 'props', \
'id' is the binary identification number of the sector, \
'r' is the sum of positive exponents and \
's' is the sum of negative exponents.";

CornerExponents::usage = "\
CornerExponents[id, ntot] gives the propagator exponents for the corner \
integral of sector 'id', where 'ntot' is the total number of propagators \
in the integral family.";

DistinguishMultiEdges::usage = "\
Reduze 2 exports graphs with a custom header MultiGraph, since they \
may contain multiedges which are not supported by Graph in Mathematica 9. \
DistinguishMultiEdges[MultiGraph[]] introduces 2-point vertices to turn a \
MultiGraph into a Graph. If you aim at an undirected graph, please use \
ToUndirected first.";



(******************************)
(* integral related functions *)
(******************************)

rofprops[p_List] := Plus @@ Select[p, Positive];
sofprops[p_List] := Plus @@ Select[-p, Positive];
tofprops[p_List] := Length@Select[p, Positive];
idofprops[p_List] := 
  Plus @@ Table[If[Positive[p[[i]]], 2^(i - 1), 0], {i, Length[p]}];

DecoratedINT[fam_String, p_List] :=
 INT[fam, tofprops[p], idofprops[p], rofprops[p], sofprops[p], p];

CornerExponents[id_Integer, ntot_Integer] := PadRight[Reverse@IntegerDigits[id, 2], ntot];



(***************************)
(* graph related functions *)
(***************************)

Vertices[MultiGraph[l_List]] := Union@Flatten[List @@@ First /@ l];

ToUndirected[MultiGraph[l_List]] := 
 MultiGraph[
  Cases[l, Labeled[a_ -> b_, s_] :> 
    Labeled[If[OrderedQ[{a, b}], a <-> b, b <-> a], s]]];

DistinguishMultiEdges[MultiGraph[l_List]] := 
 Module[{v = Abs@Last@Vertices@MultiGraph@l},
  Graph[Flatten[Table[If[Length[el] === 1, el,
      Flatten[
       Table[{ReplacePart[e, {1, 2} -> ++v], 
         ReplacePart[e, {{1, 1} -> v, 2 -> ""}]} , {e, el}], 1]],
     {el, 
      Split[Sort@l, #1[[1]] === #2[[1]] || #1[[1]] === 
          Reverse@#2[[1]] &]}], 1]]
  ];



Print["Loaded Reduze helpers. Please see usage information for:"];
Print["DecoratedINT, CornerExponents, DistinguishMultiEdges"];
