/*
 * main.cpp
 *
 *  Created on: Nov 27, 2011
 *      Author: cedric
 */

#include <iostream>
#include <sstream>
#include <fstream>
#include <vector>
#include <stdlib.h>

using namespace std;

/// convert an Reduze 1 integral string to a Reduze 2 string, prepend integral family name
string convert_integral(const string& str, const string& intfam) {
	istringstream strm(str);
	int i;
	vector<int> v;
	v.reserve(str.size() / 2 + 1);

	while (strm >> i)
		v.push_back(i);

	int t, id, r, s;
	t = r = s = id = 0;
	for (size_t i = 0; i < v.size(); ++i) {
		if (v[i] > 0) {
			id += (1 << i);
			++t;
			r += v[i];
		} else {
			s -= v[i];
		}
	}
	ostringstream out;
	out << intfam << " " << t << " " << id << "  " << r << " " << s << "  ";

	return out.str() + str;
}

void usage() {
	cout << "reduze1to2: convert reduction results"
			<< " from Reduze 1 format to Reduze 2 format." << endl;
	cout << "Usage:" << endl;
	cout << "first argument:  input file" << endl;
	cout << "second argument: output file" << endl;
	cout << "third argument:  integral family name" << endl;
	cout << endl;
}

int main(int argc, char* argv[]) {

	if (argc == 2 && (string(argv[1]) == "-h" || string(argv[1]) == "--help")) {
		usage();
		return 0;
	}

	if (argc != 4) {
		cerr << "wrong number of arguments:" << endl << endl;
		usage();
		abort();
	}

	string input(argv[1]);
	string output(argv[2]);
	string tmpoutput = output + ".tmp";
	string intfam(argv[3]);

	if (input == output) {
		cerr << "input and output file name equal" << endl;
		abort();
	}

	ifstream in(input.c_str());
	if (!in.good()) {
		cerr << "Error opening file: " << input << endl;
		in.close();
		abort();
	}

	ofstream out(output.c_str());
	if (!out.good()) {
		cerr << "Error opening file: " << output << endl;
		out.close();
		abort();
	}
	out.close();
	out.open(tmpoutput.c_str());
	if (!out.good()) {
		cerr << "Error opening file: " << tmpoutput << endl;
		out.close();
		abort();
	}

	cout << "converting reduction results from file " << input << endl;

	string str;
	bool first = true;
	while (getline(in, str)) {
		if (str.empty())
			continue;
		if (str != ";") {
			if (!first)
				out << "  ";
			out << "  " << convert_integral(str, intfam) << "\n";
			getline(in, str);
			if (str.empty()) {
				cerr << "empty line after integral detected" << endl;
				abort();
			}
			if (!first)
				out << "  ";
			out << "   " << str << "\n";
			first = false;
		} else {
			out << ";\n";
			first = true;
		}
	}

	in.close();
	out.close();

	if (rename(tmpoutput.c_str(), output.c_str()) != 0) {
		cerr << "error renaming file " << tmpoutput << " to " << output << endl;
		abort();
	}

	cout << "generated file " << output << endl;

	return 0;
}
