//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// G4Polyhedra
//
// Class description:
//
//   Class implementing a CSG-like type "PGON" Geant 3.21 volume,
//   inherited from class G4VCSGfaceted:
//
//   G4Polyhedra( const G4String& name,
//                G4double phiStart,         - initial phi starting angle
//                G4double phiTotal,         - total phi angle
//                G4int numSide,             - number sides
//                G4int numZPlanes,          - number of z planes
//                const G4double zPlane[],   - position of z planes
//                const G4double rInner[],   - tangent distance to inner surface
//                const G4double rOuter[]  ) - tangent distance to outer surface
//
//   G4Polyhedra( const G4String& name,
//                G4double phiStart,    - initial phi starting angle
//                G4double phiTotal,    - total phi angle
//                G4int    numSide,     - number sides
//                G4int    numRZ,       - number corners in r,z space
//                const G4double r[],   - r coordinate of these corners
//                const G4double z[] )  - z coordinate of these corners

// Author: David C. Williams (davidw@scipp.ucsc.edu)
// --------------------------------------------------------------------
#ifndef G4POLYHEDRA_HH
#define G4POLYHEDRA_HH

#include "G4GeomTypes.hh"

#if defined(G4GEOM_USE_USOLIDS)
#define G4GEOM_USE_UPOLYHEDRA 1
#endif

#if defined(G4GEOM_USE_UPOLYHEDRA)
  #define G4UPolyhedra G4Polyhedra
  #include "G4UPolyhedra.hh"
#else

#include "G4VCSGfaceted.hh"
#include "G4PolyhedraSide.hh"
#include "G4PolyhedraHistorical.hh"
#include "G4Polyhedron.hh"

class G4EnclosingCylinder;
class G4ReduciblePolygon;

class G4Polyhedra : public G4VCSGfaceted
{
  public:

    G4Polyhedra(const G4String& name,
                      G4double phiStart,   // initial phi starting angle
                      G4double phiTotal,   // total phi angle
                      G4int numSide,       // number sides
                      G4int numZPlanes,    // number of z planes
                const G4double zPlane[],   // position of z planes
                const G4double rInner[],   // tangent distance to inner surface
                const G4double rOuter[] ); // tangent distance to outer surface

    G4Polyhedra(const G4String& name,
                      G4double phiStart,   // initial phi starting angle
                      G4double phiTotal,   // total phi angle
                      G4int    numSide,    // number sides
                      G4int    numRZ,      // number corners in r,z space
                const G4double r[],        // r coordinate of these corners
                const G4double z[] );      // z coordinate of these corners

    ~G4Polyhedra() override;

    EInside Inside( const G4ThreeVector& p ) const override;
    G4double DistanceToIn( const G4ThreeVector& p,
                           const G4ThreeVector& v ) const override;
    G4double DistanceToIn( const G4ThreeVector& p ) const override;

    void BoundingLimits(G4ThreeVector& pMin, G4ThreeVector& pMax) const override;
    G4bool CalculateExtent(const EAxis pAxis,
                           const G4VoxelLimits& pVoxelLimit,
                           const G4AffineTransform& pTransform,
                                 G4double& pmin, G4double& pmax) const override;

    void ComputeDimensions(       G4VPVParameterisation* p,
                            const G4int n,
                            const G4VPhysicalVolume* pRep) override;

    G4GeometryType  GetEntityType() const override;

    G4bool IsFaceted () const override;

    G4VSolid* Clone() const override;

    G4double GetCubicVolume() override;
    G4double GetSurfaceArea() override;

    G4ThreeVector GetPointOnSurface() const override;

    std::ostream& StreamInfo( std::ostream& os ) const override;

    G4Polyhedron* CreatePolyhedron() const override;

    G4bool Reset();

    // Accessors

    inline G4int GetNumSide()        const;
    inline G4double GetStartPhi()    const;
    inline G4double GetEndPhi()      const;
    inline G4double GetSinStartPhi() const;
    inline G4double GetCosStartPhi() const;
    inline G4double GetSinEndPhi()   const;
    inline G4double GetCosEndPhi()   const;
    inline G4bool IsOpen()           const;
    inline G4bool IsGeneric()        const;
    inline G4int GetNumRZCorner()    const;
    inline G4PolyhedraSideRZ GetCorner( const G4int index ) const;

    inline G4PolyhedraHistorical* GetOriginalParameters() const;
      // Returns internal scaled parameters.
    inline void SetOriginalParameters(G4PolyhedraHistorical* pars);
      // Sets internal parameters. Parameters 'Rmin' and 'Rmax' in input must
      // be scaled first by a factor computed as 'cos(0.5*phiTotal/theNumSide)',
      // if not already scaled.

    G4Polyhedra(__void__&);
      // Fake default constructor for usage restricted to direct object
      // persistency for clients requiring preallocation of memory for
      // persistifiable objects.

    G4Polyhedra( const G4Polyhedra& source );
    G4Polyhedra& operator=( const G4Polyhedra& source );
      // Copy constructor and assignment operator.

  protected:

    void SetOriginalParameters(G4ReduciblePolygon* rz);
      // Sets internal parameters for the generic constructor.

    void Create( G4double phiStart,           // initial phi starting angle
                 G4double phiTotal,           // total phi angle
                 G4int    numSide,            // number sides
                 G4ReduciblePolygon* rz );    // rz coordinates
      // Generates the shape and is called by each constructor, after the
      // conversion of the arguments

    void CopyStuff( const G4Polyhedra& source );
    void DeleteStuff();

    // Methods for generation of random points on surface

    void SetSurfaceElements() const;

  protected:

    G4int numSide = 0;    // Number of sides
    G4double startPhi;    // Starting phi value (0 < phiStart < 2pi)
    G4double endPhi;      // end phi value (0 < endPhi-phiStart < 2pi)
    G4bool phiIsOpen = false;   // true if there is a phi segment
    G4bool genericPgon = false; // true if created through 2nd generic ctor
    G4int numCorner = 0;  // number RZ points
    G4PolyhedraSideRZ* corners = nullptr;  // our corners
    G4PolyhedraHistorical* original_parameters = nullptr; // original input pars

    G4EnclosingCylinder* enclosingCylinder = nullptr;

    struct surface_element { G4double area=0.; G4int i0=0, i1=0, i2=0; };
    mutable std::vector<surface_element>* fElements = nullptr;
};

#include "G4Polyhedra.icc"

#endif

#endif
